/********************************************************************************************************/
/*	Copyright Intel Corporation, 1992								*/
/*	Peter Hazen, Intel Corporation, June 4, 1992, Revision 0.0					*/
/*													*/
/*	Revision History:										*/
/*	Revision 0.0	Adapted from "28F008SA Software Drivers", AP-360				*/
/*													*/
/*	The following drivers control the Command and Status Registers of the listed memories 		*/
/*	in x16 mode (BYTE# is high) to drive word write, block erase, Status Register 			*/
/*	read and clear and array read algorithms. This code can be easily modified to access the 28Fx00 products 	*/
/*	in x8 mode (BYTE# is low) as commented throughout.

/*	Applicable products: 	28F800BV-T/B, 28F800CV-T/B, 28F800CE-T/B				*/
/*				28F400BV-T/B, 28F400CV-T/B, 28F400CE-T/B				*/
/*				28F400BX-T/B, 28F400BL-T/B						*/
/*				28F200BV-T/B, 28F200CV-T/B, 28F200CE-T/B				*/
/*				28F400BX-T/B, 28F200BL-T/B						*/	

/*         												*/
/* 	Sample Vpp and RP# control blocks are also included as are example programs combining drivers into full	*/
/*	 algorithms. The functions listed below are included:						*/
/*		erasbgn(): Begins block erasure								*/
/*		erassusp(): Suspends block erase to allow reading data from a block of the flash memory other 		*/
/*			than that being erased								*/
/*		erasres(): Resumes block erase if suspended						*/
/*		end(): Polls the Write State Machine to determine if block erase or word write have	*/
/*			completed									*/
/*		eraschk(): Executes full status check after block erase completion			*/
/*		writebgn(): Begins word write								*/
/*		writechk(): Executes full status check after word write completion			*/
/*		idread(): Reads and returns the manufacturer and device IDs of the target flash memory	*/
/*		statrd(): Reads and returns the contents of the Status Register				*/
/*				statclr(): Clears the Status Register					*/
/*		rdmode (): Puts the flash memory in Read Array mode					*/
/*		rdword (): Reads and returns a specified word from the target flash memory		*/
/*		vppup(): Enables high voltage Vpph							*/
/*		vppdown(): Disables Vpph								*/
/*		rpen(): Enables active low signal RP#							*/
/*		rpdis(): Disables active low signal RP#							*/
/*													*/
/*	Addresses are transferred to functions as pointers to far words (ie long integers).  An alternate		*/
/*		approach is to create a global array the size of the flash memory and located "over" the		*/
/*		flash memory in the system memory map.  Accessing specific locations of the flash memory is then	*/
/*		accomplished by passing the chosen function an offset from the array base versus a			*/
/*		specific address.  Different microprocessor architectures will require different array			*/
/*		definitions; ie for the x86 architecture, define it as "word twomeg[4][10000]"  for the 28F200, 	*/
/*		"word fourmeg[8][10000]" for the 28F400, and pass each function TWO offsets to access a specific 	*/
/*		location.  MCS-96 architectures are limited to "word twomeg[10000]" or "word fourmeg[10000]".  		*/
/*		Alternate approaches such as using  port pins for paging will be required to access the full flash array*/
/*															*/
/*	To create a far pointer, a function such as MK_FP() can be used, given a segment and offset in			*/
/*		the x86 architecture.  See your compiler reference manual for additional				*/
/*		information.												*/
/**********************************************************************************************************/

typedef	unsigned int word;			

/********************************************************************************************************/
/*	Function: Main											*/
/*	Description: The following code (commented out) shows examples of word write and block erase 	*/
/*		     algorithms that can be modified to fit the specific application and hardware design*/
/**********************************************************************************************************/
main()
{
/*	word far *address;										*/
/*	word data,status;										*/
/*													*/
/*	The following code gives an example of a possible word write algorithm.				*/
/*	Note that Vpp does not need to be cycled between word writes when a string of word writes occurs.  		*/
/*	Ramp Vpp to program voltage before the first word write and leave at VPPH (Program/Erase Voltage)  	*/
/*	until after completion of the last word write.							*/
/*	Doing so minimizes Vpp ramp up-down delay and maximizes word write throughput			*/
/*	Note also that the device ID does not need to be checked between each word write, however it	*/
/*	should be done before the first word write, to ensure the part is not in deep power-down mode.	*/				
/*	vppup();											*/
/*	"INSERT SOFTWARE DELAY FOR VPP RAMP IF REQUIRED	"						*/
/*	rpdis();											*/
/*	"INSERT CODE TO SET RP# TO 12V or WP# to logic high IF BOOT BLOCK WRITES ARE DESIRED"  	 	*/
/*	The boot block can be "un-locked" by writing a routine which is passed the address to be written, and sets RP# 	*/
/*	to 12V or WP# to logic high for addresses within the boot block address range.						*/
/*	address	= 0XxxxxxL;									*/
/*	data	= 0Xyyyy;									*/
/*	if (writebgn(data,address) == 1)							*/
/*		"RECOVERY CODE-POWER NOT APPLIED (ID CHECK FAIL)"				*/
/*	else											*/
/*	{											*/
/*		while (end(&status) )								*/
/*			;									*/
/*		switch (writechk(status))							*/
/*		{										*/
/*			case 0:									*/
/*				break;								*/
/*			case 1:									*/
/*				"RECOVERY CODE-VPP LOW DETECT ERROR"				*/
/*				break;								*/
/*			case 2:									*/
/*				"RECOVERY CODE-WORD WRITE ERROR"				*/
/*				break;								*/
/*		}										*/
/*		statclr();									*/
/*	}											*/
/*	vppdown();										*/
/*												*/
/*	"INSERT CODE TO RE-LOCK THE BOOT BLOCK IF A BOOT BLOCK WRITE OCCURRED."			*/
/*	After a write operation, the boot should be locked (if previously unlocked) by taking RP# from 12V 		*/
/*	(un-lock voltage) to 5V or WP# to logic low.  						*/
/*	However, the  boot block does not need to be locked inbetween writes during a string of	*/
/*	consecutive writes to the boot block.							*/
/*												*/
/*												*/
/*	The following code gives an example of a possible block erase algorithm.		*/
/*	Note that Vpp does not need to be cycled between block erases when a string of block erases occurs.  Ramp Vpp 	*/
/*	to VPPH (Program/Erase Voltage) before the first block erase and leave at VPPH 		*/
/*	until after completion of the last block erase.  Doing so minimizes
/*	Vpp ramp up-down delay and maximizes block erase throughput				*/
/*	Note also that the device ID does not need to be checked between each word write, however it			*/
/*	should be done before the first word write, to ensure the part is not in deep power-down mode.			*/				
/*	vppup();										*/
/*	"INSERT SOFTWARE DELAY FOR VPP RAMP IF REQUIRED	"					*/
/*	rpdis();										*/
/*	"INSERT CODE TO SET RP# TO 12V or WP# to logic high IF BOOT BLOCK ERASE IS DESIRED"  	 			*/
/*	The boot block can be "un-locked" by writing a routine which is passed the address to be erased, and sets RP# 	*/
/*	to 12V or WP# to logic high for addresses within the boot block address range.					*/
/*	address	= 0XxxxxxL;									*/
/*	if (erasbgn(address) == 1)								*/
/*		"RECOVERY CODE-POWER NOT APPLIED (ID CHECK FAIL)"				*/
/*	else											*/
/*	{											*/
/*		while (end(&status) )								*/
/*			;									*/
/*		switch (eraschk(status))							*/
/*		{										*/
/*			case 0:									*/
/*				break;								*/
/*			case 1:									*/
/*				"RECOVERY CODE-VPP LOW DETECT ERROR"				*/
/*				break;								*/
/*			case 2:									*/
/*				"RECOVERY CODE-BLOCK ERASE ERROR"				*/
/*				break;								*/
/*			case 3:									*/
/*				"RECOVERY CODE-ERASE SEQUENCE ERROR"				*/
/*				break;								*/
/*		}										*/
/*		statclr();									*/
/*	}											*/
/*	vppdown();										*/
/*												*/
}

/********************************************************************************************************/
/*	Function: Erasgbn										*/
/*	Description: Begins erase of a block.								*/
/*	Inputs:	blckaddr: System address within the block to be erased					*/
/*	Outputs:	None										*/
/*	Returns:	0 = Block erase successfully initiated						*/
/*			1 = Block erase not initiated (ID check error)					*/
/*	Device Read Mode on Return: Status Register (ID if returns 1)					*/
/**********************************************************************************************************/

#define	ERASETUP	0X0020		/* Erase Setup command						*/
#define	ERASCONF	0X00D0		/* Erase Confirm command					*/

int erasbgn(blckaddr)

word far *blckaddr;				/* blckaddr is an address within the block to be erased	*/

{
	word mfgrid,deviceid;

	if (idread(&mfgrid,&deviceid)==1)	/* ID read error; device not powered up?		*/
		return (1);
	*blckaddr	= ERASETUP;		/* Write Erase Setup command to block address		*/
	*blckaddr	= ERASCONF;		/* Write Erase Confirm command to block address		*/
	return (0);
}

/********************************************************************************************************/
/*	Function: Erassusp										*/
/*	Description: Suspends block erase to read from another block					*/
/*	Inputs:		None										*/
/*	Outputs:	None										*/
/*	Returns:	0 = Block erase suspended							*/
/*			1 = Error; Write State Machine not busy (block erase suspend not possible)	*/
/*	Device Read Mode on Return: Status Register							*/
/********************************************************************************************************/

#define	RDYMASK	0X0080			/* Mask to isolate the WSM Status bit of the Status Register		*/
#define	WSMRDY	0X0080			/* Status Register value after masking, signifying that the WSM is	*/
					/*	no longer busy							*/
#define	SUSPMASK	0X0040		/* Mask to isolate the erase suspend status bit of the Status Register	*/
#define	ESUSPYES	0X0040		/* Status Register value after masking, signifying that block erase has	*/
					/*	been suspended							*/
#define	STATREAD	0X0070		/* Read Status Register command						*/
#define	SYSADDR	0			/* This constant can be initialized to any address within the memory map*/
					/* 	of the target  and is alterable depending on the		*/
					/*	system architecture						*/
#define	SUSPCMD	0X00B0			/* Erase Suspend command						*/

int erassusp()

{
	word far *stataddr;			/* Pointer variable used to write commands to device			*/

	stataddr	= (word far *)SYSADDR;
	*stataddr	= SUSPCMD;		/* Write Erase Suspend command to the device				*/
	*stataddr	= STATREAD;		/* Write Read Status Register command..necessary in case erase is	*/
						/*	already completed						*/
	while ((*stataddr & RDYMASK) != WSMRDY)
		;				/* Will remain in while loop until bit 7 of the Status Register goes to 1,	*/
						/*	signifying that the WSM is no longer busy			*/
	if ((*stataddr & SUSPMASK) == ESUSPYES)
		return (0);			/* Erase is suspended...return code "0"					*/
	return(1);				/* Erase has already completed; suspend not possible.  Error code "1"	*/
}

/**********************************************************************************************************/
/*	Function: Erasres										*/
/*	Description: Resumes block erase previously suspended						*/
/*	Inputs:		None										*/
/*	Outputs:	None										*/
/*	Returns:	0 = Block erase resumed								*/
/*			1 = Error; Block erase not suspended when function called				*/
/*	Device Read Mode on Return: Status Register							*/
/**********************************************************************************************************/

#define	RDYMASK	0X0080			/* Mask to isolate the WSM Status bit of the Status Register		*/
#define	WSMRDY	0X0080			/* Status Register value after masking, signifying that the WSM is	*/
					/*	no longer busy							*/
#define	SUSPMASK	0X0040		/* Mask to isolate the erase suspend status bit of the Status Register	*/
#define	ESUSPYES	0X0040		/* Status Register value after masking, signifying that block erase	*/
					/*	has been suspended						*/
#define	STATREAD	0X0070		/* Read Status Register command						*/
#define	SYSADDR	0			/* This constant can be initialized to any address within the memory map*/
					/* 	of the target  and is alterable depending on the		*/
					/*	system architecture						*/
#define	RESUMCMD	0X00D0		/* Erase Resume command							*/

int erasres()

{
	word far *stataddr;			/* Pointer variable used to write commands to device			*/

	stataddr	= (word far *)SYSADDR;
	*stataddr	= STATREAD;		/* Write Read Status Register command to 				*/
	if ((*stataddr & SUSPMASK) != ESUSPYES)
		return (1);			/* Block erase not suspended.  Error code "1"			*/
	*stataddr	= RESUMCMD;		/* Write Erase Resume command to the device			*/
	while ((*stataddr & SUSPMASK) == ESUSPYES)
		;			/* Will remain in while loop until bit 6 of the Status Register goes to 0,	*/
					/*	signifying block erase resumption			*/
	while ((*stataddr & RDYMASK) == WSMRDY)
		;			/* Will remain in while loop until bit 7 of the Status Register goes to 0,	*/
					/*	signifying that the WSM is once again busy		*/
	return (0);
}

/**********************************************************************************************************/
/*	Function: End											*/
/*	Description: Checks to see if the WSM is busy (is word write/block erase completed?)		*/
/*	Inputs:		None										*/
/*	Outputs:	statdata: Status Register data read from device					*/
/*	Returns:	0 = Word Write/Block Erase completed						*/
/*			1 = Word Write/Block Erase still in progress					*/
/*	Device Read Mode on Return: Status Register							*/
/**********************************************************************************************************/

#define	RDYMASK	0X0080			/* Mask to isolate the WSM Status bit of the Status Register		*/
#define	WSMRDY	0X0080			/* Status Register value after masking, signifying that the WSM		*/
					/*	is no longer busy						*/
#define	STATREAD	0X0070		/* Read Status Register command						*/
#define	SYSADDR	0			/* This constant can be initialized to any address within the memory map*/
					/* 	of the target  and is alterable depending on the		*/
					/*	system architecture						*/

int end(statdata)

word *statdata;				/* Allows Status Register data to be passed back to the main program	*/
					/*	for further analysis						*/

{
	word far *stataddr;			/* Pointer variable used to write commands to device			*/

	stataddr	= (word far *)SYSADDR;
	*stataddr	= STATREAD;		/* Write Read Status Register command to 				*/
	if (((*statdata = *stataddr) & RDYMASK) != WSMRDY)
		return (1);			/* Word write/block erasure still in progress...code "1"		*/
	return (0);				/* Word write/block erase attempt completed...code "0"			*/
}

/**********************************************************************************************************/
/*	Function: Eraschk										*/
/*	Description: Completes full Status Register check for block erase (proper command sequence, Vpp low detect,	*/
/*		block erase success).  This routine assumes that block erase completion has already been checked in	*/
/*		 function end(), and therefore does not check the WSM Status bit of the Status Register			*/
/*	Inputs:	statdata: Status Register data read in function end						*/
/*	Outputs:	None											*/
/*	Returns:	0 = Block erase completed successfully							*/
/*			1 = Error; Vpp low detect								*/
/*			2 = Error; Block erase error								*/
/*			3 = Error; Improper command sequencing							*/
/*	Device Read Mode on Return: Same as when entered							*/
/**********************************************************************************************************/

#define	ESEQMASK	0X0030		/* Mask to isolate the erase and word write status bits of the Status Register	*/
#define	ESEQFAIL	0X0030		/* Status Register value after masking if block erase command sequence	*/
					/* error has been detected						*/
#define	EERRMSK	0X0020			/* Mask to isolate the erase status bit of the Status Register		*/
#define	ERASERR	0X0020			/* Status Register value after masking if block erase error		*/
					/*	has been detected						*/
#define	VLOWMASK	0X0008		/* Mask to isolate the Vpp status bit of the Status Register		*/
#define	VPPLOW		0X0008		/* Status Register value after masking if Vpp low has been detected	*/

int eraschk(statdata)

word statdata;				/* Status Register data that has been already read from the 		*/
					/*	in function end()						*/

{
	if ((statdata & VLOWMASK) == VPPLOW)
		return (1);			/* Vpp low detect error, return code "1"			*/
	if ((statdata & EERRMSK) == ERASERR)
		return (2);			/* Block erase error detect, return code "2"			*/
	if ((statdata & ESEQMASK) == ESEQFAIL)
		return (3);			/* Block erase command sequence error, return code "3"		*/
	return (0);				/* Block erase success, return code "0"				*/
}

/********************************************************************************************************/
/*	Function: Writebgn										*/
/*	Description: Begins word write sequence								*/
/*	Inputs:		wdata: Data to be written into the device					*/
/*			waddr: Target address to be written						*/
/*	Outputs:	None										*/
/*	Returns:	0 = Word write successfully initiated						*/
/*			1 = Word write not initiated (ID check error)					*/
/*	Device Read Mode on Return: Status Register (ID if returns 1)					*/
/********************************************************************************************************/

#define	SETUPCMD	0X0040		/* Word Write Setup command					*/

int writebgn(wdata,waddr)

word wdata;					/* Data to be written into the 					*/
word far *waddr;				/* waddr is the destination address for the data to be written		*/

{
	word mfgrid,deviceid;

	if (idread(&mfgrid,&deviceid)==1)	/* Device ID read error...powered up?				*/
		return (1);
	*waddr 	= SETUPCMD;			/* Write Word Write Setup command and destination address		*/
	*waddr 	= wdata;			/* Write word write data and destination address			*/
	return (0);
}

/**********************************************************************************************************/
/*	Function: Writechk										*/
/*	Description: Completes full Status Register check for word write (Vpp low detect, word write success). 		*/
/*		This routine assumes that word write completion has already been checked in function end()		*/
/*		and therefore does not check the WSM Status bit of the Status Register				*/
/*	Inputs:		statdata: Status Register data read in function end()						*/
/*	Outputs:	None										*/
/*	Returns:	0 = Word write completed successfully							*/
/*			1 = Error; Vpp low detect								*/
/*			2 = Error; Word write error								*/
/*	Device Read Mode on Return: Status Register							*/
/**********************************************************************************************************/

#define	WERRMSK	0X0010			/* Mask to isolate the word write error bit of the Status Register		*/
#define	WRITERR	0X0010			/* Status Register value after masking if word write error has been detected	*/
#define	VLOWMASK	0X0008		/* Mask to isolate the Vpp status bit of the Status Register		*/
#define	VPPLOW		0X0008		/* Status Register value after masking if Vpp low has been detected	*/

int writechk(statdata)

word statdata;				/* Status Register data that has been already read from the 		*/
					/*	in function end()					*/

{
	if ((statdata & VLOWMASK) == VPPLOW)
		return (1);			/* Vpp low detect error, return code "1"				*/
	if ((statdata & WERRMSK) == WRITERR)
		return (2);			/* Word write error detect, return code "2"				*/
	return (0);				/* Word/string write success, return code "0"			*/
}

/**********************************************************************************************************/
/*	Function: Idread										*/
/*	Description: Reads the manufacturer and device IDs from the target 					*/
/*	Inputs:		None										*/
/*	Outputs:	mfgrid: Returned manufacturer ID							*/
/*			deviceid: Returned device ID								*/
/*	Returns:	0 = ID read correct									*/
/*			1 = Wrong or no ID									*/
/*	Device Read Mode on Return: inteligent Identifier							*/
/**********************************************************************************************************/

#define	MFGRADDR	0		/* Address "0" for the target ...alterable depending on the		*/
					/*	system architecture					*/
#define	DEVICADD	1		/* Address "1" for the target ...alterable depending on the		*/
					/*	system architecture					*/
#define	IDRDCOMM	0X0090		/* Inteligent Identifier command					*/
#define 	INTELID		0X0089	/* Manufacturer ID for Intel devices				*/
#define 	DVCID		0X4470	/* This should be modified for the device you are using: 	*/

/*				  28F800BV-T, 28F800CV-T, 28F800CE-T				889CH */
/*				  28F800BV-B, 28F800CV-B, 28F800CE-B				889DH */
/*				  28F400BV-T, 28F400CV-T, 28F400CE-T, 28F400BX-T, 28F400BL-T	4470H */
/*				  28F400BV-B, 28F400CV-B, 28F400CE-B, 28F400BX-B, 28F400BL-B	4471H */
/*				  28F200BV-T, 28F200CV-T, 28F200CE-T, 28F200BX-T, 28F200BL-T	2274H */ 
/*				  28F200BV-B, 28F200CV-B, 28F200CE-B, 28F200BX-B, 28F200BL-B	2275H */


int idread(mfgrid,deviceid)

word *mfgrid;				/* The manufacturer ID read by this function, to be transferred back to	*/
					/*	the calling program					*/
word *deviceid;				/* The device ID read by this function, to be transferred back to the	*/
					/*	calling function					*/

{
	word far *tempaddr;			/* Pointer address variable used to read IDs			*/

	tempaddr	= (word far *)MFGRADDR;
	*tempaddr	= IDRDCOMM;		/* Write inteligent Identifier command to an address within the */
						/*	memory map (in this case 00 hex)			*/
	*mfgrid	= *tempaddr;			/* Read mfgr ID, tempaddr still points at address "0"		*/
	tempaddr	= (word far *)DEVICADD;	/* Point to address "1" for the device specific ID		*/
	*deviceid	= *tempaddr;		/* Read device ID						*/
	if ((*mfgrid != INTELID) || (*deviceid != DVCID) )
		return (1);			/* ID read error; device powered up?				*/
	return (0);
}

/**********************************************************************************************************/
/*	Function: Statrd										*/
/*	Description: Returns contents of the target  Status Register					*/
/*	Inputs:		None										*/
/*	Outputs:	statdata: Returned Status Register data						*/
/*	Returns:	Nothing										*/
/*	Device Read Mode on Return: Status Register							*/
/**********************************************************************************************************/

#define	STATREAD	0X0070		/* Read Status Register command				*/
#define	SYSADDR	0			/* This constant can be initialized to any address within the memory map	*/
					/* 	of the target  and is alterable depending on the		*/
					/*	system architecture					*/

int statrd(statdata)

word *statdata;				/* Allows Status Register data to be passed back to the calling program	*/
					/*	for further analysis					*/

{
	word far *stataddr;			/* Pointer variable used to write commands to device			*/

	stataddr	= (word far *)SYSADDR;
	*stataddr	= STATREAD;		/* Write Read Status Register command to 				*/
	*statdata = *stataddr;
	return;
}

/**********************************************************************************************************/
/*	Function: Statclr										*/
/*	Description: Clears the  Status Register							*/
/*	Inputs:		None										*/
/*	Outputs:	None										*/
/*	Returns:	Nothing										*/
/*	Device Read Mode on Return: Array								*/
/**********************************************************************************************************/

#define	STATCLER	0X0050		/* Clear Status Register command				*/
#define	SYSADDR	0			/* This constant can be initialized to any address within the memory map	*/
					/* 	of the target  and is alterable depending on the		*/
					/*	system architecture					*/

int statclr()

{
	word far *stataddr;			/* Pointer variable used to write commands to device			*/

	stataddr	= (word far *)SYSADDR;
	*stataddr	= STATCLER;		/* Write Clear Status Register command to 			*/
	return;
}

/**********************************************************************************************************/
/*	Function: Rdmode										*/
/*	Description: Puts the target  in Read Array Mode.  This function might be used, for example, to prepare		*/
/*		the system for return to code execution out of the Flash memory after word write or block erase algorithms	*/
/*		have been executed off-chip								*/
/*	Inputs:	None										*/
/*	Outputs:	None										*/
/*	Returns:	Nothing										*/
/*	Device Read Mode on Return: Array								*/
/**********************************************************************************************************/

#define	RDARRAY	0X00FF			/* Read Array command					*/
#define	SYSADDR	0			/* This constant can be initialized to any address within the memory map	*/
					/* 	of the target  and is alterable depending on the		*/
					/*	system architecture					*/

int rdmode()

{
	word far *tempaddr;			/* Pointer variable used to write commands to the device		*/

	tempaddr	= (word far *)SYSADDR;
	*tempaddr	= RDARRAY;		/* Write Read Array command to 				*/
	return;
}

/**********************************************************************************************************/
/*	Function: Rdword										*/
/*	Description: Reads a word of data from a specified address and returns it to the calling program			*/
/*	Inputs:		raddr: Target address to be read from							*/
/*	Outputs:	rdata: Data at the specified address							*/
/*	Returns:	Nothing										*/
/*	Device Read Mode on Return: Array								*/
/**********************************************************************************************************/

#define	RDARRAY	0X00FF		/* Read array command					*/

int rdword(rdata,raddr)

word *rdata;				/* Returns data read from the device at specified address		*/
word far *raddr;				/* Raddr is the target address to be read from			*/

{
	*raddr	= RDARRAY;		/* Write read array command to an address within the  (in this		*/
					/*	case the target address)				*/
	*rdata	= *raddr;			/* Read from the specified address and store			*/
	return;
}

/**********************************************************************************************************/
/*	Function: Vppup										*/
/*	Description: Ramps the Vpp supply to the target  to enable word write or block erase.  This routine can be		*/
/*		tailored to the individual system architecture.  For purposes of this example, I assumed that a system 	*/
/*		Control Register existed at system address 20000 hex, with the following definitions:			*/
/*												*/
/*			Bit 7: Vpph Control:	1 = Enabled						*/
/*						0 = Disabled						*/
/*			Bit 6: RP# Control:	1 = PowerDown Enabled					*/
/*						0 = PowerDown Disabled					*/
/*			Bits 5-0: Undefined								*/
/*												*/
/*	Inputs:		None										*/
/*	Outputs:	None										*/
/*	Returns:	Nothing										*/
/*	Device Read Mode on Return: As existed before entering the function.  Part is now ready for program or erase	*/
/**********************************************************************************************************/

#define	VPPHIGH		0X0080		/* Bit 7 = 1, Vpp elevated to Vpph				*/
#define	SYSCADDR	0X20000		/* Assumed system Control Register Address			*/

int vppup()

{
	word far *contaddr;			/* Pointer variable used to write data to the system Control Register	*/

	contaddr	= (word far *)SYSCADDR;
	*contaddr	= *contaddr | VPPHIGH;	/* Read current Control Register data, "OR" with constant to ramp Vpp	*/
	return;
}

/**********************************************************************************************************/
/*	Function: Vppdown										*/
/*	Description: Ramps down the Vpp supply to the target  to disable word write/block erase.			*/
/*		See above for a description of the assumed system Control Register.				*/
/*	Inputs:		None										*/
/*	Outputs:	None										*/
/*	Returns:	Nothing										*/
/*	Device Read Mode on Return: As existed before entering the function.  Part now has high Vpp disabled.  If word	*/
/*		 write or block erase was in progress when this function was called, it will complete unsuccessfully with	*/
/*		 Vpp low error in the Status Register.							*/
/**********************************************************************************************************/

#define	VPPDWN		0X007F		/* Bit 7 = 0, Vpp lowered to Vppl				*/
#define	SYSCADDR	0X20000		/* Assumed system Control Register Address			*/

int vppdown()

{
	word far *contaddr;			/* Pointer variable used to write data to the system Control Register	*/

	contaddr	= (word far *)SYSCADDR;
	*contaddr	= *contaddr & VPPDWN;	/* Read current Control Register data, "AND" with constant to lower Vpp	*/
	return;
}

/**********************************************************************************************************/
/*	Function: rpen											*/
/*	Description: "RP# enable" Toggles the  RP# pin low to put the device in Deep PowerDown mode.				*/
/*		See above for a description of the assumed system Control Register.				*/
/*	Inputs:		None										*/
/*	Outputs:	None										*/
/*	Returns:	Nothing										*/
/*	Device Read Mode on Return: The part is powered down.  If word write or block erase was in progress when this	*/
/*		 function was called, it will abort with resulting partially written or erased data.  Recovery in the form of	*/
/*		 repeat of word write or block erase will be required once the part transitions out of powerdown,		*/
/*		to initialize data to a known state.							*/
/**********************************************************************************************************/

#define	RP		0X0040		/* Bit 6 = 1, RP# enabled					*/
#define	SYSCADDR	0X20000		/* Assumed system Control Register Address			*/

int rpen()

{
	word far *contaddr;			/* Pointer variable used to write data to the system Control Register	*/

	contaddr	= (word far *)SYSCADDR;
	*contaddr	= *contaddr | RP;	/* Read current Control Register data, "OR" with constant to 		*/
					/* enable Deep PowerDown					*/
	return;
}

/**********************************************************************************************************/
/*	Function: Rpdis											*/
/*	Description: "RP# Disable" Toggles the  RP# pin high to transition the part out of Deep PowerDown.			*/
/*		See above for a description of the assumed system Control Register.				*/
/*	Inputs:	None											*/
/*	Outputs:	None										*/
/*	Returns:	Nothing										*/
/*	Device Read Mode on Return: Read Array mode.  Low voltage is removed from the RP# pin.   Output		*/
/*		pins will output valid data time tPHQV after the RP# pin transitions high (reference the datasheet AC Read	*/
/*		Characteristics) assuming valid states on all other control and power supply pins.			*/
/**********************************************************************************************************/

#define	RPOFF		0X00BF		/* Bit 6 = 0, RP# disabled					*/
#define	SYSCADDR	0X20000		/* Assumed system Control Register Address			*/

int Rpdis()

{
	word far *contaddr;			/* Pointer variable used to write data to the system Control Register	*/

	contaddr	= (word far *)SYSCADDR;
	*contaddr	= *contaddr & RPOFF;	/* Read current Control Register data, "AND" with constant to disable	*/
						/*	Deep PowerDown					*/
	return;
}



