%this example was written and tested in MATLAB R2018a
clear all;
close all;

PORT = "/dev/cu.usbmodem00000000029A1";                                     %specify your port here, like "COM1" for win

START = 1500000000;                                                         %All frequency values are in Hz.
STOP = 1700000000;
STEP = 1000000;
ATTENUATION = 0;

POINT_TIMEOUT = 200;                                                        %delay for frequency synth setting. should be keept at 200.
POINT_ADC_SAMPLES = 20;                                                     %adc sample count. affects speed.

INTERMEDIATE_FREQ = 10700000;                                               %intermediate frequency. Hardware specific, should be keept at 10700000

GEN_FREQ = 1100000000;
GEN_OUTPUT = -25;

command_index = 0;

s = serial(PORT);
s.InputBufferSize = 409600;                                                 %set bigger buffer size for large spans
s.BaudRate = 115200;
s.Terminator = "CR/LF";                                                     %default termitanor for commands
                                      
fopen(s);
flushinput(s);
flushoutput(s);

generator_on(s, command_index);
generator_set_output(s, GEN_OUTPUT, command_index);
generator_set_frequency(s, GEN_FREQ, command_index);

plot_figure = figure('Name', 'Spectrum', 'Visible', 'On');
plot_axes = axes('Parent', plot_figure);
freq_range = linspace(START, STOP,(STOP - START)/STEP + 1);

iteration_counter = 10;
while(iteration_counter ~= 0)
    amplitude_data = scan_data(s, START, STOP, STEP, ATTENUATION,...
        INTERMEDIATE_FREQ, POINT_ADC_SAMPLES, POINT_TIMEOUT,...
        command_index, false);
    command_index = command_index + 1;
    plot(plot_axes, freq_range, amplitude_data);
    ylim(plot_axes,[-110 -30]); 
    grid(plot_axes, 'on');
    xlabel('Freq, MHz');
    ylabel('Amplitude, dBm');
    pause(0.01);
    iteration_counter = iteration_counter - 1;
end
fclose(s);

function generator_on(serial_port, command_number)                          %turn on generator
    fprintf(serial_port,"gon %d\n",command_number);                         
    command_number = command_number + 1;
    
    fscanf(serial_port)
    fscanf(serial_port)
    flushinput(serial_port);
    flushoutput(serial_port);
end

function generator_off(serial_port, command_number)                         %turn on generator
    fprintf(serial_port,"gof %d\n",command_number);                         
    command_number = command_number + 1;
    
    fscanf(serial_port)
    fscanf(serial_port)
    flushinput(serial_port);
    flushoutput(serial_port);
end

function generator_set_output(serial_port, output_amplitude, command_number)
    command_att = (100 * 100) + ((output_amplitude + 15) * 100);            %format generator output for command

    fprintf(serial_port,"sga %d %d\r\n", [command_att, command_number]);    %set generator attenuation
    command_number = command_number + 1;

    fscanf(serial_port)
    fscanf(serial_port)
    flushinput(serial_port);
    flushoutput(serial_port);
end

function generator_set_frequency(serial_port, frequency, command_number)
    fprintf(serial_port,"scf %d %d\r\n",[frequency, command_number]);       %set generator frequency
    command_number = command_number+1;
    
    fscanf(serial_port)                                                                   
    fscanf(serial_port)
    fscanf(serial_port)
    flushinput(serial_port);
    flushoutput(serial_port);
end

function amplitude_result = scan_data(serial_port, Start, Stop, Step, Attenuation,...
        intermid_freq, adc_samples, timeout, command_number, tracking)
    amplitude_result = zeros((Stop - Start)/Step + 1);
    command_att = (100*100)+((Attenuation)*100);                            %format attenuation value for command
    
    if(~tracking)
        fprintf(serial_port,"scn20 %d %d %d %d %d %d %d %d\r\n",...
            [Start, Stop, Step, timeout, adc_samples,...
            intermid_freq, command_att, command_number]);                   %scan without tracking
    else
        fprintf(s,"scn22 %d %d %d %d %d %d %d %d\r\n",...
            [Start, Stop, Step, timeout, adc_samples,...
            intermid_freq, command_att, command_number]);                   %scan with tracking
    end
    command_number = command_number + 1;
    fscanf(serial_port)                                                     %receiving \n\r
    fscanf(serial_port)                                                     %receiving command header
    scan_ans =  fscanf(serial_port)                                         %receiving binary data
    fscanf(serial_port)                                                     %receiving command tail
    
    flushinput(serial_port);
    flushoutput(serial_port);
    index = length(scan_ans);
    while(scan_ans(index) ~= 255)                                           %search for binary stram end (i.e 0XFF byte)
        index = index-1;
    end
    index = index - 2;
    stopindex = index;
    
    iterator = 1;
    index = 1;
    while(index < stopindex )

        binaryval = (bitsll(uint16(scan_ans(index)),8)...
            + uint16(scan_ans(index+1)));
        
        for i = 12:15                                                       %getting rid of indexing data
            binaryval = bitset(binaryval,i,0);
        end
        
        amplitude_result(iterator) = -Attenuation + ...
            double(((10.0*80 - double(binaryval))/10.0));
        
        iterator = iterator + 1;
        index = index + 2;
    end
    fprintf("elapsed time %s",scan_ans(index + 2:length(scan_ans)))
end

