package sorcererII;

/*
 * @(#)Sorcerer.java
 */
import java.awt.Color;
import java.awt.Container;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.event.KeyEvent;
import java.io.BufferedInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.LinkedList;
import java.util.Map;
import java.util.Random;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * The Sorcerer class extends the Z80 class implementing the supporting
 * hardware emulation which was specific to the Exidy Sorcerer. This
 * includes the memory mapped screen and the IO ports which were used
 * to read the keyboard.<P>
 *
 *
 * @version 1.0 14 July 2000
 * @author <A HREF="http://www.liaquay.co.uk/">Philip M. Scull</A>
 *
 * @see Sorcerer
 * @see Z80
 */

public class Sorcerer extends Z80 implements Runnable
{
  public static Logger LOGGER = Logger.getLogger("SorcererLogger");
  
  public Graphics     parentGraphics = null;
  public Container    parent      = null;

  /** Since execute runs as a tight loop, some Java VM implementations
   *  don't allow any other threads to get a look in. This give the
   *  GUI time to update.
   */
  public  int     sleepHack = 0;
  public  int     refreshRate = 1;  // refresh every 'n' interrupts

  private int     interruptCounter = 0;
  private boolean resetAtNextInterrupt = false;
  private boolean refreshNextInterrupt = true;
  private boolean takeSnapshotNextInterrupt = false;

  public  long    timeOfLastInterrupt = 0;
  private long    timeOfLastSample = 0;
  private long    runningAverage = 40;
  private int     delayPeriod = 10;
  private boolean useHardKeyMap = true;

  //
  // Sorcerer memory space is divided up into 64 1KByte segments.
  // An array of flags to indicate which segments of memory are ROM.
  //
  private boolean[] romFlags = new boolean[ 64 ];

  //
  // To reduce the number of calls to draw image flags are kept to
  // indicate which characters have been altered on the screen.
  //
  private boolean[] screenFlags = new boolean[ nCharsWide * nCharsHigh ];

  //
  // Flags to indicate which user-defined characters have changed
  //
  private boolean[] characterFlags = new boolean[ 128 ];



  //
  // The tape device
  //
  private SorcererTape sorcererTape1 = null;
  private SorcererTape sorcererTape2 = null;
  private SorcererTape sorcererSerial = null;
  
  //
  // Flags to indicate satus of the tape motors.
  // true is for on, false is for off.
  //
  private boolean tape1Running = false;
  private boolean tape2Running = false;

  //
  // A random number generator to add noise if no tape selected
  //
  private Random rand = new Random();

  //
  // The sorcerer disk-drive system
  //
  private SorcererDiskSystem _diskSystem = new SorcererDiskSystem();

  //
  // Offset to graphics screen drawing
  //
  private int offsetX = 0;
  private int offsetY = 0;

  public Sorcerer(
    Container _parent,
    int _refreshRate,
    int _sleepHack,
    int _pixelScale,
    int _offsetX,
    int _offsetY ) 
  {
    // Sorcerer runs at 2Mhz
    //
    // Set this so screen update is ok!
    //
    super( 2.0 );

    parent = _parent;
    offsetX = _offsetX;
    offsetY = _offsetY;

    sleepHack = _sleepHack;
    pixelScale = _pixelScale;
    refreshRate = _refreshRate;

    setRomFlags();

    //
    // Initially the entire screen will need refreshing.
    //
    for( int i = 0; i < screenFlags.length; ++i )
    {
      screenFlags[ i ] = true;
    }

    resetKeyboard();

  }

  public void attachTapeUnit( SorcererTape tape, int unit) {
    switch(unit) {
    case 0:attachTapeUnit1(tape);
    case 1:attachTapeUnit2(tape);
      default:break;
    }
  }
  
  public void attachTapeUnit1( SorcererTape tape )
  {
    sorcererTape1 = tape;
  }

  public void attachTapeUnit2( SorcererTape tape )
  {
    sorcererTape2 = tape;
  }
  
  public void attachSerial( SorcererTape tape )
  {
    sorcererSerial = tape;
  }
  
  public void insertDiskIntoDrive( ArraySorcererDisk disk, int drive )
  {
    _diskSystem.insertDisk( disk, drive );
  }

  public SorcererDiskDrive getSorcererDiskDrive(int drive) {
    return _diskSystem.getSorcererDiskDrive(drive);
  }
  
  public void setPixelScale( int scale )
  {
    pixelScale = scale;
  }
  
  public void useHardKeys(final boolean useHardKeys) {
    useHardKeyMap = useHardKeys;
    resetKeyboard();
  }
  
  public void abort()
  {
    super.abort();
    if(sorcererSerial != null) sorcererSerial.motorOff();
    if(sorcererTape1 != null) sorcererTape1.motorOff();
    if(sorcererTape2 != null) sorcererTape2.motorOff();    
  }
  void resetNextInterrupt() {
    resetAtNextInterrupt = true;
  }
  /**
   * Z80 hardware interface
   */
  public final int inb( int port )
  {
//System.out.println( "inb " + Z80.toHexWord( port ) );
    int res = 0xff;
    int bport = port & 0xFF;

    if( bport == 0xFE )
    {
      //
      // Parallel port control lines:
      // ==================================
      // Bit 7 - Input ready - active high
      // Bit 6 - Output ready - active high
      //
      res =  _keyboard.getKeyBits();

      res |= 0xe0;
    }
    else if( bport == 0xFC )
    {
//System.out.println( "inb " + Z80.toHexWord( port ) );
      //
      // This is the tape/serial data input port
      //
      try
      {
        if( tape1Running && ( sorcererTape1 != null ) )
        {
          res = sorcererTape1.getByte() & 0xff;
        }
        else if( tape2Running && ( sorcererTape2 != null ) )
        {
          res = sorcererTape2.getByte() & 0xff;
        }
        else if( sorcererSerial != null ) 
        {
          res = sorcererSerial.getByte();
        }
        else
        {
          res = rand.nextInt() & 0xFF;
        }
      }
      catch( IOException e )
      {
        res = rand.nextInt() & 0xFF;
      }
    }
    else if( bport == 0xFD )
    {
//System.out.println( "inb " + Z80.toHexWord( port ) );

      //
      // This is the tape control line port
      //
      // Bit Function
      // ==================================
      // 0   Ready for input
      // 1   Ready for output
      //
      // Bits are active high
      //

      //
      // Rig the system so that input and output can always happen.
      //
      res = 0xff;
    }
    else if( bport == 0xFF ){
      // Parallel port
//      System.out.println( "Parallel read" );
      
      // Centronics 
      // ===================================
      // Ready to write - Bit 7 - active low
      
      // Joy Stick
      // ===================================
      // Left - Bit 7 - active low
      
      res = 0xFF;
    }
    else {
System.out.println( "unhandled inb " + Z80.toHexWord( port ) );
    }

    return(res);
  }

  public final void outb( int port, int outByte, int tstates )
  {
//System.out.println( "outb " + Z80.toHexWord( port ) + " " + Z80.toHexByte( outByte ) );
    int bport = port & 0xFF;

    if( bport == 0xFE )
    {
      //
      // This is the row in the keyboard the Sorcerer is scanning
      //
      _keyboard.setScanLine( outByte & 0xf );

      //
      // Look for the motor control bits changing
      //
      boolean motor1Bit = ( outByte & 0x10 ) != 0;
      boolean motor2Bit = ( outByte & 0x20 ) != 0;

      if( tape1Running && !motor1Bit )
      {
        tape1Running = false;
        if( sorcererTape1 != null ) sorcererTape1.motorOff();
      }
      else if( !tape1Running && motor1Bit )
      {
        tape1Running  = true;
        if( sorcererTape1 != null ) sorcererTape1.motorOn((outByte & 0x40) == 0 ? 300 : 1200);
      }

      if( tape2Running && !motor2Bit )
      {
        tape2Running = false;
        if( sorcererTape2 != null ) sorcererTape2.motorOff();
      }
      else if( !tape2Running && motor2Bit )
      {
        tape2Running  = true;
        if( sorcererTape2 != null ) sorcererTape2.motorOn((outByte & 0x40) == 0 ? 300 : 1200);
      }
    }
    else if( bport == 0xFC )
    {
//System.out.println( "outb " + Z80.toHexWord( port ) + " " + Z80.toHexByte( outByte ) );
      //
      // This is the tape/serial data output port
      //

      if( tape1Running )
      {
        try {
          if( sorcererTape1 != null ) sorcererTape1.writeByte( (byte)( outByte & 0xff ) );
        }
        catch(final IOException e) {
          LOGGER.log(Level.SEVERE, "Error writing to tape unit 1", e);
          sorcererTape1 = null;
        }
      }
      else if( tape2Running )
      {
        try {
          if( sorcererTape2 != null ) sorcererTape2.writeByte( (byte)( outByte & 0xff ) );
        }
        catch(final IOException e) {
          LOGGER.log(Level.SEVERE, "Error writing to tape unit 2", e);
          sorcererTape2 = null;
        }
      }
      else if( sorcererSerial != null ) {
        try {
          sorcererSerial.writeByte( (byte)( outByte & 0xff ) );
        }
        catch(final IOException e) {
          LOGGER.log(Level.SEVERE, "Error writing to tape unit 2", e);
          sorcererSerial = null;
        }
      }                
    }
    else if( bport == 0xFF ){
      // Parallel port
//      System.out.println( "Parallel write" +  Z80.toHexByte( outByte ));
    }
    else {
//System.out.println( "outb " + Z80.toHexWord( port ) + " " + Z80.toHexByte( outByte ) );
    }
  }

  //
  // Refresh the whole screen
  //
  public final void refreshWholeScreen()
  {
    for( int i = 0xF080; i < 0xF800; ++i )
    {
      screenFlags[ i - 0xF080 ] = true;
    }
  }

  //
  // Tag a particular character for refresh.
  // This is required when a user defined character changes.
  //
  public final void refreshCharacter( int c )
  {
    for( int i = 0xF080; i < 0xF800; ++i )
    {
      if( mem[ i ] == c )
      {
        screenFlags[ i - 0xF080 ] = true;
      }
    }
  }

  /** Byte access */
  public final int peekb( int address )
  {
//    if( ( address >= 0xBD00 ) && ( address <= 0xE000 ) )
//    {
//      System.out.println( "Reading " + Z80.toHexWord( address ) );
//    }
if( trace ) System.out.println( "Reading " + Z80.toHexWord( address ) + " " +  Z80.toHexByte( mem[ address ] ) );

    //
    // Check for the disk system
    //
    if( ( address >= 0xBE00 ) && ( address <= 0xBE02 ) )
    {
      return _diskSystem.read( address - 0xBE00 );
    }
    else
    {
      return mem[ address ];
    }
  }

  public final void pokeb( int address, int newByte )
  {
    if( trace ) System.out.println( "Writing " + Z80.toHexByte( newByte ) + " to " + Z80.toHexWord( address ) );
    //
    // To determine if an address resides in ROM first calculate the
    // segment being addressed.
    //
    int memorySegment = address / 1024;

    if( romFlags[ memorySegment ] )
    {
//      if( ( address >= 0xBD00 ) && ( address < 0xE000 ) )
//      {
//        System.out.println( "Writing " + Z80.toHexByte( newByte ) + " to " + Z80.toHexWord( address ) );
//      }

      //
      // Check for the disk system
      //
      if( ( address >= 0xBE00 ) && ( address <= 0xBE02 ) )
      {
        _diskSystem.write( address - 0xBE00, newByte );
      }

      //
      // If the ROM flag is set for the segment do not write to it.
      //
      return;
    }

    //
    // Handle writes to video memory
    //
    if( ( address >= 0xF080 ) && ( address < 0xF800 ) )
    {
      screenFlags[ address - 0xF080 ] = true;
    }
    //
    // Handle user defined characters single byte version only.
    //
    // Check for user defined character addresses
    //
    else if( address >= 0xFC00 )
    {
      //
      // Determine which character has been altered
      //
      int characterIndex = ( ( address - 0xFC00 ) >> 3 ) + 128;

      //
      // Determine which row in the charater has been altered
      //
      int characterRow = address & 7;

      //
      // Blank the affected row
      //
      if( characterImages == null ) setupScreenCharacters();
      Graphics g = characterImages.getGraphics();

      g.setColor( Color.black );
      g.translate( characterIndex << 3, characterRow );
      g.fillRect( 0, 0, 8, 1 );

      //
      // Set the bits in the new byte
      //
      g.setColor( Color.white );
      int b = newByte;
      for( int i = 0; i < 8; ++i )
      {
        if( ( b & 0x80 ) != 0 )
        {
        g.fillRect( i, 0, 1, 1 );
        }
        b = b << 1;
      }
      g.translate( 0, 0 );

      //
      // Make sure the required characters are refreshed
      // Watch out for sign extend if this is a byte!
      //
      //refreshCharacter( newByte );
      characterFlags[ characterIndex - 128 ] = true;
    }
    //
    // Now write the new byte to memory
    //
    mem[ address ] = newByte;
  }

  private Thread thread = null;

  public void start() {
      thread = new Thread(this, "JSorcerer");
      thread.start();
    }

  public void stop() {
    abort();
  
    //
    // Wait for the thread to end.
    //
    try {
      thread.join();
    } catch (Exception e) {
    }
    thread = null;
  
  }
  
  public void run()
  {
    try {
    timeOfLastInterrupt = System.currentTimeMillis();
    timeOfLastSample = timeOfLastInterrupt;
    execute();
    }
    catch(final Throwable t) {
      t.printStackTrace();
    }
  }

  public final void loadCom(final InputStream is, final boolean autorun) throws IOException {
    try {
      int m = 0x100;
      while (true) {
        final int b= is.read();
        if(b == -1) break;
        pokeb(m++, b);
      }
      final int l = m - 0x100;
      LOGGER.log(Level.INFO, "Read .COM to 0x100 with length 0x" + Integer.toHexString(l));

      final int b = (l + 255) >>8;
      LOGGER.log(Level.INFO, "If you are in CP/M you may want to save this file to disk with the following: SAVE " + b + " <filename>.COM");
      if (autorun) PC(0x100);
    }
    finally {
      is.close();
    }
  }

  public final void loadBin(final InputStream is, final boolean autorun) throws IOException {
    final byte[] preamble = new byte[7];
    if(is.read(preamble) != preamble.length) throw new IOException("Failed to load BIN file preamble");
    final StringBuilder name = new StringBuilder();
    while(true){
      final int b = is.read();
      if(b == -1) throw new IOException("Failed to load BIN file name");
      if(b == 0x1a) break;
      if(b != 0) {
        name.append((char)b);
      }
    }
    final byte[] args = new byte[6];
    if(is.read(args) != args.length) throw new IOException("Failed to load BIN file args");
    final int exec = (args[0] & 0xff) + ((args[1] & 0xff) <<8);
    final int start = (args[2] & 0xff) + ((args[3] & 0xff) <<8);
    final int end = (args[4] & 0xff) + ((args[5] & 0xff) <<8);
    final int size = (end - start + 1) & 0xffff;
    
    LOGGER.log(Level.INFO, "Reading " + name + 
        " exec=" + Integer.toHexString(exec) +
        " start=" + Integer.toHexString(start) +
        " end=" + Integer.toHexString(end) +
        " size=" + Integer.toHexString(size)); 

    try {
      int m = start;
      while (m <= end) {
        final int b= is.read();
        if(b == -1) break;
        pokeb(m++, b);
      }
      final int l = m - start;
      //        System.out.println("Read IN.BIN to 0x100 with length 0x" + Integer.toHexString(l));
      //
      //        final int b = (l + 255) >>8;
      //        System.out.println("If you are in CP/M you may want to save this file to disk with the following:");
      //        System.out.println("SAVE " + b + " IN.COM");
      
      /* 
       * Borrowed from mess... http://mess.dorando.at/svn/?rev=2025
       * 
       * Since Exidy Basic is by Microsoft, it needs some pre-processing before it can be run.
         1. A start address of 01D5 indicates a basic program which needs its pointers fixed up.
         2. If autorunning, jump to C689 (command processor), else jump to C3DD (READY prompt).
         Important addresses:
            01D5 = start (load) address of a conventional basic program
            C858 = an autorun basic program will have this exec address on the tape
            C3DD = part of basic that displays READY and lets user enter input */      
      
      if ((start == 0x1d5) || (exec == 0xc858)){
        final int[] fixupData = {
           0xcd, 0x26, 0xc4,     // CALL C426    ;set up other pointer
           0x21, 0xd4, 1,        // LD HL,01D4   ;start of program address (used by C689)
           0x36, 0,              // LD (HL),00   ;make sure dummy end-of-line is there
           0xc3, 0x89, 0xc6      // JP C689   ;run program
        };
        for (int i = 0; i < 11; i++) pokeb(0xf01f + i, fixupData[i]);
        if (!autorun) pokew(0xf028,0xc3dd);
        pokeb(0x1b7, end&0xff);      /* Tell BASIC where program ends */
        pokeb(0x1b8, end>>8);
        
        if (autorun && exec != 0xc858)  pokew(0xf028,exec); // only if autorun
        PC(0xf01f);
      }
      else {
        if (autorun) PC( start );
      }
    }
    finally {
      is.close();
    }
  }  
  
  
  public final int interrupt()
  {
    if ( refreshNextInterrupt )
    {
      refreshNextInterrupt = false;
      refreshWholeScreen();
    }

    if ( resetAtNextInterrupt )
    {
      resetAtNextInterrupt = false;
      reset();
    }

    if( takeSnapshotNextInterrupt )
    {
      takeSnapshotNextInterrupt = false;
      try
      {
        saveSNP( new FileOutputStream( "first.snp" ) );
      }
      catch( Exception e )
      {
        e.printStackTrace();
      }
    }
    
    interruptCounter++;

    if(parentGraphics == null) {
      parentGraphics = parent.getGraphics();
    }    
    writeScreenImage(parentGraphics);

    // Refresh every interrupt by default
    if ( (interruptCounter & 1) == 0 )
    {
      _diskSystem.tick();
    }

    timeOfLastInterrupt = System.currentTimeMillis();

    // Trying to slow to 100%, browsers resolution on the system
    // time is not accurate enough to check every interrurpt.

    long durOfLastInterrupt = (timeOfLastInterrupt - timeOfLastSample);
    timeOfLastSample = timeOfLastInterrupt;

    if ( !runAtFullSpeed )
    {
      runningAverage -= runningAverage/16;
      runningAverage += durOfLastInterrupt;

      if( ( runningAverage > 320 ) && ( delayPeriod > 0 ) ) --delayPeriod;
      if( runningAverage < 310 ) ++delayPeriod;

      if( delayPeriod > 0 )
      {
        try { Thread.sleep( delayPeriod ); }
        catch ( Exception ignored ) {}
      }
    }

    // This was put in to handle Netscape 2 which was prone to
    // locking up if one thread never gave up its timeslice.
    if ( sleepHack > 0 )
    {
      try { Thread.sleep( sleepHack ); }
      catch ( Exception ignored ) {}
    }

    return super.interrupt();
  }

  public void repaint()
  {
    refreshNextInterrupt = true;
  }

  public void reset()
  {
    super.reset();
    PC( 0xE000 );
  }

  /**
   * Screen stuff
   */
  public        int pixelScale  = 1;    // scales pixels in main screen

  public static final int nCharsWide  = 64;
  public static final int nCharsHigh  = 30;
  public static final int nPixelsWide = nCharsWide * 8;
  public static final int nPixelsHigh = nCharsHigh * 8;

  public boolean  runAtFullSpeed = false;

  public final void toggleSpeed() {
    runAtFullSpeed = !runAtFullSpeed;
  }
  
  public final void runFullSpeed(final boolean fullSpeed) {
    runAtFullSpeed = fullSpeed;
  }

  public void showMessage( String m )
  {
System.out.println( m );
  }

  Image characterImages = null;
  public void setupScreenCharacters()
  {
    if( characterImages == null )
    {
      characterImages = parent.createImage( 256 * 8, 8 );
    }

    int p = 0xF800;

    for( int i = 0; i < 256; ++i )
    {
      Graphics g = characterImages.getGraphics();
      g.translate( i * 8, 0 );

      g.setColor( Color.black );
      g.fillRect( 0,0,8,8 );
      g.setColor( Color.white );


      for( int j = 0; j < 8; j++ )
      {
        int b = mem[ p++ ];


        for( int k = 0; k < 8; ++k )
        {
          if( ( b & 0x80 ) != 0 )
          {
            g.fillRect( k, j, 1, 1 );
          }
          b = b << 1;
        }
      }
    }

    for( int i = 0; i < 128; ++i )
    {
      characterFlags[ i ] = false;
    }
  }

  public void writeScreenImage(final Graphics parentGraphics)
  {

    
    for( int i = 0; i < 128; ++i )
    {
      if( characterFlags[ i ] )
      {
        characterFlags[ i ] = false;
        refreshCharacter( i + 128 );
      }
    }

    parentGraphics.translate( offsetX, offsetY );

    //
    // Paint the screen image only updating characters that
    // have been changed.
    //
    int nChars = nCharsWide * nCharsHigh;

    for( int p = 0; p < nChars; ++p )
    {
      if( screenFlags[ p ] )
      {
        int row = (p >> 6) << 4;
        int col = (p & 63) << 3;
        int c = (mem[ p + 0xF080 ]) << 3;
        parentGraphics.drawImage(
          characterImages,
          col,
          row,
          col + 8,
          row + 16,
          c,
          0,
          c + 8,
          8,
          null );

        screenFlags[ p ] = false;
      }
    }

    parentGraphics.translate( -offsetX, -offsetY );
  }





  public void resetKeyboard()
  { 
    // TODO This is messy. Sort out encapsulation of key handlers
    _keyCharHandlers = new TreeMap<Character, KeyHanlder>();
    _keyCodeHandlers = new TreeMap<Integer, KeyHanlder>();
    _keyboard.reset();
    
    setupKeyHandlers();
    
    if(useHardKeyMap) {
      setupKeyCodeHandlers();
    }
    else {
      setupKeyCharHandlers();
    }
  }

  private interface KeyHanlder {
    public void doKey(final boolean down, final int code  );
  }
   
  private class CompoundKeyHandler implements KeyHanlder {
    private KeyHanlder[] _keyHandlers;
    
    public CompoundKeyHandler(final KeyHanlder kh1, final KeyHanlder kh2) {
      _keyHandlers = new KeyHanlder[] {kh1, kh2};
    }
    
    public CompoundKeyHandler(final KeyHanlder kh1, final KeyHanlder kh2, final KeyHanlder kh3) {
      _keyHandlers = new KeyHanlder[] {kh1, kh2, kh3};
    }
    
    public CompoundKeyHandler(final KeyHanlder[] keyHandler){
      _keyHandlers = keyHandler;
    }
    
    @Override
    public void doKey(final boolean down, final int code) {
      if(down) {
        for(int i = 0; i < _keyHandlers.length; ++i){
          _keyHandlers[i].doKey(down, code);
        }
      }
      else {
        for(int i = 0; i < _keyHandlers.length; ++i){
          _keyHandlers[_keyHandlers.length - i - 1].doKey(down, code);
        }        
      }
      
    }
  }
  
  private class KeyLineAndBitHandler implements KeyHanlder {
    private int _line;
    private int _bit;
    private String _desc;
    
    public KeyLineAndBitHandler(final int line, final int bit, final String desc) {
      _line = line;
      _bit = bit;
      _desc = desc;
    }

    @Override
    public void doKey(final boolean down, final int code) {
//      System.out.println("Key " + (down ? "pressed" : "released") + " line=" + _line + " bit=" + _bit + " description=" + _desc);
      _keyboard.addKeyEvent(_line, _bit, down, code);
    }
  }
  
  private class AmbiguousCharKeyHandler implements KeyHanlder {
    private KeyHanlder[] _keyHandlers;
    private int _keyCodes[];

    public AmbiguousCharKeyHandler(final int code1, final KeyHanlder kh1, final int code2, final KeyHanlder kh2) {
      _keyHandlers = new KeyHanlder[] {kh1, kh2};
      _keyCodes = new int[] {code1, code2};
    }
    
    @Override
    public void doKey(final boolean down, final int keyCode) {
      for(int i = 0; i < _keyHandlers.length; ++i) {
        if(_keyCodes[i] == keyCode) {
          _keyHandlers[i].doKey(down, keyCode);
          return;
        }
      }
       System.out.println("Error  could not disambiguate key " +keyCode);
      _keyHandlers[0].doKey(down, keyCode); 
    }    
  }
    
  private final class Keyboard 
  {

    // The state of the keyboard input lines must be maintained
    private int[][] _keyLines = new int[ 16 ][5];
    private int _keyScan = 0;
    
    private final class SorcererKeyEvent {
      private final int _line;
      private final int _bit;
      private final boolean _pressed;
      private final int _code;
      
      public SorcererKeyEvent(final int line, final int bit, final boolean pressed, final int code) {
        _line = line;
        _bit = bit;
        _pressed = pressed;
        _code = code;
      }
      
      public void apply() {
//        System.out.println("apply line=" + _line + " bit=" + _bit + " pressed=" + _pressed );
        _keyLines[ _line ][_bit] += _pressed ? 1 : -1;
      }
    }
    
    public void reset()
    {
      //
      // Initially all the keyline are high i.e. 1
      // There are 5 bits per row
      //
      for( int i = 0; i < _keyLines.length; ++i )
      {
        _keyLines[ i ][0] = 0;
        _keyLines[ i ][1] = 0;
        _keyLines[ i ][2] = 0;
        _keyLines[ i ][3] = 0;
        _keyLines[ i ][4] = 0;
      }
      
      /** Handle Keyboard */
      capslock = false;
    }
    
    /** Handle Keyboard */
    boolean capslock = false;

    public void setCapsLock()
    {
      capslock = true;
      _keyLines[ 0 ][3]++;
    }

    public void clearCapsLock()
    {
      capslock = false;
      _keyLines[ 0 ][3]--;
    }
    
    public void toggleCapsLock()
    {
      if( capslock )
      {
        clearCapsLock();
      }
      else
      {
        setCapsLock();
      }
    }
    
    public final void setScanLine(final int line) {
      _keyScan = line;
    }
    
    private LinkedList<SorcererKeyEvent> _fifo = new LinkedList<SorcererKeyEvent>();
    
    public final synchronized void addKeyEvent(final int line, final int bit, final boolean down, final int code) {
      _fifo.addLast(new SorcererKeyEvent(line, bit, down, code));
    }
    
    long _last = 0;
    
    public final synchronized int getKeyBits() {
    
      if(_fifo.size() > 0 && _last > 0) {
        final SorcererKeyEvent e = _fifo.peekFirst();
        
        if((e._line == _keyScan) || (_last > 16) ) {
          _fifo.removeFirst().apply();   
          // Add a delay after a change to shift etc
          _last = e._line == 0 ? -48 : 0;
        }
      }
      ++_last;
      
      return ((_keyLines[ _keyScan ][0] > 0) ? 0 : 1) +
             ((_keyLines[ _keyScan ][1] > 0) ? 0 : 2) +
             ((_keyLines[ _keyScan ][2] > 0) ? 0 : 4) +
             ((_keyLines[ _keyScan ][3] > 0) ? 0 : 8) +
             ((_keyLines[ _keyScan ][4] > 0) ? 0 : 16);
    }
  }
  
  private Keyboard _keyboard = new Keyboard();
  
  private Map<Character, KeyHanlder> _keyCharHandlers = new TreeMap<Character, KeyHanlder>();

  private void setupKeyCharHandlers() {
    _keyCharHandlers = new TreeMap<Character, KeyHanlder>();
    
    final KeyLineAndBitHandler shiftKeyhandler = new KeyLineAndBitHandler(0,4,"SHIFT"){
      @Override
      public void doKey(final boolean down, final int code) {
        super.doKey(down, KeyEvent.VK_SHIFT);
      }      
    };
    
    final KeyLineAndBitHandler ctrlKeyhandler = new KeyLineAndBitHandler(0,2,"CTRL"){
      @Override
      public void doKey(final boolean down, final int code) {
        super.doKey(down, KeyEvent.VK_CONTROL);
      }      
    };
    
    _keyCharHandlers.put('\033',                               new KeyLineAndBitHandler(0,0,"ESC"));
    _keyCharHandlers.put('\t',                                 new KeyLineAndBitHandler(1,3,"Tab"));

    final KeyLineAndBitHandler xKeyHandler = new KeyLineAndBitHandler(2,0,"X");
    _keyCharHandlers.put('X',                                  new CompoundKeyHandler(shiftKeyhandler, xKeyHandler));
    _keyCharHandlers.put('x',                                  xKeyHandler);
    _keyCharHandlers.put((char)24 ,                            new CompoundKeyHandler(ctrlKeyhandler,  xKeyHandler)); 

    final KeyLineAndBitHandler zKeyHandler = new KeyLineAndBitHandler(2,1,"Z");
    _keyCharHandlers.put('Z',                                  new CompoundKeyHandler(shiftKeyhandler, zKeyHandler));
    _keyCharHandlers.put('z',                                  zKeyHandler);
    _keyCharHandlers.put((char)26 ,                            new CompoundKeyHandler(ctrlKeyhandler,  zKeyHandler)); 

    final KeyLineAndBitHandler aKeyHandler = new KeyLineAndBitHandler(2,2,"A");
    _keyCharHandlers.put('A',                                  new CompoundKeyHandler(shiftKeyhandler, aKeyHandler));
    _keyCharHandlers.put('a',                                  aKeyHandler);
    _keyCharHandlers.put((char)01 ,                            new CompoundKeyHandler(ctrlKeyhandler,  aKeyHandler)); 
    
    final KeyLineAndBitHandler qKeyHandler = new KeyLineAndBitHandler(2,3,"Q");
    _keyCharHandlers.put('Q',                                  new CompoundKeyHandler(shiftKeyhandler, qKeyHandler));
    _keyCharHandlers.put('q',                                  qKeyHandler);
    _keyCharHandlers.put((char)17 ,                            new CompoundKeyHandler(ctrlKeyhandler,  qKeyHandler)); 

    final KeyLineAndBitHandler cKeyHandler = new KeyLineAndBitHandler(3,0,"C");
    _keyCharHandlers.put('C',                                  new CompoundKeyHandler(shiftKeyhandler, cKeyHandler));
    _keyCharHandlers.put('c',                                  cKeyHandler);
    _keyCharHandlers.put((char)03 ,                            new CompoundKeyHandler(ctrlKeyhandler,  cKeyHandler)); 
    
    final KeyLineAndBitHandler dKeyHandler = new KeyLineAndBitHandler(3,1,"D");
    _keyCharHandlers.put('D',                                  new CompoundKeyHandler(shiftKeyhandler, dKeyHandler));
    _keyCharHandlers.put('d',                                  dKeyHandler);
    _keyCharHandlers.put((char)04 ,                            new CompoundKeyHandler(ctrlKeyhandler,  dKeyHandler)); 
    
    final KeyLineAndBitHandler sKeyHandler = new KeyLineAndBitHandler(3,2,"S");
    _keyCharHandlers.put('S',                                  new CompoundKeyHandler(shiftKeyhandler, sKeyHandler));
    _keyCharHandlers.put('s',                                  sKeyHandler);
    _keyCharHandlers.put((char)19 ,                            new CompoundKeyHandler(ctrlKeyhandler,  sKeyHandler)); 
    
    final KeyLineAndBitHandler wKeyHandler = new KeyLineAndBitHandler(3,3,"W");
    _keyCharHandlers.put('W',                                  new CompoundKeyHandler(shiftKeyhandler, wKeyHandler));
    _keyCharHandlers.put('w',                                  wKeyHandler);
    _keyCharHandlers.put((char)23 ,                            new CompoundKeyHandler(ctrlKeyhandler,  wKeyHandler)); 

    final KeyLineAndBitHandler fKeyHandler = new KeyLineAndBitHandler(4,0,"F");
    _keyCharHandlers.put('F',                                  new CompoundKeyHandler(shiftKeyhandler, fKeyHandler));
    _keyCharHandlers.put('f',                                  fKeyHandler);
    _keyCharHandlers.put((char)06 ,                            new CompoundKeyHandler(ctrlKeyhandler,  fKeyHandler)); 

    final KeyLineAndBitHandler rKeyHandler = new KeyLineAndBitHandler(4,1,"R");
    _keyCharHandlers.put('R',                                  new CompoundKeyHandler(shiftKeyhandler, rKeyHandler));
    _keyCharHandlers.put('r',                                  rKeyHandler);
    _keyCharHandlers.put((char)18 ,                            new CompoundKeyHandler(ctrlKeyhandler,  rKeyHandler)); 

    final KeyLineAndBitHandler eKeyHandler = new KeyLineAndBitHandler(4,2,"E");
    _keyCharHandlers.put('E',                                  new CompoundKeyHandler(shiftKeyhandler, eKeyHandler));
    _keyCharHandlers.put('e',                                  eKeyHandler);
    _keyCharHandlers.put((char)05 ,                            new CompoundKeyHandler(ctrlKeyhandler,  eKeyHandler)); 

    final KeyLineAndBitHandler bKeyHandler = new KeyLineAndBitHandler(5,0,"B");
    _keyCharHandlers.put('B',                                  new CompoundKeyHandler(shiftKeyhandler, bKeyHandler));
    _keyCharHandlers.put('b',                                  bKeyHandler);
    _keyCharHandlers.put((char)02 ,                            new CompoundKeyHandler(ctrlKeyhandler,  bKeyHandler)); 

    final KeyLineAndBitHandler vKeyHandler = new KeyLineAndBitHandler(5,1,"V");
    _keyCharHandlers.put('V',                                  new CompoundKeyHandler(shiftKeyhandler, vKeyHandler));
    _keyCharHandlers.put('v',                                  vKeyHandler);
    _keyCharHandlers.put((char)22 ,                            new CompoundKeyHandler(ctrlKeyhandler,  vKeyHandler)); 

    final KeyLineAndBitHandler gKeyHandler = new KeyLineAndBitHandler(5,2,"G");
    _keyCharHandlers.put('G',                                  new CompoundKeyHandler(shiftKeyhandler, gKeyHandler));
    _keyCharHandlers.put('g',                                  gKeyHandler);
    _keyCharHandlers.put((char)07 ,                            new CompoundKeyHandler(ctrlKeyhandler,  gKeyHandler)); 

    final KeyLineAndBitHandler tKeyHandler = new KeyLineAndBitHandler(5,3,"T");
    _keyCharHandlers.put('T',                                  new CompoundKeyHandler(shiftKeyhandler, tKeyHandler));
    _keyCharHandlers.put('t',                                  tKeyHandler);
    _keyCharHandlers.put((char)20 ,                            new CompoundKeyHandler(ctrlKeyhandler,  tKeyHandler)); 

    final KeyLineAndBitHandler mKeyHandler = new KeyLineAndBitHandler(6,0,"M");
    _keyCharHandlers.put('M',                                  new CompoundKeyHandler(shiftKeyhandler, mKeyHandler));
    _keyCharHandlers.put('m',                                  mKeyHandler);
    _keyCharHandlers.put((char)13 ,                            new CompoundKeyHandler(ctrlKeyhandler,  mKeyHandler)); 

    final KeyLineAndBitHandler nKeyHandler = new KeyLineAndBitHandler(6,1,"N");
    _keyCharHandlers.put('N',                                  new CompoundKeyHandler(shiftKeyhandler, nKeyHandler));
    _keyCharHandlers.put('n',                                  nKeyHandler);
    _keyCharHandlers.put((char)14 ,                            new CompoundKeyHandler(ctrlKeyhandler,  nKeyHandler)); 

    final KeyLineAndBitHandler hKeyHandler = new KeyLineAndBitHandler(6,2,"H");
    _keyCharHandlers.put('H',                                  new CompoundKeyHandler(shiftKeyhandler, hKeyHandler));
    _keyCharHandlers.put('h',                                  hKeyHandler);
    
    final KeyLineAndBitHandler yKeyHandler = new KeyLineAndBitHandler(6,3,"Y");
    _keyCharHandlers.put('Y',                                  new CompoundKeyHandler(shiftKeyhandler, yKeyHandler));
    _keyCharHandlers.put('y',                                  yKeyHandler);
    _keyCharHandlers.put((char)25 ,                            new CompoundKeyHandler(ctrlKeyhandler,  yKeyHandler)); 

    final KeyLineAndBitHandler kKeyHandler = new KeyLineAndBitHandler(7,0,"K");
    _keyCharHandlers.put('K',                                  new CompoundKeyHandler(shiftKeyhandler, kKeyHandler));
    _keyCharHandlers.put('k',                                  kKeyHandler);
    _keyCharHandlers.put((char)11 ,                            new CompoundKeyHandler(ctrlKeyhandler,  kKeyHandler)); 

    final KeyLineAndBitHandler iKeyHandler = new KeyLineAndBitHandler(7,1,"I");
    _keyCharHandlers.put('I',                                  new CompoundKeyHandler(shiftKeyhandler, iKeyHandler));
    _keyCharHandlers.put('i',                                  iKeyHandler);

    final KeyLineAndBitHandler jKeyHandler = new KeyLineAndBitHandler(7,2,"J");
    _keyCharHandlers.put('J',                                  new CompoundKeyHandler(shiftKeyhandler, jKeyHandler));
    _keyCharHandlers.put('j',                                  jKeyHandler);

    final KeyLineAndBitHandler uKeyHandler = new KeyLineAndBitHandler(7,3,"U");
    _keyCharHandlers.put('U',                                  new CompoundKeyHandler(shiftKeyhandler, uKeyHandler));
    _keyCharHandlers.put('u',                                  uKeyHandler);
    _keyCharHandlers.put((char)21 ,                            new CompoundKeyHandler(ctrlKeyhandler,  uKeyHandler)); 

    _keyCharHandlers.put(',',                                  new KeyLineAndBitHandler(8,0,","));
    _keyCharHandlers.put('<',                                  new CompoundKeyHandler(shiftKeyhandler, new KeyLineAndBitHandler(8,0,"<")));

    final KeyLineAndBitHandler lKeyHandler = new KeyLineAndBitHandler(8,1,"L");
    _keyCharHandlers.put('L',                                  new CompoundKeyHandler(shiftKeyhandler, lKeyHandler));
    _keyCharHandlers.put('l',                                  lKeyHandler);
    _keyCharHandlers.put((char)12 ,                            new CompoundKeyHandler(ctrlKeyhandler,  lKeyHandler)); 

    final KeyLineAndBitHandler oKeyHandler = new KeyLineAndBitHandler(8,2,"O");
    _keyCharHandlers.put('O',                                  new CompoundKeyHandler(shiftKeyhandler, oKeyHandler));
    _keyCharHandlers.put('o',                                  oKeyHandler);
    _keyCharHandlers.put((char)15 ,                            new CompoundKeyHandler(ctrlKeyhandler,  oKeyHandler)); 

    _keyCharHandlers.put('1',                                  new AmbiguousCharKeyHandler(KeyEvent.VK_1, new KeyLineAndBitHandler(2,4,"1"), KeyEvent.VK_NUMPAD1, new KeyLineAndBitHandler(13,1,"NUMPAD1")));
    _keyCharHandlers.put('!',                                  new CompoundKeyHandler(shiftKeyhandler, new KeyLineAndBitHandler(2,4,"!")));
        
    _keyCharHandlers.put('2',                                  new AmbiguousCharKeyHandler(KeyEvent.VK_2, new KeyLineAndBitHandler(3,4,"2"), KeyEvent.VK_NUMPAD2, new KeyLineAndBitHandler(14,1,"NUMPAD2")));
    _keyCharHandlers.put('"',                                  new CompoundKeyHandler(shiftKeyhandler, new KeyLineAndBitHandler(3,4,"\"")));

    _keyCharHandlers.put('3',                                  new AmbiguousCharKeyHandler(KeyEvent.VK_3, new KeyLineAndBitHandler(4,4,"3"), KeyEvent.VK_NUMPAD3, new KeyLineAndBitHandler(15,4,"NUMPAD3")));
    _keyCharHandlers.put('#',                                  new CompoundKeyHandler(shiftKeyhandler, new KeyLineAndBitHandler(4,4,"#")));
    
    _keyCharHandlers.put('4',                                  new AmbiguousCharKeyHandler(KeyEvent.VK_4, new KeyLineAndBitHandler(4,3,"4"), KeyEvent.VK_NUMPAD4, new KeyLineAndBitHandler(13,2,"NUMPAD4")));
    _keyCharHandlers.put('$',                                  new CompoundKeyHandler(shiftKeyhandler, new KeyLineAndBitHandler(4,3,"$")));
    
    _keyCharHandlers.put('5',                                  new AmbiguousCharKeyHandler(KeyEvent.VK_5, new KeyLineAndBitHandler(5,4,"5"), KeyEvent.VK_NUMPAD5, new KeyLineAndBitHandler(14,2,"NUMPAD5")));
    _keyCharHandlers.put('%',                                  new CompoundKeyHandler(shiftKeyhandler, new KeyLineAndBitHandler(5,4,"%")));
    
    _keyCharHandlers.put('6',                                  new AmbiguousCharKeyHandler(KeyEvent.VK_6, new KeyLineAndBitHandler(6,4,"6"), KeyEvent.VK_NUMPAD6, new KeyLineAndBitHandler(14,3,"NUMPAD6")));
    _keyCharHandlers.put('&',                                  new CompoundKeyHandler(shiftKeyhandler, new KeyLineAndBitHandler(6,4,"&")));

    _keyCharHandlers.put('7',                                  new AmbiguousCharKeyHandler(KeyEvent.VK_7, new KeyLineAndBitHandler(7,4,"7"), KeyEvent.VK_NUMPAD7, new KeyLineAndBitHandler(13,4,"NUMPAD7")));
    _keyCharHandlers.put('\'',                                 new CompoundKeyHandler(shiftKeyhandler, new KeyLineAndBitHandler(7,4,"\'")));
    
    _keyCharHandlers.put('8',                                  new AmbiguousCharKeyHandler(KeyEvent.VK_8, new KeyLineAndBitHandler(8,4,"8"), KeyEvent.VK_NUMPAD8, new KeyLineAndBitHandler(13,3,"NUMPAD8")));
    _keyCharHandlers.put('(',                                  new CompoundKeyHandler(shiftKeyhandler, new KeyLineAndBitHandler(8,4,"(")));
   
    _keyCharHandlers.put('9',                                  new AmbiguousCharKeyHandler(KeyEvent.VK_9, new KeyLineAndBitHandler(8,3,"9"), KeyEvent.VK_NUMPAD9, new KeyLineAndBitHandler(14,4,"NUMPAD9")));
    _keyCharHandlers.put(')',                                  new CompoundKeyHandler(shiftKeyhandler, new KeyLineAndBitHandler(8,3,")")));
    
    
    _keyCharHandlers.put('/',                                  new KeyLineAndBitHandler(9,0,"/"));
    _keyCharHandlers.put('?',                                  new CompoundKeyHandler(shiftKeyhandler, new KeyLineAndBitHandler(9,0,"?")));
    _keyCharHandlers.put('.',                                  new KeyLineAndBitHandler(9,1,"."));
    _keyCharHandlers.put('>',                                  new CompoundKeyHandler(shiftKeyhandler, new KeyLineAndBitHandler(9,1,">")));
    _keyCharHandlers.put(';',                                  new KeyLineAndBitHandler(9,2,";"));

    final KeyLineAndBitHandler pKeyHandler = new KeyLineAndBitHandler(9,3,"P");
    _keyCharHandlers.put('P',                                  new CompoundKeyHandler(shiftKeyhandler, pKeyHandler));
    _keyCharHandlers.put('p',                                  pKeyHandler);
    _keyCharHandlers.put((char)16 ,                            new CompoundKeyHandler(ctrlKeyhandler,  pKeyHandler)); 

    _keyCharHandlers.put('0',                                  new AmbiguousCharKeyHandler(KeyEvent.VK_0, new KeyLineAndBitHandler(9,4,"0"), KeyEvent.VK_NUMPAD0, new KeyLineAndBitHandler(13,0,"NUMPAD0")));

    _keyCharHandlers.put('\\',                                 new KeyLineAndBitHandler(10,0,"\\"));
    _keyCharHandlers.put('|',                                  new CompoundKeyHandler(shiftKeyhandler, new KeyLineAndBitHandler(10,0,"|")));
    _keyCharHandlers.put('@',                                  new KeyLineAndBitHandler(10,1,"@"));
    _keyCharHandlers.put('`',                                  new CompoundKeyHandler(shiftKeyhandler, new KeyLineAndBitHandler(10,1,"`")));
    _keyCharHandlers.put('~',                                  new CompoundKeyHandler(shiftKeyhandler, new KeyLineAndBitHandler(11,3,"~")));
    
    _keyCharHandlers.put(']',                                  new KeyLineAndBitHandler(10,2,"]"));
    _keyCharHandlers.put('}',                                  new CompoundKeyHandler(shiftKeyhandler, new KeyLineAndBitHandler(10,2,"}")));
    _keyCharHandlers.put('[',                                  new KeyLineAndBitHandler(10,3,"["));
    _keyCharHandlers.put('{',                                  new CompoundKeyHandler(shiftKeyhandler, new KeyLineAndBitHandler(10,3,"{")));
    
//    _keyCharHandlers.put('\b',                                 new CompoundKeyHandler(shiftKeyhandler, new KeyLineAndBitHandler(11,0,"BACKSPACE")));
    _keyCharHandlers.put('\b',                                 new CompoundKeyHandler(ctrlKeyhandler, hKeyHandler));
    _keyCharHandlers.put('_',                                  new KeyLineAndBitHandler(11,0,"UNDERSCORE"));
    _keyCharHandlers.put('\n',                                 new KeyLineAndBitHandler(11,1,"ENTER"));

//    _keyCodeHandlers.put(new Integer(KeyEvent.VK_DOWN),        new KeyLineAndBitHandler(11,2,"DOWN"));
    
    _keyCharHandlers.put('^',                                  new KeyLineAndBitHandler(11,3,"^"));
    _keyCharHandlers.put(' ',                                  new KeyLineAndBitHandler(1,2," "));
    
    
    _keyCharHandlers.put('=',                                  new KeyLineAndBitHandler(15,3,"="));
    _keyCharHandlers.put('-',                                  new KeyLineAndBitHandler(11,4,"-"));
    _keyCharHandlers.put('+',                                  new KeyLineAndBitHandler(12,0,"+"));
    _keyCharHandlers.put(':',                                  new KeyLineAndBitHandler(10,4,":"));
    _keyCharHandlers.put('*',                                  new KeyLineAndBitHandler(12,1,"*"));
    
    _keyCharHandlers.put('.',                                  new KeyLineAndBitHandler(14,0,"."));

    _keyCodeHandlers.put(new Integer(KeyEvent.VK_F1),          new KeyLineAndBitHandler(0,1,"Graphics")); 
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_F10),         new KeyLineAndBitHandler(0,2,"CTRL"));
  }
  
  
  private Map<Integer, KeyHanlder> _keyCodeHandlers = new TreeMap<Integer, KeyHanlder>();
  
  private void setupKeyCodeHandlers() {
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_ESCAPE),      new KeyLineAndBitHandler(0,0,"ESC"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_F1),          new KeyLineAndBitHandler(0,1,"Graphics")); 
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_CONTROL),     new KeyLineAndBitHandler(0,2,"CTRL"));
// 0,3 shift lock
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_SHIFT),       new KeyLineAndBitHandler(0,4,"SHIFT"));

    _keyCodeHandlers.put(new Integer(KeyEvent.VK_F6),          new KeyLineAndBitHandler(1,0,"Clear")); 
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_F7),          new KeyLineAndBitHandler(1,1,"Repeat")); 
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_F8),          new KeyLineAndBitHandler(1,3,"Tab"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_HOME),        new KeyLineAndBitHandler(1,4,"Unknown"));

    _keyCodeHandlers.put(new Integer(KeyEvent.VK_X),           new KeyLineAndBitHandler(2,0,"X"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_Z),           new KeyLineAndBitHandler(2,1,"Z"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_A),           new KeyLineAndBitHandler(2,2,"A"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_Q),           new KeyLineAndBitHandler(2,3,"Q"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_1),           new KeyLineAndBitHandler(2,4,"1"));

    _keyCodeHandlers.put(new Integer(KeyEvent.VK_C),           new KeyLineAndBitHandler(3,0,"C"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_D),           new KeyLineAndBitHandler(3,1,"D"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_S),           new KeyLineAndBitHandler(3,2,"S"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_W),           new KeyLineAndBitHandler(3,3,"W"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_2),           new KeyLineAndBitHandler(3,4,"Z"));

    _keyCodeHandlers.put(new Integer(KeyEvent.VK_F),           new KeyLineAndBitHandler(4,0,"F"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_R),           new KeyLineAndBitHandler(4,1,"R"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_E),           new KeyLineAndBitHandler(4,2,"E"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_4),           new KeyLineAndBitHandler(4,3,"4"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_3),           new KeyLineAndBitHandler(4,4,"3"));

    _keyCodeHandlers.put(new Integer(KeyEvent.VK_B),           new KeyLineAndBitHandler(5,0,"B"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_V),           new KeyLineAndBitHandler(5,1,"V"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_G),           new KeyLineAndBitHandler(5,2,"G"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_T),           new KeyLineAndBitHandler(5,3,"T"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_5),           new KeyLineAndBitHandler(5,4,"5"));

    _keyCodeHandlers.put(new Integer(KeyEvent.VK_M),           new KeyLineAndBitHandler(6,0,"M"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_N),           new KeyLineAndBitHandler(6,1,"N"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_H),           new KeyLineAndBitHandler(6,2,"H"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_Y),           new KeyLineAndBitHandler(6,3,"Y"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_6),           new KeyLineAndBitHandler(6,4,"6"));

    _keyCodeHandlers.put(new Integer(KeyEvent.VK_K),           new KeyLineAndBitHandler(7,0,"K"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_I),           new KeyLineAndBitHandler(7,1,"I"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_J),           new KeyLineAndBitHandler(7,2,"J"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_U),           new KeyLineAndBitHandler(7,3,"U"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_7),           new KeyLineAndBitHandler(7,4,"7"));

    _keyCodeHandlers.put(new Integer(188),                     new KeyLineAndBitHandler(8,0,","));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_COMMA),       new KeyLineAndBitHandler(8,0,","));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_L),           new KeyLineAndBitHandler(8,1,"L"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_O),           new KeyLineAndBitHandler(8,2,"O"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_9),           new KeyLineAndBitHandler(8,3,"9"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_8),           new KeyLineAndBitHandler(8,4,"8"));

    _keyCodeHandlers.put(new Integer(191),                     new KeyLineAndBitHandler(9,0,"/"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_SLASH),       new KeyLineAndBitHandler(9,0,"/"));
    _keyCodeHandlers.put(new Integer(190),                     new KeyLineAndBitHandler(9,1,"Unknown"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_PERIOD),      new KeyLineAndBitHandler(9,1,"Unknown"));

    _keyCodeHandlers.put(new Integer(KeyEvent.VK_SEMICOLON),   new KeyLineAndBitHandler(9,2,";"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_P),           new KeyLineAndBitHandler(9,3,"P"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_0),           new KeyLineAndBitHandler(9,4,"0"));

    _keyCodeHandlers.put(new Integer(220),                     new KeyLineAndBitHandler(10,0,"\\"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_BACK_SLASH),  new KeyLineAndBitHandler(10,0,"\\"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_NUMBER_SIGN), new KeyLineAndBitHandler(10,1,"@")); // @`
    _keyCodeHandlers.put(new Integer(221),                     new KeyLineAndBitHandler(10,2,")"));
    _keyCodeHandlers.put(new Integer(93),                      new KeyLineAndBitHandler(10,2,")"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_BRACERIGHT),  new KeyLineAndBitHandler(10,2,")"));
    _keyCodeHandlers.put(new Integer(219),                     new KeyLineAndBitHandler(10,3,"("));
    _keyCodeHandlers.put(new Integer(91),                      new KeyLineAndBitHandler(10,3,"("));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_BRACELEFT),   new KeyLineAndBitHandler(10,3,"("));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_F12),         new KeyLineAndBitHandler(10,4,"Quote"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_QUOTE),       new KeyLineAndBitHandler(10,4,"Quote"));

    _keyCodeHandlers.put(new Integer(KeyEvent.VK_BACK_SPACE),  new KeyLineAndBitHandler(11,0,"BACK SPACE")); //_ del
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_ENTER),       new KeyLineAndBitHandler(11,1,"ENTER"));
    _keyCodeHandlers.put(new Integer(192),                     new KeyLineAndBitHandler(11,3,"^")); //^~
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_MINUS),       new KeyLineAndBitHandler(11,4,"-"));

    _keyCodeHandlers.put(new Integer(107),                     new KeyLineAndBitHandler(12,0,"+"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_PLUS),        new KeyLineAndBitHandler(12,0,"+"));
    _keyCodeHandlers.put(new Integer(106),                     new KeyLineAndBitHandler(12,1,"*"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_ASTERISK),    new KeyLineAndBitHandler(12,1,"*"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_DIVIDE),      new KeyLineAndBitHandler(12,2,"/"));
    _keyCodeHandlers.put(new Integer(189),                     new KeyLineAndBitHandler(12,3,"-"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_SUBTRACT),    new KeyLineAndBitHandler(12,3,"-"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_SPACE),       new KeyLineAndBitHandler(1,2,"SPACE"));

    _keyCodeHandlers.put(new Integer(KeyEvent.VK_NUMPAD0),     new KeyLineAndBitHandler(13,0,"NUMPAD0"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_NUMPAD1),     new KeyLineAndBitHandler(13,1,"NUMPAD1"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_NUMPAD4),     new KeyLineAndBitHandler(13,2,"NUMPAD4"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_NUMPAD8),     new KeyLineAndBitHandler(13,3,"NUMPAD8"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_NUMPAD7),     new KeyLineAndBitHandler(13,4,"NUMPAD7"));

    _keyCodeHandlers.put(new Integer(KeyEvent.VK_STOP),        new KeyLineAndBitHandler(14,0,"."));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_NUMPAD2),     new KeyLineAndBitHandler(14,1,"NUMPAD2"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_NUMPAD5),     new KeyLineAndBitHandler(14,2,"NUMPAD5"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_NUMPAD6),     new KeyLineAndBitHandler(14,3,"NUMPAD6"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_NUMPAD9),     new KeyLineAndBitHandler(14,4,"NUMPAD"));

    _keyCodeHandlers.put(new Integer(KeyEvent.VK_DELETE),      new KeyLineAndBitHandler(15,0,"DELETE"));
//    _keyCodeHandlers.put(new Integer(KeyEvent.VK_F4),          new KeyLineAndBitHandler(15,1,"Unknown"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_F5),          new KeyLineAndBitHandler(15,2,"Unknown"));
    
    _keyCodeHandlers.put(new Integer(187),                     new KeyLineAndBitHandler(15,3,"Numpad ="));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_EQUALS),      new KeyLineAndBitHandler(15,3,"Numpad ="));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_NUMPAD3),     new KeyLineAndBitHandler(15,4,"NUMPAD3"));
    
//    _keyCodeHandlers.put(new Integer(KeyEvent.VK_DOWN),        new KeyLineAndBitHandler(11,2,"DOWN"));
    
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_CAPS_LOCK), new KeyHanlder() {
      @Override
      public void doKey(boolean down, final int keyCode) {
        if( down )
        {
          _keyboard.toggleCapsLock();
        }
      }
    });      
  }
  
  private void setupKeyHandlers() {
    _keyCodeHandlers = new TreeMap<Integer, KeyHanlder>();
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_DOWN),        new KeyLineAndBitHandler(14,1,"NUMPAD2"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_UP),          new KeyLineAndBitHandler(13,3,"NUMPAD8"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_LEFT),        new KeyLineAndBitHandler(13,2,"NUMPAD4"));
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_RIGHT),       new KeyLineAndBitHandler(14,3,"NUMPAD6"));

    
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_F2), new KeyHanlder() {
      @Override
      public void doKey(boolean down, final int keyCode) {
        if( down )  toggleTrace();
      }
    });
        
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_END), new KeyHanlder() {
      @Override
      public void doKey(boolean down, final int keyCode) {
        if( down )
        {
          reset();
          PC( 0xE000 );
        }
      }
    });
    
    _keyCodeHandlers.put(new Integer(KeyEvent.VK_F3), new KeyHanlder() {
      @Override
      public void doKey(boolean down, final int keyCode) {
        if( down )
        {
          takeSnapshotNextInterrupt = true;
        }
      }
    });
  }
  
  
  public final boolean doKey( final  boolean down, final KeyEvent e )
  {
    final int code = e.getKeyCode();
    final Sorcerer.KeyHanlder charHandler = _keyCharHandlers.get(e.getKeyChar());
    if(charHandler != null) {
      charHandler.doKey(down, e.getKeyCode());
    }

    final Sorcerer.KeyHanlder handler = _keyCodeHandlers.get(code);
    if(handler != null) {
      handler.doKey(down, e.getKeyCode());
    }
    if(charHandler != null && handler != null) {
      System.out.println("Rehandled key event " + e);
    }
    return true;
  }

  public void loadROMs( URL baseURL ) throws IOException {
// This ROM causes character input problems with some games (CHOMP)
//    readBytes( new URL( baseURL, "roms/exmon12-1.dat" ), mem, 0xE000, 0x0800 );
    readBytes( new URL( baseURL, "roms/exmo1-1.dat" ), mem, 0xE000, 0x0800 );
// This ROM causes character input problems with some games (CHOMP)
//  readBytes( new URL( baseURL, "roms/exmon12-2.dat" ), mem, 0xE800, 0x0800 );
    readBytes( new URL( baseURL, "roms/exmo1-2.dat" ), mem, 0xE800, 0x0800 );
    readBytes( new URL( baseURL, "roms/exchr-1.dat" ), mem, 0xF800, 0x0400 );
    readBytes( new URL( baseURL, "roms/diskboot.dat" ), mem, 0xBC00, 0x0100 );

    setupScreenCharacters();

    reset();
    PC( 0xE000 );

  }

  public void load8kRomPack( final URL baseURL, final String romFileName ) throws IOException {
    final URL url = new URL( baseURL, romFileName );
    final InputStream is = url.openStream();
    load8kRomPack(is);
  }
  
  public void clear8kRomPack() {
    for(int i = 0; i < 0x2000; ++i ){
      mem[i+0xC000] = 0;
    }
  }
  
  private void setRomFlags() {    
    //
    // Initially set all the ROM flags to false.
    //
    for( int i = 0; i < romFlags.length; ++i )
    {
      romFlags[ i ] = false;
    }
    
    romFlags[ 56 ] = true;
    romFlags[ 57 ] = true;
    romFlags[ 58 ] = true;
    romFlags[ 59 ] = true;
    romFlags[ 62 ] = true;
    romFlags[ 47 ] = true;

    romFlags[ 48 ] = true;
    romFlags[ 49 ] = true;
    romFlags[ 50 ] = true;
    romFlags[ 51 ] = true;
    romFlags[ 52 ] = true;
    romFlags[ 53 ] = true;
    romFlags[ 54 ] = true;
    romFlags[ 55 ] = true;
  }
  
  public void load8kRomPack( final InputStream is ) throws IOException
  {
    readBytes( is, mem, 0xC000, 0x2000 );
  }

  private int readBytes( final URL url, final int a[], final int off, final int n ) throws IOException {
    final InputStream is = url.openStream();
    try 
    {
      return readBytes(is, a, off, n);
    }
    finally
    {
      is.close();
    }
  }
  
  private int readBytes( final InputStream is, final int a[], final int off, final int n ) throws IOException
  {
    BufferedInputStream bis = new BufferedInputStream( is, n );

    byte buff[] = new byte[ n ];
    int toRead = n;
    while ( toRead > 0 )
    {
      int nRead = bis.read( buff, n-toRead, toRead );
      if(nRead == -1) return -1;
      toRead -= nRead;
    }

    for ( int i = 0; i < n; i++ ) {
      a[ i+off ] = (buff[i]+256)&0xff;
    }

    return n;
  }

  private void writeBytes( OutputStream os, int a[], int off, int n ) throws IOException
  {
    for( int i = 0; i < n; ++i )
    {
      os.write( a[ off + i ] );
    }
  }

  public void saveSNP( OutputStream os ) throws IOException
  {
    byte header[] = new byte[28];

    exx();
    ex_af_af();

    header[ 0 ] = (byte)I();
    header[ 1 ] = (byte)L();
    header[ 2 ] = (byte)H();
    header[ 3 ] = (byte)E();
    header[ 4 ] = (byte)D();
    header[ 5 ] = (byte)C();
    header[ 6 ] = (byte)B();
    header[ 7 ] = (byte)F();
    header[ 8 ] = (byte)A();

    exx();
    ex_af_af();

    header[ 9  ] = (byte)L();
    header[ 10 ] = (byte)H();
    header[ 11 ] = (byte)E();
    header[ 12 ] = (byte)D();
    header[ 13 ] = (byte)C();
    header[ 14 ] = (byte)B();
    header[ 15 ] = (byte)( IY() & 0xff );
    header[ 16 ] = (byte)( ( IY() >> 8 ) & 0xff );
    header[ 17 ] = (byte)( IX() & 0xff );
    header[ 18 ] = (byte)( ( IX() >> 8 ) & 0xff );
    header[ 19 ] = (byte)0;
    if( IFF2() )
    {
      header[ 19 ] |= (byte)0x04;
    }
    if( IFF1() )
    {
      header[ 19 ] |= (byte)0x01;
    }
    header[ 20 ] = (byte)R();
    header[ 21 ] = (byte)F();
    header[ 22 ] = (byte)A();
    header[ 23 ] = (byte)( SP() & 0xff );
    header[ 24 ] = (byte)( ( SP() >> 8 ) & 0xff );
    header[ 25 ] = (byte)IM();
    header[ 26 ] = (byte)( PC() & 0xff );
    header[ 27 ] = (byte)( ( PC() >> 8 ) & 0xff );

    os.write( header );

    writeBytes( os, mem, 0, 65536 );

    os.close();
  }

  public void loadSNP(final URL url) throws IOException 
  {
    final InputStream is = url.openStream();
    try
    {
      loadSNP(is);      
    }
    finally
    {
      is.close();
    }
  }
  
  public void loadSNP(final InputStream is ) throws IOException
  {
    int header[] = new int[28];

    readBytes( is, header, 0,    28 );
    readBytes( is, mem,    0, 65536 );

    I( header[0] );

    HL( header[1] | (header[2]<<8) );
    DE( header[3] | (header[4]<<8) );
    BC( header[5] | (header[6]<<8) );
    AF( header[7] | (header[8]<<8) );

    exx();
    ex_af_af();

    HL( header[9]  | (header[10]<<8) );
    DE( header[11] | (header[12]<<8) );
    BC( header[13] | (header[14]<<8) );

    IY( header[15] | (header[16]<<8) );
    IX( header[17] | (header[18]<<8) );

    if ( (header[19] & 0x04)!= 0 )
    {
      IFF2( true );
    }
    else {
      IFF2( false );
    }

    if ( (header[19] & 0x02)!= 0 )
    {
      IFF1( true );
    }
    else {
      IFF1( false );
    }

    R( header[20] );

    AF( header[21] | (header[22]<<8) );
    SP( header[23] | (header[24]<<8) );

    switch( header[25] ) {
    case 0:
      IM( IM0 );
      break;
    case 1:
      IM( IM1 );
      break;
    default:
      IM( IM2 );
      break;
    }

    PC( header[26] | (header[27]<<8) );

    setupScreenCharacters();
  }
}



