/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2000 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   DecideUSPowerCutback_bis.c
 *
 *   Functions implementing states in the Channel Analysis phase of
 *   Initialization for RX.
 *
 *
 *
 *------------------------------------------------------------------------
 */

#include "common.h"
#include "rt_tones.h"
#include "snr.h"
#include "gdata.h"
#include "gdata_bis.h"

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: int16 DecideUSPowerCutback(int32 *pla_EchoPower)
 *
 *  Description: Decide absolute upstream power cutback based on
 * a nonlinear echo power measurement for different levels
 *  of TX power. The US power cutback level is chosen according to the following
 * algorithm:
 *
 * If the total nonlinear echo power (measured for 2nd and 3rd harmonics of line probe tones)
 *     is less than a preselected absolute threshold for at least one Tx cutback level,
 *    the smallest such Tx cutback level (highest Tx power) is chosen,
 *
 * else:
 *
 *    The differences in echo power for successive Tx cutback levels is considered.
 *    If the reduction in echo power  between two adjacent TX power levels is smaller than
 *    a predetermined fraction of the echo power, the disadvantage of the higher cutback
 *    is considered to outweight the benefit of the reduced echo, and the lower cutback (higher
 *    Tx power level) is chosen.
 *
 * else
 *    The maximum Tx power cutback is chosen.
 *
 *------------------------------------------------------------------------
 *
 *  Input Arguments:
 *      pla_EchoPower -- pointer to an array of echo powers each of which
 *                is correspoding to one TX power level
 *
 *  Output Arguments:
 *    none
 *
 *  Return:
 *    index which indicates the US power cutback level
 *
 *  Global Variables Used:
 *    gl_NLEchoThresh_bis -- Absolute nonlinear echo power threshold
 *    gs_NLEchoDiffThresh_bis -- Difference threshold of nonlinear echo power (mantissa)
 *    gs_NLEchoDiffThresh_rsh_bis -- exponent (right shift count) of gs_NLEchoDiffThresh_rsh
 *------------------------------------------------------------------------
 *^^^
 */



int16 DecideUSPowerCutback(int32 *pla_EchoPower)
{
   int16 i, s_thre, s_diff;
   int32 l_Acc;
   int16 sa_EchoPower[NUM_LINEPROBE_CUTBACK_CNT];

   for(i=0; i<NUM_LINEPROBE_CUTBACK_CNT; i++){

      // Need to convert digital power values to absolute dBm.
      sa_EchoPower[i] = ConvertToDB(pla_EchoPower[i]);

      //If nonlinear echo power is less or equal to the absolute threshold
      //choose this power cutback level.
      if(sa_EchoPower[i] <= gs_NLEchoThresh_bis)
         return(i);
   }

   // If nonlinear echo power is above threshold for all cutback levels,
   // examine echo power values and decide if at any point the marginal decrease in echo power
   // is too small to warrant the marginal power cutback.  If such a point is found
   // it determines the power cutback.  If such a point is not found the power cutback
   // will be set to the maximum possible.

   for(i=1; i<NUM_LINEPROBE_CUTBACK_CNT; i++) {

      //Compute the difference threshold which is a percentage
      //of echo power of previous step.
      l_Acc = sa_EchoPower[i-1]*gs_NLEchoDiffThresh_bis; //mutliply it by percentage

      s_thre = (int16)(l_Acc>>gs_NLEchoDiffThresh_rsh_bis); //convert back to 16-bit

      //Compute absolute value of (sa_EchoPower[i-1] - sa_EchoPower[i])
      s_diff = sa_EchoPower[i-1] - sa_EchoPower[i];
      if(s_diff < 0)
         s_diff = - s_diff;

      // If the absolute difference is less than the threshold, marginal echo power reduction
      // is not considered worth the Tx power reduction, so choose the previous cutback level.
      if(s_diff <= s_thre)
         return(i-1);

   } //for(i=1; i<NUM_LINEPROBE_CUTBACK_CNT; i++)

   // Choose maximum power cutback.

   return(NUM_LINEPROBE_CUTBACK_CNT-1);
}



