/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2001 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   logtolin.c
*
*   functions to convert logtolin for test-parameters specific for G.bis Diagnositics.
*
*-------------------------------------------------------------------------
*/

#include "common.h"

#include "stdio.h"
#include "vecpwr.h"
#include "gdata_bis.h"
#include "dsp_op.h"
#include "diagparam_bis.h"
#include <string.h>                    //for memset
#include "noiseacc.h"

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : logtolin()
 *
 *  Description:  Converts input value s_value[0] in dB Q8.8 to linear scale as
 *          mantissa s_value[0] and exponent s_exp_out[0].  Intermediate Q4.28
 *          value representation limits the  input s_value[0] to be within
 *          the range [-20log10(2^28), 20log10(8)]  = [-168.5768 18.0618].
 *          Returns 10^(s_value[0]/20) with in-place operation as
 *                s_value[0]*2^-s_expout[0]
 *
 *  Prototype:
 *       void logtolin(int16 *s_value, int16 *s_exp_out, FlagT ft_getexpflag)
 *
 *  Input/Output Arguments:
 *      int16  *s_value    - pointer to input/output frequency sample Q8.8
 *      int16  *s_exp_out  - pointer to output exponent for output s_sample
 *    FlagT ft_getexpflag  - flag to determine whether this sample is used to
 *                   determine s_exp_out (i.e. maximum sample in array)
 *                   or simply return result with given s_exp_out (ft_getexpflag=0)
 *
 *  Output Arguments:
 *
 *  Return:
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

C_SCOPE void logtolin(int16 *s_value, int16 *s_exp_out, FlagT ft_getexpflag)
{
   int16 sa_pow10xCoef[POW10X_ORDER+1];
   int16 s_ScaleBy10pow;
   int16 s_x, s_y, j, s_in, s_leftshift = 0;
   int32 l_acc0, l_acc1, l_ScaleDenom = 0;

   // Initialize pow(10,x) coefficient array
   sa_pow10xCoef[0] = POW10X_C0_MANT;
   sa_pow10xCoef[1] = POW10X_C1_MANT;
   sa_pow10xCoef[2] = POW10X_C2_MANT;
   sa_pow10xCoef[3] = POW10X_C3_MANT;
   sa_pow10xCoef[4] = POW10X_C4_MANT;


   if(!ft_getexpflag)
      s_leftshift = *s_exp_out - (28-16);  // get amount to left-shift results

   // Only perform conversion from dB to linear scale if non-zero value
   if(*s_value == 0)
   {
      //Normalize (int32)0x1 (Q4.28) and represent it in 16-bit
      if(ft_getexpflag)
      {  // if s_max find s_x_exp_out
         s_leftshift = (30-28);    //norm_l((int32)(0x1<<28));
      }

      // s_x = 1.0*2^-(28+s_x_exp_out-16)
      *s_value = (int16)(0x1<<(28+(s_leftshift)-16));

   }
   // Range of *s_value that can be represented in intermediate Q4.28 representation
   // is [-20log10(2^28), 20log10(8)]  = [LOG2LINMIN LOG2LINMAX] = [-168.5768 18.0618]
   // The lower bound is larger than what can be repesented with input Q8.8.
   else if(*s_value>(int16)LOG2LINMAX)
   {
      // Saturate to maximum intermediate value in (int16)(0x7fffffff>>16).
      // Set exponent to what it would be if the input was LOG2LINMAX.
      // Compensated result will be out of range to be represented in Diagnostics
      *s_value = MAX_16;
      s_leftshift = 0;
   }
#if 0  // minimum allowed value can not be represented in Q8.8
   else if(*s_value<(int16)LOG2LINMIN)
   {
      // Saturate to minimum intermediate value in (int16)(0x00000001>>16).
      // Set exponent to what it would be if the input was LOG2LINMAX.
      // Compensated result will be out of range to be represented in Diagnostics

      //Normalize (int32)0x00000001 (Q4.28) and represent it in 16-bit
      if(ft_getexpflag)
      { // if s_max find s_x_exp_out
         s_leftshift = (30);    //norm_l((int32)(0x1));
      }

      // s_x = 2^-28*2^-(s_x_exp_out-16)
      if(s_leftshift >= 16)
         *s_value = (int16)(0x1<<((s_leftshift)-16));
      else
         *s_value = 0;

   }
#endif
   else
   {
      // Need to scale *s_value by by 1/20 for dB definition to get|x|
      // Apply this scale following check if *s_value is in range [-20,0] in Q8.8.
      // since the pow(10,x) performed in this function is only valid for -1<x<=0;


      //If *s_value < -20 in Q8.8, then perform (with y = *s_value):
      //x=pow(10,floor((y+1)/(256*20))) * pow(10,((y+1)/(256*20) - floor((y+1)/(256*20))))
      //First pow(10,integer) simple multiplication. Second term can use this function.
      if (*s_value <= (-256*20))
      {
         if(*s_value == (int16)MIN_16)
         {
            // can not use computation below
            s_ScaleBy10pow = 6;
         }
         else
         {
            // round in the right direction to handle border cases (*s_value = N*20)
            // Use of parenthesis below is important for input equalt to MIN_16+1
            s_ScaleBy10pow = (int16) -((*s_value-1)/(256*20));  //final scaling 10^-s_ScaleBy10pow
         }

         l_ScaleDenom = 10;
         for(j=1; j < s_ScaleBy10pow; j++)
            l_ScaleDenom *= 10;
      }
      else if (*s_value > 0)
      {
         /* For range of input allowed, max input = (LOG2LINMIN/20) < 1 */
         s_ScaleBy10pow = -1;
      }
      else
      {
         s_ScaleBy10pow=0;          //final scaling 10^-s_ScaleBy10pow=1
      }

      s_in = *s_value + (256*s_ScaleBy10pow*20);


      // Convert to Q1.15 format (mult by 128/20 or 32/5)
      s_x = s_in / 5;
      s_x <<= 5;

      /***********************************************************************/
      /* Compute pow(10,s_x) = C4*s_x^4 + C3*s_x^3 + C2*s_x^2 + C1*s_x + C0  */
      /***********************************************************************/

      // Compute C0
      l_acc0 = ((int32)sa_pow10xCoef[0])<<15; // Left shift C0 by 15 to convert C0 from Q3.13 to Q4.28

      // Initialize s_y to s_x
      s_y = s_x;

      // Compute (C4*x^4 + C3*x^3 + C2*x^2 + C1*x + C0) in Q4.28
      // Each iteration of outer loop generates C(j)*x^j
      for(j=1;j<=POW10X_ORDER;j++)
      {
         if(j>1)
         {
            l_acc1 = ((int32)s_y * s_x);
            s_y = (int16)round(l_acc1, 15);
         }

         // Compute C(j)*x^j in Q4.28
         l_acc0 += ((int32)sa_pow10xCoef[j] * s_y);
      }

      // Scale by 10^-s_ScaleBy10pow
      if(s_ScaleBy10pow > 0)
         l_acc0 /= l_ScaleDenom;
      else if(s_ScaleBy10pow == -1)
         l_acc0 *= 10;

      //Normalize l_acc0 (Q4.28) and represent it in 16-bit
      if(ft_getexpflag)
      {  // if s_max find s_x_exp_out
         s_leftshift = norm_l(l_acc0);
      }
      l_acc0 = l_acc0<<s_leftshift;

      // Extract upper word for output value
        // s_x = l_acc0*2^-(28+s_x_exp_out-16)
      *s_value = (int16)round(l_acc0, 16);
   }


   if(ft_getexpflag)
   {  // if s_max find s_x_exp_out
      // set true total s_x_exp_out
      *s_exp_out = s_leftshift + (28-16);
   }
}
