/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 2003 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   TSSI_RX.C
 *
 *   Description:   Routines used for generating the inverse DS tssi data then
 *    applying it to the DS signal for estimating the DS channel impulse response.
 *
 *------------------------------------------------------------------------
 */
// ******************************************************************
// tssi_rx.c
//
// History
// 07/09/2011 AdeelJ: Set DS TSSi values to 0 if Looplength is higher than 9kft
//              and the CO Sends >100dB attenuation for tone indexes < 60.
//              This high Attenuation value causes problems when we scale the receive
//              signal by the inverse TSSi value for Frame Alignment etc. the lower tone
//              indices usually have noise power. At high looplengths, this noise power is
//              comparable to to the valid signal power on the DS tones. when this is Scaled up
//              by the inverse TSSi, then the valid signal is scaled down, and we get a high
//              Power in the lower frequencies, this disrupts our frame alignment calculation
//              as we cannot fish out the Channel Impulse Response energy out of the Low
//              Frequency noise. This was resulting in Bad MEDLEY SNR and Long Training Times.
//              We discovered a similar fix in Vinax Code and This fix was ported to The
//              ASE/DNB/AR9 Code Base Since the code has been tested against different DSLAMs.
//              This is not cmv controlled and not limited to specific DSLAMs.
//           Grep for XDSLRTFW-231 IOP_A_DS_BisPlus_LongLoopTSSIModificationFrameAlignment
//
// 14/04/2014 Hanyu: Ported ADSLRTFW-1391: Modified the DS TSSi IOP fix in ADSLRTFW-1347 (XDSLRTFW-231) for Ciena CO CTNW on
//              long loop over 15 kft to improve training times in ADSL2/2+ against CNX5 at AT&T.
//           Grep for XDSLRTFW-1595 ADSLRTFW-1391 IOP_DS_BisPlus_CTNW_ModifyTssiLongLoop
//
// ******************************************************************
#include "common.h"
#include "gdata.h"
#include "gdata_bis.h"
#include "spectral_shape_bis.h"
#include "const.h"
#include "bitload_const.h"
#include "cmv.h"
#include "dsp_op.h"
#include "dsp_op2.h"
#include "tssi_rx.h"

/*^^^
*------------------------------------------------------------------------
*
*  Name : BgCalculateTssi
*
*   Prototype:
*       void BgCalculateTssi(void)
*
*   Abstract:
*     Does two things:
*     1) Call to function that defines the US supported set.
*     2) Call to function that calculates DS inverse tssi.
  *-----------------------------------------------------------------------------
^^^*/
void BgCalculateTssi(void)
{
   DefineUsTssiSupportedSet();
   DefineDsTssiSupportedSet();

   guc_BkgdTaskState = TRAINING_DONE;
}

/*^^^
*------------------------------------------------------------------------
*
*  Name : DefineDsTssiSupportedSet
*
*   Prototype:
*       void DefineDsTssiSupportedSet(void)
*
*   Abstract:
*     This function defines the US supported set array based on the US breakpoints.
*     and also interpolates the them to define US tssi
*
*     Tssi values are represented in logarithmic scale as a 7 bit unsigned value
*     in -0.5 dB steps ranging from 0 to -62.5 dB (value 125). Value 126 is a special
*     value, indicating the log_tssi should be calculated by interpolating between the
*     two neighboring breakpoints.  Value 127 is another special
*     value, indicating the sub-carrier is not transmitted (i.e. tssi=0 in linear scale).
*
*     The tssi values for all subcarriers are interpolated from samples of frequency
*     breakpoints and the log of the tssi^2 values.
*
*     The interpolation follows 3 rules:
*
*     1. Flat extension to low frequencies from lowest frequency breakpoint
*     2. Flat extension to high frequencies from highest frequency breakpoint
*     3. For a given frequency which falls in between two consecutive
*        breakpoints A and B, interpolate as follows:
*
*        for A < i < B,
*        tss(i) = tss(A) + slope * (i - A)
*
*        where    slope = (tss(B) - tss(A)) / (B - A);
*
*     The interpolated log tssi values are converted to a linear scale then inverted
*     later in Channel Discovery after we have determined the DS power cutback
*
*   Input Parameters:
*
*  Global Variables:
*     gt_RxInfoSave.t_USPMD_Params.usa_3A_BreakFreq - tssi breakpoint frequencies
*     gt_RxInfoSave.t_USPMD_Params.usa_3A_tssi - log tssi values
*     gusa_DS_Inverse_Tssi_Value - inverse US tssi values
*-----------------------------------------------------------------------------
^^^*/
FlagT gft_ANFPNotchPresent = 0;
#define NOTCH_NUM_TONES_THRESHOLD 30
void DefineDsTssiSupportedSet(void)
{
   int i, s_NumTonesInShallowNotch, s_NumTonesInDeepNotch, s_NumTonesInANFPNotch, s_Notch_StartTone;
   uint8* puca_DS_Tssi;
   int16* psa_DS_BreakFreq;
   FlagT *fta_DS_SupportSet;
   //XDSLRTFW-231 IOP_A_DS_BisPlus_LongLoopTSSIModificationFrameAlignment (START_END)
   FlagT ft_ClearDSTssiHighAttenuationLowFreq = 0;
   int16 s_CombSetSize, s_NumCombSubcarriersInspected = NUM_COMB_SUBCARRIERS_INSPECTED_ADSL2;

   // Attach pointers to Handshake tssi arrays
   // STAT_ConfigMode_G992_3x
   if (( gl_SelectedMode & (MODE_G992_3)  ))
   {
      s_CombSetSize = 16;
   }
   // STAT_ConfigMode_G992_5x
   else
   {
      s_CombSetSize = 26;
   }

   psa_DS_BreakFreq = (int16 *)(void *)gt_GHS_DSTssiInfo.usa_Brkpt;
   puca_DS_Tssi = gt_GHS_DSTssiInfo.uca_Tssi;
   fta_DS_SupportSet = gt_GHS_DSTssiInfo.uca_Si;

   /* Hack to get around BRCM bug in setting tssi for notch with no mask profile case     */
   /* This bug present in CO v4.5.11, but fixed in 4.5.12, 4.6 and later               */
   /* Specifically, BRCM sends a breakpoint index of 32 with log tssi value of 80(-40dB)  */
   /* This is a bug and hence we hack through it on the CPE side by forcing log tssi value   */
   /* for tone index 32 as 0                                               */
   if (gs_CurrentCoChipset == BDCM_CO_CHIPSET)
   {
      for(i = 0; i < gt_GHS_DSTssiInfo.s_NumBrkptkept; i++)
      {
         if (psa_DS_BreakFreq[i] == 32 &&
             (puca_DS_Tssi[i] >= 50 && puca_DS_Tssi[i] != INTERP_LOG_TSSI_CODE
              && puca_DS_Tssi[i] != NO_XMIT_TSSI_CODE))
         {
            puca_DS_Tssi[i] = 0;
         }
      }
   }
   //XDSLRTFW-231 IOP_A_DS_BisPlus_LongLoopTSSIModificationFrameAlignment (START)
#ifndef ISDN      // Annex A
//XDSLRTFW-1595 ADSLRTFW-1391 IOP_DS_BisPlus_CTNW_ModifyTssiLongLoop (START)
//DS TSSI from UNH CNX5 in ADSL2+:
//breakpoint = [9,10,28,33,36,256,376,510,511];
//attenuation (dB) = [63.5,30,20,15,0,0,10,11.5,11.5];
//DS TSSI from AT&T CNX5 on loop 16kft in READSL:
//breakpoint = [9,10,28,33,36,150,255,258,263,281,282];
//attenuation (dB) = [63.5,30,20,15,0,0,0,15,20,30,63.5];
// The CNX5 DS TSSi on lower band must be igonored to avoid long training times.
if ( (gs_CurrentCoChipset == CTNW_CO_CHIPSET) &&
     (gs_hsk_tone_power_dB_PCB <= MINIMUM_GHS_TONE_PWR_AT_15000F) )  // Loop >= 15kft
 {
  for(i = 0; i < gt_GHS_DSTssiInfo.s_NumBrkptkept; i++)
  {
   if ( (psa_DS_BreakFreq[i] <= 200) &&       //tone number
        ( (puca_DS_Tssi[i]> 0)         &&       //Tssi Value (Atten dB)
        (puca_DS_Tssi[i]!=INTERP_LOG_TSSI_CODE) ))      //Avoid Special interpolation value  126
                                                        //But Include Special Max Atten Value 127
     {
       puca_DS_Tssi[i] = 0;
     }
  }
 } // if ( (gs_CurrentCoChipset == CTNW_CO_CHIPSET) &&
 else
 {
   for(i = 0; i < gt_GHS_DSTssiInfo.s_NumBrkptkept; i++)
   {
      if ( (psa_DS_BreakFreq[i] <= 60) &&                //tone number
         ( (puca_DS_Tssi[i]>=100)                  &&       //Tssi Value (Atten dB)
            (puca_DS_Tssi[i]!=INTERP_LOG_TSSI_CODE) ))      //Avoid Special interpolation value    126
      {
           ft_ClearDSTssiHighAttenuationLowFreq = 1;

      }
   }
   if ((ft_ClearDSTssiHighAttenuationLowFreq==1)  &&
      (gs_hsk_tone_power_dB_PCB < MINIMUM_GHS_TONE_PWR_AT_9000F))
   {
      for(i=0; i<gt_GHS_DSTssiInfo.s_NumBrkptkept; i++)
      {
         puca_DS_Tssi[i] = 0;
      }

   }
 } // else
#endif //ifndef ISDN
//XDSLRTFW-1595 ADSLRTFW-1391 IOP_DS_BisPlus_CTNW_ModifyTssiLongLoop (End)
   //XDSLRTFW-231 IOP_A_DS_BisPlus_LongLoopTSSIModificationFrameAlignment (END)

   // Initialization, clear DS SupportedSet and MedleySet Arrays
   for (i=0 ; i<RX_NUM_TONES ; i++){
      CLEARTONEFLAG(p_MEDLEYset_DS, i);
      CLEARTONEFLAG(p_SUPPORTEDset_DS, i);
   }

   // Define DS Log Tssi
   DefineLogTssiSet(gusa_DS_Tssi_Value, gs_RxNumTones, psa_DS_BreakFreq, puca_DS_Tssi, gt_GHS_DSTssiInfo.s_NumBrkptkept);

   // Define DS SupportedSet and MedleySet
   DefineSupportedSet(p_SUPPORTEDset_DS, gs_RxNumTones, psa_DS_BreakFreq, fta_DS_SupportSet, gt_GHS_DSTssiInfo.s_NumBrkptkept);

   //Initialize # of tones that could be in a notch
   s_NumTonesInDeepNotch = 0;
   s_NumTonesInShallowNotch = 0;
   s_NumTonesInANFPNotch = 0;
   s_Notch_StartTone = 350;
   gs_firstSupportedTone = 0;
   // DS MEDLEYset ~ (DS SUPPORTEDset - DS BLACKOUTTones)
   for (i=0 ; i<gs_RxNumTones ; i++){
      if (IS_TONEFLAGSET(p_SUPPORTEDset_DS, i) && !IS_TONEFLAGSET(guca_RMsgPCBTab, (32+i)))
      {
         SETTONEFLAG(p_MEDLEYset_DS, i);
         if ( gs_firstSupportedTone == 0 )  gs_firstSupportedTone = i ;
         //check Hamband notch
         if((i > s_Notch_StartTone) && (gusa_DS_Tssi_Value[i] > gs_tssi_ShallowNotchthreshold))
         {
            //check whether there is shallow notch presenting in the supported set, NOTCH_TSSI_THRESHOLD is 0x3C00 (30dB)
            s_NumTonesInShallowNotch++;
            //check whether there is deep notch presenting in the supported set, NOTCH_TSSI_THRESHOLD is 0x6400 (50dB)
            if(gusa_DS_Tssi_Value[i] > gs_tssi_DeepNotchthreshold)
            {
               s_NumTonesInDeepNotch++;
            }
         }

         //check ANFP notch
         if ((i <= s_Notch_StartTone) && (gusa_DS_Tssi_Value[i] > gs_tssi_DeepNotchthreshold))
         {
            //check whether
            s_NumTonesInANFPNotch++;
         }
      }
      else
         CLEARTONEFLAG(p_MEDLEYset_DS, i);
   }

   //deep notch has higher priority
   if (s_NumTonesInDeepNotch > NOTCH_NUM_TONES_THRESHOLD)
      gft_NotchPresent = 1;
   else if (s_NumTonesInShallowNotch > NOTCH_NUM_TONES_THRESHOLD)
      gft_NotchPresent = 2;

   if (s_NumTonesInANFPNotch > NOTCH_NUM_TONES_THRESHOLD)
      gft_ANFPNotchPresent = 1;

   // Define set of tones used for pre-Comb2SNR detections in channel discovery.
   gs_NumCombSubcarriersInspected=0;

   for (i=2; (i < s_CombSetSize && gs_NumCombSubcarriersInspected < s_NumCombSubcarriersInspected); i++)
   {
      if ((IS_TONEFLAGSET(p_SUPPORTEDset_DS, gsa_C_COMB_index[i])) &&
         gusa_DS_Tssi_Value[gsa_C_COMB_index[i]] < gs_tssi_threshold)
      {
         gsa_Inspected_COMB_Subcarriers[gs_NumCombSubcarriersInspected++] = gsa_C_COMB_index[i];
      }
   }
}

/*^^^
*------------------------------------------------------------------------
*
*  Name : DefineUsTssiSupportedSet
*
*   Prototype:
*       void DefineUsTssiSupportedSet(void)
*
*   Abstract:
*     This function defines the US supported set array based on the US breakpoints.
*     and also interpolates the them to define US tssi
*
*     Tssi values are represented in logarithmic scale as a 7 bit unsigned value
*     in -0.5 dB steps ranging from 0 to -62.5 dB (value 125). Value 126 is a special
*     value, indicating the log_tssi should be calculated by interpolating between the
*     two neighboring breakpoints.  Value 127 is another special
*     value, indicating the sub-carrier is not transmitted (i.e. tssi=0 in linear scale).
*
*     The tssi values for all subcarriers are interpolated from samples of frequency
*     breakpoints and the log of the tssi^2 values.
*
*     The interpolation follows 3 rules:
*
*     1. Flat extension to low frequencies from lowest frequency breakpoint
*     2. Flat extension to high frequencies from highest frequency breakpoint
*     3. For a given frequency which falls in between two consecutive
*        breakpoints A and B, interpolate as follows:
*
*        for A < i < B,
*        tss(i) = tss(A) + slope * (i - A)
*
*        where    slope = (tss(B) - tss(A)) / (B - A);
*
*     The interpolated log tssi values are then converted to a linear scale.
*
*   Input Parameters:
*
*  Global Variables:
*
*-----------------------------------------------------------------------------
^^^*/
void DefineUsTssiSupportedSet(void)
{
   int i;
   uint8 *puca_US_Tssi;
   FlagT *pfta_US_SprtSet;
   int16 usa_US_BreakFreq[MAX_NUM_US_TSSI_VALUES];

   for (i = 0; i < gt_GHS_USTssiInfo.s_NumBrkptkept; i++)
      usa_US_BreakFreq[i] = gt_GHS_USTssiInfo.uca_Brkpt[i];

   puca_US_Tssi = gt_GHS_USTssiInfo.uca_Tssi;
   pfta_US_SprtSet = gt_GHS_USTssiInfo.uca_Si;

   // Initialization, clear US SupportedSet and MedleySet Arrays
   for (i=0 ; i<TX_NUM_TONES ; i++){
      CLEARTONEFLAG(gp_MEDLEYset_US, i);
      CLEARTONEFLAG(gp_SUPPORTEDset_US, i);
   }

   // Define US Log Tssi
   DefineLogTssiSet(gusa_US_Tssi_Value, TX_NUM_TONES, usa_US_BreakFreq, puca_US_Tssi, gt_GHS_USTssiInfo.s_NumBrkptkept);
   // Convert Log Tssi values to Linear format
   DefineLinearTssiSet(gusa_US_Tssi_Value, gusa_US_Tssi_Value, TX_NUM_TONES);

   // Debug option -- disable applying of US tssi
   if((OPTNArray[OPTN_AlgControl] & OPTN_USTssiDisable) != 0)
   {
      // "Disable" US tssi by setting values to zero for tones
      // outside the supported set of tones.  This is not standard compliant.
      // It is for testing only.
      for (i=0 ; i<gs_TxFirstChannel; i++)
         gusa_US_Tssi_Value[i] = 0;

      for (i=gs_TxLastChannel+1 ; i<TX_NUM_TONES; i++)
         gusa_US_Tssi_Value[i] = 0;
   }


#if 0 //Amara: TBD For PSD Measurement disable TSSI
    if (TESTArray[TEST_FREEZE_CPE_TX] & TEST_FREEZE_CPE_TX_R_REVERB_2) {
      for (i=0 ; i<gs_TxFirstChannel; i++)
         gusa_US_Tssi_Value[i] = 0;

       for (i= gs_TxFirstChannel; i<= gs_TxLastChannel; i++) {
          gusa_US_Tssi_Value[i] = 1024;
       }

   for (i=gs_TxLastChannel+1 ; i<TX_NUM_TONES; i++)
      gusa_US_Tssi_Value[i] = 0;
    }
#endif

   // Define US SupportedSet
   DefineSupportedSet(gp_SUPPORTEDset_US, gs_TxNumTones, usa_US_BreakFreq, pfta_US_SprtSet, gt_GHS_USTssiInfo.s_NumBrkptkept);
#ifndef ISDN
   // Conexant CO:
   // send Annex-A spectrum in Annex-L training for increased upstream performance
   // supported set: according to Annex-L
   // tssi-values:      according to Annex-A
   if ( (gs_CurrentCoChipset == GSI_CO_CHIPSET) && (gl_SelectedMode & (MODE_G992_3)) && (gl_SelectedMode & (ANNEX_L)) )
   {
      // move upper limit of upstream band to tone 32
      usa_US_BreakFreq[4] = guca_US_TssiIndex_G9923A[0][4];
      usa_US_BreakFreq[5] = guca_US_TssiIndex_G9923A[0][5];

      // re-calculate tssi values
      // Define US Log Tssi
      DefineLogTssiSet(gusa_US_Tssi_Value, TX_NUM_TONES, usa_US_BreakFreq, puca_US_Tssi, gt_GHS_USTssiInfo.s_NumBrkptkept);
      // Convert Log Tssi values to Linear format
      DefineLinearTssiSet(gusa_US_Tssi_Value, gusa_US_Tssi_Value, TX_NUM_TONES);
   }
#endif
}


void DefineLogTssiSet(uint16 *pusa_LogTssiOut, int16 s_NumTones, int16 *psa_BreakFreq, uint8 *puca_LogTssiIn, int16 s_NumBrkpts)
{
   int i, j, k;
   int16 s_slope, s_slope_exp;
   int16 s_FreqDelta, s_Denom_exp, s_Num_exp;
   int32 l_TssiDeltaThisTone, l_TssiDelta;
   int16 result, s_Sign;
   int16 s_FirstBrkptIndex = 0, s_LastBrkptIndex = 0; // First and last breakpoints, excluding those with log_tssi=126.

   if (s_NumBrkpts > 0)
   {
      // Find first and last breakpoints.  Ignore breakpoints whose log_tssi value is 126, a special code.
      for(i=0; i < s_NumBrkpts; i++) {
         if (puca_LogTssiIn[i] != INTERP_LOG_TSSI_CODE)
            s_LastBrkptIndex = i;
         if (puca_LogTssiIn[s_NumBrkpts-i-1] != INTERP_LOG_TSSI_CODE)
            s_FirstBrkptIndex = s_NumBrkpts-i-1;
      }

      // Flat extension of log_tssi to lower frequencies, convert to Q8.8, multiply by -1 to convert tssi format
      // to proper scale
      for(i=0; i<=psa_BreakFreq[s_FirstBrkptIndex]; i++)
         pusa_LogTssiOut[i] = (uint16)((int32)puca_LogTssiIn[s_FirstBrkptIndex] << 8);

      // Flat extension of log_tssi to higher frequencies, convert to Q8.8, multiply by -1 to convert tssi format
      // to proper scale
      for(i=psa_BreakFreq[s_LastBrkptIndex]; i< s_NumTones; i++)
         pusa_LogTssiOut[i] = (uint16)((int32)puca_LogTssiIn[s_LastBrkptIndex] << 8);

      // Interpolate log_tssi for all subcarriers.
      // By their definition, breakpoints with log_tssi=126 can be ignored for this stage.
      for (i=(psa_BreakFreq[s_FirstBrkptIndex]+1); i < psa_BreakFreq[s_LastBrkptIndex]; i++)
      {
         for(j=0; j < s_LastBrkptIndex; j++)
         {
            if (puca_LogTssiIn[j] == INTERP_LOG_TSSI_CODE)
               continue;

            // Find the next breakpoint after bkpt[j], skipping over those with log_tssi=126.
            for (k=j+1 ; k<= s_LastBrkptIndex ; k++) {
               if (puca_LogTssiIn[k] != INTERP_LOG_TSSI_CODE)
                  break;
            }

            // Check whether subcarrier i is between brkpt[j] and brkpt[k]
            if ((i >= psa_BreakFreq[j]) && (i < psa_BreakFreq[k]))
            {
               // Not clear how we should interpolate between breakpoints when the log_tssi for one of the
               // breakpoints is 127, meaning tssi=0 (linear).  For now we interpolate as if 127 meant -63.5dB,
               // which doesn't require us to do anything special.  The breakpoint with log_tssi=127 will still wind up
               // with tssi(linear) = 0.

               // Compute the slope in Q8.8
               // Note: The delta range of breakpoint frequencies of consecutive
               // breakpoints is [1,gs_RxNumTones-1] while that for the tssi values is [-63,63].

               s_FreqDelta = psa_BreakFreq[k] - psa_BreakFreq[j];
               s_Denom_exp = norm_16bit(s_FreqDelta);

               l_TssiDelta = puca_LogTssiIn[k] - puca_LogTssiIn[j];
               s_Sign = 1;
               if (l_TssiDelta < 0)
               {
                  l_TssiDelta = -l_TssiDelta;
                  s_Sign = -1;
               }
               s_Num_exp = norm_l((int32)l_TssiDelta);

               result = Divide_32by16bit(l_TssiDelta<<s_Num_exp, (int16)(-s_Num_exp),
                  (int16)(s_FreqDelta<<s_Denom_exp), (int16)(-s_Denom_exp),
                  &s_slope, &s_slope_exp);

               if (s_Sign == -1)
                  s_slope = -s_slope;

               // Compute interpolated tssi in Q8.8
               l_TssiDeltaThisTone = ((int32)s_slope*(i - psa_BreakFreq[j]));
               l_TssiDeltaThisTone = round((int32)l_TssiDeltaThisTone, (int16)(-s_slope_exp-8));
               pusa_LogTssiOut[i] = (uint16)( ((int32)puca_LogTssiIn[j]<<8) + l_TssiDeltaThisTone);

               break;
            }
         }
      }
   }  //    if (s_NumBrkpts > 0)

   // Number of breakpoints is zero.
   else
   {

      for(i=0; i<s_NumTones; i++)
      {
         // Default is tssi=1.0 (0dB).
         pusa_LogTssiOut[i] = 0;
      }
   }
}


#undef NOTCH_NUM_TONES_THRESHOLD
