/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 2007 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   FDExtrapolate.c
*
*-------------------------------------------------------------------------
*/
/*********************************************************************************/
// Test code to verify Frequency Domain Extrapolation on Channel Estimation
// Standalone code
// Takes lab collected DMT channel estimation
// Runs through the extrapolation and generate the new data
/********************************************************************************/
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "typedef.h"
#include "gdata.h"
#include "matrix.h"
#include "ieee_flt.h"
#include "data_alloc.h"

/*=============Local Variables==================================*/
/* Hard-coded matrices for extrapolation. For doing an extrapolation, we need to:
a) Select a range for reference and extrapolated data,
b) Select order
c) Select basis functions for extrapolations.

The goal is to have a set of functions T(x,n) so that we can express any point x as
x ~= c(0)*T(x,0)+c(1)*T(x,2)+...c(p)*T(x,p)

Functions T(n,x) are chosen to be the n'th Chebyshev polynomial. Why?
- Empirically, polynomial interpolation gets good performance
- Taking simply T(n,x) = x^n is numerically unstable, especially around the axis edges,
because x^n and x^(n+1) are very close if x is very close to 0 or 1.
- Chebyshev polynomials are much better behaved numerically, as they are orthogonal between
-1 and 1.

As described in function DoInterpolation, we can see extrapolation as a least-squares problem.
With matrix T defined as above (each row is a reference/interpolation point, each column is
the polynomial order), given a vector of reference data x, we can solve for c above by solving
the over-determined system T*c = x.

In order to preserve numerical stability, we form matrix T off-line (which only depends on
number of interpolation points, reference points and interpolation order).
Since Chebyshev polynomials are only orthogonal between -1 and 1, we rescale the x axis
as (Matlab notation):

DataPoints = FirstTone:1:LastTone
b = DataPoints(end);
a = DataPoints(1);

Y = (DataPoints-(b+a)/2)/(b-a)*2; % axis on which to form T

and then we forn T(n,Y).

T can be defined recursively as

T(0,x) = 1;
T(1,x) = x;
T(n,x) = 2*x*T(n-1,x)-T(n-2,x)


We then do a QR factorization on T off-line, so that
Q*R*c = x <---> R*c = Q'*x

and solve for c on-line using back-substitution.
To get new interpolated points, we can just compute T(ind,:)*c where ind = index set of interpolated tones.

*/

/*
Data set for the following interpolation parameters:
First reference tone = 33
Last reference tone = 38
First interpolated tone = 30
Last interpolated tone = 32
Interpolation order: Cubic
*/

/* MatrixMult assumes row-wise data storage so fa_Q is stored that way */
DATA_MAP_deILV3_BIS
Float32 fa_Q[36] = {
0xbed105ec, 0xbf18fd40, 0x3f0ba8d2, 0xbebecfa6, 0xbe3169b2, 0xbdc95805,
0xbed105ec, 0xbeb7964d, 0xbddf7483, 0x3f05915b, 0x3efda8b5, 0x3ed64c86,
0xbed105ec, 0xbdf4c867, 0xbedf7483, 0x3e98a61f, 0xbe7f24b9, 0xbf30eb06,
0xbed105ec, 0x3df4c867, 0xbedf7483, 0xbe98a61f, 0xbefc2cb2, 0x3f0b8986,
0xbed105ec, 0x3eb7964d, 0xbddf7483, 0xbf05915b, 0x3f1dfaf0, 0xbe4c500c,
0xbed105ec, 0x3f18fd40, 0x3f0ba8d2, 0x3ebecfa6, 0xbe4a555c, 0x3ccf4813
};

//new 5pt
Float32 fa_R_Extrap[10] = {
0xc01cc471, 0xbf9cc471, 0x3f2737f0, 0x3f3c1eee,
0x3f562f5a, 0x3fd62f5a, 0x3f2d0fbb, 0x3efa450d,
0x3fbbb3ca, 0x3e83e37f
};

Float32 fa_T[72] = {
0x3f800000, 0xc0666666, 0x41c75c29, 0xc32fd1ec,
0x3f800000, 0xc059999a, 0x41b0f5c3, 0xc313051f,
0x3f800000, 0xc04ccccd, 0x419bd70a, 0xc2f2f0a4,
0x3f800000, 0xc0400000, 0x41880000, 0xc2c60000,
0x3f800000, 0xc0333333, 0x416ae148, 0xc29ed0e5,
0x3f800000, 0xc0266666, 0x414851ec, 0xc27a0419,
0x3f800000, 0xc019999a, 0x412851ec, 0xc240624e,
0x3f800000, 0xc00ccccd, 0x410ae148, 0xc20ff7cf,
0x3f800000, 0xc0000000, 0x40e00000, 0xc1d00000,
0x3f800000, 0xbfe66666, 0x40af5c29, 0xc18f6c8b,
0x3f800000, 0xbfcccccd, 0x4083d70a, 0xc1395810,
0x3f800000, 0xbfb33333, 0x403ae148, 0xc0d8d4fe,
0x3f800000, 0xbf99999a, 0x3ff0a3d7, 0xc053f7cf,
0x3f800000, 0xbf800000, 0x3f800000, 0xbf800000,
0x3f800000, 0xbf4ccccd, 0x3e8f5c29, 0x3eb43958,
0x3f800000, 0xbf19999a, 0xbe8f5c29, 0x3f6f9db2,
0x3f800000, 0xbecccccd, 0xbf2e147b, 0x3f71a9fc,
0x3f800000, 0xbe4ccccd, 0xbf6b851f, 0x3f116873
};

DATA_MAP_END


/***************************************************************************
**
** FUNCTION     :   BackSubstitution
**
** PROTOTYPE    :   void BackSubstitution2(Float32 *pfa_L, Float32 *pfa_result, int16 s_len);
**
**
** DESCRIPTION  :   Implements matrix backsubstitution
**
**
** PARAMETERS   :   r5 = pfa_L, r3 = pfa_result, a0 = s_len
**
**
** RETURN-VALUE :
**
**
** NOTES        :   This is a general version of BackSubstiution() in matrix.asm
**                  The latter cannot be reused because: a) it assumes that diagonals of R are 1,
**                  b) it assumes that input vector is stored as an extra column of L
**                  This makes its application very awkward and it's more efficient to do another routine.
**                  Another solution is to generalize BackSubstitution() but its still more efficient to
**                  do another routine from scratch.
**
***************************************************************************/
void BackSubstitution2(Float32 *pfa_L, Float32 *pfa_result, int16 s_len)
{
   int16 i, j;
   Float32 f_sum;

     for (i = s_len-1; i >=0; i--) {
         f_sum = pfa_result[i];

         for (j=i+1; j <=s_len-1; j++)
             f_sum = subf32(f_sum, mpyf32(RdLMat(pfa_L, i, j, s_len),pfa_result[j]));

         pfa_result[i] = divf32(f_sum,RdLMat(pfa_L,i,i,s_len));

     }
}


/****************************************************************************
**
** FUNCTION     :   Extrapolate
**
** PROTOTYPE    :   void Extrapolate(Float32* fa_h_ref, Float32* fa_Hinterp, int16 s_NumRefPoints, int16 s_NumExtrapolationPoints, int16 s_ExtrapType, int16 s_ExtrapOrder)
**
** DESCRIPTION  :   extrapolates numerically on (absphase) of H to improve TDQ performance.
**                  Interpolation matrices are computed offline to save mips/code
**                  By default, the interpolation matrices stored offline determine interpolation range.
**                  See comments above the definitions of fa_Q, fa_R, fa_T for details.
**
** RETURN-VALUE :
**
**
**
***************************************************************************/
#define ONE_THIRD (0x3eaaaaab)
void Extrapolate3(Float32* fa_h_ref, Float32* fa_Hinterp, int16 s_NumRefPoints, int16 s_NumExtrapolationPoints, int16 s_ExtrapType, int16 s_ExtrapOrder)
{
   Float32 fa_coef[6], f_sum;
   int16 i;

   if (s_ExtrapType == 0)  //cubic extrapolation
   {
      // Compute w = Q'*y;
      MatrixMult(fa_Q, fa_h_ref, fa_coef, s_NumRefPoints, 1, s_NumRefPoints, 1);

      // Solve R*c = w
      BackSubstitution2(fa_R_Extrap, fa_coef, (int16)(s_ExtrapOrder+1));

      // get T*c. T is dimension p*N, p = # of interpolated points
      MatrixMult(fa_T+(18-s_NumExtrapolationPoints)*4, fa_coef, fa_Hinterp, s_NumExtrapolationPoints, 1, (int16)(s_ExtrapOrder+1), 0);
   }
   else if (s_ExtrapType == 1)  //linear extrapolation
   {
      //Linear Extrapolate
      //computes a least-squares linear extrapolation based on 3 references tones, as follows:
      //given 3 data points, say -1,0,1, if we want to fit a line through points y(-1), y(0), y(1),
      //the optimal fit for line y = a*x + b is given by
      //a = (-y(-1) + y(1))/2
      //b = (y(-1)+y(0)+y(1))/3
      //Assume the fa_h_ref[0] is y(-1), fa_h_ref[1] is y(0) and fa_h_ref[2] is y(1)
      // calculate a
      fa_coef[0] = subf32(fa_h_ref[2], fa_h_ref[0]);
      fa_coef[0] = mpyf32(ONE_HALF, fa_coef[0]);

      //calculate b
      fa_coef[1] = addf32(fa_h_ref[0], fa_h_ref[1]);
      fa_coef[1] = addf32(fa_h_ref[2], fa_coef[1]);
      fa_coef[1] = mpyf32(ONE_THIRD, fa_coef[1]);

      f_sum = subf32(fa_coef[1], fa_coef[0]);  //-a+b

      for (i = 0; i < s_NumExtrapolationPoints; i++)
      {
         f_sum = subf32(f_sum, fa_coef[0]);
         fa_Hinterp[s_NumExtrapolationPoints-1-i] = f_sum;
      }


   }
}



