/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 2007 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   convert.c
*
*-------------------------------------------------------------------------
*/
// ******************************************************************
// convert.c
//
// History
//
// 15/01/2018 Chih-Wen: XDSLRTFW-3475
//            BER failed in TR100A 5T1 MV test against Lucent Stinger.
//            After 5dB noise boost, BER failed even with new NMS disabled. Some bits were swapped to lower tones (33~48)
//            because of higher margin, which caused lots of CRC.
//            The workaround was to disable NMS and avoid bits being swapped to tone 33~48.
//
//            To kick in the workaround, the following conditions must be met.
//            1. ADSL2+ AnnexA.
//            2. CNXT CO.
//            3. STAT[STAT_Performance] set "STAT_5T1Noise".
//            4. Loop lengths are between 4K and 6K feet.
//
//            Grep for XDSLRTFW_3475_TR100A_MV_5T1_AvoidBitswapAddBitsOnLower16Tones
//
// 24/07/2019 Stefan: XDSLRTFW-4206  CTL, ADSL2p: Downstream low perf on 5-7kft Xtalk loop - NMS algo disabled
//                NMS gain is now turned on by default for all loops / noises (gft_AddNMS is always
//                "TRUE" in Medley and Showtime)
//                correct Noise detection (false detection of line-noise was disabling NMS on multiple loops)
// *****************************************************************************

/*********************************************************************************
 * Utility functions:
 * ConvertComplexToAbsPhase:  convert reference data from int (real, imag) to float (abs, phase)
 * ConvertAbsPhaseToComplex:    convert float (abs, phase) to int (real, imag)
 ********************************************************************************/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "typedef.h"
#include "fft_tab.h"
#include "gdata.h"
#include "ieee_flt.h"
#include "mul.h"
#include "arctan.h"
#include "find_sintbl.h"
#include "snr.h"
#include "cmv.h"
#include "Decimalgain.h"
#include "memsetbf.h"
#include "bitload_const.h"
#include "states.h"
#include "hndshk_Data.h"


//#define N256 1024

extern int16 *gsa_sk_tab256;
extern int16 gs_CPilotTone;   /* cpilot tone index */
extern FlagT gft_ForceUnityTDQ;

//XDSLRTFW_3475_TR100A_MV_5T1_AvoidBitswapAddBitsOnLower16Tones (START)
FlagT gft_BitswapNotAddingBitsToTones = FALSE;
RxToneFlags gp_BitswapNotAddingBitsToTones = {0x00000000, 0x0000000, 0x00000000, 0x00000000};
//XDSLRTFW_3475_TR100A_MV_5T1_AvoidBitswapAddBitsOnLower16Tones (END)

/****************************************************************************
 ;
 ; Subroutine Name: ConvertComplexToAbsPhase(int16 s_Real, int16 s_Imag, Float32 f_Abs, Float32 f_Phase, , FlagT ft_doLogAbs)
 ;
 ;  Description:
 ;  convert reference data from int (real, imag) to float (abs, phase)
 ;  ft_doLogAbs: 1: convert abs to 20*log(abs); 0: abs
 ;
 ; Prototype:
 ;    void ConvertComplexToAbsPhase(int16 s_Real, int16 s_Imag, Float32 f_Abs, Float32 f_Phase)
 ;
 ;  Input Arguments:
 ;     s_Real, s_Imag -- int16
 ;
 ;
 ;   Output Arguments:
 ;    f_Abs -- Float32
 ;    f_Phase -- Float32
 ;
 ;  Return Value:
 ;
  *****************************************************************************/
void ConvertComplexToAbsPhase(int16 s_Real, int16 s_Imag, Float32* f_Abs, Float32* f_Phase, FlagT ft_doLogAbs)
{
   int32 l_prod_real, l_prod_imag, l_abs2;

   // Abs
   // l_prod_real = s_Real^2; l_prod_imag = s_Image^2
   MULS16(l_prod_real, s_Real, s_Real);
   MULS16(l_prod_imag, s_Imag, s_Imag);

   // f_abs = sqrt(abs^2)
   l_abs2 = l_prod_real + l_prod_imag;

   if (ft_doLogAbs)
   {
      //convert magnitude to log
      //compute 20*log(abs)  and fit to that - it yields better performance!
      *f_Abs = int2f32(ConvertToDB(l_abs2));

      //the output of ConvertToDB is in Q8.8, needs to divide 256 (0x43800000)
      *f_Abs = divf32(*f_Abs, 0x43800000);
   }
   else
      *f_Abs = sqrtf32(int32toFloat32(l_abs2));

   // Phase
   *f_Phase = int2f32(FastAtan(s_Imag, s_Real));
   *f_Phase = divf32(*f_Phase, int2f32(8192));


}
/*
*-------------------------------------------------------------------------------
*
*   Prototype: void SelectRxBandLimit(int16 *psa_SnrBuf, int16 s_WindowSize,
*      int16 *ps_HighestTone, int16 *ps_LowestTone)
*
*   This function selects the highest RX tone indices which has
*   the reasonable SNR.
*
*   Input Arguments:
*      psa_SnrBuf - pointer to the SNR buffer
*      s_WindowSize - size of the window to compute the average SNR
*
*   Output Arguments:
*      ps_HighestTone - pointer to the highest tone index
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void SelectRxBandLimit(int16 *psa_SnrBuf, int16 *ps_HighestTone, int16 s_Windowsize, int16 s_snr_thr)
{
   int16 s_ch, s_BandStart;
   int32 l_Metric, l_MinBandMetric;
   FlagT ft_FirstTime;


   // set the default values for highest tone
   *ps_HighestTone = gs_RxNumTones-1;
   l_MinBandMetric = s_snr_thr*s_Windowsize;
   ft_FirstTime = 1;
   s_BandStart = gs_RxFirstChannel;
   l_Metric = 0;

      // Loop over all tones of the actual Rx band
      for (s_ch = s_BandStart; s_ch < gs_RxNumTones; s_ch++)
      {
         if ((gl_SelectedMode & (MODE_ADSL2))&&(!IS_TONEFLAGSET(p_MEDLEYset_DS, s_ch)))
         {
            // Reset the sliding window
            s_BandStart = s_ch + 1;
            l_Metric = 0;
         }
         else
         {
            l_Metric += psa_SnrBuf[s_ch];

            // Accumulate the SNR first for a number of "WINDOW_SIZE" tones.
            // When the "l_Metric" contains the "WINDOW_SIZE" SNR then the evaluation per tone will start.
            // The sliding window algorithm is comparing for every new tone against the "WINDOW_SIZE" threshold.
            // Therefore the SNR of first tone (s_BandStart) gets be removed and new tone (s_ch ) gets be added.
            if(((s_ch-s_BandStart)+1) >= s_Windowsize)
            {
               if (l_Metric < l_MinBandMetric)
               {
                  if (ft_FirstTime == 1)
                  {
                     // Set the Highest RX tone
                     *ps_HighestTone = s_ch-1;              // Use the actual processed tone-1 as the last tone (be conservative)
                     // make sure the pilot tone is not cutoff
                     if (*ps_HighestTone < gs_CPilotTone+10)
                     {
                        *ps_HighestTone = gs_CPilotTone+10;
                        // check if it is over the band edge
                        if (*ps_HighestTone > gs_RxNumTones)
                        {
                           *ps_HighestTone = gs_RxNumTones;
                        }
                     }
                     ft_FirstTime = 0;
                  }
               }
               // if metric goes up again, renew the search
               else
               {
                  ft_FirstTime = 1;
                  *ps_HighestTone = gs_RxNumTones;
               }

               l_Metric -= psa_SnrBuf[s_BandStart];
               s_BandStart++;
            }
         }
      }
}

/*^^^
*-----------------------------------------------------------------------------
*
*   Prototype:
*       void MovingAverage( int16 *sa_InBuf,int16 *sa_OutBuf,int16 s_firstCh,int16 s_LastCh,int16 s_WindowLen,int16 s_log2WindowLen)
*
*   Description:
*     Computes the average acorss tones for ex
*        sa_OutBuf(4) = (sa_InBuf(1) + sa_InBuf(2) + sa_InBuf(3) + sa_InBuf(4) + sa_InBuf(5)+sa_InBuf(6) +sa_InBuf(7)+sa_InBuf(8)) /8
*
*  Input:
*     sa_InBuf - pointer to the Input buffer
*        s_firstCh - first Channel
*        s_LastCh - last channel
*        s_WindowLen - Window length for averaging
*        s_log2WindowLen - log2 of window length
*
*  Output:
*     sa_OutBuf - Output buffer
*-----------------------------------------------------------------------------
^^^*/

void MovingAverage( int16 *sa_InBuf,int16 *sa_OutBuf,int16 s_firstCh,int16 s_LastCh,int16 s_WindowLen,int16 s_log2WindowLen)
{

   int16 i,j,idx,sf_ext,se_ext;
   int32 l_Accum = 0;
   int16 s_RunAvg;
   int16 s_round;
   s_round = 1<<(s_log2WindowLen-1);

   sf_ext = ((s_WindowLen>>1)-1); //(windowLen/2-1) for windowLen = 8,sf_ext = 3
   se_ext = ((s_WindowLen>>1));   //(windowLen/2) for windowLen = 8,se_ext = 4

   // Initialise first tones and last tones
   for (i = (s_firstCh-1);i >= (s_firstCh-sf_ext);i--)
   {
      sa_InBuf[i] = sa_InBuf[s_firstCh];
   }
   for (i = s_LastCh;i < (s_LastCh+se_ext);i++)
   {
      sa_InBuf[i] = sa_InBuf[s_LastCh-1];
   }

   idx = s_firstCh; //for ex firstCh 32,
   j =  idx-sf_ext; // start from 29
   for ( ;idx < (s_LastCh);idx++,j++)
   {
      l_Accum = 0;
      for (i = j;i <(j+s_WindowLen);i++)
      {
         l_Accum += sa_InBuf[i];
      }
      s_RunAvg = (int16)((l_Accum+s_round)>>s_log2WindowLen);
      sa_OutBuf[idx] = s_RunAvg;
   }
}

/*^^^
*-------------------------------------------------------------------
*
*  int16 DecimalGain_24to_M14dB(int16 s_gain_dB);
*
*  Description:
*      Computes the decimal equivalent of the fine gain in 8.8 format,
*     left-justified,
*      Assumes the the input gain_dB is in 8.8 format and is within the
*      range [-14.0 dB, +24 dB].
*      Returns 10^(gain_dB/20) in 10.6
*
*      Algorithm:
*
*          1) Adjust approximation if original input x was not in
*              6144 (>24*256dB)
*
*          2) RightShift the input s_gain_dB by 3, that is s_gain_dB/8;
*          3) Call DecimalGain with input "s_gain_dB/8"
*          4) Adjust the output for the divide by /3.
*          5) from "10^((s_gain_dB*8)/8/10) = [10^((s_gain_dB)/8/10)]^8";
*
*  Input Arguments:
*     s_gain_db -- file gain in dB (Q8.8 format)
*
*  Return:
*     decimal gain (in Q10.6 format)
*
*-------------------------------------------------------------------
*^^^
*/
  int16 DecimalGain_24to_M14dB(int16 s_gain_dB)
  {
      int16 s_DecimalValue;
      int32 l_tmp_2,l_tmp_4,l_tmp_8;
      int16 s_tmp_2,s_tmp_4;

      if (s_gain_dB > 24*256)
         s_gain_dB = 24*256;
      /* s_gain_dB should be changed to  (2*s_gain_dB)/8 for uisng the "DecimalGain" function.*/
      s_gain_dB = s_gain_dB >> 2;
      // Here "s_DecimalValue" returns in 3.13 format
      s_DecimalValue = DecimalGain(s_gain_dB);
      l_tmp_2 = s_DecimalValue*s_DecimalValue; //3.13*3.13
      s_tmp_2 = (int16)((l_tmp_2 + 0x8000) >> 16);        //Q6.10
      l_tmp_4 = s_tmp_2*s_tmp_2;               //6.10 *6.10
      //6.10*6.10  => 12.20 =>convert to 6.10
      //               => 6.10, 2^6/2^16
      s_tmp_4 = (int16)((l_tmp_4 +0x200 )>>10);        //Q6.10 format
      l_tmp_8 = s_tmp_4 * s_tmp_4;
      //6.10*6.10  => 12.20 =>convert to 10.6          //Q12.20
      //               =>  10.6, 2^2/2^16
      s_DecimalValue = (int16)((l_tmp_8 + 0x2000) >> 14);
      return (s_DecimalValue);
  }

/*^^^
*-----------------------------------------------------------------------------
*
*   Prototype:
*       void FindNoiseMarginSeparation()
*
*   Description:
*     Computes Noise Margin vector from ADC/Tx noise/ ISI
*
*
*  Input:
*
*
*  Output:
*     sa_OutBuf - Output buffer
*-----------------------------------------------------------------------------
^^^*/

void FindNoiseMarginSeparation()
{
   int16 s_TxSilence,s_ADCNoise,s_ISI,s_sum_Noise,s_MaxNMS;
   int16 s_ch,s_snr1,s_snr2,s_fineGainAdj,s_DesiredMargin_lin;
   int32 l_snr1,l_snr2;
   int16 *ps_SNRPtr;
   int16 *ps_src_1;
   int16 *psa_src;

   int16 s_TxNoiseDelta;
   int16 s_ADCNoiseDelta;
   int16 s_ISINoiseDelta;
   int16 s_SNR_THR_PAR_Tone;
   int16 s_SnrReq_1bit;
   int16 s_NMS = 0;
   FlagT ft_AWGN130_140Noise;

   // XDSLRTFW-4206  CTL, ADSL2p: Downstream low perf on 5-7kft Xtalk loop - NMS algo disabled
   //                NMS gain is now turned on by default for all loops / noises (gft_AddNMS is always "TRUE" in Medley and Showtime)
   //                correct Noise detection (false detection of line-noise was diabling NMS on multiple loops)
   gft_AddNMS = TRUE;
   s_fineGainAdj = 640; // 2.5 in 8.8 format
   s_SnrReq_1bit = gsa_ConstellationSNR[1];
   s_SNR_THR_PAR_Tone = s_SnrReq_1bit+gs_RxDesiredMargin-SNR_THRESH_LOAD_MIN_BITS+gs_min_fine_gain;
   s_TxNoiseDelta = gt_HercADSL_OPTNMap_MarginControl.s_TxNoise_delta;   // for IOP and debug in 8.8 format
   s_ADCNoiseDelta = gt_HercADSL_OPTNMap_MarginControl.s_ADCNoise_delta; // for IOP and debug in 8.8 format
   s_ISINoiseDelta = gt_HercADSL_OPTNMap_MarginControl.s_ISINoise_delta; // for IOP and debug in 8.8 format
   //memset(p_PARRTONEset, 0,(sizeof(uint8)) * ((int16) (RX_NUM_TONES >> 3)));
   //gpsa_NMS_Vector = gpsa_NoiseSeparationSnrAdj;
   //gpsa_NoiseSeparationSnrAdj = gpsa_New_NMS_Vector;
   MemSetBuffer(gpsa_New_NMS_Vector, 0, 0, (int16)(sizeof(int16)*gs_RxNumTones));
   MemSetBuffer(gpsa_NMS_Vector, 0, 0, (int16)(sizeof(int16)*RX_NUM_TONES));
   MemSetBuffer((int16 *)(void *)gla_RxAccumBuf, 0, 0, (int16)(sizeof(int32)*(2*gs_RxNumTones)));
   psa_src = (int16 *)(void *)gla_RxAccumBuf;
   psa_src += gs_RxNumTones;

   s_MaxNMS = (gs_RxDesiredMargin - 0x0100);
   // s_MaxNMS = ((gs_RxDesiredMargin >> 1)+ 0x0000);
   gt_HercADSL_OPTNMap_MarginControl.s_deltaMarginMax = MIN(gt_HercADSL_OPTNMap_MarginControl.s_deltaMarginMax,s_MaxNMS);


   if(gt_HercADSL_OPTNMap_MarginControl.us_NMS_Ctrl & OPTN_NoiseMarginChange_NM_Ctrl_Tx_ReverbEcho_Meas)
      ps_SNRPtr  =  gsa_ReverbEchoSnrBuf;
   else
      ps_SNRPtr  =  gpsa_RCReverb_Quiet_Snr;

   gs_HighestAllowedRxTone = gs_RxNumTones-1;
   if(gt_HercADSL_OPTNMap_MarginControl.us_NMS_Ctrl & OPTN_NoiseMarginChange_NM_Ctrl_UseEchoSNR_For_Medley)
   {
      ps_src_1  =  gsa_ReverbEchoSnrBuf;
   }
   else
   {
      ps_src_1 = gt_StateMachCntrl.psa_RXSNRBuf;
      SelectRxBandLimit(ps_src_1, &gs_HighestAllowedRxTone, 5, gsa_ConstellationSNR[0]);
   }
   if(gs_brk_dbg & 0x4000)
      Pause(0x4000);

   for (s_ch = gs_RxFirstChannel; s_ch < gs_HighestAllowedRxTone; s_ch++)
   {
      s_snr1 = gpsa_RCReverb_Silence_Snr[s_ch];
      s_snr2 = gpsa_RCReverb_Quiet_Snr[s_ch];
      s_TxSilence = DecimalGain_24to_M14dB((int16)(s_snr1 - s_snr2-s_TxNoiseDelta))-(0x0040); // here 0x0040 is 1 in 10.6 format
      if((s_TxSilence < 0)||(gt_HercADSL_OPTNMap_MarginControl.us_NMS_Ctrl & OPTN_NoiseMarginChange_NM_Ctrl_No_TxNoise_Add))
         s_TxSilence = 0;

      s_snr1 = gpsa_deltaPGA_Snr[s_ch];
      s_snr2 = gpsa_PGA_Snr[s_ch];
      s_ADCNoise = DecimalGain_24to_M14dB((int16)(s_snr1 - s_snr2-s_ADCNoiseDelta))-(0x0040); // here 0x0040 is 1 in 10.6 format
      if((s_ADCNoise < 0)||(gt_HercADSL_OPTNMap_MarginControl.us_NMS_Ctrl & OPTN_NoiseMarginChange_NM_Ctrl_No_ADCNoise_Add))
      s_ADCNoise = 0;

      s_snr1 = ps_SNRPtr[s_ch];
      s_snr2 = ps_src_1[s_ch];
      s_ISI = DecimalGain_24to_M14dB((int16)(s_snr1 - s_snr2 - s_fineGainAdj -s_ISINoiseDelta))-(0x0040); // here 0x0040 is 1 in 10.6 format
      if((s_ISI < 0)||(gt_HercADSL_OPTNMap_MarginControl.us_NMS_Ctrl & OPTN_NoiseMarginChange_NM_Ctrl_No_ISINoise_Add))
         s_ISI = 0;

      s_sum_Noise = (s_TxSilence+s_ADCNoise+s_ISI);
      psa_src[s_ch] = s_sum_Noise;
      //ignore PAR tones set detection
      if((((s_snr1-s_snr2 )> (10*256)) && (s_snr2 < s_SNR_THR_PAR_Tone))
               ||(((s_snr1-s_snr2 )> (6*256)) && (s_ch  > 100)) // Echo effect can be seen till 100 tone
         )
      {
         psa_src[s_ch] = psa_src[s_ch-1];
         //SETTONEFLAG(p_PARRTONEset, s_ch);
      }
      // if tone is not set in MEDLEY set take previous one Required for smooth operation
      if ((gl_SelectedMode & (MODE_ADSL2))&&(!IS_TONEFLAGSET(p_MEDLEYset_DS, s_ch)))
         psa_src[s_ch] = psa_src[s_ch-1];
   }

   // smoothing operation
   // To get smooth Averge around pilot
   psa_src[gs_CPilotTone] = psa_src[gs_CPilotTone-1];
   MovingAverage(psa_src,gpsa_New_NMS_Vector,gs_RxFirstChannel,(gs_HighestAllowedRxTone),8,3);
   s_DesiredMargin_lin = DecimalGain_24to_M14dB(gs_RxDesiredMargin);   // in Q10.6 format

   //CNXT DSLAM gft_M140WhiteNoise getting set for all 24HDSL/24DSL also, not able to use "gft_M140WhiteNoise" effectively
   ft_AWGN130_140Noise = FALSE;
   if (    (STATArray[STAT_Performance] == STAT_M140WhiteNoise)
        || (STATArray[STAT_Performance] == (STAT_M140WhiteNoise|STAT_M130M140WhiteNoise))
      )
   {
      ft_AWGN130_140Noise = TRUE;
   }
   if((ft_AWGN130_140Noise == TRUE) || (gft_5T1Noise != FALSE)||(gft_ForceUnityTDQ == TRUE))
   {
      gft_AddNMS = TRUE;
      //XDSLRTFW_3475_TR100A_MV_5T1_AvoidBitswapAddBitsOnLower16Tones (START)
#ifndef ISDN
      if ( (gl_SelectedMode & MODE_G992_5) &&
            (gs_CurrentCoChipset == GSI_CO_CHIPSET) &&
            (STATArray[STAT_Performance] & STAT_5T1Noise) &&
            (gs_hsk_tone_power_dB < MINIMUM_GHS_TONE_PWR_AT_4000F) &&
            (gs_hsk_tone_power_dB > MINIMUM_GHS_TONE_PWR_AT_6000F) )
      {
         gft_AddNMS = FALSE;

         gft_BitswapNotAddingBitsToTones = TRUE;

         //memset(gp_BitswapNotAddingBitsToTones, 0,(sizeof(uint8)) * ((int16) (gs_RxNumTones >> 3)));
         for (s_ch = gs_RxFirstChannel; s_ch < gs_RxFirstChannel+16; s_ch++)
            SETTONEFLAG(gp_BitswapNotAddingBitsToTones, s_ch);
      }
#endif
   //XDSLRTFW_3475_TR100A_MV_5T1_AvoidBitswapAddBitsOnLower16Tones (END)
   }

   if (    (gft_24HDSLNoise == TRUE)
        && (gs_hsk_tone_power_dB < MAXIMUM_GHS_TONE_PWR_24HDSL_NMS_Add)
        && (gs_hsk_tone_power_dB > MINIMUM_GHS_TONE_PWR_24HDSL_NMS_Add)
        && (gl_SelectedMode & (MODE_ADSL2))
      )
   {
      gft_AddNMS = TRUE;
   }

   for (s_ch = gs_RxFirstChannel; s_ch < gs_HighestAllowedRxTone; s_ch++)
   {
      //extra_margin_est_M3 = 10*log10(s_sum_Noise);
      //NMS_Vec_dB = 6-(sum_dB(6,extra_margin_est_M3)-sum_dB(extra_margin_est_M3,0));
      // optimise the above
      //x1 = s_sum_Noise+1;
      //x2 = s_sum_Noise+4;
      //NMS_Vec_dB_3 = 6-(10*log10(x2)-10*log10(x1));
      s_sum_Noise = gpsa_New_NMS_Vector[s_ch];
      l_snr1 = s_sum_Noise+(s_DesiredMargin_lin);  // 4 => ideally we should use DecimalGain(2*(gs_RxDesiredMargin));
      l_snr2 = s_sum_Noise+(1<<6);// 1 in 10.6 format
      // We should use "gs_RxDesiredMargin" in place of 0x0600. To save some cycles we are
      gpsa_New_NMS_Vector[s_ch] = (gs_RxDesiredMargin) - (ConvertToDB(l_snr1)-ConvertToDB(l_snr2));

      if (s_ch == gs_CPilotTone)
         continue;

      if(gft_AddNMS == TRUE)
      {
         s_NMS = MIN(gpsa_New_NMS_Vector[s_ch],gt_HercADSL_OPTNMap_MarginControl.s_deltaMarginMax);
         gpsa_NMS_Vector[s_ch] = s_NMS;
         // "gpsa_New_NMS_Vector" will have NMS vector without capping."gpsa_Medley_Vector" will have Medley SNR without NMS.
         //gsa_MedleySnrBuf[s_ch] += MIN(gpsa_New_NMS_Vector[s_ch],gt_HercADSL_OPTNMap_MarginControl.s_deltaMarginMax);
         gsa_MedleySnrBuf[s_ch] += s_NMS;
      }
   }
}




/*****************************************************************************
*
* FUNCTION     :   UnwrapPhase
*
* PROTOTYPE    :   void UnwrapPhase(Float32 *pfa_Array, int16 s_pts);
*
* DESCRIPTION  :   Unwraps an array of phases (in radians) assumed to be between 0 and 2*Pi
*                  so that relative jumps between values are no bigger than Pi
*
*
* PARAMETERS   :   r0 = pfa_Array , a0 = s_pts
*
*
* RETURN-VALUE :
*
*
* NOTES        :   Similar to Matlab's unwrap function
*****************************************************************************/
#define F_ZERO (0x0)
#define F_ONE (0x3f800000)

#define F_PI (0x40490fdb)
#define F_MINUS_PI (0xc0490fdb)
#define F_PI_OVER_2 (0x3fc90fdb)
#define F_MINUS_PI_OVER_2 (0xbfc90fdb)
#define F_PI_TIMES_2 (0x40c90fdb)

void UnwrapPhase(Float32 *pfa_Array, int16 s_pts)
{
    Float32 f_LastValue, f_Delta, f_Correction = F_ZERO;
   int16 i;

    f_LastValue = *pfa_Array++; // point to second element in array
    for (i=1; i < s_pts; i++)
    {
        f_Delta = subf32(*pfa_Array,f_LastValue);
        if (cmpgtf32(f_Delta,F_PI))
            f_Correction = subf32(f_Correction, F_PI_TIMES_2);
        else if (cmpgtf32(F_MINUS_PI, f_Delta))
            f_Correction = addf32(f_Correction, F_PI_TIMES_2);

        f_LastValue = *pfa_Array;
        (*pfa_Array++) = addf32(*pfa_Array,f_Correction);
   }

}


/************************************************************************
**
** FUNCTION     :   Delogf32
** DATE         :   2004-04-27
** AUTHOR       :   AWARE - Ling Zheng
**
** PROTOTYPE    :   Float32 log2linf32(Float32 f_x)
**
** DESCRIPTION  :   Convert dB to linear scale.
**                  pow(10,x) = 0.3937*x^4 + 1.4807*x^3 + 2.4667*x^2 + 2.2785*x + 0.9993
**                  (-1<=x<0)
**
** CONSTANTS    :   -
**
** GLOBALS      :
**
**
** STATICS      :   None.
**
** RETURN-VALUE :   a0 = 10^(f_x/20)
**
** NOTES        :   -
**
**
**********************************************************************/
Float32 log2linf32(Float32 f_x)
{
   Float32 f_y, f_adj, f_acc0, fa_Power10f32Coef[5];
   int16 j;

   //Initialization
   f_adj = F_ONE;
   f_acc0 = F_ZERO;

   fa_Power10f32Coef[0] = 0x3f7fd220; //C0 = 0.9993
   fa_Power10f32Coef[1] = 0x4011d2f2; //C1 = 2.2785
   fa_Power10f32Coef[2] = 0x401dde6a; //C2 = 2.4667
   fa_Power10f32Coef[3] = 0x3fbd8794; //C3 = 1.4807
   fa_Power10f32Coef[4] = 0x3ec9930c; //C4 = 0.3937

    //Check the input value to make sure -20<=f_x < 0
   if (cmpgtf32(f_x, F_ZERO))
   {
      while(cmpgtf32(f_x, F_ZERO))
      {
         //f_x -= 20; 0xc1a00000 = -20.0
         f_x = addf32(f_x, 0xc1a00000);

         //f_adj *= 10; 0x41200000 = 10.0
         f_adj = mpyf32(f_adj, 0x41200000);
      }
   }
   else
   {
      while(cmpgtf32(0xc1a00000, f_x))
      {
         //f_x += 20; 0x41a00000 = 20.0
         f_x = addf32(f_x, 0x41a00000);

         //f_adj /= 10;
         f_adj = divf32(f_adj, 0x41200000);
      }
   }

   //f_x = f_x/20
   f_x = divf32(f_x, 0x41a00000);

   /***********************************************************************/
   /* Compute pow(10,s_x) = C4*s_x^4 + C3*s_x^3 + C2*s_x^2 + C1*s_x + C0  */
   /***********************************************************************/
   // calculate pow(10,x) = 0.9993 + 2.2785*x + 2.4667*x^2 + 1.4807*x^3 + 0.3937*x^4
   //           = sum(Ci*x^i)
   // (-1 <= x < 0)
   // Compute C0
   f_acc0 = fa_Power10f32Coef[0]; // Left shift C0 by 15 to convert C0 from Q3.13 to Q4.28

   // Initialize f_y to f_x
   f_y = f_x;

   // Compute (C4*x^4 + C3*x^3 + C2*x^2 + C1*x + C0)
   // Each iteration of outer loop generates C(j)*x^j
   for(j=1;j<=POW10X_ORDER;j++)
   {
      if(j>1)
         f_y = mpyf32(f_y, f_x);

      // Compute C(j)*x^j
      f_acc0 = addf32(f_acc0, mpyf32(fa_Power10f32Coef[j], f_y));
   }

   // do adjustment
   f_acc0 = mpyf32(f_acc0, f_adj);

   return(f_acc0);

}

/****************************************************************************
**
** FUNCTION     :   ConvertAbsPhaseToComplex
**
** PROTOTYPE    :   void ConvertAbsPhaseToComplex(Float32 f_Abs, Float32 f_Phase,
**                      int16 *psa_h, FlagT ft_doLogAbs);
**             ft_doLogAbs: 1: f_Abs is in 20*log(abs) format; 0: abs
**
**
** DESCRIPTION  :   Converts from (magnitude, phase) to (real,imag):
**                      X = Z*cos(theta), Y = Z*sin(theta)
**
** PARAMETERS   :
**
** NOTES        :   Used for converting to (phase,angle) from (real,imag) representations
**
**
***************************************************************************/
void ConvertAbsPhaseToComplex(Float32 f_Abs, Float32 f_Phase, int16 *psa_h, FlagT ft_doLogAbs)
{
   Float32 f_out;
   int16 s_index, sa_value[2], i;

   // Sinf32
    // 1) Calc mod(f_Phase, 2*pi):
    while (cmpgtf32(f_Phase,F_PI_TIMES_2))
        f_Phase = subf32(f_Phase,F_PI_TIMES_2);

    while (cmpgtf32(F_ZERO,f_Phase))
        f_Phase = addf32(f_Phase,F_PI_TIMES_2);

   // 2) solve for k: phase = 2*pi*k/N ie k = N*Phase/(2*Pi)
    //gsa_sk_tab256 = sin(2*Pi*K/N), N = 512: 512 = 0x44000000
   f_out = mpyf32(0x44000000, f_Phase);

   f_out = divf32(f_out, F_PI_TIMES_2);
   s_index = f32toint16(f_out, 0);

   // Cosf32
    // s_k is between 0 and 512
   //gsa_twid_real256 = (int16 *)gsa_CommonTwidMem;
   //gsa_twid_imag256 = (int16 *)gsa_twid_real256 + N256/4;
   //gsa_sk_tab256 = (int16 *) gsa_twid_imag256 + N256/4;

   sa_value[1] = findSinTbl(s_index);

    s_index += 128 ;               // cos(angle) = sin(angle+pi/2)
    if (s_index > 511) {
         s_index -= 512 ;
    }

   if(gs_RxFftLength == 1024)
      sa_value[0] = findSinTbl((int16)(2*s_index));
   else
      sa_value[0] = findSinTbl(s_index);


   if (ft_doLogAbs)
   {
      f_Abs = log2linf32(f_Abs);
   }

   for (i = 0; i < 2; i++)
   {
      //convert from Q1.15 int16 to Float32, a0 = a0/32768; 32768 = 0x47000000
      f_out = int2f32(sa_value[i]);
      f_out = divf32(f_out, 0x47000000);

      psa_h[i] = f32toint16(mpyf32(f_Abs, f_out), 0);
   }

}
