/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2003 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;   Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413
;   Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;
;   File Name: RxBtswpDecision.c
;
;   This file contains functions for implementing G.992.1/G.992.2 bit-swapping.

*
*-------------------------------------------------------------------------
*/
// ******************************************************************
// RxBtswpDecision.c
//
// History
//
// 17/03/08 Palaksha:   Data on Pilot in DMT mode feature
//       grep for  "R3_P1:PERF_DS_DMT_DataOnPilot"
//
//  29/08/08     AdeelJ  : modified code to incorporate gainswaps
//
//        This code removes the original Danube DMT bitswap algorithm with a new one
//        based on the AR7 DMT algorithm. The older Danube algorithm did not support
//        gainswaps.
//
//        The control flow starts with the function 'RxBitSwapDecision'
//        here, in case of automatic bitswaps, the search function 'BitSwapToneSearch'
//        is called after which depending on the search results, an
//        AOC message is formulated in function 'RxBitSwapComposeMessage'
//
//        In 'BitSwapToneSearch()' we essentially search for
//        1) tone with worst snr
//        2) two Tones with best margin
//
//        over here is there is a huge difference in Margin, then direct bitswaps are triggered
//        and the 'GnswpBypassMode' is enabled. This deals with the case of sudden noise or sudden RFI where it
//        is sure that gainswaps would not help.
//        if this is not the case then the following are searched.
//
//        4) tone which can be used for gain swap (can lose 1dB FG and has 1.5dB higher margin than worst tone) 'puca_Pswap[0]'
//        5) 2 tones which can lose 3dB FG and lose bit 'puca_spare_BitDonor'
//        6) 2 tones which can gain 2dB FG and gain bit 'puca_Bswap'
//        7) 2 tones which can gain 3dB FG and gain bit 'puca_spare_BitReceiver'
//
//        Now the function 'RxBitSwapComposeMessage' is called. This function uses the
//        outputs of the search function. one of a possible 7 messages are selected
//        depending on the availability of the searched tones.
//
//        the highest priority is given to sudden RFI scenarios covered by 'GnswpBypassMode'.
//        after that, in case one tone has lost all bits then priority task is to drain all
//        residual FineGain on that tone and distribute it to the worst tones.
//        The next priority is of the Gainswap in which a high margin tone gives 1 dB FineGain
//        to the worst tone.
//        Then if these conditions are not met, we try to equalize the Per Tone margins by a mixture
//        of bits and gainswaps.
//
//    Grep for SMS00746477 PERF_DS_DMT_ALL_Gainswap
//
// 10/08/2010 Nihar: Rewrote the finegain threshold definitions in Hex to
//                        improve the readability.
//                        It's a known issue that ADSL1 CTLM CO has problem in handling
//                        DS bitswap requests if CPE sends them too often and will drop
//                        the line during showtime.
//                        If the CO is CTLM, disabled the finegain swap
//                        so that the bitswap will be triggered less frequently.
//                        Current shortcomings is that there are CRCs seen, because
//                        only bitswap information is transmitted but not gainswap though
//                        the gainswaps are calculated and applied on the h/w
//                        Grep for IOP_DS_DMT_CTLM_Bitswap
//
// 10/08/2010 Nihar: Some ECI ADI Anaconda DSLAMs have problem handling one of the
//                       Extended bitswap requests and will drop the line. The fix is to
//                       avoid sending out this message if the CO is Anaconda.
//                       Grep for IOP_A_DS_DMT_ANCDA_ExtBitswap
//
// 10/08/2010 Nihar: Bug fix for DMT bitswap algorithm, which is to avoid 1-bit loaded tone after bitswap.
//                       Grep for BugFix_DS_DMT_ALL_Bitswap
//
// 10/08/2010 Nihar: Three bug fixes for KTL and BT qual tests:
//            (1) Avoid bitswap on aux pilot in G.dmt/T1.413 mode.
//            (2) No (i.e. 0dB) finegain on aux pilot in G.dmt/T1.413 mode.
//            (3) Skip pilot FDQ magnitude adaptation during sync symbol since CO Tx pilot with average finegain power.
//             Grep for PERF_DS_AB_ALL_ALL_PilotBugfixes
//26-03-2013 Anantha: Logic added to trigger bitswap over stored SNR to increase bitswap speed
//             during RFI test
//          grep pattern XDSLRTFW-686: PERF_DS_DMT_ALL_Bitswap
// 01-04-2013 Anantha:Ikanos DSLAM NVLTC doesnot support 2->0 bit transition. During RFI test VR9 asks 2->0 bit
//    transition and the DSLAM always rejects the bitswap and VR9 stuck in the state of 2->0 bit
//    transition,  which prevents bitswap in other tones and link drop. Vendor ID specific fix
//    is added to prevent 2->0 bit transition for ikanos DSLAMs
//          grep pattern XDSLRTFW-686: PERF_DS_DMT_IKNS_Bitswap
//***************************************************************************

#include <string.h>
#include "config.h"
#include "common.h"
#include "aoc.h"
#include "dsp_op.h"
#include "gdata.h"
#include "bitload.h"
#include "tone_ord.h"
#include "fifo.h"
#include "dsp_op2.h"
#include "cmv.h"
#include "DSLEngin.h"
#include "tx_aoc.h"
#include "bitswap.h"
#include "decimalgain.h"
#include "gdata_bis.h"
#include "bitload_const.h"
#include "dec_gain.h"
#include "minmaxmargin.h"
#include "pll.h"
#include "bitload2.h"


//SMS00746477 PERF_DS_DMT_ALL_Gainswap (Start)
// included new defines, one typedef and added two functions 'BitSwapToneSearch'
// and 'RxBitSwapComposeMessage'
//SMS00844998 IOP_DS_DMT_CTLM_Bitswap (START)
#define PHY_BSWP_MAX_FINE_GAIN_MINUS_1DB                   (int16)(0x0180) //  2.5 - 1.0 =  1.5 dB  in Q8.8 format
#define PHY_BSWP_MAX_FINE_GAIN_MINUS_2DB                   (int16)(0x0080) //  2.5 - 2.0 =  0.5 dB  in Q8.8 format
#define PHY_BSWP_MAX_FINE_GAIN_MINUS_3DB                   (int16)(0xFF80) //  2.5 - 3.0 = -0.5 dB  in Q8.8 format
#define PHY_BSWP_MIN_FINE_GAIN_PLUS_3DB                    (int16)(0x0080) // -2.5 + 3.0 =  0.5 dB  in Q8.8 format
#define PHY_BSWP_MIN_FINE_GAIN_PLUS_2DB                    (int16)(0xFF80) // -2.5 + 2.0 = -0.5 dB  in Q8.8 format
#define PHY_BSWP_MIN_FINE_GAIN_PLUS_1DB                    (int16)(0xFE80) // -2.5 + 1.0 = -1.5 dB  in Q8.8 format

//SMS01419722 BugFix_DS_DMT_ALL_Bitswap (START)
//Threshold on the margin difference to trigger bitswap are changed as below.
//- Threshold on margin difference to remove 2 bits from/to tones is changed to 10 dB.
//- Threshold on margin difference to remove 1 bit from/to tones is changed to 7 dB.
#define PHY_BSWP_FAST_THRESH_2BIT                          (int16)(0x0A00) // 256*10 (10dB)  in Q8.8 format
#define PHY_BSWP_FAST_THRESH_1BIT                          (int16)(0x0700) // 256*7 (7dB)  in Q8.8 format
//SMS01419722 BugFix_DS_DMT_ALL_Bitswap (END)
//SMS00844998 IOP_DS_DMT_CTLM_Bitswap (END)

#define BEST_ONE_TONE_GAIN_2BIT                               3
#define BEST_TWO_TONE_GAIN_1BIT                               2
#define BEST_ONE_TONE_GAIN_1BIT                               1

typedef struct
{
 uint8 uc_worstTone;                           // tone with worst MSE
 uint8 uca_bswapTones[2];                       // Bitswap Tones
 uint8 uca_pswapTones[2];                       // power swap tones
 uint8 uca_spare_BitDonorTones[2];               // Tones that can lose 1 bit and 3dB FG
 uint8 uca_spare_BitReceiverTones[2];            // Tones that can gain 1 bit and 3dB FG

}OVHD_AOC_bitswapTones_t;


/*****************************************************************************
;  Subroutine Name: BitSwapToneSearch( )
;
;  Purpose: Based on the fine gain on the tone (i_max) with the worst MSE, this function
;       searches for the best tones for performing power,bit,and extended bitswaps.
;       we essentially search for
;        1) tone with worst snr
;        2) two Tones with best margin
;
;
;  Prototype:
;
;
;  Input Arguments:
;     OVHD_AOC_bitswapTones_t   *pt_OVHD_AOC_bitswapTones,
;               uint8                     *puc_GnswpBypassMode
;
;  Output Arguments:
;     none
;
;  Return:
;     TRUE
;
;  Global Variables:
;
;
;       Revision history:
;
*******************************************************************************/
uint8 BitSwapToneSearch(  OVHD_AOC_bitswapTones_t   *pt_OVHD_AOC_bitswapTones,
                          uint8                     *puc_GnswpBypassMode)
{

  int i,k;
  int16 *psa_fineGain;
  int16  s_localGain;
  int16  s_worstSnr;
  int16  sa_bestSnr[2];
  int16  sa_bestSnr_Idx[2];
  uint8 *puca_Bswap = pt_OVHD_AOC_bitswapTones->uca_bswapTones;
  uint8 *puca_Pswap = pt_OVHD_AOC_bitswapTones->uca_pswapTones;
  uint8 *puca_spare_BitDonor = pt_OVHD_AOC_bitswapTones->uca_spare_BitDonorTones;
  uint8 *puca_spare_BitReceiver = pt_OVHD_AOC_bitswapTones->uca_spare_BitReceiverTones;
  uint8 uc_spare_BitDonorIdx = 0;
  uint8 uc_spare_BitReceiverIdx = 0;

  int16 sa_bswap_SNR[2] = {NEG_INFINITY_DB,NEG_INFINITY_DB}; // smallest gain used to start search
  int16 sa_pswap_SNR[2] = {NEG_INFINITY_DB,NEG_INFINITY_DB}; // smallest gain used to start search
  int16 s_localMargin;

  // threshold array helpful in finding bitswap tones in 8.8 dB format. these are values taken from
  // original AR7 code probably derived through experimentation
  int16 sa_bswap_threshold[15]  = {0x0E11,0x0E11,0x03CD,0x0176,
                                   0x020B,0x024E,0x020B,0x024E,
                                   0x020B,0x024E,0x020B,0x024E,
                                   0x020B,0x024E,0x020B};

                                //dec: {3601, 3601, 973, 374, 523, 590, 523, 590, 523, 590, 523, 590, 523, 590, 523};

  //as per Mikes suggestion - change thres value sa_spareTone_thresh[1] from 2.04dB to 2.3dB to include ALL tones that follow ~3dB/bit approximation
  // used for selecting spare tones which can lose or gain bit
  int16 sa_spareTone_thresh[2] = {0x020B,0x024E};// in dB 8.8 format, dec: {523, 590}
  // set variables to initial values
  for (i = 0; i < 2; i++)
  {
    puca_Bswap[i] = 0;
    puca_Pswap[i] = 0;
    puca_spare_BitDonor[i] = 0;
    puca_spare_BitReceiver[i] = 0;
    sa_bestSnr[i] = -32768;   // worst possible SNR
    sa_bestSnr_Idx[i] = 0;
  }


  // local pointer to fine gain table
  psa_fineGain = gsa_RxFineGains;

  s_worstSnr = 32767; //start from highest value

  // This for loop searches for
  //     1) the tone with worst margin
  //     2) two Tones with best margin
  // both margin and tone index are stored
  for (i=gs_RxBitLoadFirstChannel ; i<=gs_RxBitLoadLastChannel ; i++)
  {
//SMS01406492 PERF_DS_AB_ALL_ALL_PilotBugfixes (START)
     //Avoid bitswap on aux/pilot tones.
     if((i == gs_AuxPilotToneIdx) ||
        (i == gs_CPilotTone))
        continue;
     // compute SNR Margin for this tone. This will be a relative value. We donot require the constants for absolute value.
     s_localMargin = gpsa_RxShowtimeSnrBuf[i] + psa_fineGain[i] - gsa_ConstellationSNR[guca_RxBat[i]];
     if (psa_fineGain[i] != NEG_INFINITY_DB)  // is a monitored tone
     {
         //XDSLRTFW-686: PERF_DS_DMT_IKNS_Bitswap(start_end)
         if ((s_localMargin<s_worstSnr)&& (guca_RxBat[i])&&  // if there are bits loaded
         (!((gs_CurrentCoChipset == IKNS_CO_CHIPSET) && (guca_RxBat[i]<=2))))
         {
             s_worstSnr =  s_localMargin;
             pt_OVHD_AOC_bitswapTones->uc_worstTone = i; //*i_max = i;
         }
         if ((s_localMargin > sa_bestSnr[1] )  && // search for two tones with best margins
             (guca_RxBat[i]<15))            // has space for a bit
//SMS01406492 PERF_DS_AB_ALL_ALL_PilotBugfixes (END)
         {
             k = 1;
             if(s_localMargin> sa_bestSnr[0])
             {
                k--;
                sa_bestSnr[1] = sa_bestSnr[0];
                sa_bestSnr_Idx[1] = sa_bestSnr_Idx[0];
             }
             sa_bestSnr[k] = s_localMargin;
             sa_bestSnr_Idx[k] = i;
         }
     }
  }

  // In case of sudden RFI, This condition will be met
  // in case the difference in Margins between the best and worst tones is too much
  // gainswaps would not help since they are slow (1dB step) and can statistically only change 2dB
//SMS01419722 BugFix_DS_DMT_ALL_Bitswap (START)
  if( (sa_bestSnr[0]>(s_worstSnr+ PHY_BSWP_FAST_THRESH_1BIT)) &&  // thresh: 7dB difference
   (guca_RxBat[pt_OVHD_AOC_bitswapTones->uc_worstTone]>=3) )
  {
//Case 1: Remove 2 bits from the worst tone onto the best tone with 0 bits.
//    To avoid the case in which the worst tone is loaded with 3 bits,
//    and the best case is loaded with 0 bit. Bitswap removing 2 bits
//    from the worst tone will result in 1-bit loaded tone.
      if ( (guca_RxBat[sa_bestSnr_Idx[0]]==0) &&
            (sa_bestSnr[0]>(s_worstSnr+ PHY_BSWP_FAST_THRESH_2BIT)) &&
            (guca_RxBat[pt_OVHD_AOC_bitswapTones->uc_worstTone]!=3)
            && (gs_CurrentCoChipset != ANCDA_CO_CHIPSET)   //XDSLRTFW-2144
            )
      {
           sa_bestSnr_Idx[1] = sa_bestSnr_Idx[0];  // load 2 bits on best tone
           *puc_GnswpBypassMode = BEST_ONE_TONE_GAIN_2BIT; // remove 2 bits from worst tone. load 2 bits on this tone. num_loaded_tones increased
      }
//Case 2: Remove 2 bits from the worst tone onto the best two tones with non-zero bits.
      else if( (guca_RxBat[pt_OVHD_AOC_bitswapTones->uc_worstTone]!=3)   &&     //  can lose 2 bits while retaining bits
            (guca_RxBat[sa_bestSnr_Idx[0]]>0) &&
                  (sa_bestSnr[1]>(s_worstSnr+ PHY_BSWP_FAST_THRESH_1BIT))   &&   //  and margin diff >thresh 7 dB
               (guca_RxBat[sa_bestSnr_Idx[1]]>0)
               && (gs_CurrentCoChipset != ANCDA_CO_CHIPSET)    //XDSLRTFW-2144
              )                            //  there should be a bit loaded on the 2nd best tone
       {
          *puc_GnswpBypassMode = BEST_TWO_TONE_GAIN_1BIT;                 //  swap 2 bits out of worst tone and distribute them among the two best tones
       }
//Case 3: Remove 1 bit from the worst tone onto either one of the best two tones with non-zero bits
//    To avoid the case in which the worst tone is loaded with 2 bits,
//    Bitswap removing 1 bit from the worst tone will result in 1-bit loaded tone.
      else
      {
      if (guca_RxBat[sa_bestSnr_Idx[0]]>0)
               *puc_GnswpBypassMode = BEST_ONE_TONE_GAIN_1BIT; //  swap 1 bit out of worst tone and give it to the best tone
            else if ((guca_RxBat[sa_bestSnr_Idx[1]]>0) && (sa_bestSnr[1]>(s_worstSnr+ PHY_BSWP_FAST_THRESH_1BIT)))
            {
               sa_bestSnr_Idx[0] = sa_bestSnr_Idx[1];
               *puc_GnswpBypassMode = BEST_ONE_TONE_GAIN_1BIT; //  swap 1 bit out of worst tone and give it to the second best tone
            }
        }
//SMS01419722 BugFix_DS_DMT_ALL_Bitswap (END)

      if(*puc_GnswpBypassMode)
      {
          puca_Bswap[0] = sa_bestSnr_Idx[0];
          puca_Bswap[1] = sa_bestSnr_Idx[1];
          return(1); // jump out of function: no extra search requierd. decision for bit-swapping is already made
                                 // gain swap is not required
      }
  }


  // In this for loop the following are searched
  // 1) tone which can be used for gain swap (can lose 1dB FG and has 1.5dB higher margin than worst tone) 'puca_Pswap[0]'
  // 2) 2 tones which can lose 3dB FG and lose bit 'puca_spare_BitDonor'
  // 3) 2 tones which can gain 2dB FG and gain bit 'puca_Bswap'
  // 4) 2 tones which can gain 3dB FG and gain bit 'puca_spare_BitReceiver'
  for (i=gs_RxBitLoadFirstChannel ; i<=gs_RxBitLoadLastChannel ; i++)
  {
     // don't consider bitswap on g=0 tones
     if ((psa_fineGain[i] != NEG_INFINITY_DB)       && // is a monitored tone
         (i!=pt_OVHD_AOC_bitswapTones->uc_worstTone) && //not the uc_worstToneIdx && has some bits allocated and is not the worst snr tone
         (i!=gs_AuxPilotToneIdx) && (i!=gs_CPilotTone)) // is not one of the pilot tones
     {
         s_localGain = psa_fineGain[i];

         if(!(guca_RxBat[i])        &&                        // tone has lost all bits through bitswap
             (s_localGain >= PHY_BSWP_MIN_FINE_GAIN_PLUS_1DB))  // tone can lose 1dB finegain
         {
             puca_Pswap[0] = i;
             puca_Pswap[1] = 1;    // puca_Pswap[1] is used as a flag to indicate that there is a tone which has no bits
                              // but still some donatable fine gain on it. in this case gainswap is made to worst tone
         }
         else if(guca_RxBat[i])  // only if bits are loaded
         {
             s_localMargin = gpsa_RxShowtimeSnrBuf[i] + s_localGain - gsa_ConstellationSNR[guca_RxBat[i]];
             if(s_localGain >= PHY_BSWP_MIN_FINE_GAIN_PLUS_1DB) // tone can lose 1dB finegain
             {
                if((s_localMargin > (s_worstSnr + 384)) && // (s_localMargin > worst + 1.5dB) try pswaps in order to limit bswaps
                    (!puca_Pswap[1]))                       // no bypass because of new 0 bit tones with residual FG
                {
                   // store  highest SNR tone if (highest SNR >= (lowest SNR + 1.5dB))
                   if(s_localMargin > sa_pswap_SNR[0])
                   {
                      sa_pswap_SNR[0] = s_localMargin;
                      puca_Pswap[0] = i;                 // tone index for gainswap
                   }
                } // if(s_localMargin > s_worstSnr + 1.5)

                if ((s_localGain>=PHY_BSWP_MIN_FINE_GAIN_PLUS_3DB)        &&
                    (guca_RxBat[i]>RX_MIN_BITS_PER_TONE) &&
                    (sa_bswap_threshold[guca_RxBat[i]-1]>=sa_spareTone_thresh[0]))
                {
                    // store two tones (can lose 3dB FG & lose bit)
                    sa_spareTone_thresh[0] = sa_bswap_threshold[guca_RxBat[i]-1];
                    if(uc_spare_BitDonorIdx<2)
                       puca_spare_BitDonor[uc_spare_BitDonorIdx++] = i;
                }

             } //(s_localGain >= min_gain_plus1dB)

             if( (guca_RxBat[i]<guc_MaxAllocBitsPerTone) &&
                 (s_localGain<= PHY_BSWP_MAX_FINE_GAIN_MINUS_2DB))//max_gain_less2dB) )
             {
                  // store two tones (can gain 2 dB & gain bit)
                  if(s_localMargin > (s_worstSnr + sa_bswap_threshold[guca_RxBat[i]]))
                  {
                      if(s_localMargin > sa_bswap_SNR[1])
                      {
                         k = 1;
                         if(s_localMargin>sa_bswap_SNR[0])
                         {
                            k--;
                            sa_bswap_SNR[1] = sa_bswap_SNR[0];
                            puca_Bswap[1]=puca_Bswap[0];
                         }
                         sa_bswap_SNR[k] = s_localMargin;
                         puca_Bswap[k] = i;
                      }
                  } // if(s_localMargin > (s_worstSnr + sa_bswap_threshold[guca_RxBat[i]]))

                  if((s_localGain <= PHY_BSWP_MAX_FINE_GAIN_MINUS_3DB)&&
                     (sa_bswap_threshold[guca_RxBat[i]]<=sa_spareTone_thresh[1]))
                  // tone can gain 3dB FG and bits are between 3 and 14
                  {
                     // store two tones (can gain 3 dB & gain bit)
                     sa_spareTone_thresh[1] = sa_bswap_threshold[guca_RxBat[i]];
                     if(uc_spare_BitReceiverIdx<2)
                        puca_spare_BitReceiver[uc_spare_BitReceiverIdx++] = i;
                  }
             } //( (guca_RxBat[i]<guc_MaxAllocBitsPerTone) && (s_localGain<= max_gain_less2dB) )
         }
     } // if(tone_to_consider && bits_loaded)

  }//for ( i = DOWN_FIRSTBIN; i < numDsTones; i++ )

  return(1);
}


/*****************************************************************************
;  Subroutine Name: RxBitSwapComposeMessage( )
;
;  This subroutine composes Bitswap Message for automatic bitswap
;
;  Prototype:
;  void RxBitSwapComposeMessage(int16 *psa_SnrBuf)
;
;  Input Arguments:
;     psa_SnrBuf -- pointer to the SNR buffer computed during SHOWTIME
;
;  Output Arguments:
;     none
;
;  Return:
;     none
;
;  Global Variables:
;     gl_rx_sframe_count      -- (I) RX superframe count
;     gt_RxBitSwapInfo     -- (I) structure containing RX bit swap info
;
;     gsa_RxBat[]          -- (O) RX Bit Allocation Table
;     gsa_RxFineGains[]    -- (O) RX fine gain table
;     gsa_RxCombinedGains[]   -- (O) RX combined gain table
;
*******************************************************************************/
uint8 RxBitSwapComposeMessage(OVHD_AOC_bitswapTones_t *pt_OVHD_AOC_bitswapTones,
                              uint8 *puca_msg_body,
                              int16 *s_ActualNumBitsToSwap,
                              uint8 uc_GnswpBypassMode)
{

   uint8 uc_worstToneIdx;
        int16 s_worst_gain;
        uint8 uc_automatic_bitswap = 0;

        uint8 *puca_Bswap = pt_OVHD_AOC_bitswapTones->uca_bswapTones;
        uint8 *puca_Pswap = pt_OVHD_AOC_bitswapTones->uca_pswapTones;
        uint8 *puca_spare_BitDonor = pt_OVHD_AOC_bitswapTones->uca_spare_BitDonorTones;
        uint8 *puca_spare_BitReceiver = pt_OVHD_AOC_bitswapTones->uca_spare_BitReceiverTones;


        uc_worstToneIdx = pt_OVHD_AOC_bitswapTones->uc_worstTone;

   s_worst_gain = gsa_RxFineGains[uc_worstToneIdx];

        if(puca_spare_BitReceiver[0] == puca_Pswap[0])
             puca_spare_BitReceiver[0] = puca_spare_BitReceiver[1];

        if(puca_spare_BitDonor[0]==puca_Bswap[0])  // should be very unlikely but...
             puca_spare_BitDonor[0] = puca_spare_BitDonor[1];


        //MESSAGE01: two best tones gain a bit each
        //           worst tone sheds either 1 or 2 bits depending on Trellis option
        if(uc_GnswpBypassMode>1) // either best tone has to gain 2 bits from zero BEST_ONE_TONE_GAIN_2BIT
                                // or 2 best tones will gain a bit each. worst tone has > 3 bits
        {
            *puca_msg_body++ = AOC_CMD_BIT_INC1;
            *puca_msg_body++ = (uint8)puca_Bswap[0];
            *puca_msg_body++ = AOC_CMD_BIT_INC1;
            *puca_msg_body++ = (uint8)puca_Bswap[1];
            *puca_msg_body++ = AOC_CMD_BIT_DEC1;
            *puca_msg_body++ = (uint8)uc_worstToneIdx;
            // choose between removing 1 bit or 2 bits from the worst tone
            if(  (guca_RxBat[uc_worstToneIdx] == RX_MIN_BITS_PER_TONE) ||  // one tone will go empty so Trellis equation will be fine
                 !gft_TcmFlag                                     ||  // Trellis coding not enabled no need to worry about trellis bits
                 (gus_ncloaded  & 0x0001)                         ||
                 (uc_GnswpBypassMode== BEST_TWO_TONE_GAIN_1BIT))

            {
               *puca_msg_body++ = AOC_CMD_BIT_DEC1;
               *puca_msg_body = (uint8)uc_worstToneIdx;    // worst tone: remove 2 bits
            }
            else
            {
               *puca_msg_body++ = AOC_CMD_DO_NOTHING;
               *puca_msg_body = (uint8)puca_spare_BitDonor[0];
            }
            uc_automatic_bitswap = 1;
            *s_ActualNumBitsToSwap = 1; // Normal BSR
            return(uc_automatic_bitswap);
        }

        //MESSAGE02: best tone gaina a bit
        //           worst tone loses a bit
        else if(uc_GnswpBypassMode == BEST_ONE_TONE_GAIN_1BIT)
        {
            *puca_msg_body++ = AOC_CMD_BIT_INC1;
            *puca_msg_body++ = (uint8)puca_Bswap[0];         // bswap tone: Add bit
            *puca_msg_body++ = AOC_CMD_BIT_DEC1;
            *puca_msg_body++ = (uint8)uc_worstToneIdx;       // Worst tone: remove bit
            *puca_msg_body++ = AOC_CMD_DO_NOTHING;
            *puca_msg_body++ = (uint8)puca_spare_BitDonor[0];
            *puca_msg_body++ = AOC_CMD_DO_NOTHING;
            *puca_msg_body = (uint8)puca_spare_BitDonor[0];
             uc_automatic_bitswap = 1;
            *s_ActualNumBitsToSwap = 1; // Normal BSR
             return(uc_automatic_bitswap);
        }

        //SMS00844998 IOP_DS_DMT_CTLM_Bitswap (START)
        //Skip the finegain swap if the CO is CTLM
        if (gs_CurrentCoChipset == CTLM_CO_CHIPSET)
           return(uc_automatic_bitswap);
        //SMS00844998 IOP_DS_DMT_CTLM_Bitswap (END)

        if (s_worst_gain < PHY_BSWP_MAX_FINE_GAIN_MINUS_1DB)//max_gain_less1dB)
        {
             //MESSAGE03: best tone loses 1dB FineGain
             //           worst tone gains 1dB FineGain
             if ((puca_Pswap[0] != 0)
                  && (gs_CurrentCoChipset != ANCDA_CO_CHIPSET) //XDSLRTFW-2144
                )
             {
                 *puca_msg_body++ = AOC_CMD_PWR_DEC1;
                 *puca_msg_body++ = (uint8)puca_Pswap[0];  // good tone : remove 1dB
                 *puca_msg_body++ = AOC_CMD_PWR_INC1;
                 *puca_msg_body++ = (uint8)uc_worstToneIdx;  // worst tone: Add 1dB
                 *puca_msg_body++ = AOC_CMD_DO_NOTHING;
                 *puca_msg_body++ = (uint8)puca_spare_BitDonor[0];
                 *puca_msg_body++ = AOC_CMD_DO_NOTHING;
                 *puca_msg_body = (uint8)puca_spare_BitDonor[0];
                 uc_automatic_bitswap = 1;
                 *s_ActualNumBitsToSwap = 1; // Normal BSR
                 return(uc_automatic_bitswap);
             }

             //MESSAGE03: best tone gains bit and gains 2dB FineGain
             //           spare Donor loses bit and loses 3dB FineGain
             //           worst tone gains 1dB FineGain
             //SMS01225902/SMS01222062 IOP_A_DS_DMT_ANCDA_ExtBitswap (START)
             //Some ADI Anaconda COs have problem handling this message. Large
             //number of DS CRCs will occur and the CO will drop the link.
             else if ((puca_Bswap[0] != 0) && (puca_spare_BitDonor[0] != 0)
                      && (gs_CurrentCoChipset != ANCDA_CO_CHIPSET)
                      )
             //SMS01225902/SMS01222062 IOP_A_DS_DMT_ANCDA_ExtBitswap (END)
             {
                 *puca_msg_body++ = AOC_CMD_BIT_INC1;
                 *puca_msg_body++ = (uint8)puca_Bswap[0];
                 *puca_msg_body++ = AOC_CMD_BIT_DEC1;
                 *puca_msg_body++ = (uint8)puca_spare_BitDonor[0];
                 *puca_msg_body++ = AOC_CMD_PWR_INC2;
                 *puca_msg_body++ = (uint8)puca_Bswap[0];  // puca_Bswap[0]  : Add 2dB add 1 bit
                 *puca_msg_body++ = AOC_CMD_PWR_DEC2;
                 *puca_msg_body++ = (uint8)puca_spare_BitDonor[0];
                 *puca_msg_body++ = AOC_CMD_PWR_DEC1;
                 *puca_msg_body++ = (uint8)puca_spare_BitDonor[0]; // spare_BitDonor   : remove 3dB remove 1 bit
                 *puca_msg_body++ = AOC_CMD_PWR_INC1;
                 *puca_msg_body = (uint8)uc_worstToneIdx;  // worst tone: Add 1dB
                 uc_automatic_bitswap = 1;
                 *s_ActualNumBitsToSwap = 2; // Extended BSR
                 return(uc_automatic_bitswap);
             }
         } // if (s_worst_gain < max_gain_less1dB)

         if (s_worst_gain > PHY_BSWP_MIN_FINE_GAIN_PLUS_2DB)
         { // s_worst_gain > min_gain_plus2db --> try to swap a bit off this tone

             if (guca_RxBat[uc_worstToneIdx] > RX_MIN_BITS_PER_TONE)
             {
                 //MESSAGE04: best tone gains bit and gains 2dB FineGain
                 //           worst tone loses 2dB FineGain and loses 1 bit
                 if (puca_Bswap[0] != 0)
                 {
                    *puca_msg_body++ = AOC_CMD_BIT_INC1;
                    *puca_msg_body++ = (uint8)puca_Bswap[0];
                    *puca_msg_body++ = AOC_CMD_BIT_DEC1;
                    *puca_msg_body++ = (uint8)uc_worstToneIdx;
                    *puca_msg_body++ = AOC_CMD_PWR_INC2;
                    *puca_msg_body++ = (uint8)puca_Bswap[0];   // puca_Bswap[0] : Add 2dB Add 1 bit
                    *puca_msg_body++ = AOC_CMD_PWR_DEC2;
                    *puca_msg_body = (uint8)uc_worstToneIdx;   // worst tone: remove 2dB remove 1 bit
                    uc_automatic_bitswap = 1;
                    *s_ActualNumBitsToSwap = 1; //Normal BSR
                    return(uc_automatic_bitswap);
                 }
                 else
                 {  // can't do a simple bitswap -- try to swap to a spareTone bin
                    //MESSAGE05: best tone loses 1dB FineGain
                    //           worst tone loses 2dB FineGain and loses 1 bit
                    //           spare Receiver gains 3dB and gains 1 bit
                    if ((puca_Pswap[0] != 0) && (puca_spare_BitReceiver[0] != 0))
                    {

                        *puca_msg_body++ = AOC_CMD_BIT_INC1;
                        *puca_msg_body++ = (uint8)puca_spare_BitReceiver[0];
                        *puca_msg_body++ = AOC_CMD_BIT_DEC1;
                        *puca_msg_body++ = (uint8)uc_worstToneIdx;
                        *puca_msg_body++ = AOC_CMD_PWR_INC3;
                        *puca_msg_body++ = (uint8)puca_spare_BitReceiver[0];  // spare_BitReceiver:  Add 3dB Add 1 bit
                        *puca_msg_body++ = AOC_CMD_PWR_DEC2;
                        *puca_msg_body++ = (uint8)uc_worstToneIdx;   // worst tone: remove 2dB remove 1 bit
                        *puca_msg_body++ = AOC_CMD_PWR_DEC1;
                        *puca_msg_body++ = (uint8)puca_Pswap[0];   // pswap tone: remove 1dB
                        *puca_msg_body++ = AOC_CMD_DO_NOTHING;
                        *puca_msg_body = (uint8)puca_spare_BitDonor[0];
                        uc_automatic_bitswap = 1;
                        *s_ActualNumBitsToSwap = 2; // Extended BSR
                        return(uc_automatic_bitswap);
                    }
                 }
              }// (guca_RxBat[uc_worstToneIdx] > RX_MIN_BITS_PER_TONE): Rcv_nbits[uc_worstToneIdx[0]] > 2

              else
              { // worst tone has 2 bits.. and the s_worst_gain > max_gain_less1db
                 //MESSAGE06: best tone gains 1 bit
                 //           worst tone loses 2dB FineGain and loses 2 bits
                 //           2nd best tone gains 2dB and gains bit
                 if (((!gft_TcmFlag) || ((gus_ncloaded & 0x0001) == 0)) && (puca_Bswap[1] != 0))
                 {

                      *puca_msg_body++ = AOC_CMD_BIT_INC1;
                      *puca_msg_body++ = (uint8)puca_Bswap[0];   // puca_Bswap[0] : Add 1 bit
                      *puca_msg_body++ = AOC_CMD_BIT_INC1;
                      *puca_msg_body++ = (uint8)puca_Bswap[1];
                      *puca_msg_body++ = AOC_CMD_BIT_DEC1;
                      *puca_msg_body++ = (uint8)uc_worstToneIdx;
                      *puca_msg_body++ = AOC_CMD_BIT_DEC1;
                      *puca_msg_body++ = (uint8)uc_worstToneIdx;
                      //only increase 2dB in 2nd best bswap tone
                      //if the tone with now zero bits will have FG remaining it will be swapped on priority basis
                      *puca_msg_body++ = AOC_CMD_PWR_DEC2;
                      *puca_msg_body++ = (uint8)uc_worstToneIdx;    // worst tone: remove 2 bits remove 2dB
                      *puca_msg_body++ = AOC_CMD_PWR_INC2;
                      *puca_msg_body = (uint8)puca_Bswap[1];    // puca_Bswap[1] : Add 2dB Add 1 bit
                      uc_automatic_bitswap = 1;
                      *s_ActualNumBitsToSwap = 2; // Extended BSR
                      return(uc_automatic_bitswap);
                 }
                 //MESSAGE07: best tone gains 1 bit and gains 2dB FineGain
                 //           worst tone loses 2dB FineGain and loses 2 bits
                 //           only one bit is gained because Trellis bit is reduced
                 //           as number of loaded tones will decrement
                 else if((gft_TcmFlag && (gus_ncloaded & 0x0001)) && (puca_Bswap[0] != 0))
                 {

                      // here the bits added and removed are not equal as number of tones are going
                      // to be less after bswap. Total_bits = Lp + Trellis_bits
                      // Trellis bits = fn(no. of bitloaded tones)... (1 for every 2 tones)
                      *puca_msg_body++ = AOC_CMD_BIT_INC1;
                      *puca_msg_body++ = (uint8)puca_Bswap[0];
                      *puca_msg_body++ = AOC_CMD_PWR_INC2;
                      *puca_msg_body++ = (uint8)puca_Bswap[0];   // puca_Bswap[0] : Add 2dB Add 1 bit
                      *puca_msg_body++ = AOC_CMD_BIT_DEC1;
                      *puca_msg_body++ = (uint8)uc_worstToneIdx;
                      *puca_msg_body++ = AOC_CMD_BIT_DEC1;
                      *puca_msg_body++ = (uint8)uc_worstToneIdx;
                      // also remove FG from worst tone as it is still monitored
                      // this will also keep the FG equation intact
                      *puca_msg_body++ = AOC_CMD_PWR_DEC2;
                      *puca_msg_body++ = (uint8)uc_worstToneIdx;    // worst tone: remove 2 bits remove 2dB
                      *puca_msg_body++ = AOC_CMD_DO_NOTHING;
                      *puca_msg_body = (uint8)puca_spare_BitDonor[0];
                      uc_automatic_bitswap = 1;
                      *s_ActualNumBitsToSwap = 2; // Extended BSR
                      return(uc_automatic_bitswap);
                 }
              } //else part of Rcv_nbits[uc_worstToneIdx[0]] > 2
         } // if(s_worst_gain > min_gain_plus2db)

         return(uc_automatic_bitswap);
 }
//SMS00746477 PERF_DS_DMT_ALL_Gainswap (End)

/*****************************************************************************
;   Subroutine Name: RxBitSwapDecision( )
;
;   This subroutine computes the SNR margin and decide if the bit-swap
;   should be initiated or not.
;
;   Prototype:
;   void RxBitSwapDecision(int16 *psa_SnrBuf)
;
;   Input Arguments:
;      psa_SnrBuf -- pointer to the SNR buffer computed during SHOWTIME
;
;   Output Arguments:
;      none
;
;   Return:
;     TRUE if bitswap message formulated
;               FALSE otherwise
;
;   Global Variables:
;      gl_rx_sframe_count      -- (I) RX superframe count
;      gt_RxBitSwapInfo      -- (I) structure containing RX bit swap info
;
;      gsa_RxBat[]            -- (O) RX Bit Allocation Table
;      gsa_RxFineGains[]      -- (O) RX fine gain table
;      gsa_RxCombinedGains[]   -- (O) RX combined gain table
;
*******************************************************************************/
int16 RxBitSwapDecision(void)
{
   int16 i, j, s_ActualNumBitsToSwap, s_AOCRxMsgId;
   uint8 uca_msg_body[AOC_MSG_EBSR_LEN - 1];
   uint8 *puca_msg_body;

//SMS00746477 PERF_DS_DMT_ALL_Gainswap (Start)
        OVHD_AOC_bitswapTones_t t_OVHD_AOC_bitswapTones;

        uint8  uc_Bitswap_type = 0;
        uint8  uc_GnswpBypassMode = 0;


        uint8 *puca_spare_BitDonorTone = t_OVHD_AOC_bitswapTones.uca_spare_BitDonorTones;                       // spareTone tones
        uint8 *puca_spare_BitReceiverTone = t_OVHD_AOC_bitswapTones.uca_spare_BitReceiverTones;
//SMS00746477 PERF_DS_DMT_ALL_Gainswap (End)
   s_ActualNumBitsToSwap = 0;



   // forced bitswap
   if (TESTArray[TEST_Control] & TEST_TestBitSwap)
   {
         //From Tone
      j = TESTArray[TEST_BitSwapFromTone];
      if ((j>= gs_RxNumTones) || (j==gs_PilotToneIdx)) {
         return (FALSE);
      }

      i = guca_RxBat[j];
      if (i > 2)
         s_ActualNumBitsToSwap = 1;
      else if (i == 2)
         s_ActualNumBitsToSwap = 2;
      else
         return (FALSE);

      // To tone
      j = TESTArray[TEST_BitSwapToTone];
      if ((j >= gs_RxNumTones) || (j == gs_PilotToneIdx))
         return (FALSE);

      i = guca_RxBat[j];
      if (i + s_ActualNumBitsToSwap > RX_MAX_BITS_PER_TONE)
         return (FALSE);

      // Set
      puca_spare_BitReceiverTone[0] = TESTArray[TEST_BitSwapToTone];
      puca_spare_BitReceiverTone[1] = TESTArray[TEST_BitSwapToTone];
      puca_spare_BitDonorTone[0] = TESTArray[TEST_BitSwapFromTone];
      puca_spare_BitDonorTone[1] = TESTArray[TEST_BitSwapFromTone];

      TESTArray[TEST_Control] &= ~TEST_TestBitSwap;
      uc_Bitswap_type = 2;  //set flag
   /* If no bits can be added, return FALSE */
   if (s_ActualNumBitsToSwap == 0)
      return (FALSE);

                // Formulate AOC message for Bitswap
   /* Form bit swap request message to swap 1 or 2 bit */
   puca_msg_body = &uca_msg_body[0];
   *puca_msg_body++ = AOC_CMD_BIT_DEC1;
                *puca_msg_body++ = (uint8)puca_spare_BitDonorTone[0];

   if (s_ActualNumBitsToSwap == 1)
   {
      *puca_msg_body++ = AOC_CMD_BIT_INC1;
         *puca_msg_body++ = (uint8)puca_spare_BitReceiverTone[0];
   }
   else
   {   /* if(s_ActualNumBitsToSwap == 2) */
            *puca_msg_body++ = AOC_CMD_BIT_DEC1;
         *puca_msg_body++ = (uint8)puca_spare_BitDonorTone[1] ;

         *puca_msg_body++ = AOC_CMD_BIT_INC1;
         *puca_msg_body++ = (uint8)puca_spare_BitReceiverTone[0];

         *puca_msg_body++ = AOC_CMD_BIT_INC1;
         *puca_msg_body++ = (uint8)puca_spare_BitReceiverTone[1];
   }   /* else */

   /* Fill in fine gain parameters (the fine gain change is not implemented) */
   *puca_msg_body++ = AOC_CMD_DO_NOTHING;
                 *puca_msg_body++ = (uint8)puca_spare_BitDonorTone[0];
   *puca_msg_body++ = AOC_CMD_DO_NOTHING;
                 *puca_msg_body = (uint8)puca_spare_BitDonorTone[0];

        }   // manual bitswap
//SMS00746477 PERF_DS_DMT_ALL_Gainswap (Start)
   else  // non-manual (i.e. automatic, based on SNRs bitswap)
   {
      // call function which searches for tones which will be used for Bitswap and gainswap
      BitSwapToneSearch( &t_OVHD_AOC_bitswapTones,
                     &uc_GnswpBypassMode);

      // now we have
      // 1) worst snr tone
      // 2) two high snr tones  if 1.5dB higher than worst tone
      // 3) two tones which can lose 3dB and bit
      // 4) two tones which can gain 2dB and bit
      // 5) two tones which can gain 3dB and bit

      puca_msg_body = &uca_msg_body[0]; // place pointer to message buffer

      // compose an AOC request for bits/gainswap (only payload)
      uc_Bitswap_type = RxBitSwapComposeMessage(   &t_OVHD_AOC_bitswapTones,
                                       puca_msg_body,
                                       &s_ActualNumBitsToSwap,
                                       uc_GnswpBypassMode);
         // XDSLRTFW-2028(Start)
         // Don't Request Bitswap if  RxMinMargin above 12dB.[BugFix for Jira : XDSLRTFW-1713]
         // We need Bitswap below 12dB for RFI Test [TR067 8.1.1 BitSwap functional Test] for  stable link
         if(((gs_CurrentCoChipset == ANCDA_CO_CHIPSET) && (gs_RxMinMargin > 0x0C00))
            ||((gs_CurrentCoChipset == TI_CO_CHIPSET)&&(gs_RxMinMargin > 0x0300))
           )
         {
            uc_Bitswap_type = 0;
         }
         // XDSLRTFW-2028(End)
   } // end else (i.e. non-manual BitSwap)
   //XDSLRTFW-686: PERF_DS_DMT_ALL_Bitswap(start)
   if(gs_RxMinMargin < -256){
      gft_InterimStrdSNRBitswapOn = 1;
   } else {
      gft_InterimStrdSNRBitswapOn = 0;
   }
   gft_BitswapAlgoExcecutionInProgress = 0;

    if(!uc_Bitswap_type || gft_InvalidSNR)  // neither manual nor automatic bit swap
      return(FALSE);  // no bitswap jumps out of function
   //XDSLRTFW-686: PERF_DS_DMT_ALL_Bitswap(end)
//SMS00746477 PERF_DS_DMT_ALL_Gainswap (End)


   /* Form Bit Swap Request Message (header etc)*/
   s_AOCRxMsgId = TxAocMsgBSR((uint8) s_ActualNumBitsToSwap, uca_msg_body, &gt_TxAocMsgBSR);

        // if bitswap message formed, set outstanding flag for transmission
   gft_SendAocMsg_Flag = OUTSTANDING;



   return (TRUE);
}
//SMS00746477 PERF_DS_DMT_ALL_Gainswap (Start)
#undef PHY_BSWP_MAX_FINE_GAIN_MINUS_1DB
#undef PHY_BSWP_MAX_FINE_GAIN_MINUS_2DB
#undef PHY_BSWP_MAX_FINE_GAIN_MINUS_3DB
#undef PHY_BSWP_MIN_FINE_GAIN_PLUS_3DB
#undef PHY_BSWP_MIN_FINE_GAIN_PLUS_2DB
#undef PHY_BSWP_MIN_FINE_GAIN_PLUS_1DB
#undef PHY_BSWP_FAST_THRESH_1BIT
#undef PHY_BSWP_FAST_THRESH_2BIT

#undef BEST_ONE_TONE_GAIN_2BIT
#undef BEST_TWO_TONE_GAIN_1BIT
#undef BEST_ONE_TONE_GAIN_1BIT
//SMS00746477 PERF_DS_DMT_ALL_Gainswap (End)
