/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2003 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;  Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413
;  Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;
;  File Name: bitswap.c
;
;  This file contains functions for implementing G.992.1/G.992.2 bit-swapping.
;
;***************************************************************************/

#include <string.h>
#include "config.h"
#include "common.h"
#include "aoc.h"
#include "dsp_op.h"
#include "gdata.h"
#include "bitload.h"
#include "tone_ord.h"
#include "fifo.h"
#include "dsp_op2.h"
#include "cmv.h"
#include "DSLEngin.h"
#include "tx_aoc.h"
#include "bitswap.h"
#include "decimalgain.h"
#include "gdata_bis.h"
#include "bitload_const.h"
#include "dec_gain.h"
#include "olr_Bpn_check.h"
#include "minmaxmargin.h"
#include "showinit_olr.h"
#include "data_alloc.h"
#include "pll.h"
#include "trail.h"


/* =========================================================================== */
/* Global Variables */
/* =========================================================================== */
/*****************************************************************************
;  Subroutine Name: StoreRxBitSwapInfo( )
;  This subroutine stores the RX side bit swap information.
;
;  Prototype:
;  void StoreRxBitSwapInfo(AocMsgStruc_t *pt_TxMsg, AocMsgStruc_t *pt_RxMsg)
;
;  Input Arguments:
;     pt_TxMsg -- pointer to the structure containing the BSR or EBSR message
;              being sent by the near-end modem
;
;     pt_RxMsg -- pointer to the structure containing the BSA message sent
;              from the far-end modem
;
;  Output Arguments:
;     none
;
;  Return:
;     none
;
;  Global Variables:
;     gt_RxBitSwapInfo containing:
;        uca_msg_body   -- (O) BSR or EBSR message sent from the near-end modem
;        uc_SuperframeCount-- (O) super frame count after which the RX bit swap
;                          should take place
*******************************************************************************/
void StoreRxBitSwapInfo(AocMsgStruc_t *pt_TxMsg, AocMsgStruc_t *pt_RxMsg)
{

   int16 i;
   uint8 *puc_src, *puc_dest;

   /* Copy the message body to gt_RxBitSwapInfo; */
   puc_src = pt_TxMsg->uca_msg_body;
   puc_dest = gt_RxBitSwapInfo.uca_msg_body;
   for(i=pt_TxMsg->uc_msg_length; i>0; i--)
      *puc_dest++ = *puc_src++;

   /* Store super frame count */
   gt_RxBitSwapInfo.uc_SuperframeCount = pt_RxMsg->uca_msg_body[2];

   // Log Bitswap info
   if (gs_BitSwapLogIndex < (2*MAX_STATES - 20))
   {
      gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) 0xBBBB;        /* delimiter indicating DS Bitswap Ack information */
      gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) gus_ShowtimeSNRUpdateCount;   /* rx symbol count */
      gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) AOC_MSG_ID_BSA;   /* aoc message id */
      gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) gt_RxBitSwapInfo.uc_SuperframeCount;/* Bitswap SuperFrame Cnt */
   }
}

/*****************************************************************************
;  Subroutine Name: StoreTxBitSwapInfo( )
;  This subroutine stores the TX side bit swap information.
;
;  Prototype:
;  void StoreTxBitSwapInfo(AocMsgStruc_t *pt_RxMsg)
;
;  Input Arguments:
;     pt_RxMsg -- pointer to the structure containing the BSR or EBSR message
;              being sent by the far-end modem
;
;  Output Arguments:
;     none
;
;  Return:
;     SUCCEED  -- if the received message is valid
;     FAIL  -- if the received message is not valid
;
;  Global Variables:
;     gt_TxBitSwapInfo containing:
;        uca_msg_body   -- (O) BSR or EBSR message sent from the far-end modem
;        uc_SuperframeCount-- (O) super frame count after which the TX bit swap
;                          should take place
;     gs_bitswap_tx_sframe_count    -- (I) current TX superframe counter number
;
*******************************************************************************/
int16 StoreTxBitSwapInfo(AocMsgStruc_t *pt_RxMsg)
{
   int32 i, l_Acc;
   int32 la_TxBitSwapGainTbl[5] = {BS_GAIN_INC1, BS_GAIN_INC2, BS_GAIN_INC3, BS_GAIN_DEC1, BS_GAIN_DEC2};

   int16 s_TotalBits;
   int16 s_TotalTonesBefore = 0;
    int16 s_TotalBitsGain = 0, s_TotalTonesGain = 0;

    uint8 uca_BAT[TX_NUM_TONES], *puca_Bat, *puca_LocalBat;      // TX Bit Allocation Table copy

   uint8 *puc_src, *puc_dest;
    uint8  uc_command, uc_tone;


    puca_Bat = (uint8*)(guca_TxBat+gs_TxFirstChannel);
   puca_LocalBat = (uint8*)(uca_BAT+gs_TxFirstChannel);

   for(i=gs_TxLastChannel-gs_TxFirstChannel+1; i>0; i--) {
      if(*puca_Bat> 0) {
         s_TotalTonesBefore++;
      }
      *puca_LocalBat++ = *puca_Bat++;
   }

   /* Check the validality of the message */
   s_TotalBits = 0;
   puc_src=pt_RxMsg->uca_msg_body;
   puc_src++;

   // Log Bitswap info
   if (gs_BitSwapLogIndex < (2*MAX_STATES - 20))
   {
      gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) 0xCCCC;        /* delimiter indicating US Bitswap Request information */
      gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) gus_ShowtimeSNRUpdateCount;   /* rx symbol count */
      gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) gs_AOCRxMsgId; /* aoc message id */
      for (i=0; i<pt_RxMsg->uc_msg_length-1; i++)
         gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16)(*(puc_src+i)) ;   /* aoc message, variable length */

   }

   for(i=(pt_RxMsg->uc_msg_length >> 1); i>0; i--) {

      uc_command = *puc_src++;
      uc_tone = *puc_src++;

      if(uc_command != AOC_CMD_DO_NOTHING) {       // otherwise ignore

         if((uc_tone < gs_TxFirstChannel) ||
            (uc_tone > gs_TxLastChannel)) {
            gs_DMT_USBitSwapErrorCode = REQ_INVALID_TONE;
            return(FAIL);
         }

         switch(uc_command) {

            /* Increase 1 bit for the designated subchannel */
            case AOC_CMD_BIT_INC1:
               s_TotalBitsGain++;
               uca_BAT[uc_tone]++;
               if (uca_BAT[uc_tone]>RX_MAX_BITS_PER_TONE) {
                  gs_DMT_USBitSwapErrorCode = REQ_INVALID_INC_Bi;
                  return(FAIL) ;
               }
               else if (uca_BAT[uc_tone]==1) {
                  s_TotalTonesGain++;
               }
               break;

            /* Decrease 1 bit for the designated subchannel */
            case AOC_CMD_BIT_DEC1:
               s_TotalBitsGain--;
               if (uca_BAT[uc_tone]==0) {
                  gs_DMT_USBitSwapErrorCode = REQ_INVALID_DEC_Bi;
                  return(FAIL) ;
               }
               else if (uca_BAT[uc_tone]==1) {
                  s_TotalTonesGain--;
               }
               uca_BAT[uc_tone]--;
               break;

            /* Check for the gain error */
            case AOC_CMD_PWR_INC1: case AOC_CMD_PWR_INC2:
            case AOC_CMD_PWR_INC3: case AOC_CMD_PWR_DEC1:
            case AOC_CMD_PWR_DEC2:
               /* Find requested new fine gain by multiplying delta gain and fine gain */
               /* for designated tone. */
               l_Acc = la_TxBitSwapGainTbl[uc_command - 3] * gsa_TxFineGains[uc_tone];
               /* Round since la_TxBitSwapGainTbl[] is in Q18.14 format. */
               l_Acc = round(l_Acc, 14);
               /* Reject if new fine gain exceeds the maximum (0x2AB0 = 1.334*8192 = 2.503dB) */
               /* Maximum fine gain check is relaxed by 0.003dB to account for small roundoff */
               /* errors the other side might have in their fine gain calculation. */
               if (l_Acc  > 0x2AB0)
               {
                  gs_DMT_USBitSwapErrorCode = REQ_INVALID_INC_Gi;
                  return(FAIL);
               }
               break;

            default: /* Ignore message */
               gs_DMT_USBitSwapErrorCode = REQ_INVALID_COMMAND;
               return(FAIL);
         }  /* switch(uc_command) */
      }

   }

   //DMT doesn't support 1bit tone
   puca_LocalBat = (uint8*)(uca_BAT+gs_TxFirstChannel);
   for(i=gs_TxLastChannel-gs_TxFirstChannel+1; i>0; i--) {
      if(*puca_LocalBat++==1) {
         gs_DMT_USBitSwapErrorCode = REQ_INVALID_1BITTONE;
         return(FAIL);
      }
   }

    /* If swap-in bits is not equal to swap-out bits, do the test */
   if (s_TotalBitsGain != 0 )
   {
      if (gft_TcmFlag != TRUE) {
         gs_DMT_USBitSwapErrorCode = REQ_INVALID_L_TCM_OFF;
         return(FAIL);
      }

      if(s_TotalTonesBefore & 1) {
         // For an odd number of loaded tones before bit-swap,
         // the only valid possibility is to loose 1 tone and 1 bit.
         if((s_TotalTonesGain != -1) || (s_TotalBitsGain != -1)) {
            gs_DMT_USBitSwapErrorCode = REQ_INVALID_L_TCM_ON;
            return(FAIL);
         }
      }
      else {
         // For an even number of loaded tones before bit-swap,
         // the only valid possibility is to gain 1 tone and 1 bit.
         if((s_TotalTonesGain != 1) || (s_TotalBitsGain != 1)) {
            gs_DMT_USBitSwapErrorCode = REQ_INVALID_L_TCM_ON;
            return(FAIL);
         }
      }
   }

   /* Copy the message body to gt_RxBitSwapInfo; */
   puc_src = pt_RxMsg->uca_msg_body;
   puc_dest = gt_TxBitSwapInfo.uca_msg_body;
   for(i=pt_RxMsg->uc_msg_length; i>0; i--)
      *puc_dest++ = *puc_src++;

   /* Store super frame count */
   int16 s_Temp = gs_bitswap_tx_sframe_count + NUM_SUPERFRAMES_WT_FOR_BITSWAP;
   if (s_Temp >= MAX_SFRAME_COUNT)
   {
      s_Temp -= MAX_SFRAME_COUNT;
   }
   gt_TxBitSwapInfo.uc_SuperframeCount =  (uint8)s_Temp;

// gt_TxBitSwapInfo.uc_SuperframeCount = gs_bitswap_tx_sframe_count + NUM_SUPERFRAMES_WT_FOR_BITSWAP;
// if(gt_TxBitSwapInfo.uc_SuperframeCount >= (uint8)MAX_SFRAME_COUNT)
   // gt_TxBitSwapInfo.uc_SuperframeCount -= (uint8)MAX_SFRAME_COUNT;

   return(SUCCEED);
}

/*****************************************************************************
;  Subroutine Name: PerformRxBitSwap( )
;  This subroutine performs bit swap operation.
;
;  Prototype:
;  void PerformRxBitSwap()
;
;  Output Arguments:
;     none
;
;  Return:
;     none
;
;  Global Variables:
;     guca_RxBat[]            -- (O) RX Bit Allocation Table
;     gsa_RxFineGains[]    -- (O) RX fine gain table
;     gsa_RxCombinedGains[]   -- (O) RX combined gain table
;
*******************************************************************************/
void PerformRxBitSwap()
{
   int16 i;
   uint8 uc_msg_length, uc_command, uc_tone, *puc_msg_byte;
   int16 sa_RxBitSwapGainTbl[6] = {1,2,3,-1,-2}; // Fine gain adjustments (dB) allowed by bitswap commands.

   // Disable SNR, FDQ update (critical in Decision Direct mode and SNRM update in both DD and SS mode)
   OLRPM_RxPreProcessing();

   //clear bitswap tone array
   memset(guca_RxBitswapTones, 0, sizeof(RxToneFlags));

   puc_msg_byte = gt_RxBitSwapInfo.uca_msg_body;
   if(*puc_msg_byte++ == AOC_HEADER_BSR)
      uc_msg_length = AOC_MSG_BSR_LEN;
   else
      uc_msg_length = AOC_MSG_EBSR_LEN;

   for(i=1; i<uc_msg_length; i += 2) {

      uc_command = *puc_msg_byte++;
      uc_tone = *puc_msg_byte++;

      switch(uc_command) {

         /* Increase 1 bit for the designated subchannel */
         case AOC_CMD_BIT_INC1:
            SETTONEFLAG(guca_RxBitswapTones, uc_tone);
            if (guca_RxBat[uc_tone]==0)
               gus_ncloaded++;

            guca_RxBat[uc_tone]++;
            break;

         /* Decrease 1 bit for the designated subchannel */
         case AOC_CMD_BIT_DEC1:
            SETTONEFLAG(guca_RxBitswapTones, uc_tone);

            guca_RxBat[uc_tone]--;

            if (guca_RxBat[uc_tone]==0)
               gus_ncloaded--;
            break;

         /* Increase/decrease fine gain for the designated subchannel */
         case AOC_CMD_PWR_INC1: case AOC_CMD_PWR_INC2: case AOC_CMD_PWR_INC3:
         case AOC_CMD_PWR_DEC1: case AOC_CMD_PWR_DEC2:
            SETTONEFLAG(guca_RxBitswapTones, uc_tone);

            gsa_RxFineGains[uc_tone] += (sa_RxBitSwapGainTbl[uc_command - 3] << 8);
            break;

         default: /* Ignore it */
            break;
      }  /* switch(pt_BitSwapInfo->uca_msg_body[i]) */
   }
   // Calculate Combined gain
   CalcDecodGain(guca_RxBat, gsa_RxFineGains, gsa_RxCombinedGains, 0, gs_RxNumTones, FALSE, guca_RxBitswapTones);

   //Perform tone-ordering
   ToneOrdering(guca_RxBat, gs_RxNumTones, (int16)RX_MAX_BITS_PER_TONE, gsa_RxToneOrder);
   gs_FirstNonzeroRxTone = gs_FirstNonzeroTone;

   // Log Bitswap info
   if (gs_BitSwapLogIndex < (2*MAX_STATES - 20))
   {
      gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) 0xEEEE;        /* delimiter indicating US Bitswap Ack information */
      gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) gus_ShowtimeSNRUpdateCount;   /* rx symbol count */

      for (i = 0; i < 256 ; i++)
      {
         if (IS_TONEFLAGSET(guca_RxBitswapTones, i))
         {
            gsa_StatesTrail[gs_BitSwapLogIndex++] = i;
            gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16)guca_RxBat[i];
            gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16)gsa_RxFineGains[i];
         }
      }
   }

   gft_BitSwapPerform = BKGDTASK_DONE;
}

/*****************************************************************************
;  Subroutine Name: PerformTxBitSwap( )
;  This subroutine performs bit swap operation.
;
;  Prototype:
;  void PerformTxBitSwap()
;
;  Output Arguments:
;     none
;
;  Return:
;     none
;
;  Global Variables:
;     guca_TxBat[]            -- (O) TX Bit Allocation Table
;     gsa_TxFineGains[]    -- (O) TX fine gain table
;     gsa_TxCombinedGains[]   -- (O) TX combined gain table
;
*******************************************************************************/
void PerformTxBitSwap()
{
   int32 i, l_Acc;
   int32 la_TxBitSwapGainTbl[5] = {BS_GAIN_INC1, BS_GAIN_INC2, BS_GAIN_INC3, BS_GAIN_DEC1, BS_GAIN_DEC2};
   int16 s_qc;
   uint8 uc_msg_length, uc_command, uc_tone, *puc_msg_byte;
   uint8 uca_swap_tone[AOC_MSG_EBSR_LEN>>1];

   puc_msg_byte = gt_TxBitSwapInfo.uca_msg_body;
   if(*puc_msg_byte++ == AOC_HEADER_BSR)
      uc_msg_length = AOC_MSG_BSR_LEN;
   else
      uc_msg_length = AOC_MSG_EBSR_LEN;

   for(i=1; i<uc_msg_length; i += 2) {

      uc_command = *puc_msg_byte++;
      uc_tone = *puc_msg_byte++;
      uca_swap_tone[i>>1] = uc_tone;

      switch(uc_command) {

         /* Increase 1 bit for the designated subchannel */
         case AOC_CMD_BIT_INC1:
            guca_TxBat[uc_tone]++;
            break;

         /* Decrease 1 bit for the designated subchannel */
         case AOC_CMD_BIT_DEC1:
            guca_TxBat[uc_tone]--;
            break;

         /* Increase/decrease fine gain for the designated subchannel */
         case AOC_CMD_PWR_INC1: case AOC_CMD_PWR_INC2: case AOC_CMD_PWR_INC3:
         case AOC_CMD_PWR_DEC1: case AOC_CMD_PWR_DEC2:
            /* Update the fine gain by multiplying delta gain and fine gain for designated tone */
            l_Acc = la_TxBitSwapGainTbl[uc_command - 3] * gsa_TxFineGains[uc_tone];
            l_Acc = round(l_Acc, 14);     /* since la_TxBitSwapGainTbl[] is in Q18.14 format */
            gsa_TxFineGains[uc_tone] = sature16(l_Acc);
            break;

         default: /* Ignore it */
            break;
      }  /* switch(pt_BitSwapInfo->uca_msg_body[i]) */
   }

   //update combined gain
   for(i=0; i < (uc_msg_length>>1); i++) {
      uc_tone = uca_swap_tone[i];

      //Get the contellation size (bit loading) for this channel
      s_qc = guca_TxBat[uc_tone];

      if(s_qc == 0) {
         gsa_TxCombinedGains[uc_tone] = 0;
      }
      else {

         //Get the new QAM decoder constellation gain for this channel
         l_Acc = (int32) gsa_QAMEncGainTab[s_qc];

         //=========================================================
         //Compute the new combined gain = s_gain*QAMGain
         //=========================================================
         l_Acc *= gsa_TxFineGains[uc_tone];
         l_Acc = round(l_Acc, 13);
         gsa_TxCombinedGains[uc_tone] = sature16(l_Acc);
      }
   }

   //Perform tone-ordering
   ToneOrdering(guca_TxBat, gs_TxNumTones, (int16)TX_MAX_BITS_PER_TONE, gsa_TxToneOrder);
   gs_FirstNonzeroTxTone = gs_FirstNonzeroTone;    /* gs_FirstNonzeroTone is set by ToneOrdering() */

   //Add functions to load BAT, Tone Reordering Tables and Gains to fifo
   AddFunctionToFifo(gp_TxLoadingFunctionFifo,LoadTxBitSwap);

}

/*****************************************************************************
;  Subroutine Name: CheckForTxBitSwap( )
;
;  This subroutine check to see if it is time to perform the TX bit swap and
;  then performs the bit swap if time arrives
;
;  Prototype:
;  void CheckForTxBitSwap(void)
;
;  Input Arguments:
;     none
;
;  Output Arguments:
;     none
;
;  Return:
;     none
;
;  Global Variables:
;     gs_bitswap_tx_sframe_count    -- (I) TX superframe count
;     gt_TxBitSwapInfo     -- (I) structure containing TX bit swap info
;
;     gft_TxBitSwapFlag    -- (O) TX bit swap flag
;     guca_TxBat[]            -- (O) TX Bit Allocation Table
;     gsa_TxFineGains[]    -- (O) TX fine gain table
;     gsa_TxCombinedGains[]   -- (O) TX combined gain table
;
*******************************************************************************/
void CheckForTxBitSwap(void)
{
   if(gs_bitswap_tx_sframe_count == gt_TxBitSwapInfo.uc_SuperframeCount) {

      /* Perform bit swap */
      PerformTxBitSwap();

      /* Clear TX bit swap flag */
      gft_TxBitSwapFlag = FALSE;
   }
}

/*****************************************************************************
;  Subroutine Name: CheckForRxBitSwap( )
;
;  This subroutine check to see if it is time to perform the RX bit swap and
;  then performs the bit swap if time arrives
;
;  Prototype:
;  void CheckForRxBitSwap(void)
;
;  Input Arguments:
;     none
;
;  Output Arguments:
;     none
;
;  Return:
;     none
;
;  Global Variables:
;     gl_rx_sframe_count      -- (I) RX superframe count
;     gt_RxBitSwapInfo     -- (I) structure containing RX bit swap info
;
;     gsa_RxBat[]          -- (O) RX Bit Allocation Table
;     gsa_RxFineGains[]    -- (O) RX fine gain table
;     gsa_RxCombinedGains[]   -- (O) RX combined gain table
;
*******************************************************************************/
#define LOG_BITSWAP  //by default, we turn on debug log of Rx BITSWAP. This is good for understanding the detail.
               //We can map the log to states trail, which would be better.

#ifdef LOG_BITSWAP
DATA_MAP_DMT
int16 adr_BsDbgArray[12];
DATA_MAP_END
#endif

void CheckForRxBitSwap(void)
{
   int16 s_SuperFrameCnt;

   //bitswap superframe counter is MOD(256)
   s_SuperFrameCnt = gt_RxBitSwapInfo.uc_SuperframeCount-4;
   if (s_SuperFrameCnt<0)
      s_SuperFrameCnt += (int16)MAX_SFRAME_COUNT;

   // PerformRxBitswap is now used as background function. We should start it earlier than the required bitswap
   // reconfiguration time. Here we think 4 superframe ealier is long enough for finishing it.

   if(gs_bitswap_rx_sframe_count == s_SuperFrameCnt) {


      /* Perform bit swap */
      AddFunctionToBkgdFifo((PtrToBkgdFunc)PerformRxBitSwap);

      gft_BitSwapPerform = BKGDTASK_IN_PROGRESS;
   }

   // We should load b/g/trb table at the right required bitswap reconfiguration time though.

   if (gs_bitswap_rx_sframe_count == gt_RxBitSwapInfo.uc_SuperframeCount) {

      if (gft_BitSwapPerform == BKGDTASK_DONE)
      {
         //Add functions to load BAT, Tone Reordering Tables and Gains to fifo
         AddFunctionToFifo(gp_RxLoadingFunctionFifo,LoadRxBitSwap);
         // Bitswap changes bi/gi table -- setup Iridia Reverb Segue Detector and f/w RS detection bins again
         // We do not use Iridia Revrb Segue Detector in DMT, but we update it anyways since we are assured that this
         // loadind function will conveniently be executed during sync symbol FDQ TC task (ie. when FDQ is not active)
         PreSelectReverbSegueDetector();
         AddFunctionToFifo(gp_RxLoadingFunctionFifo,ReverbSegueDetectorSetup);
         gft_BitSwapPerform = BKGDTASK_WAITING;
      }

      //else //something bad happen Is it necessary(?)
         //we should restore the initial SWEngine b/g/trb table
   }

      //bitswap superframe counter is MOD(256)
   s_SuperFrameCnt = gt_RxBitSwapInfo.uc_SuperframeCount+1;
   if (s_SuperFrameCnt>= (int16)MAX_SFRAME_COUNT)
      s_SuperFrameCnt -= (int16)MAX_SFRAME_COUNT;


   if (gs_bitswap_rx_sframe_count == s_SuperFrameCnt) {


      /* reset flag to false */
      gft_ShowtimeSnrUpdateFlag = FALSE;

      // Enable SNR, FDQ update (critical in Decision Direct mode and SNRM update in both DD and SS mode)
      OLRPM_RxPostProcessing(0);

      /* Clear RX bit swap flag */
      gft_RxBitSwapFlag = FALSE;

      /* Set the flag to IDLE so the local modem can initiate another RX bit swap */
      gft_SendAocMsg_Flag = IDLE;

#ifdef LOG_BITSWAP
      adr_BsDbgArray[11]++ ;
#endif
   }
}

/*********************************

Following code should be called every symbol in ShowtimeRX function (or sych byte hander) to initiate BitSwap:

1)
   if(gft_SendAocMsg_Flag == OUTSTANDING) {
      if(TxAocMsgPut(&gt_TxAocMsgBSR) == SUCCEED)
         gft_SendAocMsg_Flag = SUCCEED;
   }

======================================
Following code should be called at the very beginning of each RX or TX superframe:

1)
   if(gft_TxBitSwapFlag == TRUE)
      CheckForTxBitSwap()

2)
   if(gft_RxBitSwapFlag == TRUE)
      CheckForRxBitSwap()


//=====================================
Following code should be called every time SNRs are recomputed:

   if((gft_SendAocMsg_Flag == IDLE)
      RxBitSwapDecision(gsa_RxShowtimeSnrBuf);

**********************************/

/*****************************************************************************
;  Subroutine Name: TxAocMsgBSR(...)
;
;  This subroutine forms Bit Swap Request Message or Extended Bit Swap Message.
;  Here, it is assumed that the number of bits removed from one sub-channel is always
;  equal to the number of bits added to another subchannel. However, the fine gains
;  of any two sub-channels can be modified with either positive or negative.
;
;  Prototype:
;     void TxAocMsgBSR(uint8 uc_Bits, uint8 uc_BitIncChanIdx, uint8 uc_BitDecChanIdx,
;                 int8 *pca_Gain, uint8 puca_GainChanIdx, AocMsgStruc_t *pt_msg)
;
;  Input Arguments:
;     uc_Bits           -- number of bits to be swapped
;     uc_BitIncChanIdx  -- index of subchannel with bit increased
;     uc_BitDecChanIdx  -- index of subchannel with bit decreased
;     pca_Gain       -- pointer to a pair of fine gain change values (in dB)
;     puca_GainChanIdx  -- pointer to a pair of subchannels associated with fine gain
;
;  Output Arguments:
;     pt_msg      -- pointer to the output message structure
;
;  Return:
;
;  Global Variables:
;
*******************************************************************************/
uint8 TxAocMsgBSR(uint8 uc_Bits, uint8 *puc_msg_body, AocMsgStruc_t *pt_msg)
{
   int16 i;
   uint8 uc_msg_id, *puc_msg_dest;

   puc_msg_dest = pt_msg->uca_msg_body;

   /* ====================================================================== */
   /* Form message header and length */
   /* ====================================================================== */
   switch(uc_Bits) {

      /* For 0 and 1 bit swap, form BSR message */
      case 0: case 1:

         /* set message length */
         i = pt_msg->uc_msg_length = AOC_MSG_BSR_LEN;

         /* set message header */
         *puc_msg_dest++ = AOC_HEADER_BSR;

         uc_msg_id = AOC_MSG_ID_BSR;

         //increment the counter
         gt_DSOLRPMCnt.t_BitswapCnt.us_SimpleOLRPMReq_Cnt++;

         break;

      /* For 2 bit swap, form EBSR message */
      case 2:

         /* set message length */
         i = pt_msg->uc_msg_length = AOC_MSG_EBSR_LEN;

         /* set message header */
         *puc_msg_dest++ = AOC_HEADER_EBSR;

         uc_msg_id = AOC_MSG_ID_EBSR;

         //increment the counter
         gt_DSOLRPMCnt.t_BitswapCnt.us_ExtendOLRPMReq_Cnt++;

         break;

      /* Cannot swap more than 2 bits */
      default:
         uc_msg_id = AOC_MSG_ID_UNDEFINED;
         goto _end;

   } /* switch(uc_Bits) */

   /* ====================================================================== */
   /* Copy the message body */
   /* ====================================================================== */
   for(i-=1; i>0; i--)
      *puc_msg_dest++ = *puc_msg_body++;

_end:
   /* set message type */
   pt_msg->uc_msg_id = uc_msg_id;
   return(uc_msg_id);

}

/*****************************************************************************
;  Subroutine Name: TxAocMsgBSA(...)
;
;  This subroutine forms Bit Swap Acknowledgement Message.
;
;  Prototype:
;     void TxAocMsgBSA(uint8 uc_SuperFrameCount, AocMsgStruc_t *pt_msg)
;
;  Input Arguments:
;     uc_SuperFrameCount -- Bit swap supperframe counter number
;
;  Output Arguments:
;     pt_msg      -- pointer to the output message structure
;
;  Return:
;
;  Global Variables (defined in other files):
;
;   Global Variables (defined in this file):
;
;
*******************************************************************************/
void TxAocMsgBSA(uint8 uc_SuperFrameCount, AocMsgStruc_t *pt_msg)
{
   /* set message length */
   pt_msg->uc_msg_length = AOC_MSG_BSA_LEN;

   /* set message body */
   pt_msg->uca_msg_body[0] = AOC_HEADER_BSA;
   pt_msg->uca_msg_body[1] = AOC_CMD_BSA;
   pt_msg->uca_msg_body[2] = uc_SuperFrameCount;

   /* set message type */
   pt_msg->uc_msg_id = AOC_MSG_ID_BSA;

   // Log Bitswap info
   if (gs_BitSwapLogIndex < (2*MAX_STATES - 20))
   {
      gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) 0xDDDD;        /* delimiter indicating US Bitswap Ack information */
      gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) gus_ShowtimeSNRUpdateCount;   /* rx symbol count */
      gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) AOC_MSG_ID_BSA;   /* aoc message id */
      gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) uc_SuperFrameCount;/* Bitswap SuperFrame Cnt */
   }
}

