/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2001 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;  Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413
;  Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;
;  File Name: prf_man.c
;
;  This file contains functions for implementing profile management.
;
;***************************************************************************/
#include <string.h>
#include "config.h"
#include "common.h"
#include "gdata.h"
#include "xgdata.h"
#include "aoc.h"
#include "tx_aoc.h"



/*****************************************************************************
;  Subroutine Name: StoreTxProfile(...)
;
;  This subroutine stores the current TX profile to TX profile memory.
;
;  Prototype:
;     void StoreTxProfile(uint8 uc_profile_number)
;
;  Input Arguments:
;     uc_profile_number -- number of profile to be stored
;
;  Output Arguments:
;     none
;
;  Return:
;     SUCCEED  -- if the selected profile is store
;     FAIL  -- if profile number is greater than the maximum profile number
;
;  Global Variables (defined in other files):
;     gt_tx_config -- (I) structure of current TX configuration parameters
;     gta_TXProfile -- (O) a array of structures for storing TX profiles
;
;   Global Variables (defined in this file):
;
*******************************************************************************/

void StoreTxProfile(uint8 uc_profile_number)
{
   TXProfileStruc_t *pt_profile;
   int16 i, s_BitsPerSymbol;
   int16 *ps_fg_src, *ps_fg_dest;
   uint8 *ps_bat_src, *ps_bat_dest;

   /* Update the total number of TX profiles stored */
   pt_profile = (TXProfileStruc_t*) (gta_TXProfile + uc_profile_number);
   if(pt_profile->s_ReqdBits == 0)
      guc_num_tx_profiles_stored++;

   /* Compute the total number of bits per symbol */
   s_BitsPerSymbol = (gt_tx_config.sa_Bpn[INTERLEAVE_DATA_PATH][0] +
      (gt_tx_config.s_Rp[INTERLEAVE_DATA_PATH]/gt_tx_config.s_Mp[INTERLEAVE_DATA_PATH]) + 1) << 3;

   /* ========================================================================== */
   /* Save the current TX profile */
   /* ========================================================================== */
   pt_profile->s_ReqdBits = s_BitsPerSymbol;
   if(gt_tx_config.s_Mp[INTERLEAVE_DATA_PATH] == S_HALF) {
      pt_profile->s_R = gt_tx_config.s_Rp[INTERLEAVE_DATA_PATH];
   } else {
      pt_profile->s_R = gt_tx_config.s_Rp[INTERLEAVE_DATA_PATH]/gt_tx_config.s_Mp[INTERLEAVE_DATA_PATH];
   }
   pt_profile->s_S = gt_tx_config.s_Mp[INTERLEAVE_DATA_PATH];
   pt_profile->s_D = gt_tx_config.s_Dp[INTERLEAVE_DATA_PATH];

   {


      ps_bat_src = guca_TxBat;
      ps_bat_dest = pt_profile->sa_bat;
      ps_fg_src = gsa_TxFineGains;
      ps_fg_dest = pt_profile->sa_gain;
      for(i=gs_TxNumTones; i>0; i--) {
         *ps_bat_dest++ = *ps_bat_src++;
         *ps_fg_dest++ = *ps_fg_src++;
      }
   }


}

/*****************************************************************************
;  Subroutine Name: StoreRxProfile(...)
;
;  This subroutine first checks to see if the profile number in PMA matches
;  to that in PMR. If the match is not found, return FAIL. Otherwise,
;  it will store the current RX profile to the profile memory and update
;  the profile index array which contains the profile numbers
;  in descending bits per symbol.
;
;  Prototype:
;     uint8 StoreRxProfile(AocMsgStruc_t *pt_TxMsg, AocMsgStruc_t *pt_RxMsg)
;
;  Input Arguments:
;     pt_TxMsg -- pointer to the current TX outstanding message
;     pt_RxMsg -- pointer to the current RX outstanding message
;
;  Output Arguments:
;     none
;
;  Return:
;     SUCCEED  -- if the selected profile is store
;     FAIL  -- if profile numbers do not match
;
;  Global Variables (defined in other files):
;     gt_rx_config      -- (I) structure containing TX configuration parameters
;     gta_RXProfile     -- (O) a array of structures for storing profiles
;     gsa_rx_profile_ord   -- (I/O) sorted profile numbers
;
;   Global Variables (defined in this file):
;     guc_remote_max_profiles -- (O) maximum number of profiles supported by
;                          remote terminal
;
*******************************************************************************/
uint8 StoreRxProfile(AocMsgStruc_t *pt_TxMsg, AocMsgStruc_t *pt_RxMsg)
{
   RXProfileStruc_t *pt_profile;
   uint8 uc_profile_number;
   int16 i, s_BitsPerSymbol;
   int16 *ps_fg_src, *ps_bg_dest;
   uint8 *ps_bat_src;

   /* If the RX profile number is not equal to the TX profile number */
   /* declare unmatch */
   uc_profile_number = pt_RxMsg->uca_msg_body[2] & 0x0F;
   if(uc_profile_number != pt_TxMsg->uca_msg_body[2])
      return(FAIL);

   /* Store the maximum profile number the remote terminal can save */
   guc_remote_max_profiles = (pt_RxMsg->uca_msg_body[2]>>4)+1;

   /* ========================================================================= */
   /* Store RX profiles */
   /* ========================================================================= */

   /* Compute the total number of bits per symbol */
   if(gt_rx_config.s_Mp[INTERLEAVE_DATA_PATH] == S_HALF) {
      s_BitsPerSymbol = (gt_rx_config.sa_Bpn[INTERLEAVE_DATA_PATH][0] +
                     gt_rx_config.s_Rp[INTERLEAVE_DATA_PATH] + 1) << 3;
   } else {
      s_BitsPerSymbol = (gt_rx_config.sa_Bpn[INTERLEAVE_DATA_PATH][0] +
         (gt_rx_config.s_Rp[INTERLEAVE_DATA_PATH]/gt_rx_config.s_Mp[INTERLEAVE_DATA_PATH]) + 1) << 3;
   }

   /* Save the current RX profile */
   pt_profile = (RXProfileStruc_t*) (gta_RXProfile + uc_profile_number);
   pt_profile->s_ReqdBits = s_BitsPerSymbol;
   if(gt_rx_config.s_Mp[INTERLEAVE_DATA_PATH] == S_HALF) {
      pt_profile->s_R = gt_rx_config.s_Rp[INTERLEAVE_DATA_PATH];
   } else {
      pt_profile->s_R = gt_rx_config.s_Rp[INTERLEAVE_DATA_PATH]/gt_rx_config.s_Mp[INTERLEAVE_DATA_PATH];
   }
   pt_profile->s_S = gt_rx_config.s_Mp[INTERLEAVE_DATA_PATH];
   pt_profile->s_D = gt_rx_config.s_Dp[INTERLEAVE_DATA_PATH];

   {

      ps_bat_src = guca_RxBat;
      ps_fg_src = gsa_RxFineGains;
      ps_bg_dest = pt_profile->sa_BG;
      for(i=gs_RxNumTones; i>0; i--)
         *ps_bg_dest++ = (*ps_fg_src++ | *ps_bat_src++);
   }



   /* ============================================================================ */
   /* Store profile numbers in an array such that the first entry of this array */
   /* contains the profile number of the maximum bit loading and the last entry */
   /* contains the profile number of the minimum bit loading */
   /* ============================================================================ */

   {
      int16 *ps_order;

      /* First check to see if this profile number has been stored previously */
      ps_order = gsa_rx_profile_ord;
      for(i=0; i<guc_num_rx_profiles_stored; i++, ps_order++) {

         if(*ps_order == uc_profile_number) {

            guc_num_rx_profiles_stored--;
            break;
         }
      }

      /* Shift the data in gsa_rx_profile_ord[] to remove the pre-stored uc_profile_number */
      for(; i<guc_num_rx_profiles_stored; i++, ps_order++)
         *ps_order = *(ps_order+1);

      /* Insert new profile index in gsa_rx_profile_ord[]. */
      ps_order = (int16*) (gsa_rx_profile_ord + guc_num_rx_profiles_stored-1);
      for(; ps_order>=gsa_rx_profile_ord; ps_order--) {

         if(pt_profile->s_ReqdBits > gta_RXProfile[*ps_order].s_ReqdBits)
            *(ps_order+1) = *ps_order;
         else
            break;
      }
      *(ps_order+1) = uc_profile_number;
      guc_num_rx_profiles_stored++;
   }

   return(SUCCEED);
}

/*****************************************************************************
;  Subroutine Name: EraseAllProfiles
;
;  Prototype:
;     void EraseAllProfiles(FlagT ft_indicator)
;
;  Input Arguments:
;     ft_indicator --   TX or RX
;
;  Output Arguments:
;     none
;
;  Return:
;     N/A
;
;  Global Variables:
;     gta_TXProfile  -- (O) array of Tx profiles
;     gta_RXProfile  -- (O) array of Rx profiles
;     guc_num_tx_profiles_stored -- (0)
;     guc_num_rx_profiles_stored -- (0)
;
*******************************************************************************/
void EraseAllProfiles(FlagT ft_indicator)
{
   /* Erase all TX profiles */
   if(ft_indicator == TX) {

      memset(gta_TXProfile, 0, NUM_PROFILES*sizeof(TXProfileStruc_t));
      guc_num_tx_profiles_stored = 0;
   }
   else { /* Erase all RX profiles */

      memset(gta_RXProfile, 0, NUM_PROFILES*sizeof(RXProfileStruc_t));
      guc_num_rx_profiles_stored = 0;
   }
}

/*****************************************************************************
;  Subroutine Name: TxAocMsgPMR(...)
;
;  This subroutine forms a Profile Management Request Message.
;
;  Prototype:
;     void TxAocMsgPMR(uint8 uc_profile_number, AocMsgStruc_t *pt_msg)
;
;  Input Arguments:
;     uc_profile_number -- current profile number
;
;  Output Arguments:
;     pt_msg      -- pointer to the output message structure
;
;  Return:
;
;  Global Variables (defined in other files):
;
;   Global Variables (defined in this file):
;
;
*******************************************************************************/

void TxAocMsgPMR(uint8 uc_profile_number, AocMsgStruc_t *pt_msg)
{
   /* set message length */
   pt_msg->uc_msg_length = AOC_MSG_PMR_LEN;

   /* set message body */
   pt_msg->uca_msg_body[0] = (uint8)AOC_HEADER_PMR;
   pt_msg->uca_msg_body[1] = (uint8)AOC_CMD_PMR;
   pt_msg->uca_msg_body[2] = uc_profile_number;

}

/*****************************************************************************
;  Subroutine Name: TxAocMsgPMA(...)
;
;  This subroutine forms a Profile Management Acknowledge Message.
;
;  Prototype:
;     void TxAocMsgPMA(uint8 uc_profile_number,
;                 uint8 uc_max_num_profiles,
;                 AocMsgStruc_t *pt_msg)
;
;  Input Arguments:
;     uc_profile_number -- current profile number
;     uc_max_num_profiles -- max. number of profiles
;
;  Output Arguments:
;     pt_msg            -- pointer to the output message structure
;
;  Return:
;
;  Global Variables (defined in other files):
;
;   Global Variables (defined in this file):
;
;
*******************************************************************************/

void TxAocMsgPMA(uint8 uc_profile_number, uint8 uc_max_num_profiles, AocMsgStruc_t *pt_msg)
{
   /* set message length */
   pt_msg->uc_msg_length = AOC_MSG_PMA_LEN;

   /* set message body */
   pt_msg->uca_msg_body[0] = (uint8)AOC_HEADER_PMA;
   pt_msg->uca_msg_body[1] = (uint8)AOC_CMD_PMA;
   pt_msg->uca_msg_body[2] = (((uc_max_num_profiles-1)<<4) | uc_profile_number);

}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : Check_SendPMR()
 *
 *  Description:  Check to see if it is necessary to send a PMR.  A PMR
 *
 *  Prototype:  void Check_SendPMR(void)
 *
 *  Input Arguments:
 *      None
 *
 *  Output Arguments:
 *      None
 *
 *  Return:
 *      None
 *
 *  Global Variables Used:
 *      gs_bitswap_tx_sframe_count          - (I)   TX Super Frame Count
 *      gft_SendAocMsg_Flag            - (I/O) Flag to indicate if a PMR should be sent
 *                                              IDLE:       do not send;
 *                                              OUTSTANDING:   send;
 *                                              SUCCEED:    already sent
 *      guc_next_rx_profile_number  - (I/O) Next RX Profile to be stored
 *      gt_TxMsgPMR                 - (O)   PMR message structure
 *      guc_remote_max_profiles     - (I)   Number of profiles that other
 *                                          ATU-x terminal can support
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void Check_SendPMR(void)
{
    /* Send a message to request the remote terminal to store the current RX profile */
   /* after SHOWTIME starts */
   if((gs_bitswap_tx_sframe_count >= TIME_TO_SEND_PMR) && (gft_SendAocMsg_Flag == OUTSTANDING))
    {
        /* Form the PMR message and store it in gt_TxMsgPMR */
      TxAocMsgPMR(guc_next_rx_profile_number, &gt_TxMsgPMR);

      /* Put the message in AOC TX buffer */
      if(TxAocMsgPut(&gt_TxMsgPMR) == SUCCEED)
        {
            /* Set it to 2 so PMR will not be sent again after this time */
         gft_SendAocMsg_Flag = SUCCEED;
      }
   }
}
