/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2002 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   RCPilotFR1RxF.c
 *
 *   Fast Retrain RX State Functions for ATU-R.
 *   Covering the RX states: R_C_PILOT_FR1,
 *
 *   Notes:
 *
 *   R_C_MSG_FR1 is implemented in analysi2.c
 *
 *------------------------------------------------------------------------
 */

#include "common.h"
#include "rt_tones.h"
#include "snr.h"
#include "pga_set.h"
#include "rx_ops.h"
#include "dsp_op.h"
#include "pll.h"
#include "gdata.h"
#include "frm_sync.h"
#include "tdq_init.h"
#include "fdq_init.h"
#include "fifo.h"
#include "fretrain.h"
#include "accum32.h"
#include "xrtstate.h"
#include "cmv.h"
#include "profile2.h"
#include "crc16.h"
#include <string.h>
#include "vecpwr.h"
#include "detect.h"
#include "memsetbf.h"
#include "bufmisc.h"
#include "afe.h"
#include "frm_sync_b.h"
#include "states.h"
#include "RCReverbFR2RxF.h"

#include "VRX_AfeCommonConst.h"
#include "VRX_AfeCommonData.h"
#include "AFED_Constants.h"
#include "AFED_Data.h"
#include "AFED_Functions.h"
#include "AFED_ReadWriteModify.h"

/*^^^
*-------------------------------------------------------------------
########                 ########
######## LOCAL FUNCTIONS ########
########                 ########
*-------------------------------------------------------------------
*^^^
*/


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: int16 DecideUC(int32 *pla_EchoPower)
 *
 *  Description: Decide absolute upstream power cutback based on
 * a nonlinear echo power measurement for different levels
 *  of TX power. The US power cutback level is chosen according to the following
 * algorithm:
 *
 * If the total nonlinear echo power (measured for 2nd and 3rd harmonics of line probe tones)
 *     is less than a preselected absolute threshold for at least one Tx cutback level,
 *    the smallest such Tx cutback level (highest Tx power) is chosen,
 *
 * else:
 *
 *    The differences in echo power for successive Tx cutback levels is considered.
 *    If the reduction in echo power  between two adjacent TX power levels is smaller than
 *    a predetermined fraction of the echo power, the disadvantage of the higher cutback
 *    is considered to outweight the benefit of the reduced echo, and the lower cutback (higher
 *    Tx power level) is chosen.
 *
 * else
 *    The maximum Tx power cutback is chosen.
 *
 *------------------------------------------------------------------------
 *
 *  Input Arguments:
 *      pla_EchoPower -- pointer to an array of echo powers each of which
 *                is correspoding to one TX power level
 *
 *  Output Arguments:
 *    none
 *
 *  Return:
 *    index which indicates the US power cutback level
 *
 *  Global Variables Used:
 *    gl_NLEchoThresh -- Absolute nonlinear echo power threshold
 *    gs_NLEchoDiffThresh_rsh -- Difference threshold of nonlinear echo power (mantissa)
 *    gs_NLEchoDiffThresh_rsh -- exponent (right shift count) of gs_NLEchoDiffThresh_rsh
 *------------------------------------------------------------------------
 *^^^
 */

/*===================================================================
WARNING??? The following two threshold values should be decided experimentally
and current values are used for test only and has no real value.
===================================================================*/

/*Absolute nonlinear echo power threshold:
If nonlinear echo power is less than this threshold,
it assumes nonlinear distortion has no effect to normal phone operation */
int16 gs_NLEchoThresh = (5<<8)>>LOGNUM_LINEPROBE_HARMONIC_TONES_TOTAL;     /*in dB and Q8.8 format*/

/*Nonlinear echo power difference threshold in percentage, represented by mantissa and exponent as:
A = (gs_NLEchoDiffThresh>>gs_NLEchoDiffThresh_rsh) (e.g., 1/8 as given in this setting)
If the percentage of decrease between previous and current
nonlinear echo power level is less than A, it indicates further TX power
cutback may not significantly reduce nonlinear echo */
int16 gs_NLEchoDiffThresh = 1;
int16 gs_NLEchoDiffThresh_rsh = 5;

int16 DecideUC(int32 *pla_EchoPower)
{
   int16 i, s_thre, s_diff;
   int32 l_Acc;
   int16 sa_EchoPower[NUM_LINEPROBE_CUTBACK_CNT];

   for(i=0; i<NUM_LINEPROBE_CUTBACK_CNT; i++){

      // Need to convert digital power values to absolute dBm.
      sa_EchoPower[i] = ConvertToDB(pla_EchoPower[i]);

      //If nonlinear echo power is less or equal to the absolute threshold
      //choose this power cutback level.
      if(sa_EchoPower[i] <= gs_NLEchoThresh)
         return(i);
   }

   // If nonlinear echo power is above threshold for all cutback levels,
   // examine echo power values and decide if at any point the marginal decrease in echo power
   // is too small to warrant the marginal power cutback.  If such a point is found
   // it determines the power cutback.  If such a point is not found the power cutback
   // will be set to the maximum possible.

   for(i=1; i<NUM_LINEPROBE_CUTBACK_CNT; i++) {

      //Compute the difference threshold which is a percentage
      //of echo power of previous step.
      l_Acc = sa_EchoPower[i-1]*gs_NLEchoDiffThresh;  //mutliply it by percentage

      s_thre = (int16)(l_Acc>>gs_NLEchoDiffThresh_rsh);  //convert back to 16-bit

      //Compute absolute value of (sa_EchoPower[i-1] - sa_EchoPower[i])
      s_diff = sa_EchoPower[i-1] - sa_EchoPower[i];
      if(s_diff < 0)
         s_diff = - s_diff;

      // If the absolute difference is less than the threshold, marginal echo power reduction
      // is not considered worth the Tx power reduction, so choose the previous cutback level.
      if(s_diff <= s_thre)
         return(i-1);

   } //for(i=1; i<NUM_LINEPROBE_CUTBACK_CNT; i++)

   // Choose maximum power cutback.

   return(NUM_LINEPROBE_CUTBACK_CNT-1);
}

/*^^^
*-------------------------------------------------------------------
########               ########
######## RX ROUTINES   ########
########               ########
*-------------------------------------------------------------------
*^^^
*/



/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: RCPilotFR1RxF
 *
 *  Description: During this time, ATU_R maintains phase and frequency lock.
 *  Rx AGC is set to a fixed value and the original value is restored
 *  at the end of line probing operations.
 *  when the TX is in the line probe state, then the power in the 2nd
 *  and 3rd harmonics is measured and is averaged over 64 symbols. Based
 *  on this measurement, the amount of cutback (in steps of 2dB) required is determined.
 *  This procedure is repeated 6 times and then the final absolute
 *  upstream fast retrain power cutback (in gs_UC) is determined by counting the
 *  number of 2dB cutbacks required. During this time PLL is disabled.
 *  PLL is turned on after UC is determined. Then, the symbols of R_C_PILOT_FR1 are used to
 *  establish a threshold for detection of the C_Reverb_FR2 signal.  The
 *  ATU-R shall progress to R_C_REVERB_FR2_RX only after it has detected the
 *  C_Reverb_FR2 signal from the ATU-C.
 *
 *  The detection threshold is determined by computing the average of the
 *  received power over a number of frames.  The threshold is set to the
 *  average power multiplied by a scaling factor.
 *
 *  Prototype: void RCPilotFR1RxF(void)
 *
 *  Input Arguments: none
 *
 *  Output Arguments: none
 *
 *  Return: none
 *
 *  Global Variables Used:
 *      gs_RxSubState                     - (I/O) Substate within current RX state
 *      gs_RxNextState                    - (O)   Rx state starting next symbol period
 *      gl_RxSymbolCount                  - (I)   # of symbol period in current state
 *      gs_RxToneBuf                   - (O)   Output of FFT
 *    gl_Rx_Total_Power                - (I/O) Total received signal power
 *       gl_Rx_C_Reverb_Thr                  - (I/O) Reverb signal detection threashold
 *
 *  Substates:
 *      R_C_PILOT_FR1_RX_INITIALIZE        - Perform initialization, such as saving the
 *                                           current PGA value and setting the PGA value.
 *      R_C_PILOT_FR1_RX_COMPUTE_HARMONICS - When the transmitter is transmitting the
 *                                           line probe signal, compute the power in 2nd
 *                                           and 3rd harmonic tones. After accumulating
 *                                           energy for 64 frames, compare the energy
 *                                           with the threshold. If the energy is greater
 *                                           than the threshold, the harmonics are too
 *                                           high and hence employ cutback. Keep a count
 *                                           of the number of cutbacks in
 *                                           guc_lineprobe_cutback.
 *      R_C_PILOT_FR1_RX_COUNT_SYMBOLS     - Continue to measure RX power and declare
 *                                           C_REVERB_FR2 received if threshold exceeded
 *                                           for R_C_REVERB1_DET_CNT consecutive symbols.
 *
 *  Notes:
 * 1) The value of the scaling factor R_C_REVERB_FR1_THR_SCALE will
 *  need to be calibrated w/ the AFE of the target system.
 * 2) State: R_C_PILOT_FR2_RX
 *
 *------------------------------------------------------------------------
 *^^^
 */
/* Local Function prototype*/
int16 DecideUC(int32 *pla_EchoPower);

/* =============================================== */
/* substates */
/* =============================================== */
#define R_C_PILOT_FR1_RX_INITIALIZE        (0)
#define R_C_PILOT_FR1_RX_PGA_WAIT          (4)
#define R_C_PILOT_FR1_RX_RESET_PLL_REF     (6)
#define R_C_PILOT_FR1_RX_COMPUTE_HARMONICS (1)
#define R_C_PILOT_FR1_RX_PGA_WAIT2        (5)
#define R_C_PILOT_FR1_RX_RESET_PLL_REF2   (7)
#define R_C_PILOT_FR1_RX_COUNT_SYMBOLS     (3)

void RCPilotFR1RxF(void) {

   switch (gs_RxSubState) {

      case R_C_PILOT_FR1_RX_INITIALIZE:

         //Wait until TX side goes to R_LINE_PROBE_TX
         if (gs_TxState == R_LINE_PROBE_TX) {

            /* save the current pga value */
            gs_PGA_before_lineprobe = gs_PGA_set;

            /* set the PGA value during lineprobe to a fixed value */
            gs_PGA_required = PGA_DURING_LINEPROBE;
            guc_PgaTrainingState = TRAINING_IN_PROGRESS;
            AddFunctionToBkgdFifo((PtrToBkgdFunc)AFED_BgPGA);
            guc_lineprobe_cutback = 0;
            guc_lineprobe_iter_cnt = 0;

            gl_Pa = 0;
            gl_Rx_Total_Power = 0;

            gs_RxSubState = R_C_PILOT_FR1_RX_PGA_WAIT;
            }
         break;



      case R_C_PILOT_FR1_RX_PGA_WAIT:

         if (guc_PgaTrainingState == TRAINING_DONE){
            AddFunctionToFifo(gp_RxLoadingFunctionFifo,AFED_SetPga);

                       /* Go to next substate. */
            gs_RxSubState = R_C_PILOT_FR1_RX_RESET_PLL_REF;
            gs_RxSubStateCnt = 0;

            /* Turn off PLL. */
            gft_EnablePLL = FALSE;
         }
         break;

           case R_C_PILOT_FR1_RX_RESET_PLL_REF:

         gs_RxSubStateCnt++;

         if (gs_RxSubStateCnt == PGA_SETTLING_TIME) {

            /* Turn on PLL */
            gft_EnablePLL = TRUE;

            /* Get the reference phase of pilot signal */
            ResetPllRefTone(gsa_RxPilotTone[0], gsa_RxPilotTone[1]);

            /* Go to next substate. */
            gs_RxSubState = R_C_PILOT_FR1_RX_COMPUTE_HARMONICS;
            gs_RxSubStateCnt = 0;
         }
         break;

      case R_C_PILOT_FR1_RX_COMPUTE_HARMONICS:

         gs_RxSubStateCnt++;

         /* Measure the power of the 2nd and 3rd harmonics of the R_LineProbe signal. */
         gl_Pa += VectorPower(gsa_RxToneBuf, (int16) (2*FIRST_R_LINEPROBE_HARMONIC2_TONE),
                        (int16) (2*NUM_LINEPROBE_HARMONIC2_TONES),
                        LOG2_R_C_LINEPROBE_AC_LEN + LOGNUM_LINEPROBE_HARMONIC_TONES_TOTAL);

         gl_Pa += VectorPower(gsa_RxToneBuf, (int16) (2*FIRST_R_LINEPROBE_HARMONIC3_TONE),
                        (int16) (2*NUM_LINEPROBE_HARMONIC3_TONES),
                        LOG2_R_C_LINEPROBE_AC_LEN + LOGNUM_LINEPROBE_HARMONIC_TONES_TOTAL);

         /* After accumulating energy for R_C_LINEPROBE_AC_LEN frames. */
         if (gs_RxSubStateCnt == R_C_LINEPROBE_AC_LEN) {

            /* Save the power of harmonics of line probe signal. */
            /* Note that value is (actual power >> LOGNUM_LINEPROBE_HARMONIC_TONES_TOTAL). */
            gla_HarmonicsPower[guc_lineprobe_iter_cnt] = gl_Pa;

            guc_lineprobe_iter_cnt++;

            if(guc_lineprobe_iter_cnt < NUM_LINEPROBE_CUTBACK_CNT) {

               /* Set up parameter for next iteration. */
               gl_Pa = 0;
               gs_RxSubState = R_C_PILOT_FR1_RX_COMPUTE_HARMONICS;
               gs_RxSubStateCnt = 0;
            }
            else {

               /* Decide the best TX power cutback. */
               guc_lineprobe_cutback = (uint8) DecideUC(gla_HarmonicsPower);

               //record the absolute Fast retrain upstream cutback in variable gs_UC
               gs_UC = (int16)guc_lineprobe_cutback;

               //record gs_UC in R_MSG_FR1
               // u0 to u4, Absolute Upstream Fast Retrain Power Cutback
               guca_RMsgFR1Tab[1] |= (guc_lineprobe_cutback & 0x1F);

               /* Restore the PGA value. */
               gs_PGA_required = gs_PGA_before_lineprobe;
               guc_PgaTrainingState = TRAINING_IN_PROGRESS;
               AddFunctionToBkgdFifo((PtrToBkgdFunc)AFED_BgPGA);

               /* Set up for next substate. */
               gl_Rx_Total_Power = 0;
               gs_RxSubState = R_C_PILOT_FR1_RX_PGA_WAIT2;
               gs_RxSubStateCnt = 0;

            } /* if(guc_lineprobe_iter_cnt < NUM_LINEPROBE_CUTBACK_CNT) else  */
         } /* if (gs_RxSubStateCnt == R_C_LINEPROBE_AC_LEN)   */
         break;

      case R_C_PILOT_FR1_RX_PGA_WAIT2:

         if (guc_PgaTrainingState  == TRAINING_DONE)
         {
            AddFunctionToFifo(gp_RxLoadingFunctionFifo,AFED_SetPga);

                        /* Go to next substate. */
            gs_RxSubState = R_C_PILOT_FR1_RX_RESET_PLL_REF2;
            gs_RxSubStateCnt = 0;

            /* Turn off PLL. */
            gft_EnablePLL = FALSE;

                  }
         break;

            case R_C_PILOT_FR1_RX_RESET_PLL_REF2:

         gs_RxSubStateCnt++;
         /* Wait for a long enough time (5+128 symbols) to avoid a
         false detection that might be caused by the echo path, then go directly
         to the substate   of R_C_PILOT_FR1_RX_COUNT_SYMBOLS.        */

         if (gs_RxSubStateCnt == (PGA_SETTLING_TIME + R_C_PILOT1_THR_LEN)) {

            /* Turn on PLL */
            gft_EnablePLL = TRUE;

            /* Get the reference phase of pilot signal */
            ResetPllRefTone(gsa_RxPilotTone[0], gsa_RxPilotTone[1]);

            /* Go to next substate. */
            gs_RxSubState = R_C_PILOT_FR1_RX_COUNT_SYMBOLS;
            gs_RxSubStateCnt = 0;
         }
         break;

      case R_C_PILOT_FR1_RX_COUNT_SYMBOLS:

      /* Detect ReverB by detecting the absence of the Pilot tone */
      if ((!DetectTone(gsa_RxToneBuf,gs_CPilotTone,(int16)(gs_CPilotTone-4),(int16)(gs_CPilotTone+4),(int16)(gs_ToneDetectScale*256)))) {
            gs_RxSubStateCnt++;
         }
         else {
            gs_RxSubStateCnt = 0;
         }

         if (gs_RxSubStateCnt == R_C_REVERB1_DET_CNT) {
            gs_RxNextState = R_C_REVERB_FR2_RX;
            gpF_RxStateFunc = (PtrToFunc)RCReverbFR2RxF;
         }


         /* Check for timeout */
         if(gl_RxSymbolCount >= (R_C_PILOT_FR1_TIMEOUT-1)) {

            gs_RxNextState = FAIL_RX;
            gpF_RxStateFunc = (PtrToFunc)ExceptionHandler;

            /* Set exception handler variables */
            gus_ExceptionState   = gs_RxState;
            gus_ExceptionCode = E_CODE_RCPilotFR1Rx_SignalEnd_Failure;
         }
         break;
   }
}

#undef R_C_PILOT_FR1_RX_INITIALIZE
#undef R_C_PILOT_FR1_RX_PGA_WAIT
#undef R_C_PILOT_FR1_RX_RESET_PLL_REF
#undef R_C_PILOT_FR1_RX_COMPUTE_HARMONICS
#undef R_C_PILOT_FR1_RX_PGA_WAIT2
#undef R_C_PILOT_FR1_RX_RESET_PLL_REF2
#undef R_C_PILOT_FR1_RX_COUNT_SYMBOLS



