/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-1998 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   profile1.c
*
*   Profile management algorithms.
*
*-------------------------------------------------------------------------
*/

#include "common.h"
#include "gdata.h"



#define PROFILE_RX_BITS    0x000F
#define PROFILE_RX_GAINS   0xFFF0

/* =============================================== */
/* static function prototypes */
/* =============================================== */
void LoadUSProfile(void);

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: DecodeCInfoFR1(void)
 *
 *  Description: Decodes the US and DS power cutbacks from the
 *  C_INFO_FR1 message.
 *
 *  Prototype: void DecodeCInfoFR1(void);
 *
 *  Arguments: none
 *
 *  Return: none
 *
 *  Global Variables Used:
 *      gs_DS_FR_CutBack    (O) Downstream cutback received from ATU-R
 *                              actual cutback applied is 2 * gs_DS_FR_cutback (dB)
 *                              from the nominal ATU-C TX PSD
 *      gs_US_FR_CutBack    (O) Upstream cutback received from ATU-R
 *                              actual cutback applied is 2 * gs_US_FR_cutback (dB)
 *      gft_FR_Truncate_Flag  (O)  = TRUE, if Fast Retrain Truncate bit received from ATU-R is set
 *                                = 0, if FR Truncate bit is not set
 *      guca_R_C_Msg_FR1[]  (I) Message received from ATU-C during R_C_MSG_FR1_RX
 *
 *------------------------------------------------------------------------
 *^^^
 */
void DecodeCInfoFR1(void) {

    /*  get Fast Retrain Politeness Power Cutback */
    gs_DPF = (int16)(guca_R_C_Msg_FR1[0] & 0x1F);

   if(guca_R_C_Msg_FR1[0] & (INVALID_PROFILE))
      gft_FR_Truncate_Flag = TRUE;
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : DecodeCInfoFR2(void)
 *
 *  Description:    Decodes the C_INFO_FR2 message.  Checks to
 *  see if the unknown profile bit was set, that the received profile
 *  number is not higher than the maximum number of profiles, and that
 *  there is a valid profile stored at that number.  If any of these
 *  conditions is not met then Profile Loading is declared failed.
 *  If all the conditions are met then the upstream profile is loaded.
 *
 *  Prototype:  void DecodeCInfoFR2(void);
 *
 *  Arguments: none
 *
 *  Return: none
 *
 *  Global Variables Used:
 *  gs_US_Profile       (O) Upstream profile number received
 *  guca_R_C_Msg_FR2[]  (I) message from ATU-C
 *  gs_ProfileLeadOK    (O) set to zero if profile load fails
 *  gta_TXProfile[]      (I) Table of US profiles
 *
 *------------------------------------------------------------------------
 *^^^
 */
void DecodeCInfoFR2(void) {

    gs_US_Profile = (int16)(guca_R_C_Msg_FR2[0] & 0x0F);

    /*  test if ATU-C reported unknown profile */
    if (guca_R_C_Msg_FR2[0] & UNKNOWN_PROFILE) {

        /*  ATU-C reported unknown profile */
        guc_ProfileLoadOK = FAIL;
        return;
    }

    /*  check for valid profile number */
    if (gs_US_Profile >= NUM_PROFILES) {

        /*  invalid profile number */
        guc_ProfileLoadOK = FAIL;
        return;
    }

    /*  check valid profile stored for this profile number */
    if (gta_TXProfile[gs_US_Profile].s_ReqdBits == 0) {

        /*  invalid profile stored */
        guc_ProfileLoadOK = FAIL;
        return;
    }

    /*  stored profile is valid */
    /*  load US Profile */
    LoadUSProfile();
   guc_TXProfileLoadOK  = SUCCEED;
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: LoadUSProfile
 *
 *  Description: Loads the Upstream BAT and gains table
 *
 *  Prototype: void LoadUSProfile(void);
 *
 *  Arguments: none
 *
 *  Return: none
 *
 *  Global Variables Used:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void LoadUSProfile(void) {

   int16 i,   *ps_fg_src, *ps_fg_dest;
   uint8 *ps_bat_dest, *ps_bat_src;

   ps_bat_src = gta_TXProfile[gs_US_Profile].sa_bat;
   ps_bat_dest = guca_TxBat;
   ps_fg_src = gta_TXProfile[gs_US_Profile].sa_gain;
   ps_fg_dest = gsa_TxFineGains;

   for(i=0; i<gs_TxNumTones; i++) {
      *ps_bat_dest++ = *ps_bat_src++;
      *ps_fg_dest++ = *ps_fg_src++;
   }



    return;
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: CheckStoredProfiles
 *
 *  Description: Check to see if there are any stored profiles.
 *  If there are no US and DS stored profiles then the fast retrain truncate bit is
 *  is set.
 *
 *  Prototype: void CheckStoredProfiles(void)
 *
 *  Arguments: none
 *
 *  Return: none
 *
 *  Global Variables Used:
 *  gta_TXProfile[]      (I) Table of US profiles
 *  gta_RXProfile[]      (I) Table of DS profiles
 *  guca_RMsgFR1Tab[]    (O) R_MSG_FR1_TX signal in which bit 5 is Fast retrain
 *                          truncate bit.
 *
 *------------------------------------------------------------------------
 *^^^
 */
void CheckStoredProfiles(void)
{
   guca_RMsgFR1Tab[0] = 0x00; /*  reset fast retrain truncate bit  */
   guca_RMsgFR1Tab[1] = 0x00; /*  set all other bits to zero */

   gft_FR_Truncate_Flag = FALSE; /*  reset fast retrain truncate flag */

   if((guc_num_tx_profiles_stored == 0) || (guc_num_rx_profiles_stored == 0))
   {
      guca_RMsgFR1Tab[0] |= INVALID_PROFILE; /*  set fast retrain truncate bit */
      gft_FR_Truncate_Flag = TRUE; /*  set fast retrain truncate flag */
   }
}
