/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C) 1998-2000 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *----------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Condfidential
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413
 *   Phone (781) 276-4000
 *   FAX   (781) 276-4001
 *
 *   IRI_IOf_FFT.C
 *
 *   Iridia CPE Data Transfer Layer.
 *
 *
 *----------------------------------------------------------------------------
 */

#include "iridia.h"
#include "alphaeus_const.h"

#include "const.h"
#include "gdata.h"
#include "error.h"
#include "aoc.h"

#include "LL_Iof.h"
#include "memrymap.h"
#include "string.h"
#include "IRI_Iof.h"
#include "states.h"
#include "const.h"

/*
 *------------------------------------------------------------------------
 *
 *  Name:
 *    LoadTxOutBuffer
 *
 * Prototype:
 *    LoadTxOutBuffer (int16 *ps_data, int ndata);
 *
 *  Abstract:
 *    LoadTxOutBuffer() -
 *
  *      Copies "ndata" 16-bit data from ps_data to Tx buffer.
 *
 *
 *  Returns:
 *    None
 *
 *  Global Variables Used:
 *      gs_TxFftLength, gsa_TxToneBuf,
 *    gs_Tx_OutBuf_Size, gs_TxCPLength
 *------------------------------------------------------------------------
 */
void LoadTxOutBuffer(int16 *ps_data, int16 ndata)
{
   int16 HalfSize;
   uint16 RealPart;
   uint16 ImagPart;

   HalfSize = (ndata >> 1);
   WhichIFFTBuffers(&RealPart, &ImagPart);

   /* both time domain and freq domain signals are copied in the same way */
#ifndef NEW_IRIDIA
   WriteCoreBuf((uint16) RealPart, ps_data, HalfSize, 2);
   WriteCoreBuf((uint16) ImagPart, ps_data+1, HalfSize, 2);
#else
   WriteCoreBuf32((uint16) RealPart, ps_data, HalfSize);
#endif

}

/*
 *------------------------------------------------------------------------
 *
 *  Name:
 *    GetTxOutBuffer
 *
 * Prototype:
 *    GetTxOutBuffer (int16 *ps_data, int ndata);
 *
 *  Abstract:
 *    GetTxOutBuffer() -
 *
  *      Copies "ndata" 16-bit data from Tx buffer to ps_data.
 *
 *
 *  Returns:
 *    None
 *
 *  Global Variables Used:
 *      gs_TxFftLength, gsa_TxToneBuf,
 *    gs_Tx_OutBuf_Size, gs_TxCPLength
 *------------------------------------------------------------------------
 */
void GetTxOutBuffer(int16 *ps_data, int16 ndata)
{
   int16 HalfSize;
   uint16 RealPart;
   uint16 ImagPart;

   HalfSize = ndata/2;
   WhichIFFTBuffers(&RealPart, &ImagPart);
   /* both time domain and freq domain signals are copied in the same way */

#ifndef NEW_IRIDIA
   ReadCoreBuf((uint16) RealPart, ps_data, HalfSize, 2);
   ReadCoreBuf((uint16) ImagPart, ps_data+1, HalfSize, 2);
#else
   ReadCoreBuf32((uint16) RealPart, ps_data, HalfSize);
#endif

}

/***************************************************************************************
;  Subroutine Name: GetRxTones
;
;  Description:
;     This routine copies the FFT buffer output from Iridia
;
;  Prototype:
;     void GetRxTones(int16 *psa_OutBuf, int16 s_FirstTone, int16 s_NumTones)
;
;  Input Arguments:
;
;  Output Arguments:
;     none
;
;  Return Value:
;     none
;
;  Global Variables:
;     gsa_RxToneBuf[] (O)      - DMT tones output from FDQ
;
;   Substates:
;             None
;
;**********************************************************************************************/
void GetRxTones(int16 *psa_OutBuf, int16 s_FirstTone, int16 s_NumTones)
{
   uint16 RealPart;  /* points to IRI_RAM_RX_FFT_A0 or IRI_RAM_RX_FFT_B0 */
   uint16 ImagPart;  /* points to IRI_RAM_RX_FFT_A1 or IRI_RAM_RX_FFT_B1 */
   int16 HalfSize;

   /* ===========================================================================   */
   /* if gft_CopyRxBuffer is FALSE, then do not load gsa_RxToneBuf               */
   /* ===========================================================================   */

   if (gft_CopyRxBuffer == FALSE)
   {
      return;
   }

   /* ===========================================================================   */
   /* if gft_CopyRxBuffer is TRUE, then load gsa_RxToneBuf with freq domain data */
   /* ===========================================================================   */

   HalfSize = s_NumTones;
   WhichFFTBuffers(&RealPart,&ImagPart);

   ReadCoreBuf32((uint16)(RealPart+s_FirstTone), psa_OutBuf, (uint16) HalfSize);
}

/***************************************************************************************
;  Subroutine Name: GetRxTime
;
;  Description:
;     This routine copies the time domain signal from FFT buffer.
;
;  Prototype:
;     void GetRxTime(int16 *ps_data, int16 ndata)
;
;  Input Arguments:
;     ndata - Number of samples to read from the FFT buffer
;
;  Output Arguments:
;     ps_data - Pointer to a buffer in which to store the time-domain samples
;
;  Return Value:
;     none
;
;  Global Variables:
;       gs_RxNumTones - Variable used to indicate what sampling rate is used for the TDQ
;                       filtering in Strymon.
;
;   Substates:
;
;**********************************************************************************************/
#define REAL_PART (0)
#define IMAG_PART (1)

void GetRxTime(int16 *ps_data, int16 ndata)
{
   uint16 RealPart;  /* points to IRI_RAM_RX_FFT_A0 or IRI_RAM_RX_FFT_B0 */
   uint16 ImagPart;  /* points to IRI_RAM_RX_FFT_A1 or IRI_RAM_RX_FFT_B1 */
   int16 i;
   guc_NonZeroPart = REAL_PART;
   /* ===========================================================================   */
   /* if gft_CopyRxBuffer is FALSE, then do not load gsa_RxToneBuf               */
   /* ===========================================================================   */

   if (gft_CopyRxBuffer == FALSE)
   {
      return;
   }

   /* ===========================================================================   */
   /* if gft_CopyRxBuffer is TRUE, then load gsa_RxToneBuf with time domain data */
   /* ===========================================================================   */

   WhichFFTBuffers(&RealPart,&ImagPart);

   if(ndata == 256) {

      // Every other sample of received data is a zero, which means either the Real or the
      // Imag FFT buffer is all zero.  Determine which buffer has the non-zero data.

#ifndef NEW_IRIDIA
      if(guc_NonZeroPart == REAL_PART)
         ReadCoreBuf(RealPart, ps_data, ndata, 1);
      else
         ReadCoreBuf(ImagPart, ps_data, ndata, 1);
#else
      ReadCoreBuf32LowHigh(RealPart, ps_data, ndata, guc_NonZeroPart);
#endif

      for (i=0; i<ndata; i++) {

         if (ps_data[i] != 0)
            return; // This is the buffer containing the non-zero samples.
      }

      // The other buffer contains the non-zero samples.

#ifndef NEW_IRIDIA
      if(guc_NonZeroPart == REAL_PART) {

         guc_NonZeroPart = IMAG_PART;
         ReadCoreBuf(ImagPart, ps_data, ndata, 1);
      }
      else {

         guc_NonZeroPart = REAL_PART;
         ReadCoreBuf(RealPart, ps_data, ndata, 1);
      }
#else
      if (guc_NonZeroPart == REAL_PART)
         guc_NonZeroPart = IMAG_PART;
      else
         guc_NonZeroPart = REAL_PART;

      ReadCoreBuf32LowHigh(RealPart, ps_data, ndata, guc_NonZeroPart);
#endif

   }
   else {

#ifndef NEW_IRIDIA
      ReadCoreBuf(RealPart, ps_data, (uint16)(ndata/2), 2);
      ReadCoreBuf(ImagPart, ps_data+1, (uint16)(ndata/2), 2);
#else
      ReadCoreBuf32(RealPart, ps_data, (uint16)(ndata >> 1));
#endif

   }
}
#undef REAL_PART
#undef IMAG_PART

#ifdef USE_ENGINE_FOR_BG_FFTS
/***************************************************************************************
;  Subroutine Name: LoadFFTInput
;
;  Description:
;     This loading function copies a time domain signal to be transformed from the task layer to
;     Iridia's FFT buffer.
;
;  Prototype:
;     void LoadFFTInput(void)
;
;  Input Arguments:
;     None.
;
;  Output Arguments:
;     None.
;
;  Return Value:
;     none
;
;  Global Variables:
;       gs_EngineFFTSize - Specifies size of FFT. Equal to either 128 or 256.
;     gpsa_EngineFFTInput - Pointer to time-domain real-valued buffer of size 2*gs_FastFFTSize.
;
;   Substates:
;
;**********************************************************************************************/

void LoadEngineFFTInput(void)
{
   uint16 RealPart;  /* points to IRI_RAM_RX_FFT_A0 or IRI_RAM_RX_FFT_B0 */
   uint16 ImagPart;  /* points to IRI_RAM_RX_FFT_A1 or IRI_RAM_RX_FFT_B1 */

   /* ===========================================================================   */
   /* if gft_CopyRxBuffer is TRUE, then load gsa_RxToneBuf with time domain data */
   /* ===========================================================================   */

   WhichFFTBuffers(&RealPart,&ImagPart);

   if(gs_EngineFFTSize == 128) {

#ifndef NEW_IRIDIA
      WriteCoreBuf(RealPart, gpsa_EngineFFTInput, (int16)(2*gs_EngineFFTSize), 1);
      FillCoreBuf(ImagPart, 0, gs_EngineFFTSize, 1);
#else
      WriteCoreBuf32LowHigh(RealPart, gpsa_EngineFFTInput, (int16)(2*gs_EngineFFTSize), 0);
      FillCoreBuf32LowHigh(RealPart, 0, gs_EngineFFTSize, 1);
#endif

   }
   else {   // Assumed that gs_EngineFFTSize == 256

#ifndef NEW_IRIDIA
      WriteCoreBuf(RealPart, gpsa_EngineFFTInput, gs_EngineFFTSize, 2);
      WriteCoreBuf(ImagPart, gpsa_EngineFFTInput+1, gs_EngineFFTSize, 2);
#else
      WriteCoreBuf32(RealPart, gpsa_EngineFFTInput, gs_EngineFFTSize);
#endif

   }
}

/***************************************************************************************
;  Subroutine Name: GetEngineFFTResult
;
;  Description:
;     This time-critical task copies the contents of the current FFT buffer to the
;     output buffer specified for the Fast FFT.  It is similar to GetRxTones() but is
;     especially for use in reading the results of the Fast FFT.
;
;  Prototype:
;     void GetEngineFFTResult(void)
;
;  Input Arguments:
;     None.
;
;  Output Arguments:
;     None.
;
;  Return Value:
;     none
;
;  Global Variables:
;       gs_EngineFFTSize - Specifies size of FFT. Equal to either 128 or 256.
;     gpsa_EngineFFTOutput - Pointer to buffer of size 2*gs_EngineFFTSize words.
;
;   Substates:
;
;**********************************************************************************************/
void GetEngineFFTResult(void)
{
   uint16 RealPart;  /* points to IRI_RAM_RX_FFT_A0 or IRI_RAM_RX_FFT_B0 */
   uint16 ImagPart;  /* points to IRI_RAM_RX_FFT_A1 or IRI_RAM_RX_FFT_B1 */
   int16 HalfSize;

   HalfSize = gs_EngineFFTSize;
   WhichFFTBuffers(&RealPart,&ImagPart);

#ifndef NEW_IRIDIA
   ReadCoreBuf(RealPart, gpsa_EngineFFTOutput, (uint16) HalfSize, (uint16) 2);
   ReadCoreBuf(ImagPart, gpsa_EngineFFTOutput+1, (uint16) HalfSize, (uint16) 2);
#else
   ReadCoreBuf32(RealPart, gpsa_EngineFFTOutput, (uint16) HalfSize);
#endif

}
#endif


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : CopyTone
 *
 *  Description:  tone data is copied form FFT output
 *
 *  Prototype:  void CopyTone(int16 *ps_PilotToneBuf, int16 s_toneIndex)
 *
 *
 *  Input Arguments: None.
 *    s_toneIndex :  Tone Index
 *
 *  Output Arguments:
 *      ps_PilotToneBuf:  two element array to store the pilot tone
 *
 *  Return:
 *          None
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void CopyTone(int16 *buf, int16 s_toneIndex)
{
    uint16 RealPart; /* points to IRI_RAM_RX_FFT_A0 or IRI_RAM_RX_FFT_B0 */
    uint16 ImagPart; /* points to IRI_RAM_RX_FFT_A1 or IRI_RAM_RX_FFT_B1 */

   WhichFFTBuffers(&RealPart,&ImagPart);

   ReadCoreBuf32((uint16) (RealPart+s_toneIndex), (uint16 *)&buf[0], 1);
}

/***************************************************************************************
;  Subroutine Name: GetRxCP
;
;  Description:
;     This routine copies the Cyclic Prefix (CP) from Iridia CP buffers
;
;  Prototype:
;     void GetRxCP(void)
;
;  Input Arguments:
;
;  Output Arguments:
;     none
;
;  Return Value:
;     none
;
;  Global Variables:
;     gsa_RxCPBuf[] (O)      - Cyclic Prefix (CP) for ISI analysis
;
;   Substates:
;             None
;
;**********************************************************************************************/

#ifdef ISI_ANALYSIS_ON
void GetRxCP(void)
{
   uint16 RealPart;  /* points to IRI_RAM_RX_CYCLICPREFIX_A0 or IRI_RAM_RX_CYCLICPREFIX_B0 */
   uint16 ImagPart;  /* points to IRI_RAM_RX_CYCLICPREFIX_A1 or IRI_RAM_RX_CYCLICPREFIX_B1 */
   int16 HalfSize;

   /* ===========================================================================   */
   /* if gft_CopyRxBuffer is FALSE, then do not load gsa_RxCPBuf              */
   /* ===========================================================================   */

   if (gft_CopyRxBuffer == FALSE)
   {
      return;
   }

   /* ===========================================================================   */
   /* if gft_CopyRxBuffer is TRUE, then load gsa_RxCPBuf with cyclic prefix      */
   /* ===========================================================================   */

   HalfSize = (int16) (RX_PREFIX_LENGTH>>1);
   WhichCPBuffers(&RealPart,&ImagPart);

#ifndef NEW_IRIDIA
   ReadCoreBuf(RealPart, gsa_RxCPBuf, (uint16) HalfSize, (uint16) 2);
   ReadCoreBuf(ImagPart, gsa_RxCPBuf+1, (uint16) HalfSize, (uint16) 2);
#else
   ReadCoreBuf32(RealPart, gsa_RxCPBuf, (uint16) HalfSize);
#endif

}
#endif

/******************************************************************************/



/***************************************************************************************
;  Subroutine Name: GetRxData
;
;  Description:
;     This routine copies the FFT buffer output from Iridia
;
;  Prototype:
;     void GetRxData(int16 *psa_buffer, int16 s_logicalBuffer, int16 s_offset, int16 s_numTones)
;
;  Input Arguments:
;
;  Output Arguments:
;     none
;
;  Return Value:
;     none
;
;  Global Variables:
;
;   Substates:
;             None
;
;**********************************************************************************************/

void GetRxData(int16 *psa_buffer, int16 s_logicalBuffer, int16 s_offset, int16 s_numTones)
{
   uint16 us_ibank, us_physicalBuffer;
   uint16 us_addr[3] = {IRI_RAM_RX_FFT_A0_ADDR, IRI_RAM_RX_FFT_B0_ADDR, IRI_RAM_RX_FFT_C0_ADDR};

   ReadCoreReg(IRI_REG_IBANK_ADDR,&us_ibank);
   if(s_logicalBuffer == 1)
      us_physicalBuffer = ((us_ibank & IRI_PI_RLB1_MASK) >> IRI_PI_RLB1_BITPOS);
   else if(s_logicalBuffer == 2)
      us_physicalBuffer = ((us_ibank & IRI_PI_RLB2_MASK) >> IRI_PI_RLB2_BITPOS);

   ReadCoreBuf32((int16)(us_addr[us_physicalBuffer]+s_offset), psa_buffer, s_numTones);
}
