/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2000 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;  Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413
;  Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;
;  File Name: LL_IOf.c
;
;  Low-level core interface layer.
;
;  Note: Currently these functions can access memory and
;     registers in Strymon and Iridia only.
*****************************************************************************/

#include "typedef.h"
#include "LL_Iof.h"

#include "strymon_cpe.h"
#include "iridia.h"
#include "zephyr.h"
#include "alphaeus.h"
#include "cri_cocomo_interface.h"
#include "byteflip.h"

/****************************************************************************
; Name: ReadCoreReg
;
; Prototype:
;  void ReadCoreReg(uint16 us_addr, uint16 *pus_word)
;
; Description:
;  Function for reading a register in the Strymon CPE BM.
;
; Arguments:
;  us_addr     (I)      physical address of register
;  *pus_word   (O)      16-bit data read from the register
;
;  Return Value:
;  N/A
*****************************************************************************/
void ReadCoreReg(uint16 us_addr, uint16 *pus_word)
{
   uint32 ul_word;
   if(!StrymonCPE_ReadWord(us_addr, pus_word))
      return;
   if(!IRI_ReadWord16(us_addr, pus_word)){
      return;
   }
    if(!ZEP_ReadWord(us_addr, &ul_word)) {
      /* Zephy ReadWord function now requires a pointer to a 32-bit read back data. */
      *pus_word = (uint16) ul_word;
      return;
   }
    if(!Alphaeus_ReadWord(us_addr, &ul_word)) {
      /* Alphaeus ReadWord function now requires a pointer to a 32-bit read back data. */
      *pus_word = (uint16) ul_word;
      return;
   }
   if(!CRI_ReadWord(us_addr, pus_word))
      return;
}

/****************************************************************************
; Name: ReadCoreReg32
;
; Prototype:
;  void ReadCoreReg32(uint16 us_addr, uint32 *pul_word)
;
; Description:
;  Function for reading a register in the Strymon CPE BM.
;
; Arguments:
;  us_addr     (I)      physical address of register
;  *pul_word   (O)      32-bit data read from the register
;
;  Return Value:
;  N/A
*****************************************************************************/
void ReadCoreReg32(uint16 us_addr, uint32 *pul_word)
{
   uint16 us_word;
   if(!StrymonCPE_ReadLongWord(us_addr, pul_word))
      return;
   if(!IRI_ReadWord16(us_addr, &us_word)){
      /* Iridia ReadWord function requires a pointer to a 16-bit read back data. */
      *pul_word = (uint32) us_word;
      return;
   }
    if(!ZEP_ReadWord(us_addr, pul_word)) {
      return;
   }
    if(!Alphaeus_ReadWord(us_addr, pul_word)) {
      return;
   }
   if(!CRI_ReadWord(us_addr, &us_word))
   {
      /* CRI ReadWord function requires a pointer to a 16-bit read back data. */
      *pul_word = (uint32) us_word;
      return;
   }
}


/****************************************************************************
; Name: WriteCoreBuf8
;
; Prototype:
;  void WriteCoreBuf8(uint16 us_addr, int8 *psa_words, uint16 us_num, uint16 us_SrcPitch)
;
; Description:
;  Function for writing to a buffer in the ADSL cores.
;
; Arguments:
;  us_addr     (I)      physical address at which to start writing
;  psa_words   (I)      array containing data to be written to the buffer
;  us_num      (I)      number of 16-bit words to write to the buffer
;  us_SrcPitch(I)    Amount to increment destination read pointer after each word is read.
;
;
;  Return Value:
;  N/A
*****************************************************************************/
void WriteCoreBuf8(uint16 us_addr, uint8 *psa_words, uint16 us_num)
{
   uint8 uc_Error;
   uint16 us_idx;
   uint16 us_DestAddr;
   int8 tmp;

   us_DestAddr = us_addr;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++, us_DestAddr++) {

      tmp = *psa_words++;
      if(IRI_WriteWord16(us_DestAddr, (uint16) tmp)){
         uc_Error = 1;
         break;
      }
   }
}

/****************************************************************************
; Name: WriteCoreReg
;
; Prototype:
;  void WriteCoreReg(uint16 us_addr, uint16 us_word)
;
; Description:
;  Function for writing to a register in the Strymon CPE BM.
;
; Arguments:
;  us_addr     (I)      physical address of register
;  us_word     (I)      16-bit data to be written to the register
;
;  Return Value:
;  N/A
*****************************************************************************/
void WriteCoreReg(uint16 us_addr, uint16 us_word)
{
   if(!StrymonCPE_WriteWord(us_addr, us_word))
      return;
   if(!IRI_WriteWord16(us_addr, (uint16)(us_word))){
      return;
   }
   if(!ZEP_WriteWord(us_addr, (uint32) us_word))
      return;
   if(!Alphaeus_WriteWord(us_addr, (uint32) us_word))
      return;
   if(!CRI_WriteWord(us_addr, us_word))
      return;
}


/****************************************************************************
; Name: WriteCoreReg32
;
; Prototype:
;  void WriteCoreReg32(uint16 us_addr, uint32 ul_word)
;
; Description:
;  Function for writing to a register in the Strymon CPE BM.
;
; Arguments:
;  us_addr     (I)      physical address of register
;  ul_word     (I)      32-bit data to be written to the register
;
;  Return Value:
;  N/A
*****************************************************************************/
void WriteCoreReg32(uint16 us_addr, uint32 ul_word)
{
   if(!StrymonCPE_WriteLongWord(us_addr, ul_word))
      return;
   if(!IRI_WriteWord16(us_addr, (uint16)(ul_word))){
      return;
   }
   if(!ZEP_WriteWord(us_addr, ul_word))
      return;
   if(!Alphaeus_WriteWord(us_addr, ul_word))
      return;
   if(!CRI_WriteWord(us_addr, (uint16)(ul_word)))
      return;
}


/****************************************************************************
; Name: SetCoreReg
;
; Prototype:
;  void SetCoreReg(uint16 us_addr, uint16 us_mask)
;
; Description:
;  Function for setting bits in a register in the Strymon CPE
;  BM.
;
; Arguments:
;  us_addr     (I)      physical address of register
;  us_mask     (I)      bits set to 1 cause the corresponding
;                 bits in the register to be set to 1
;
;  Return Value:
;  N/A
*****************************************************************************/
void SetCoreReg(uint16 us_addr, uint16 us_mask)
{
   uint16 tmp;

   ReadCoreReg(us_addr, &tmp);
   tmp |= us_mask;
   WriteCoreReg(us_addr, tmp);

}

/****************************************************************************
; Name: ResetCoreReg
;
; Prototype:
;  void ResetCoreReg(uint16 us_addr, uint16 us_mask)
;
; Description:
;  Function for resetting bits in a register in the Strymon CPE
;  BM.
;
; Arguments:
;  us_addr     (I)      physical address of register
;  us_mask     (I)      bits set to 1 cause the corresponding
;                 bits in the register to be set to 0
;
;  Return Value:
;  N/A
*****************************************************************************/
void ResetCoreReg(uint16 us_addr, uint16 us_mask)
{
   uint16 tmp;

   ReadCoreReg(us_addr, &tmp);
   tmp &= ~us_mask;
   WriteCoreReg(us_addr, tmp);
}

/****************************************************************************
; Name: ReadCoreBuf
;
; Prototype:
;  void ReadCoreBuf(uint16 us_addr, int16 *psa_words, uint16 us_num, uint16 us_DestPitch)
;
; Description:
;  Function for reading a buffer in the ADSL cores.
;
; Arguments:
;  us_addr     (I)      physical address at which to start reading
;  psa_words   (O)      array containing data read from the buffer
;  us_num      (I)      number of 16-bit words to read from the buffer
;  us_DestPitch(I)      Amount to increment destination write pointer after each word is written.
;
;  Return Value:
;  N/A
*****************************************************************************/
void ReadCoreBuf(uint16 us_addr, int16 *psa_words, uint16 us_num, uint16 us_DestPitch)
{
   uint8 uc_Error;
   uint16 us_idx;
   uint16 us_SrcAddr, us_DestOffset;

   us_SrcAddr = us_addr;
   us_DestOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++, us_SrcAddr++, us_DestOffset += us_DestPitch) {

      if(StrymonCPE_ReadWord(us_SrcAddr, (uint16*) &(psa_words[us_DestOffset]))){
         uc_Error = 1;
         break;
      }
   }
   if (!uc_Error)
      return;

   us_SrcAddr = us_addr;
   us_DestOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++, us_SrcAddr++, us_DestOffset += us_DestPitch) {

      if(IRI_ReadWord16(us_SrcAddr, (uint16*) (psa_words+us_DestOffset))){
         uc_Error = 1;
         break;
      }
   }
}

/****************************************************************************
; Name: ReadCoreBuf8
;
; Prototype:
;  void ReadCoreBuf(uint16 us_addr, int16 *psa_words, uint16 us_num, uint16 us_DestPitch)
;
; Description:
;  Function for reading a buffer in the ADSL cores.
;
; Arguments:
;  us_addr     (I)      physical address at which to start reading
;  psa_words   (O)      array containing data read from the buffer
;  us_num      (I)      number of 16-bit words to read from the buffer
;  us_DestPitch(I)      Amount to increment destination write pointer after each word is written.
;
;  Return Value:
;  N/A
*****************************************************************************/
void ReadCoreBuf8(uint16 us_addr, uint8 *psa_words, uint16 us_num)
{
   uint8 uc_Error;
   uint16 us_idx;
   uint16 us_SrcAddr;
   uint16 tmp;

   us_SrcAddr = us_addr;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++, us_SrcAddr++) {

      if(IRI_ReadWord16(us_SrcAddr, (uint16*) &tmp)){
         uc_Error = 1;
         break;
      }
      *psa_words++ = (uint8) tmp;
   }
}
/****************************************************************************
; Name: WriteCoreBuf
;
; Prototype:
;  void WriteCoreBuf(uint16 us_addr, int16 *psa_words, uint16 us_num, uint16 us_SrcPitch)
;
; Description:
;  Function for writing to a buffer in the ADSL cores.
;
; Arguments:
;  us_addr     (I)      physical address at which to start writing
;  psa_words   (I)      array containing data to be written to the buffer
;  us_num      (I)      number of 16-bit words to write to the buffer
;  us_SrcPitch(I)    Amount to increment destination read pointer after each word is read.
;
;
;  Return Value:
;  N/A
*****************************************************************************/
void WriteCoreBuf(uint16 us_addr, int16 *psa_words, uint16 us_num, uint16 us_SrcPitch)
{
   uint8 uc_Error;
   uint16 us_idx;
   uint16 us_SrcOffset, us_DestAddr;

   us_DestAddr = us_addr;
   us_SrcOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++, us_SrcOffset += us_SrcPitch, us_DestAddr++) {

      if(StrymonCPE_WriteWord(us_DestAddr, (uint16) psa_words[us_SrcOffset])){
         uc_Error = 1;
         break;
      }
   }

   if (!uc_Error)
      return;

   us_DestAddr = us_addr;
   us_SrcOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++, us_SrcOffset += us_SrcPitch, us_DestAddr++) {

      if(IRI_WriteWord16(us_DestAddr, (uint16) *(psa_words+us_SrcOffset))){
         uc_Error = 1;
         break;
      }
   }
}

/****************************************************************************
; Name: FillCoreBuf
;
; Prototype:
;  void FillCoreBuf(uint16 us_addr, int16 s_Value, uint16 us_num, uint16 us_DestPitch)
;
; Description:
;  Function for writing a constant to multiple locations in a buffer in the ADSL cores.
;
; Arguments:
;  us_addr     (I)      physical address at which to start writing
;  s-Value     (I)      16-bit value to be written to the buffer
;  us_num      (I)      number of 16-bit locations to write the value to
;  us_DestPitch(I)      Amount to increment destination write pointer after each word is written.
;
;
;  Return Value:
;  N/A
*****************************************************************************/
void FillCoreBuf(uint16 us_addr, int16 s_Value, uint16 us_num, uint16 us_DestPitch)
{
   uint8 uc_Error;
   uint16 us_idx;
   uint16 us_DestAddr;

   us_DestAddr = us_addr;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++, us_DestAddr += us_DestPitch) {

      if(StrymonCPE_WriteWord(us_DestAddr, s_Value )){
         uc_Error = 1;
         break;
      }
   }

   if (!uc_Error)
      return;

   us_DestAddr = us_addr;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++, us_DestAddr += us_DestPitch) {

      if(IRI_WriteWord16(us_DestAddr, s_Value)){
         uc_Error = 1;
         break;
      }
   }
}

#ifdef NEW_IRIDIA
/****************************************************************************
; Name: ReadCoreBuf32
;
; Prototype:
;  void ReadCoreBuf32(uint16 us_addr, int16 *psa_words)
;
; Description:
;  Function for 32-bit read of ADSL core buffer
;
; Arguments:
;  us_addr     (I)      physical address at which to start reading
;  psa_words   (O)      array containing data read from the buffer
;
;  Return Value:
;  N/A
*****************************************************************************/
void ReadCoreBuf32(uint16 us_addr, int16 *psa_words, uint16 us_num)
{
   uint8 uc_Error;
   uint16 us_idx;
   uint16 us_SrcAddr, us_DestOffset;
   uint32 ul_data;

#ifdef HWENGINE_30
   /* Strymon */
   us_SrcAddr = us_addr;
   us_DestOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++){

      if(StrymonCPE_ReadLongWord(us_SrcAddr, &ul_data)){
         uc_Error = 1;
         break;
      }
      *(psa_words+us_DestOffset) = (int16) ((ul_data) & 0x0000FFFF);
      us_DestOffset++;

      *(psa_words+us_DestOffset) = (int16) (((ul_data) & 0xFFFF0000) >> 16);
      us_DestOffset++;

      us_SrcAddr++;

   }

   if (!uc_Error)
      return;
#endif

   /* Iridia */
   us_SrcAddr = us_addr;
   us_DestOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++){

      if(IRI_ReadWord(us_SrcAddr, &ul_data)){
         uc_Error = 1;
         break;
      }
      *(psa_words+us_DestOffset) = (int16) ((ul_data) & 0x0000FFFF);
      us_DestOffset++;

      *(psa_words+us_DestOffset) = (int16) (((ul_data) & 0xFFFF0000) >> 16);
      us_DestOffset++;

      us_SrcAddr++;

   }

   if (!uc_Error)
      return;

   /* Zephyr */
   us_SrcAddr = us_addr;
   us_DestOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++){

      if(ZEP_ReadWord(us_SrcAddr, &ul_data)){
         uc_Error = 1;
         break;
      }
      *(psa_words+us_DestOffset) = (int16) ((ul_data) & 0x0000FFFF);
      us_DestOffset++;

      *(psa_words+us_DestOffset) = (int16) (((ul_data) & 0xFFFF0000) >> 16);
      us_DestOffset++;

      us_SrcAddr++;

   }

   if (!uc_Error)
      return;
}

/****************************************************************************
; Name: WriteCoreBuf32
;
; Prototype:
;  void WriteCoreBuf32(uint16 us_addr, int16 *psa_words, uint16 us_num)
;
; Description:
;  Function for writing 32-bit words to a 32-bit buffer in the ADSL cores.
;
; Arguments:
;  us_addr     (I)      physical address at which to start writing
;  psa_words   (I)      array containing data to be written to the buffer
;  us_num      (I)      number of 32-bit words to write to the buffer
;
;
;  Return Value:
;  N/A
*****************************************************************************/
void WriteCoreBuf32(uint16 us_addr, int16 *psa_words, uint16 us_num)
{
   uint8 uc_Error;
   uint16 us_idx;
   uint16 us_SrcOffset, us_DestAddr;
   uint32 ul_data1;
   uint32 ul_data2;

#ifdef HWENGINE_30

   /* Strymon */
   us_DestAddr = us_addr;
   us_SrcOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++){

      /* read ls 16-bits */
      ul_data1= (uint32)(*(psa_words+us_SrcOffset));
      ul_data1 = ul_data1 & 0x0000FFFF;
      us_SrcOffset++;

      /* read mw 16-bits */
      ul_data2 = (uint32)(*(psa_words+us_SrcOffset));
      ul_data2 = (ul_data2<<16) & 0xFFFF0000;
      us_SrcOffset++;

      /* combine into a 32-bit word */
      ul_data1 = ul_data1 | ul_data2;

      /* write data to hardware */
      if(StrymonCPE_WriteLongWord(us_DestAddr, ul_data1)){
         uc_Error = 1;
         break;
      }
      us_DestAddr++;

   }

   if (!uc_Error)
      return;
#endif

   /* Iridia */
   us_DestAddr = us_addr;
   us_SrcOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++){

      /* read ls 16-bits */
      ul_data1= (uint32)(*(psa_words+us_SrcOffset));
      ul_data1 = ul_data1 & 0x0000FFFF;
      us_SrcOffset++;

      /* read mw 16-bits */
      ul_data2 = (uint32)(*(psa_words+us_SrcOffset));
      ul_data2 = (ul_data2<<16) & 0xFFFF0000;
      us_SrcOffset++;

      /* combine into a 32-bit word */
      ul_data1 = ul_data1 | ul_data2;

      /* write data to hardware */
      if(IRI_WriteWord(us_DestAddr, ul_data1)){
         uc_Error = 1;
         break;
      }
      us_DestAddr++;

   }

   if (!uc_Error)
      return;

   /* Zephyr */
   us_DestAddr = us_addr;
   us_SrcOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++){

      /* read ls 16-bits */
      ul_data1= (uint32)(*(psa_words+us_SrcOffset));
      ul_data1 = ul_data1 & 0x0000FFFF;
      us_SrcOffset++;

      /* read mw 16-bits */
      ul_data2 = (uint32)(*(psa_words+us_SrcOffset));
      ul_data2 = (ul_data2<<16) & 0xFFFF0000;
      us_SrcOffset++;

      /* combine into a 32-bit word */
      ul_data1 = ul_data1 | ul_data2;

      /* write data to hardware */
      if(ZEP_WriteWord(us_DestAddr, ul_data1)){
         uc_Error = 1;
         break;
      }
      us_DestAddr++;

   }

   if (!uc_Error)
      return;

   /* Alphaeus */
   us_DestAddr = us_addr;
   us_SrcOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++){

      /* read ls 16-bits */
      ul_data1= (uint32)(*(psa_words+us_SrcOffset));
      ul_data1 = ul_data1 & 0x0000FFFF;
      us_SrcOffset++;

      /* read mw 16-bits */
      ul_data2 = (uint32)(*(psa_words+us_SrcOffset));
      ul_data2 = (ul_data2<<16) & 0xFFFF0000;
      us_SrcOffset++;

      /* combine into a 32-bit word */
      ul_data1 = ul_data1 | ul_data2;

      /* write data to hardware */
      if(Alphaeus_WriteWord(us_DestAddr, ul_data1)){
         uc_Error = 1;
         break;
      }
      us_DestAddr++;

   }

}

/****************************************************************************
; Name: WriteCoreBuf32NoDMA
;
; Prototype:
;  void WriteCoreBuf32NoDMA(uint16 us_addr, int16 *psa_words, uint16 us_num)
;
; Description:
;  Function for writing 32-bit words to a 32-bit buffer in the ADSL cores.
;
; Arguments:
;  us_addr     (I)      physical address at which to start writing
;  psa_words   (I)      array containing data to be written to the buffer
;  us_num      (I)      number of 32-bit words to write to the buffer
;
;
;  Return Value:
;  N/A
*****************************************************************************/
void WriteCoreBuf32NoDMA(uint16 us_addr, int16 *psa_words, uint16 us_num)
{
   // On cocomo, this is same as WriteCoreBuf32
   // Wrapper function needed to distinguish between DMA and NonDMA transfers
   // to 32 bit buffers on Socrates
   WriteCoreBuf32(us_addr, psa_words, us_num);
}

/****************************************************************************
; Name: FillCoreBuf32
;
; Prototype:
;  void FillCoreBuf32(uint16 us_addr, int32 s_value, uint16 us_num)
;
; Description:
;
; Arguments:
;  us_addr     (I)      physical address at which to start writing
;  s_value     (I)      data to be filled in to the buffer
;  us_num      (I)      number of 16-bit words to write to the buffer
;
;
;  Return Value:
;  N/A
*****************************************************************************/
void FillCoreBuf32(uint16 us_addr, int32 s_value, uint16 us_num)
{
   uint8 uc_Error;
   uint16 us_idx;
   uint16 us_DestAddr;

#ifdef HWENGINE_30

   us_DestAddr = us_addr;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++){

      /* write data to hardware */
      if(StrymonCPE_WriteLongWord(us_DestAddr, s_value)){
         uc_Error = 1;
         break;
      }
      us_DestAddr++;

   }

   if (!uc_Error)
      return;
#endif

   us_DestAddr = us_addr;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++){

      /* write data to hardware */
      if(IRI_WriteWord(us_DestAddr, s_value)){
         uc_Error = 1;
         break;
      }
      us_DestAddr++;

   }

}

/****************************************************************************
; Name: ReadCoreBuf32LowHigh
;
; Prototype:
;  void ReadCoreBufLowHigh(uint16 us_addr, int16 *psa_words, int16 s_ReadControl)
;
; Description:
;  Function for 16-bit read from a 32-bit ADSL core buffer
;
; Arguments:
;  us_addr     (I)      physical address at which to start reading
;  psa_words   (O)      array containing data read from the buffer
;  s_ReadControl  (I)      Selects between reads from MSW or LSW 16-bits.
;
;  Return Value:
;  N/A
*****************************************************************************/
void ReadCoreBuf32LowHigh(uint16 us_addr, int16 *psa_words, uint16 us_num, int16 s_ReadControl)
{
   uint8 uc_Error;
   uint16 us_idx;
   uint16 us_SrcAddr, us_DestOffset;
   uint32 ul_data;

#ifdef HWENGINE_30

   us_SrcAddr = us_addr;
   us_DestOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++){

      if(StrymonCPE_ReadLongWord(us_SrcAddr, &ul_data)){
         uc_Error = 1;
         break;
      }
      *(psa_words+us_DestOffset) = (int16) (ul_data & ((uint32) 0xFFFF << (s_ReadControl*16)));

      us_SrcAddr++;
      us_DestOffset++;

   }

   if (!uc_Error)
      return;

#endif

   us_SrcAddr = us_addr;
   us_DestOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++){

      if(IRI_ReadWord(us_SrcAddr, &ul_data)){
         uc_Error = 1;
         break;
      }

      *(psa_words+us_DestOffset) = (int16) (ul_data & ((uint32) 0xFFFF << (s_ReadControl*16)));

      us_SrcAddr++;
      us_DestOffset++;

   }
}

/****************************************************************************
; Name: WriteCoreBuf32LowHigh
;
; Prototype:
;  void WriteCoreBuf32LowHigh(uint16 us_addr, int16 *psa_words, uint16 us_num, int16 s_WriteControl)
;
; Description:
;  Function for writing 16-bit words to a 32-bit buffer in the ADSL cores.
;
; Arguments:
;  us_addr     (I)      physical address at which to start writing
;  psa_words   (I)      array containing data to be written to the buffer
;  us_num      (I)      number of 16-bit words to write to the buffer
;  s_WriteControl(I) Control to access either the LS or MS 16-bits.
;
;
;  Return Value:
;  N/A
*****************************************************************************/
void WriteCoreBuf32LowHigh(uint16 us_addr, int16 *psa_words, uint16 us_num, int16 s_WriteControl)
{
   uint8 uc_Error;
   uint16 us_idx;
   uint16 us_SrcOffset, us_DestAddr;
   uint32 ul_data;

#ifdef HWENGINE_30

   us_DestAddr = us_addr;
   us_SrcOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++){

      ul_data = ((uint32) (*(psa_words+us_SrcOffset) & 0xFFFF) << (s_WriteControl * 16));
      us_SrcOffset++;

      /* write data to hardware */
      if(StrymonCPE_WriteLongWord(us_DestAddr, ul_data)){
         uc_Error = 1;
         break;
      }
      us_DestAddr++;

   }

   if (!uc_Error)
      return;

#endif

   us_DestAddr = us_addr;
   us_SrcOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++){

      ul_data = ((uint32) (*(psa_words+us_SrcOffset) & 0xFFFF) << (s_WriteControl * 16));

      us_SrcOffset++;
      /* write data to hardware */
      if(IRI_WriteWord(us_DestAddr, ul_data)){
         uc_Error = 1;
         break;
      }
      us_DestAddr++;

   }

}

/****************************************************************************
; Name: FillCoreBuf32LowHigh
;
; Prototype:
;  void FillCoreBuf32LowHigh(uint16 us_addr, int16 s_value, uint16 us_num, int16 s_WriteControl)
;
; Description:
;  Function for writing 16-bit words to a 32-bit buffer in the ADSL cores.
;
; Arguments:
;  us_addr     (I)      physical address at which to start writing
;  s_value     (I)      data to be filled in to the buffer
;  us_num      (I)      number of 16-bit words to write to the buffer
;  s_WriteControl(I) Control to access either the LS or MS 16-bits.
;
;
;  Return Value:
;  N/A
*****************************************************************************/
void FillCoreBuf32LowHigh(uint16 us_addr, int16 s_value, uint16 us_num, int16 s_WriteControl)
{
   uint8 uc_Error;
   uint16 us_idx;
   uint16 us_SrcOffset, us_DestAddr;
   uint32 ul_data;

#ifdef HWENGINE_30

   us_DestAddr = us_addr;
   us_SrcOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++){

      ul_data = ((uint32) (s_value & 0xFFFF) << (s_WriteControl * 16));
      us_SrcOffset++;

      /* write data to hardware */
      if(StrymonCPE_WriteLongWord(us_DestAddr, ul_data)){
         uc_Error = 1;
         break;
      }
      us_DestAddr++;

   }

   if (!uc_Error)
      return;
#endif

   us_DestAddr = us_addr;
   us_SrcOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++){

      ul_data = ((uint32) (s_value & 0xFFFF) << (s_WriteControl * 16));

      us_SrcOffset++;
      /* write data to hardware */
      if(IRI_WriteWord(us_DestAddr, ul_data)){
         uc_Error = 1;
         break;
      }
      us_DestAddr++;

   }

}

/**************************************************************************************/
#endif // ifdef NEW_IRIDIA
