/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2000 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;  Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413
;  Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;
;  File Name: channel.c
;
;  Functions for simulating a channel.
;
*****************************************************************************/

#include <math.h>
#include <string.h>
#include "typedef.h"
#include "dsp_op.h"
#include "channel.h"
#include "compiler.h"
#include "gdata.h"
#include "mp.h"
#include "cmv.h"


#undef USE_FLOATINGPOINT_FOR_CHANNEL

float gf_Pga = 1.0;           /*unitary floating pt. PGA */

/* temporary AEC channel model */
#define AEC_CHANNEL_FILTER_LEN    (4)
#define AEC_CHANNEL_OUTPUT_SHIFT   (-15)
static int16 sa_aec_channel_state[AEC_CHANNEL_FILTER_LEN];
static float64 da_aec_channel_fir[AEC_CHANNEL_FILTER_LEN] = {0x7fff,0,0,0};

/* #define ECHO_SIM_FILTER_LEN (4)
   static float64 da_echo_fir[ECHO_SIM_FILTER_LEN] = {0x7FFF, 0, 0, 0 }; */

#define ECHO_SIM_FILTER_LEN    (100)
#define ECHO_SIM_OUTPUT_SHIFT  (-15)

static int16 sa_echo_state[ECHO_SIM_FILTER_LEN];  /* unity upsampling */

#if 0

   /* ansi 7 echo */
   /* commented out for AEC testing since it produces insufficient
       echo due to the current (0 db in passband) channel */

static float64 da_echo_fir[ECHO_SIM_FILTER_LEN] = { 0, 0, 0, 0, -1, -2, -4, -9, -15, -21,
                                    -26, -28, -28, -26, -21, -14, -5, 5, 15, 25,
                                       34, 41, 47, 51, 53, 53, 50, 46, 40, 32,
                                                   23, 14, 5, -4, -13, -21, -28, -34, -39, -41,
                                                  -43, -43, -42, -40, -36, -32, -27, -22, -17, -12,
                                                   -7, -2, 3, 6, 9, 12, 14, 15, 15, 15,
                                                   15, 14, 12, 11, 9, 7, 5, 4, 2, 1,
                                                    0, -1, -2, -2, -2, -2, -2, -2, -1, -1,
                                                    0, 1, 1, 2, 2, 3, 3, 3, 3, 3,
                                                    3, 3, 2, 2, 2, 1, 1, 0, 0, 0};
#endif

/* scaled by 10 ansi 7 echo */
/* replace with ansi 7 echo when ansi 7 channel is implemented */

static float64 da_echo_fir[ECHO_SIM_FILTER_LEN] = { 0, 0, 0, 0, -10, -20, -40, -90, -150, -210,
                                    -260, -280, -280, -260, -210, -140, -50, 50, 150, 250,
                                       340, 410, 470, 510, 530, 530, 500, 460, 400, 320,
                                                   230, 140, 50, -40, -130, -210, -280, -340, -390, -410,
                                                  -430, -430, -420, -400, -360, -320, -270, -220, -170, -120,
                                                   -70, -20, 30, 60, 90, 120, 140, 150, 150, 150,
                                                   150, 140, 120, 110, 90, 70, 50, 40, 20, 10,
                                                    0, -10, -20, -20, -20, -20, -20, -20, -10, -10,
                                                    0, 10, 10, 20, 20, 30, 30, 30, 30, 30,
                                                    30, 30, 20, 20, 20, 10, 10, 0, 0, 0};



#define FIR_LEN            (7)
#define IIR_LEN            (7)


#ifndef USE_FLOATINGPOINT_FOR_CHANNEL

   /* the following are used only by function Filter() */
   static int16 sa_x_mem[FIR_LEN];  /* FIR filter states */
   static int16 sa_y_mem[IIR_LEN];  /* IIR filter states */

   /*  This channel is a low pass channel and was designed to have a cut off at tone 56 of 128 tones or tone 112 of 256 tones */
   /* int16 gsa_b[FIR_LEN] = { (int16)0x0f94, (int16)0x2ebd, (int16)0x2ebd, (int16)0x0f94 }; */
   /* int16 gsa_a[IIR_LEN] = { (int16)0x7fff, (int16)0xd1e2, (int16)0x2f11, (int16)0xfbb0 }; */

   /* #define LOG2_ONE_DIV_A0 0   // = log2(1/1.0), because gsa_a[0] = 0x7fff = 1.0. This is the scaling factor that */
                           /*  is applied to the filter output */

   /*  This channel is a low pass channel and was designed to have a cut off at tone 75 of 128 tones or tone 150 of 256 tones */
   /* int16 gsa_b[FIR_LEN] = { (int16)0x1f10, (int16)0x5d35, (int16)0x5d35, (int16)0x1f10}; */
   /* int16 gsa_a[IIR_LEN] = { (int16)0x7fff, (int16)0x3f76, (int16)0x3302, (int16)0x0614 }; */

   /* #define LOG2_ONE_DIV_A0 0   // = log2(1/1.0), because gsa_a[0] = 0x7fff = 1.0. This is the scaling factor that */
                           /*  is applied to the filter output */

   /*  This channel is a low pass channel and was designed to have a cut off at tone 115 of 128 tones or tone 230 of 256 tones */
/* int16 gsa_b[FIR_LEN] = { (int16)0x173b, (int16)0x45ae, (int16)0x45ae, (int16)0x173b}; */
   int16 gsa_b[FIR_LEN] = { (int16)0x173b, (int16)0x0,    (int16)0x45ae, (int16)0x0,
                     (int16)0x45ae, (int16)0x0,    (int16)0x173b};
/* int16 gsa_b[FIR_LEN] = { (int16)0xfff3, (int16)0xff9b, (int16)0xfffa, (int16)0xff44,
                     (int16)0x0001, (int16)0xff78, (int16)0x0012, (int16)0xffe6,
                     (int16)0xfff7, (int16)0xffef, (int16)0x001d, (int16)0x0030,
                     (int16)0xffa2, (int16)0xff34, (int16)0x0251, (int16)0x0db9,
                     (int16)0x22c5, (int16)0x3b7e, (int16)0x4c78, (int16)0x4c78,
                     (int16)0x3b7e, (int16)0x22c5, (int16)0x0db9, (int16)0x0251,
                     (int16)0xff34, (int16)0xffa2, (int16)0x0030, (int16)0x001d,
                     (int16)0xffef, (int16)0xfff7, (int16)0xffe6, (int16)0x0012,
                     (int16)0xff78, (int16)0x0001, (int16)0xff44, (int16)0xfffa,
                     (int16)0xff9b, (int16)0xfff3}; */
/* int16 gsa_a[IIR_LEN] = { (int16)0x2000, (int16)0x4ba9, (int16)0x3d49, (int16)0x10db }; */
   int16 gsa_a[IIR_LEN] = { (int16)0x2000, (int16)0x0,    (int16)0x4ba9, (int16)0x0,
                     (int16)0x3d49, (int16)0x0,    (int16)0x10db };

   #define LOG2_ONE_DIV_A0 2   /*  = log2(1/0.25), because gsa_a[0] = 0x2000 = 0.25 This is the scaling factor that */
                        /*  is applied to the filter output */

#else /*  #ifdef USE_FLOATINGPOINT_FOR_CHANNEL */

   float gfa_x_mem[FIR_LEN] = {(float) 0.0, (float)0.0, (float)0.0, (float)0.0}; /* FIR filter states */
   float gfa_y_mem[IIR_LEN] = {(float)0.0, (float)0.0, (float)0.0, (float)0.0};  /* IIR filter states */

   /*  This channel is a low pass channel and was designed to have a cut off at tone 56 of 128 tones or tone 112 of 256 tones */
   /* float gfa_b[FIR_LEN] = { (float)0.1217, (float)0.3652, (float)0.3652, (float)0.1217 }; */
   /* float gfa_a[IIR_LEN] = { (float)1.0, (float)(-0.3603), (float)0.3677, (float)(-0.0337) }; */

   /*  This channel is a low pass channel and was designed to have a cut off at tone 75 of 128 tones or tone 150 of 256 tones */
   /* float gfa_b[FIR_LEN] = { (float)0.2427, (float)0.7282, (float)0.7282, (float)0.2427 }; */
   /* float gfa_a[IIR_LEN] = { (float)1.0, (float)0.4958, (float)0.3985, (float)0.0475 }; */

   /*  This channel is a low pass channel and was designed to have a cut off at tone 115 of 128 tones or tone 230 of 256 tones */
   float gfa_b[FIR_LEN] = { (float)0.7258, (float)2.1775, (float)2.1775, (float)0.7258 };
   float gfa_a[IIR_LEN] = { (float)1.0, (float)2.3644, (float)1.9154, (float)0.5268 };

#endif


/* Intepolating filter definitions */
#define OUTPUT_SHIFT2 -15
#define FILTER_LENGTH2 80
#define OUTPUT_SHIFT4 -15
#define FILTER_LENGTH4 79
#define MAX_FILTER_LENGTH 80

static InterpFirfp_t t_Interp;

static float64 da_Fir[FILTER_LENGTH2] =   {-5, -5, 12, 12, -21, -22, 35, 36,
                              -54, -56, 80, 82, -114, -118, 158, 164,
                              -214, -222, 283, 296, -371, -390, 480, 507,
                              -617, -657, 791, 853, -1020, -1115, 1333, 1488,
                              -1795, -2069, 2561, 3128, -4147, -5803, 9821, 29459,
                              29459, 9821, -5803, -4147, 3128, 2561, -2069, -1795,
                              1488, 1333, -1115, -1020, 853, 791, -657, -617,
                              507, 480, -390, -371, 296, 283, -222, -214,
                              164, 158, -118, -114, 82, 80, -56, -54,
                              36, 35, -22, -21, 12, 12, -5, -5};

static float64 da_Fir4[FILTER_LENGTH4] =  {-37, -69, -62, 0, 96,
                                        164, 139, 0, -194, -321, -263,
                                        0, 347, 560, 450, 0, -576,
                                        -916, -728, 0, 913, 1443, 1139,
                                        0, -1422, -2249, -1782, 0, 2259,
                                        3621, 2924, 0, -3934, -6611, -5697,
                                        0, 9711, 20745, 29460, 32768, 29460,
                                        20745, 9711, 0, -5697, -6611, -3934,
                                        0, 2924, 3621, 2259, 0, -1782,
                                        -2249, -1422, 0, 1139, 1443, 913,
                                        0, -728, -916, -576, 0, 450,
                                        560, 347, 0, -263, -321, -194,
                                        0, 139, 164, 96, 0, -62,
                                        -69, -37};

/*
static float64 da_Fir[FILTER_LENGTH] = {-69, -557, 173, 637, -307, -992, 570, 1480,
                              -1010, -2194, 1788, 3396, -3420, -6212, 9174, 29998,
                              29998, 9174, -6212, -3420, 3396, 1788, -2194, -1010,
                              1480, 570, -992, -307, 637, 173, -557, -69};
*/
static int16 sa_State[MAX_FILTER_LENGTH/MIN_UPSAMPLING_FACTOR];

#ifdef GEN_RX_OUT
FILE* hOutputBin = NULL;
#endif

/* Local function prototypes */
int16 interp_fir_fp(const int16 sa_Input[], uint16 us_InputLength, int16 sa_Output[], InterpFirfp_t* pt_InterpFirfp);
void Filter(int16 *psa_input, int16 *psa_output, int16 s_len);
int16 Noise(void);
void AecChannel(const int16 *ps_AecDacInBuf, int16 *ps_AecChannelOutBuf);
void EchoSim(const int16 *ps_TxInBuf, int16 *ps_EchoOutBuf);

InterpFirfp_t *GetInterp(void) {
    return(&t_Interp);
}


void InitInterp() {

    if ((TESTArray[TEST_Control] & TEST_UpsampleControl)) {
        /* Assume (1) CO is transmitting at 1.104 MHz and ADC_FS at 4.416 MHz or
               (2) CO is transmitting at 2.208 MHz and ADC_FS at 8.832 MHz  */
        t_Interp.us_Length = FILTER_LENGTH4;
        t_Interp.pda_Fir = da_Fir4;
        t_Interp.psa_State = sa_State;
        t_Interp.s_OutputShift = OUTPUT_SHIFT4;
        t_Interp.us_UpsamplingFactor = 4;
    }
    else {
        /* Assume (1) CO is transmitting at 2.208 MHz and ADC_FS at 4.416 MHz or
               (2) CO is transmitting at 4.416 MHz and ADC_FS at 8.832 MHz  */
      t_Interp.us_Length = FILTER_LENGTH2;
      t_Interp.pda_Fir = da_Fir;
      t_Interp.psa_State = sa_State;
      t_Interp.s_OutputShift = OUTPUT_SHIFT2;
      t_Interp.us_UpsamplingFactor = 2;
    }
}


/************************************************************************
;
; Name: Channel
;
; Prototype: void Channel(const int16 sa_InBuf[], uint16 us_InBufSize, int16 sa_OutBuf[])
;
; Description:
;  This function implements the channel.
;
; Globals:
************************************************************************/
void Channel(const int16 sa_InBuf[], uint16 us_InBufSize, int16 sa_OutBuf[])
{
   int16 i;

#ifdef GEN_RX_OUT
   // recording received CO data and generate input bin
   if (!hOutputBin)
      hOutputBin = fopen ("ghsrx_out.bin", "wb");

   if (hOutputBin)
      fwrite (sa_InBuf, sizeof(int16), us_InBufSize, hOutputBin);
#endif

   interp_fir_fp(sa_InBuf, us_InBufSize, sa_OutBuf, &t_Interp);

    if ((TESTArray[TEST_Control] & TEST_ChannelControl) != 0)
    {
      Filter(sa_OutBuf, sa_OutBuf, (int16) (us_InBufSize*t_Interp.us_UpsamplingFactor));
        for(i=0; i< us_InBufSize*t_Interp.us_UpsamplingFactor; i++)
            sa_OutBuf[i] += Noise();
    }
}

/************************************************************************
; Name: AecChannel
;
; Prototype: void AecChannel(const int16 *ps_AecDacInBuf, int16 *ps_AecChannelOutBuf);
;
; Description:
;  This function generates applies an FIR channel to the output of
;  the AEC DAC.
;
; Arguments:
;  ps_AecDacInBuf          (I)      pointer to AEC DAC output sample.
;  ps_AecChannelOutBuf  (O)      pointer to output of AEC channel sample.
; Return values:     none
************************************************************************/

void AecChannel(const int16 *ps_AecDacInBuf, int16 *ps_AecChannelOutBuf)
{
   InterpFirfp_t  t_UnityInterp;

   t_UnityInterp.us_Length = AEC_CHANNEL_FILTER_LEN;
   t_UnityInterp.pda_Fir = da_aec_channel_fir;
   t_UnityInterp.psa_State = sa_aec_channel_state;
   t_UnityInterp.s_OutputShift = AEC_CHANNEL_OUTPUT_SHIFT;
   t_UnityInterp.us_UpsamplingFactor = 1;

   interp_fir_fp(ps_AecDacInBuf, 1, ps_AecChannelOutBuf, &t_UnityInterp);

}
/************************************************************************
; Name: EchoSim
;
; Prototype: void EchoSim(const int16 *ps_TxInBuf, int16 *ps_EchoOutBuf)
;
; Description:
;  This function generates an echo by filtering a Tx sample
;   with an echo FIR filter.
;
; Arguments:
;  ps_TxInBuf        (I)      pointer to input Tx sample.
;  ps_EchoOutBuf     (O)      pointer to output Echo sample.

; Globals:
; Return values:     none
************************************************************************/

void EchoSim(const int16 *ps_TxInSample, int16 *ps_EchoOutSample)
{
   InterpFirfp_t  t_UnityInterp;

   t_UnityInterp.us_Length = ECHO_SIM_FILTER_LEN;
   t_UnityInterp.pda_Fir = da_echo_fir;
   t_UnityInterp.psa_State = sa_echo_state;
   t_UnityInterp.s_OutputShift = ECHO_SIM_OUTPUT_SHIFT;
   t_UnityInterp.us_UpsamplingFactor = 1;

/* make the echo*/
   interp_fir_fp(ps_TxInSample, 1, ps_EchoOutSample, &t_UnityInterp);

}

/************************************************************************
;
; Name: ApplyPga
;
; Prototype: void ApplyPga(int16 *ps_sample)
;
; Description:
;  Apply PGA to noisy channel output.
;
; Globals: gf_Pga
************************************************************************/
void ApplyPga(int16 *ps_sample)
{
   float f_tmp;
   float64 d_ip;  /* integral part of f_tmp */

    f_tmp = gf_Pga * (float) *ps_sample;

    /*check for OF*/
    if(f_tmp > (pow(2.0, 15.0)-1))
        *ps_sample = (int16) 0x7fff;
    else
        if(f_tmp < -pow(2.0, 15.0))
            *ps_sample = (int16) 0x8000;

        /*nonzero fractional part */
        else
            if (modf((float64) f_tmp, &d_ip) !=0){
                if(f_tmp >= 0)
                    *ps_sample = (int16) d_ip;
                else
                    *ps_sample = (int16) d_ip-1;
            }

            else
                *ps_sample= (int16) d_ip;
}


/***********************************************************************
; Name: interp_fir_fp
;
; Prototype:
;  int16 interp_fir_fp(const int16 sa_Input[], uint16 us_InputLength, int16 sa_Output[], InterpFirfp_t* pt_InterpFirfp)
;
; Description:
;  This function interpolates an input sequence by doing upsampling
;  followed by FIR filtering.
;
;                   ___________     _____
;          x[n] -->| Upsampler |-->| FIR |--> y[m]
;                  |___________|   |_____|
;
;
;  For an input sequence x[n], n = 0, 1, ..., L-1, an upsampling factor U,
;  and a filter h[m], m = 0, 1, ..., M-1, the output is a sequence y[m],
;  m = 0, 1, ..., L*U-1, given by
;
;     y[i+U*n] = h_i[0]*x[n] + h_i[1]*x[n-1] + ... + h_i[M_i-1]*x[0],
;  for i = 0, 1, ..., U-1, and n >= M_i-1;
;
;     y[i+U*n] = h_i[0]*x[n] + h_i[1]*x[n-1] + ... + h_i[n]*x[0] +
;              h_i[n+1]*s[0] + h_i[n+2]*s[1] + ... + h_i[M_i-1]*s[M_i-n-2],
;
;  for i = 0, 1, ..., U-1, and n < M_i-1. h_i[m] denotes the ith phase of
;  the filter, and M_i is the length of h_i[m]. s[m] is the initial
;  state of the filter.
;
;  Filter Implementation: All the parameters of the filter are passed to the
;  function in the form of a data structure InterpFirfp_t. The accumulator is
;  64-bit floating point. Each output sample is obtained by shifting the
;  contents of the accumulator (after M_i values have been accumulated) by an
;  amount specified in the filter data structure, and then extracting bits 15
;  to 0, where bit 0 corresponds to the 0th power of 2. The shifted contents
;  of the accumulator are saturated if necessary.
;
;              Input sample  Filter coeff.
;                 (S.15)      (float64)      (S = sign bit)
;                    \           /
;                     \         /
;                      \->(X)<-/
;                          |
;                      (float64)
;                          |
;                          v
;                         (+)<---------|
;                          |           |
;                    ______v______     |
;                   | Accumulator |    |
;                   |  (float64)  | (Delay)
;                   |_____________|    ^
;                          |           |
;           _______________|___________|
;          |               |
;     _____v______     ____v____
;    | Saturation |   | Shifter |
;    |   Logic    |   |_________|
;    |____________|        |
;          |           ____v____
;          |          | Extract |
;          |          | bits 15 |
;          |          |  to 0   |
;          |          |_________|
;          |               |
;          |               |    0x7FFF  0x8000
;          |               |       |       |
;          |              _v_______v_______v_
;          |------------>|    Multiplexer    |
;                        |___________________|
;                                  |
;                                  v
;                            Output sample
;                                (S.15)
;
; Arguments:
;  psa_Input      (I)      Pointer to array of input time samples.
;  us_InputLength (I)      Length of input time sequence.
;  psa_Output     (O)      Pointer to array of output time samples.
;                    The length of this array should at least
;                    equal s_InputLength*U, where U is the
;                    upsampling factor in the pt_InterpFir
;                    data structure.
;  pt_InterpFirfp (I)      Pointer to filter data structure.
;
; Return Value:
;  Number of output samples that overflowed/underflowed
************************************************************************/
int16 interp_fir_fp(const int16 sa_Input[], uint16 us_InputLength, int16 sa_Output[], InterpFirfp_t* pt_InterpFirfp)
{
   int16 i, j, k, s_NumOverflows, s_Phase, s_StateLength;
   float64 d_Acc, d_IntPart;

   /* Initialize overflow counter. */
   s_NumOverflows = 0;

   /* Generate the polyphase components of the output successively. */
   for(s_Phase=0; s_Phase<pt_InterpFirfp->us_UpsamplingFactor; s_Phase++) {

      /* Convolve input sequence with one phase */
      /* of the filter impulse response. */
      for(i=0; i<us_InputLength; i++) {

         /* Reset accumulator. */
         d_Acc = 0;

         /* Accumulate one output sample */
         for(k=i, j=s_Phase; (j<pt_InterpFirfp->us_Length) && (k>=0); j+=pt_InterpFirfp->us_UpsamplingFactor)
            d_Acc += ((float64) sa_Input[k--])*(pt_InterpFirfp->pda_Fir[j]);

         if(k<0) {
            for(k=0; j<pt_InterpFirfp->us_Length; j+=pt_InterpFirfp->us_UpsamplingFactor)
               d_Acc += ((float64) (pt_InterpFirfp->psa_State[k++]))*(pt_InterpFirfp->pda_Fir[j]);
         }

         /* Extract output sample from accumulator. */
         k = s_Phase + pt_InterpFirfp->us_UpsamplingFactor*i;
         d_Acc *= pow((float64) 2, (float64) pt_InterpFirfp->s_OutputShift);
         if(d_Acc > pow((float64) 2, (float64) 15)-1) {

            sa_Output[k] = (int16) 0x7fff;
            s_NumOverflows++;
         }
         else if(d_Acc < -pow((float64) 2, (float64) 15)) {

            sa_Output[k] = (int16) 0x8000;
            s_NumOverflows++;
         }
         else if(modf(d_Acc, &d_IntPart)) {

            if(d_Acc >= 0)
               sa_Output[k] = (int16) d_IntPart;
            else
               sa_Output[k] = (int16) d_IntPart-1;
         }
         else

            sa_Output[k] =  (int16) d_IntPart;
      }
   }

   /* Update the state of the filter. */
   for(s_StateLength=-1, j=0; j<pt_InterpFirfp->us_Length; j+=pt_InterpFirfp->us_UpsamplingFactor)
      s_StateLength++;

   if(us_InputLength < s_StateLength) {

      /* Keep the most recent states. */
      for(i=s_StateLength-us_InputLength; i>0; i--)
         pt_InterpFirfp->psa_State[i+us_InputLength-1] = pt_InterpFirfp->psa_State[i-1];

      /* Prefix the most recent inputs to the state vector. */
      for(i=0; i<us_InputLength; i++)
         pt_InterpFirfp->psa_State[i] = sa_Input[us_InputLength-1-i];
   }
   else {

      /* Write the most recent inputs to the state vector. */
      for(i=0; i<s_StateLength; i++)
         pt_InterpFirfp->psa_State[i] = sa_Input[us_InputLength-1-i];
   }

   return(s_NumOverflows);
}

/*^^^
*-------------------------------------------------------------------
*  Prototype:
*      void Filter(int16* psa_input, int16* psa_output, int16 s_len);
*
*   Description:
*      Helper function to be used with Channel(). Applies effects of passive
*      channel as an IIR filter.
*
*  Arguments:
*      psa_input             - filter input
*      psa_output            - filter output
*      s_len                 - length
*
*  Returns: none
*
*  Global Variables: none
*
*
*-------------------------------------------------------------------
*^^^
*/
#ifndef USE_FLOATINGPOINT_FOR_CHANNEL

void Filter(int16* psa_input, int16* psa_output, int16 s_len)
{
   int16    n, i;
   int32   l_y0;
   int32   l_Acc0;

   for (n = 0; n < s_len; n++) {

         /*  clear output */
      l_y0 = 0;

         /* ================= */
         /*  compute FIR part */
         /* ================= */

         /* update FIR memory */
      for (i = (FIR_LEN - 1); i > 0; i--) {
         sa_x_mem[i] = sa_x_mem[i-1];
      }
      sa_x_mem[0] = psa_input[n];

         /* apply FIR */
      for (i = 0; i < FIR_LEN; i++) {
         l_Acc0 = (int32)sa_x_mem[i] * gsa_b[i];
         l_y0 = l_add(l_y0, l_Acc0);
      }

         /* ================= */
         /*  compute IIR part */
         /* ================= */

         /* update IIR memory */
      for (i = (IIR_LEN - 1); i > 0; i--) {
         sa_y_mem[i] = sa_y_mem[i-1];
      }

         /* apply IIR */
      for (i = 1; i < IIR_LEN; i++) {
         l_Acc0 = (int32)sa_y_mem[i] * gsa_a[i];
         l_Acc0 *= -1;

         l_y0 = l_add(l_y0, l_Acc0);
      }
      l_y0 <<= LOG2_ONE_DIV_A0;
      sa_y_mem[0] = (int16) (round(l_y0,15));

      psa_output[n] = sa_y_mem[0];

   }

}

#else /* #ifdef USE_FLOATINGPOINT_FOR_CHANNEL */

void Filter(int16* psa_input, int16* psa_output, int16 s_len)
{
   int16    n, i;
   float   f_y0;
   float   f_Acc0;

   for (n = 0; n < s_len; n++) {

         /*  clear output */
      f_y0 = 0;

         /* ================= */
         /*  compute FIR part */
         /* ================= */

         /* update FIR memory */
      for (i = (FIR_LEN - 1); i > 0; i--) {
         gfa_x_mem[i] = gfa_x_mem[i-1];
      }
      gfa_x_mem[0] = (float) psa_input[n];

         /* apply FIR */
      for (i = 0; i < FIR_LEN; i++) {
         f_Acc0 = gfa_x_mem[i] * gfa_b[i];
         f_y0 = f_y0 + f_Acc0;
      }

         /* ================= */
         /*  compute IIR part */
         /* ================= */

         /* update IIR memory */
      for (i = (IIR_LEN - 1); i > 0; i--) {
         gfa_y_mem[i] = gfa_y_mem[i-1];
      }

         /* apply IIR */
      for (i = 1; i < IIR_LEN; i++) {
         f_Acc0 = gfa_y_mem[i] * gfa_a[i];
         f_y0 = f_y0 - f_Acc0;
      }
      f_y0 /= gfa_a[0];

      gfa_y_mem[0] = f_y0;
      psa_output[n] = (int16)gfa_y_mem[0];

   }

}
#endif /* #ifdef USE_FLOATINGPOINT_FOR_CHANNEL */

#define IA 16807
#define IM 2147483647
#define AM (1.0/IM)
#define IQ 127773
#define IR 2836
#define NTAB 32
#define NDIV (1+(IM-1)/NTAB)
#define EPS 1.2e-7
#define RNMX (1.0-EPS)

float64 ran1(int32 *idum)
{
   int16 j;
   int32 k;
   static int32 iy = 0;
   static int32 iv[NTAB];
   float64 temp;

   if ((*idum <= 0) || !iy) {
      if (-(*idum) < 1) *idum = 1;
      else *idum = -(*idum);
      for (j=NTAB+7; j>=0; j--) {
         k = (*idum)/IQ;
         *idum = IA*(*idum-k*IQ)-IR*k;
         if (*idum < 0) *idum += IM;
         if (j < NTAB) iv[j] = *idum;
      }
      iy = iv[0];
   }
   k = (*idum)/IQ;
   *idum = IA*(*idum-k*IQ)-IR*k;
   if (*idum < 0) *idum += IM;
   j = iy/NDIV;
   iy = iv[j];
   iv[j] = *idum;
   if ((temp = AM*iy) > RNMX) return RNMX;
   else return temp;
}

#undef IA
#undef IM
#undef AM
#undef IQ
#undef IR
#undef NTAB
#undef NDIV
#undef EPS
#undef RNMX

float64 gasdev(int32 *idum)
{
   float64 ran1(int32 *idum);
   static int16 iset = 0;
   static float64 gset;
   float64 fac, rsq, v1, v2;

   if (iset == 0) {
      do {
         v1 = 2.0*ran1(idum)-1.0;
         v2 = 2.0*ran1(idum)-1.0;
         rsq = v1*v1+v2*v2;
      }while ((rsq >= 1.0) || (rsq == 0.0));
      fac = sqrt(-2.0*log(rsq)/rsq);
      gset = v1*fac;
      iset = 1;
      return v2*fac;
   }
   else {
      iset = 0;
      return gset;
   }
}

#define SEED (-10) /* < 0 */
/****************************************************************************
*  Prototype:
*      int16 Noise(void);
*
*   Description:
*      Generates gaussian noise with variance 37dB below full scale.
*
*  Arguments:
*     N/A
*
*  Returns:
*     noise sample
*
*  Global Variables:
*     N/A
*****************************************************************************/
int16 Noise(void)
{
   static int32 l_seed = SEED;
   return ((int16) (72*gasdev(&l_seed)));
}
#undef SEED


/* undefine constants used by this file only */
#undef FIR_LEN
#undef IIR_LEN
#undef MAX_32
#undef MIN_32
