/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-1998 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   DebugBuffer.c
*
*  It contains functions related to Debug Buffer operation
*
*-------------------------------------------------------------------------
*/
// ****************************************************************************************************
// DebugBuffer.c
// History :
//
// 15/12/2017 Sriram shastry: DCT tool does not work in ADSL mode for VRX518 FW R5
// Added Support towards VRX518 Platform for  debug traces ( Updated address field )
// Search  Pattern XDSLRTFW-3638
// ****************************************************************************************************
#include "gdata.h"
#include <stdarg.h>
#include "dataswap.h"
#include "aux_reg_danube.h"
#include "cmv.h"
#include "pll.h"
#include "rx_plam.h"
#include "tx_plam.h"
#include "exchdata.h"
#include "gdata_bis.h"
#include "string.h"

#ifdef DEBUG_STREAMING
#include "Crc16.h"
#include "Version.h"
#include "Nmp_plfm.h"
#include "Hmp_funcs.h"
#include "Fifo.h"
void disable1_save(int32 *pul_stat);
void restore1_save(int32 ul_stat);
uint16 DSH_calcCrc16(uint16 crc, uint8 c);
#endif

#ifdef DEBUG_TRACES
#include "DebugBuffer.h"
extern FlagT GetXdmaAccessStatus(void);
/*
 *-------------------------------------------------------------------------------
 *
 * Prototype: int32 LogMessages(int16 s_count,int16 s_Write_type, ... )
 *
 * This function accepts variable number of arguments in a particular order and copies the Tx/Rx Message data (GHS +Channel Discovery +Training +Channel Analysis + Exchange) to internal buffer . DMA of this buffer is done at the end of GHS once and At the end of Exchange once)
 *
 * Input Arguments:
 *    s_count => number of arguments to be sent(excluding s_Write_type)
 *    s_Write_type =>  END_OF_THE_MSG if buffer has complete message
 *                      NOT_END_OF_MSG if part of message is sent
 *
 *    order of arguments type is fixed ---
 *    Example ---
 *    case 1 : If number of arguments passing are more than 2
 *    1st argument - s_count
 *    2nd argument - s_Write_type
 *    3nd argumnet - 16 bit
 *    4rd argument - 16 bit(length of message to be copied)
 *    5th argument - address(Message source address )
 *    6th argument - 32 bit(ideally -- Delimiter)
 *
 *    Case 2 : If number of arguments passing are equal to 2
 *    1st argument - s_count
 *    2nd argument - s_Write_type
 *    3nd argumnet - 32 bit(ideally -- Delimiter)
 *
 *    Any addition of arguments can be safely done by passing the arguments at beginning and modifying the corresponding case statement
 *
 *
 *
 *   Returns: 1) NO_SPACE_INTERNAL_BUFFER if no space available in the internal buffer
 *
 *       2) NO_DEBUG_BUFFER_ERROR if Space available
 *
 *
 * Global Variables:
 *
 *-------------------------------------------------------------------------------
 */


int32 LogMessages(int16 s_count,int16 s_Write_type, ... )
{
   //if oth bit is set then stop logging  data
   if(((gt_DebugBuffControl.us_debugBuffer_StopStartCaptureData)& GHS_TRAINING_CAPTURE_ENABLE_CHECK)!=STOP_GHS_TRAINING_CAPTURE)
   {

      va_list argp;
      uint16 us_msg_length = 0,us_msg_length_pad=0,us_delimiter_length =0;;
      uint16 us_total_buff_size=(DEBUG_INTERNAL_BUFFER_MAX_SIZE_BYTES>>1);
      uint16 us_Msg_Type = 0;

      uint8* puc_msg_data;
      int32 l_temp_msg=0;

      //maximum of 4 arguments can be sent with the current implementation and can be added if needed by
      //changing the the correponding case statement
      if(s_count>=5)
         return(EXCEEDS_MAX_ARGUMENTS);

      va_start( argp, s_Write_type);

      while(s_count-- > 0)
      {
         switch(s_count)
         {
         case 0: if(s_Write_type==END_OF_THE_MSG)
               {
                  //symbol count or delimiter
                  *((uint32 *)(void *)(gpusa_debugBuff + gt_debugBufStatus.s_debugTrailLen))= va_arg( argp, uint32 );
                  gt_debugBufStatus.s_debugTrailLen+=2;
                  gus_PrevMsg_Length=0;
               }
               else
                  gus_PrevMsg_Length+=us_msg_length+ us_msg_length_pad;

               return(NO_DEBUG_BUFFER_ERROR);

         case 1 : puc_msg_data = va_arg( argp, uint8* );
            // memcpy only if buffer space is available
               if((us_total_buff_size-gt_debugBufStatus.s_debugTrailLen) >= ((us_msg_length+us_msg_length_pad+us_delimiter_length)>>1)+2)
               {
                  *(gpusa_debugBuff+gt_debugBufStatus.s_debugTrailLen) = us_Msg_Type;
                  *(gpusa_debugBuff+gt_debugBufStatus.s_debugTrailLen+1) = us_msg_length;
                  gt_debugBufStatus.s_debugTrailLen+=2;
                  memcpy(&gpusa_debugBuff[gt_debugBufStatus.s_debugTrailLen],puc_msg_data,us_msg_length);
               }
               else
                  return(NO_SPACE_INTERNAL_BUFFER);


               if(s_Write_type==END_OF_THE_MSG)
               {
                  memcpy((uint8*)(void *)gpusa_debugBuff+(gt_debugBufStatus.s_debugTrailLen<<1)+us_msg_length,&l_temp_msg,us_msg_length_pad);

               }
               gt_debugBufStatus.s_debugTrailLen+= ((us_msg_length+ us_msg_length_pad)>>1);
               break;
         case 2 :
                     //message length
                     us_msg_length = va_arg( argp, uint16 );
                     if(s_Write_type==END_OF_THE_MSG)
                     {
                        us_msg_length_pad = ((4-(us_msg_length&3))&3);
                        us_delimiter_length=4;
                     }


                  break;
         case 4:
         case 3 :
                  //message type or symbol count
                  us_Msg_Type = va_arg( argp, uint16 );
                  break;
         }

      }

      va_end( argp );
   }
   return 0;
}

/*
 *-------------------------------------------------------------------------------
 *
 * Prototype: int32 LogMessageSegments(int16 s_Write_type,uint16 us_Msg_length,uint8 *puc_data,uint32 ul_delimiter)
 *
 * This function copies the Tx/Rx Message segment data (GHS +Channel Discovery +Training +Channel Analysis + Exchange) to internal buffer . DMA of this buffer is done at the end of GHS once and At the end of Exchange once)
 *
 * Input Arguments:
 *
 *    s_Write_type =>  END_OF_THE_MSG if buffer has complete message
 *                NOT_END_OF_MSG if part of message is sent
 *    us_Msg_length => length of message to be copied in bytes
 *    puc_data     => pointer to message data
 *    ul_delimiter => Delimiter to copy at the the end of message
 *
 *   Returns: 1) NO_SPACE_INTERNAL_BUFFER if no space available in the internal buffer
 *
 *       2) NO_DEBUG_BUFFER_ERROR if Space available
 *
 *
 *  Global Variables:
 *  gus_PrevMsg_Length => will have the length of previous segment of message data written before this call and
 *  there by helps updating the message length field dynamically and will be set to zero at the end of message
 *
 *-------------------------------------------------------------------------------
 */


int32 LogMessageSegments(int16 s_Write_type,uint16 us_Msg_length,uint8 *puc_data,uint32 ul_delimiter)
{
   //if oth bit is set then stop logging  data
   if(((gt_DebugBuffControl.us_debugBuffer_StopStartCaptureData)& GHS_TRAINING_CAPTURE_ENABLE_CHECK)!=STOP_GHS_TRAINING_CAPTURE)
   {
      uint16 us_total_buff_size=(DEBUG_INTERNAL_BUFFER_MAX_SIZE_BYTES>>1),us_msg_length_pad=0,us_delimiter_length=0;
      int32 l_temp_msg=0;

     //calculate the padding bytes if end of the message
      if(s_Write_type==END_OF_THE_MSG)
      {
         us_msg_length_pad = ((4-((gus_PrevMsg_Length+us_Msg_length)&3))&3);
         us_delimiter_length=4; //4 bytes
      }

      //copy only if buffer space available
      if((us_total_buff_size-gt_debugBufStatus.s_debugTrailLen) > ((us_Msg_length+us_msg_length_pad+us_delimiter_length)>>1))
         memcpy((uint8*)(void *)gpusa_debugBuff+(gt_debugBufStatus.s_debugTrailLen<<1)+((2-(gus_PrevMsg_Length&1))&1),puc_data,us_Msg_length);
      else
         return(NO_SPACE_INTERNAL_BUFFER);


      //increment the number of bytes of message data copied in the buffer in the message length field
      *(gpusa_debugBuff+(gt_debugBufStatus.s_debugTrailLen)-(gus_PrevMsg_Length>>1)-1)=*(gpusa_debugBuff+(gt_debugBufStatus.s_debugTrailLen)-(gus_PrevMsg_Length>>1)-1)+us_Msg_length;


      if(s_Write_type==END_OF_THE_MSG)
      {
         //copy padding bytes
         memcpy(((uint8*)(void *)gpusa_debugBuff+(gt_debugBufStatus.s_debugTrailLen<<1)+us_Msg_length+((2-(gus_PrevMsg_Length&1))&1)),&l_temp_msg,us_msg_length_pad);

         gt_debugBufStatus.s_debugTrailLen+=(us_Msg_length+((2-(gus_PrevMsg_Length&1))&1)+us_msg_length_pad)>>1 ;

         //delimiter
         *((uint32 *)(void *)(gpusa_debugBuff + gt_debugBufStatus.s_debugTrailLen))= ul_delimiter;
         gt_debugBufStatus.s_debugTrailLen+=2;
         gus_PrevMsg_Length=0;
      }
      else
      {
         gus_PrevMsg_Length+=us_Msg_length;
         gt_debugBufStatus.s_debugTrailLen+=(us_Msg_length+((2-(gus_PrevMsg_Length&1))&1))>>1 ;
      }

      return(NO_DEBUG_BUFFER_ERROR);
   }
   return 0;
}

/*
 *-------------------------------------------------------------------------------
 *
 * Prototype: void Init_DebugBuf_variables()
 *
 * This function initializes all the variables related to Debug Buffer. All variables related to Debug Buffer
 *  should be initialized here as these variable are defined only after the FW download i.e same space is being
 *  used as data buffers in FW download period and failing to follow these intructions might result in Fail state
 *  at the GHS only
 *
 *
---------------------------------------------------------------------------------
*/
void Init_DebugBuf_variables()
{

         gpusa_debugBuff=(uint16 *)(void *)&gusa_DebugBuf ; // initialize the pointer for debug buffer
         gpuca_debugBuff=(uint8 *)(void *)&gusa_DebugBuf ;
         gus_PrevMsg_Length=0;


         gt_debugBufStatus.s_debugTrailLen=0;

         gul_DMA_Busy_Error=gt_DebugBufferStatus.s_debugBuffer_GHS_TO_EXCHANGE_length=gt_DebugBufferStatus.s_debugBuffer_Showtime_length=gt_DebugBufferStatus.s_debugBuffer_Showtime_Oldested_data_Start=0;

         gt_DebugBufferStatus.s_debugBuffer_Size_of_Per_Symbol_structure_Bytes=sizeof(DebugShowTimeBuffSymbol_t);

         gt_DebugBufferStatus.s_debugBuffer_Size_of_Super_Frame_structure_Bytes=0;
         gus_LenDebugBuff=0;

         gt_DebugBuffSuperframe.l_sync_structure_start_delimiter=DEBUG_BUFFER_DELIMITER_SYC_SYM_START;
         gt_DebugBuffSuperframeReTxEnabled.l_sync_structure_start_delimiter= DEBUG_BUFFER_DELIMITER_SYC_SYM_START;

         guc_ShowtimeEnter=0;

         //for debug purpose, log GHS messages recieved/transmitted
         LogMessages(1,END_OF_THE_MSG,DEBUG_BUFFER_DELIMITER_GHS_START);

}

/*
 *-------------------------------------------------------------------------------
 *
 * Prototype: void RequestSwap_DebugBuff_GHS_OR_TRAINING_Tomem()
 *
 * This function initializes all the variables of a stucture related to DMA initialization.
 *  This function can be called when DMA of collected data till training in Debug Buffer has to be done
 *
 * Input Arguments:
 *
 *    s_Write_type =>  END_OF_THE_MSG if buffer has complete message
 *                NOT_END_OF_MSG if part of message is sent
 *    us_Msg_length => length of message to be copied in bytes
 *    puc_data     => pointer to message data
 *    ul_delimiter => Delimiter to copy at the the end of message
 *
 *
---------------------------------------------------------------------------------
*/

void RequestSwap_DebugBuff_GHS_OR_TRAINING_Tomem()
{
   if(((gt_DebugBuffControl.us_debugBuffer_StopStartCaptureData)& GHS_TRAINING_CAPTURE_ENABLE_CHECK)!=STOP_GHS_TRAINING_CAPTURE)
   {
      // XDSLRTFW-3638 (Start_End)
      static int32 l_DstAddr_update=(int32)&gula_DebugBuffer_BAR15[0];      // XDSLRTFW-1942 (Start_End)

      gt_DataSwap.c_State   = DATASWAP_REQUESTED;
      gt_DataSwap.l_SrcAddr = (int32)gusa_DebugBuf;
      gt_DataSwap.l_DstAddr = l_DstAddr_update;//s_DstAddr_update;
      //gt_debugBufStatus.s_debugTrailLen is the number of words to DMA
      gt_DataSwap.s_BufSize = ((gt_debugBufStatus.s_debugTrailLen+1)/2)*4; // byte size, longword aligned and

      gt_DataSwap.c_XferDir = XDMA_WRITE_XMEM;
      gus_DataBuf_InXmem   |= DEBUG_BUF_IN_XMEM;

      l_DstAddr_update+= ((gt_debugBufStatus.s_debugTrailLen+1)/2)*4;

      //if DMA is free
      if (GetXdmaAccessStatus()!= 0)
      gt_DebugBufferStatus.s_debugBuffer_GHS_TO_EXCHANGE_length+=((gt_debugBufStatus.s_debugTrailLen+1)/2*4)>>1;
   }

}

/*
 *-------------------------------------------------------------------------------
 *
 * Prototype: void RequestSwap_DebugBuff_ShowTime_Tomem(int32 *pl_addr_source ,int16 s_size_bytes)
 *
 * This function initializes all the variables of a stucture related to DMA initialization.
 *  This function can be called when DMA of collected data in showtime in Debug Buffer has to be done
 *
 **   Input Arguments:
 *
 *    pl_addr_source =>  Pointer to source(DATA) of which DMA to be inititated
 *    s_size_bytes => length of message to be copied in bytes
 *
 *
---------------------------------------------------------------------------------
*/
void RequestSwap_DebugBuff_ShowTime_Tomem(int32 *pl_addr_source ,int16 s_size_bytes)
{
   if(((gt_DebugBuffControl.us_debugBuffer_StopStartCaptureData) & SHOW_TIME_CAPTURE_ENABLE_CHECK)!=STOP_SHOW_TIME_CAPTURE)
   {
      // XDSLRTFW-3638 (Start)
      /* gula_DebugBuffer_BAR15 is a 32-bit  Array with sizeof 2000bytes i.e. 2000/4 .
      so gula_DebugBuffer_BAR15 start index is 500*/
      // XDSLRTFW-3638 (End)
      static int32 l_DstAddr_update = (int32)&gula_DebugBuffer_BAR15[DEBUG_BUF_GHS_TRAINING_SIZE >> 2];      // XDSLRTFW-1942 (Start_End)

      gt_DataSwap.c_State   = DATASWAP_REQUESTED;
      gt_DataSwap.l_SrcAddr = (int32)pl_addr_source;
      gt_DataSwap.l_DstAddr = l_DstAddr_update;//s_DstAddr_update;
      gt_DataSwap.s_BufSize = ((s_size_bytes+3)/4)*4; // byte size, longword aligned
      gt_DataSwap.c_XferDir = XDMA_WRITE_XMEM;
      gus_DataBuf_InXmem   |= DEBUG_BUF_IN_XMEM;

      l_DstAddr_update+= s_size_bytes;
      gt_debugBufStatus.s_debugTrailLen+=s_size_bytes;

      //if destination address is equal or greater than showtime end address => reset the address to start
      if(l_DstAddr_update>=XMEM_DEBUG_BUF_SHOWTIME_END)
      {
         // XDSLRTFW-3638 (Start_End)
         /* gula_DebugBuffer_BAR15 is a 32-bit  Array with sizeof 2000bytes i.e. 2000/4 .
         so gula_DebugBuffer_BAR15 start index is 500*/
         l_DstAddr_update= (int32)&gula_DebugBuffer_BAR15[DEBUG_BUF_GHS_TRAINING_SIZE >> 2];
         //make the length to zero
         gt_debugBufStatus.s_debugTrailLen=0;
         //set the CMV DSL 26 1 to maximum of showtime bytes available
         gt_DebugBufferStatus.s_debugBuffer_Showtime_length=(SDRAM_RAW_DUMP_SHOWTIME_SIZE_BYTES>>1);

         //flag indicating the wrap around condition
         guc_StLengthFlag=1;
      }

   //if DMA is free then only increment the length
      if (GetXdmaAccessStatus()!= 0)
      {


         if(guc_StLengthFlag==0)
         {
            //if no wrap around then increment the maximum showtime length bytes written
            gt_DebugBufferStatus.s_debugBuffer_Showtime_length=(((gt_debugBufStatus.s_debugTrailLen+3)/4)*4)>>1;
         }
         else
         {
            //increment the pointer to point to the oldest data start
            gt_DebugBufferStatus.s_debugBuffer_Showtime_Oldested_data_Start=(((gt_debugBufStatus.s_debugTrailLen+3)/4)*4)>>1;
         }
      }



   }

}


/*
 *-------------------------------------------------------------------------------
 *
 * Prototype: void updateShowtimeBuffSymbol(void)
 *
 * This function updates the Debug structure which collects data for every symbol and eventually calls a function which will copy this sturucture to internal buffer
 *
 * Input Arguments: void
 *
 * Returns: Void
 *
 *
 * Global Variables:
 *
 *-------------------------------------------------------------------------------
 */



void updateShowtimeBuffSymbol(void)
{
   //if 2nd bit set then stop logging showtime data
   if(((gt_DebugBuffControl.us_debugBuffer_StopStartCaptureData) & SHOW_TIME_CAPTURE_ENABLE_CHECK)!=STOP_SHOW_TIME_CAPTURE)
   {

      /*    int32 l_RxTimer1, l_RxTimer2;
         int32 s_MIPS;


      //Read TX timer value
      ReadRxTimer(&l_RxTimer1); */

      //copy real and imaginary part of pilottone to structure
      memcpy(&gt_DebugBuffSymbol.s_rx_PilotToneReal,gsa_RxPilotTone,4);

      LogShowtimeData((uint8 *)(void *)&gt_DebugBuffSymbol,sizeof(DebugShowTimeBuffSymbol_t));



      /*       ReadRxTimer(&l_RxTimer2);

      //Compute the MIPS for this task
      s_MIPS = (l_RxTimer2 - l_RxTimer1);

      if (s_MIPS < 0)
      {
      s_MIPS = (s_MIPS + gl_MaxRxTimerCnt);
      }

      if(s_MIPS>gl_mipsdebug2) {
      gl_mipsdebug2=s_MIPS;
      }
      gl_mipsdebug1= s_MIPS;
      } */
   }
}


/*
 * -----------------------------------------------------------------------------
 *  Prototype: void Decode_Config_Info(void)
 *
 *  This function fetches the parameters required for the profile determination
 *  and assigns it to an CMV INFO 241
 *
 *    Input arguments: Void
 *    Returns: Void
 * -----------------------------------------------------------------------------
 */

// XDSLRTFW-1782: CAPTURE_CONFIGURATION_MESSAGES(START)
void Decode_Config_Info(void)
{
   int32 i;
   gt_Decoded_ProfileInfo_t.s_connection_mode0=gt_HercADSL_STATMap_ModeSelect.s_Offset0;

   if (gt_Decoded_ProfileInfo_t.s_connection_mode0==1 || gt_Decoded_ProfileInfo_t.s_connection_mode0==4 ||
       gt_Decoded_ProfileInfo_t.s_connection_mode0==8 )        // ADSL 1
      {
         gt_Decoded_ProfileInfo_t.us_RA_MODE_DS=0;
         gt_Decoded_ProfileInfo_t.us_RA_USNRM_DS=0;
         gt_Decoded_ProfileInfo_t.us_RA_DSNRM_DS=0;
         gt_Decoded_ProfileInfo_t.us_RA_UTIME_DS=0;
         gt_Decoded_ProfileInfo_t.us_RA_DTIME_DS=0;
         gt_Decoded_ProfileInfo_t.s_MINIMUM_SNR_MARGIN_DS=gt_RCMsgRA.us_MinRequiredSSDSSnrMargin;
         gt_Decoded_ProfileInfo_t.s_MAXIMUM_SNR_MARGIN_DS=gt_RCMsgRA.us_MaxDSSnrMargin;
         gt_Decoded_ProfileInfo_t.s_TARGET_SNR_MARGIN_DS=gt_RCMsgRA.us_MinRequiredInitDSSnrMargin;
      }
   else
      {
         gt_Decoded_ProfileInfo_t.us_RA_MODE_DS=gt_RCMsgs1_bis.us_RA_MODEds;
         gt_Decoded_ProfileInfo_t.us_RA_USNRM_DS=gt_RCMsgs1_bis.us_RA_USNRMds;
         gt_Decoded_ProfileInfo_t.us_RA_DSNRM_DS=gt_RCMsgs1_bis.us_RA_DSNRMds;
         gt_Decoded_ProfileInfo_t.us_RA_UTIME_DS=gt_RCMsgs1_bis.us_RA_UTIMEds;
         gt_Decoded_ProfileInfo_t.us_RA_DTIME_DS=gt_RCMsgs1_bis.us_RA_DTIMEds;
         gt_Decoded_ProfileInfo_t.us_cut_back_C_US=gt_TxPMDControl.us_pwr_cutback_US;
         gt_Decoded_ProfileInfo_t.s_TARGET_SNR_MARGIN_DS=gt_RCMsgs1_bis.us_TARSNRMds;
         gt_Decoded_ProfileInfo_t.s_MINIMUM_SNR_MARGIN_DS=gt_RCMsgs1_bis.us_MINSNRMds;
         gt_Decoded_ProfileInfo_t.s_MAXIMUM_SNR_MARGIN_DS=gt_RCMsgs1_bis.us_MAXSNRMds;
         gt_Decoded_ProfileInfo_t.s_MAX_DATA_RATE_DS=gt_HandshakeBis.sa_DS_MaxNetDataRateBC[0];
         gt_Decoded_ProfileInfo_t.s_MIN_DATA_RATE_DS=gt_HandshakeBis.sa_DS_MinNetDataRateBC[0];
         gt_Decoded_ProfileInfo_t.s_delay_DS=gt_HandshakeBis.sa_DS_MaxLatencyBC[0];
         gt_Decoded_ProfileInfo_t.s_INP_erasure_DS=gt_HandshakeBis.sa_DS_MinINPBC[0];
   }
   gt_Decoded_ProfileInfo_t.us_cut_back_C_DS=gt_TxPMDControl.us_pwr_cutback_DS;

if (gt_Decoded_ProfileInfo_t.s_connection_mode0==8 || gt_Decoded_ProfileInfo_t.s_connection_mode0==512 ||
    gt_Decoded_ProfileInfo_t.s_connection_mode0==16384)
    {
       gt_Decoded_ProfileInfo_t.s_MAX_NOMINAL_AGGRE_POWER_DS=199;
    }
   else
   {
       gt_Decoded_ProfileInfo_t.s_MAX_NOMINAL_AGGRE_POWER_DS=204;
   }

if (gt_Decoded_ProfileInfo_t.s_connection_mode0==8 || gt_Decoded_ProfileInfo_t.s_connection_mode0==512 ||
    gt_Decoded_ProfileInfo_t.s_connection_mode0==16384 || gt_Decoded_ProfileInfo_t.s_connection_mode0==16 ||
    gt_Decoded_ProfileInfo_t.s_connection_mode0==64 )
    {
       gt_Decoded_ProfileInfo_t.s_MAX_NOMINAL_AGGRE_POWER_US=133;
    }
else if (gt_Decoded_ProfileInfo_t.s_connection_mode0==32 || gt_Decoded_ProfileInfo_t.s_connection_mode0==128 )
    {
       gt_Decoded_ProfileInfo_t.s_MAX_NOMINAL_AGGRE_POWER_US=134;
    }
    else
    {
       gt_Decoded_ProfileInfo_t.s_MAX_NOMINAL_AGGRE_POWER_US=125;
    }


   gt_Decoded_ProfileInfo_t.s_connection_support[0]=gt_HercADSL_OPTNMap_ModeControl.s_Offset0;
   gt_Decoded_ProfileInfo_t.s_connection_support[1]=gt_HercADSL_OPTNMap_ModeControl.s_Offset1;

   // Upstream Tssi or Breakpoints
   gt_Decoded_ProfileInfo_t.tssi_info_US.s_NumBrkptkept=gt_GHS_USTssiInfo.s_NumBrkptkept;
   for (i=0;i<gt_Decoded_ProfileInfo_t.tssi_info_US.s_NumBrkptkept;i++)
   {
      gt_Decoded_ProfileInfo_t.tssi_info_US.uca_Brkpt[i]=gt_GHS_USTssiInfo.uca_Brkpt[i];
   }
   for (i=0;i<gt_Decoded_ProfileInfo_t.tssi_info_US.s_NumBrkptkept;i++)
   {
      gt_Decoded_ProfileInfo_t.tssi_info_US.uca_Tssi[i]=gt_GHS_USTssiInfo.uca_Tssi[i];
   }
   //Downstream Tssi or Breakpoints
   gt_Decoded_ProfileInfo_t.tssi_info_DS.s_NumBrkptkept=gt_GHS_DSTssiInfo.s_NumBrkptkept;
   for (i=0;i<gt_Decoded_ProfileInfo_t.tssi_info_DS.s_NumBrkptkept;i++)
   {
      gt_Decoded_ProfileInfo_t.tssi_info_DS.usa_Brkpt[i]=gt_GHS_DSTssiInfo.usa_Brkpt[i];
   }
   for (i=0;i<gt_Decoded_ProfileInfo_t.tssi_info_DS.s_NumBrkptkept;i++)
   {
      gt_Decoded_ProfileInfo_t.tssi_info_DS.uca_Tssi[i]=gt_GHS_DSTssiInfo.uca_Tssi[i];
   }

   gt_Decoded_ProfileInfo_t.ft_ReTX_enable=gt_ReTxConfigInfo.ft_ReTxOn;
   gt_Decoded_ProfileInfo_t.us_RETX_ETR_MIN_DS=gt_ReTxConfigInfo.us_ETR_min;
   gt_Decoded_ProfileInfo_t.us_RETX_ETR_MAX_DS=gt_ReTxConfigInfo.us_ETR_max;
   gt_Decoded_ProfileInfo_t.us_RETX_DATARATE_MAX_DS=gt_ReTxConfigInfo.us_net_max;
   gt_Decoded_ProfileInfo_t.us_RETX_MIN_INP_DS=gt_ReTxConfigInfo.us_INP_min;
   gt_Decoded_ProfileInfo_t.us_RETX_SHINE_RATIO_DS=gt_ReTxConfigInfo.us_SHINEratio;
   gt_Decoded_ProfileInfo_t.us_RETX_MIN_INP_REIN_DS=gt_ReTxConfigInfo.us_INP_min_rein;
   gt_Decoded_ProfileInfo_t.us_RETX_MAX_DELAY_DS=gt_ReTxConfigInfo.us_delay_max;
   gt_Decoded_ProfileInfo_t.us_RETX_MIN_DELAY_DS=gt_ReTxConfigInfo.us_delay_min;
   gt_Decoded_ProfileInfo_t.us_RETX_LEFTR_THRES_DS=gt_ReTxConfigInfo.us_leftr_thresh;
}
// XDSLRTFW-1782: CAPTURE_CONFIGURATION_MESSAGES(END)

/*
 *-------------------------------------------------------------------------------
 *
 * Prototype: void updateShowtimeBuffSymbol(void)
 *
 *  This function updates the Debug structure which collects data for every Sync symbol and eventually calls a
 *  function which will copy this sturucture to internal buffer
 *
 * Input Arguments: void
 *
 * Returns: Void
 *
 *
 * Global Variables:
 *
 *-------------------------------------------------------------------------------
 */


void updateShowtimeBuffSuperframe(void)
{


   //if 2nd bit set then stop logging showtime data
   if(((gt_DebugBuffControl.us_debugBuffer_StopStartCaptureData) & SHOW_TIME_CAPTURE_ENABLE_CHECK)!=STOP_SHOW_TIME_CAPTURE)
   {
      /* int32 l_RxTimer1, l_RxTimer2;
         int32 s_MIPS;


      //Read TX timer value
      ReadRxTimer(&l_RxTimer1); */

      //is ReTx Enabled
      if(gt_ReTxConfigInfo.ft_ReTxOn)
      {
         gt_DebugBuffSuperframeReTxEnabled.l_symbolCountBuff=gl_TxSymbolCount;

         //copy Pilot tone Real and Imaginary parts
         memcpy(&gt_DebugBuffSuperframeReTxEnabled.s_rx_PilotToneReal,gsa_RxPilotTone,4);

         //copy Eoc timeout counts
         gt_DebugBuffSuperframeReTxEnabled.usa_EocNumTimeouts[0]=gta_TxHdlcControlInfo[0].s_TimedOutMsgCount;
         gt_DebugBuffSuperframeReTxEnabled.usa_EocNumTimeouts[1]=gta_TxHdlcControlInfo[1].s_TimedOutMsgCount;
         gt_DebugBuffSuperframeReTxEnabled.usa_EocNumTimeouts[2]=gta_TxHdlcControlInfo[2].s_TimedOutMsgCount;

         //copy US and DS Bitswap counts
         gt_DebugBuffSuperframeReTxEnabled.OLR_PM_COUNT_US.us_SimpleOLRPMReq_Cnt=gt_USOLRPMCnt.t_BitswapCnt.us_SimpleOLRPMReq_Cnt;
         gt_DebugBuffSuperframeReTxEnabled.OLR_PM_COUNT_US.us_OLRPMUTCResp_Cnt=gt_USOLRPMCnt.t_BitswapCnt.us_OLRPMUTCResp_Cnt;
         gt_DebugBuffSuperframeReTxEnabled.OLR_PM_COUNT_US.us_ActualOLRPMPerform_Cnt=gt_USOLRPMCnt.t_BitswapCnt.us_ActualOLRPMPerform_Cnt;
         gt_DebugBuffSuperframeReTxEnabled.OLR_PM_COUNT_US.us_RespTimeOut_Cnt=gt_USOLRPMCnt.t_BitswapCnt.us_RespTimeOut_Cnt;

         gt_DebugBuffSuperframeReTxEnabled.OLR_PM_COUNT_DS.us_SimpleOLRPMReq_Cnt=gt_DSOLRPMCnt.t_BitswapCnt.us_OLRPMUTCResp_Cnt;
         gt_DebugBuffSuperframeReTxEnabled.OLR_PM_COUNT_DS.us_OLRPMUTCResp_Cnt=gt_DSOLRPMCnt.t_BitswapCnt.us_OLRPMUTCResp_Cnt;
         gt_DebugBuffSuperframeReTxEnabled.OLR_PM_COUNT_DS.us_ActualOLRPMPerform_Cnt=gt_DSOLRPMCnt.t_BitswapCnt.us_ActualOLRPMPerform_Cnt;

         gt_DebugBuffSuperframeReTxEnabled.us_NeSefDefectCounter=gs_sef_cnt;

         //copy Margin parameters helps to identify failure to LOM
         gt_DebugBuffSuperframeReTxEnabled.s_SnrMargin=gt_NearEndParam.s_SnrMargin;
         gt_DebugBuffSuperframeReTxEnabled.s_RxMinMargin=gs_RxMinMargin;
         gt_DebugBuffSuperframeReTxEnabled.s_RxMinMarginTone=gs_RxMinMarginTone;

         //Copy NE and FE failure values
         gt_DebugBuffSuperframeReTxEnabled.s_g997_Failure_Flag[0]=gs_NearEndFailureReported;
         gt_DebugBuffSuperframeReTxEnabled.s_g997_Failure_Flag[1]=gs_FarEndFailure;

         //copy ReTx variables and copy only lower 16 bit as it is sufficient to identify changes within last 15 seconds
         //XDSLRTFW-1634 (Start_End)
         gt_DebugBuffSuperframeReTxEnabled.ReTx_Status.s_RetransmittedDtuCnt =(uint16)gt_ReTXStats.ul_RetransmittedDtuCnt;
         gt_DebugBuffSuperframeReTxEnabled.ReTx_Status.us_CorrectedDtuCnt=(uint16)gt_ReTXStats.ul_CorrectedDtuCnt;
         gt_DebugBuffSuperframeReTxEnabled.ReTx_Status.us_UncorrectedDtuCnt=(uint16)gt_ReTXStats.ul_UncorrectedDtuCnt;
         gt_DebugBuffSuperframeReTxEnabled.ReTx_Status.ul_TotalDtuCnt =(uint16)gt_ReTXStats.ul_TotalDtuCnt;

         //copy the entire structure formed to internal buffer
         LogShowtimeData((uint8 *)(void *)&gt_DebugBuffSuperframeReTxEnabled,gt_DebugBufferStatus.s_debugBuffer_Size_of_Super_Frame_structure_Bytes);

      }
      else
      {
         gt_DebugBuffSuperframe.l_symbolCountBuff=gl_TxSymbolCount;
         //copy Pilot tone Real and Imaginary parts
         memcpy(&gt_DebugBuffSuperframe.s_rx_PilotToneReal,gsa_RxPilotTone,4);
         //copy DS CRC for LPO and LP1
         gt_DebugBuffSuperframe.s_dsCrcLP0=(int16)gt_HercADSL_CNTRMap_g997_NE_ChPerf_CRCStats.ula_g997_CRC_RunningCnt[0];
         gt_DebugBuffSuperframe.s_dsCrcLP1=(int16)gt_HercADSL_CNTRMap_g997_NE_ChPerf_CRCStats.ula_g997_CRC_RunningCnt[1];

         gt_DebugBuffSuperframe.usa_EocNumTimeouts[0]=gta_TxHdlcControlInfo[0].s_TimedOutMsgCount;
         gt_DebugBuffSuperframe.usa_EocNumTimeouts[1]=gta_TxHdlcControlInfo[1].s_TimedOutMsgCount;
           gt_DebugBuffSuperframe.usa_EocNumTimeouts[2]=gta_TxHdlcControlInfo[2].s_TimedOutMsgCount;

         //copy US and DS Bitswap variables
         gt_DebugBuffSuperframe.OLR_PM_COUNT_US.us_SimpleOLRPMReq_Cnt=gt_USOLRPMCnt.t_BitswapCnt.us_SimpleOLRPMReq_Cnt;
         gt_DebugBuffSuperframe.OLR_PM_COUNT_US.us_OLRPMUTCResp_Cnt=gt_USOLRPMCnt.t_BitswapCnt.us_OLRPMUTCResp_Cnt;
         gt_DebugBuffSuperframe.OLR_PM_COUNT_US.us_ActualOLRPMPerform_Cnt=gt_USOLRPMCnt.t_BitswapCnt.us_ActualOLRPMPerform_Cnt;
         gt_DebugBuffSuperframe.OLR_PM_COUNT_US.us_RespTimeOut_Cnt=gt_USOLRPMCnt.t_BitswapCnt.us_RespTimeOut_Cnt;

         gt_DebugBuffSuperframe.OLR_PM_COUNT_DS.us_SimpleOLRPMReq_Cnt=gt_DSOLRPMCnt.t_BitswapCnt.us_OLRPMUTCResp_Cnt;
         gt_DebugBuffSuperframe.OLR_PM_COUNT_DS.us_OLRPMUTCResp_Cnt=gt_DSOLRPMCnt.t_BitswapCnt.us_OLRPMUTCResp_Cnt;
         gt_DebugBuffSuperframe.OLR_PM_COUNT_DS.us_ActualOLRPMPerform_Cnt=gt_DSOLRPMCnt.t_BitswapCnt.us_ActualOLRPMPerform_Cnt;

         gt_DebugBuffSuperframe.us_NeSefDefectCounter=gs_sef_cnt;
         gt_DebugBuffSuperframe.s_SnrMargin=gt_NearEndParam.s_SnrMargin;
         gt_DebugBuffSuperframe.s_RxMinMargin=gs_RxMinMargin;
         gt_DebugBuffSuperframe.s_RxMinMarginTone=gs_RxMinMarginTone;
         gt_DebugBuffSuperframe.s_g997_Failure_Flag[0]=gs_NearEndFailureReported;
         gt_DebugBuffSuperframe.s_g997_Failure_Flag[1]=gs_FarEndFailure;

         //copy the entire structure formed to internal buffer
         LogShowtimeData((uint8 *)(void *)&gt_DebugBuffSuperframe,gt_DebugBufferStatus.s_debugBuffer_Size_of_Super_Frame_structure_Bytes);
      }


   /*       ReadRxTimer(&l_RxTimer2);

      //Compute the MIPS for this task
      s_MIPS = (l_RxTimer2 - l_RxTimer1);

      if (s_MIPS < 0)
      {
      s_MIPS = (s_MIPS + gl_MaxRxTimerCnt);
      }

      if(gl_mipsdebug3<s_MIPS+gl_mipsdebug1) {
      gl_mipsdebug3=s_MIPS+gl_mipsdebug1;
      }
      if(gl_mipsdebug4<s_MIPS) {
      gl_mipsdebug4=s_MIPS;
      }
      gl_mipsdebug6=s_MIPS;
      // gl_mipsdebug1=1;  */
   }


}


/*
 *-------------------------------------------------------------------------------
 *
 * Prototype: int32 LogShowtimeData(uint8 *puc_base_addr , uint16 us_num_bytes)
 *
 * This function updates the internal Buffer(guca_DebugBuf) with the showtime debug data colleted in symbol(DebugShowTimeBuffSymbol_t) and superframe(DebugShowTimeBuffSuperframe_t) structure ... Internal Buffer(size=4000 bytes) is divided into Ping(2000 bytes) and Pong(2000 bytes) buffers . Keep filling ping untill it is full and then switch the buffer start filling the pong , parallelly initiate the DMA of ping ... Similarly once Pong is full , DMA the pong and start filling the ping ... If initiated DMA(of ping or pong) not completed by the time we want to DMA the other buffer it returns "Buffer FULL condition error" and wait untill DMA becomes free.
 *
 * Input Arguments:
 *
 *   puc_base_addr => source address of structure to be copied
 *   us_num_bytes => Number of Bytes to be copied
 *
 *   Returns: 'NO_DEBUG_BUFFER_ERROR' if no error
 *       'INTERNAL_BUFFER_FULL' if Buffer full condition
 *
 *
 * Global Variables:
 *
 *
      +-------------+ ^ ^
      |             | | |
      |  Ping Buf   | | | DEBUG_INTERNAL_PING_OR_PONG_BUFFER_MAX_SIZE_BYTES
      |             | | |
      +-------------+ | -
      |             | |
      |  Pong Buf   | | DEBUG_INTERNAL_BUFFER_MAX_SIZE_BYTES
      |             | |
      +-------------+ -

 *-------------------------------------------------------------------------------
 */


int32 LogShowtimeData(uint8 *puc_base_addr , uint16 us_num_bytes)
{

   static int32 l_Ping_DMA_once=0; //Flag which enables dma of Ping buffer to be done only once before DMA of        pong buffer is done


   //calculate the Total available length in the internal buffer
   int16 s_avail_length=(DEBUG_INTERNAL_BUFFER_MAX_SIZE_BYTES-gus_LenDebugBuff);

   //available length in ping buffer
   int16 s_avail_length_ping=(DEBUG_INTERNAL_PING_OR_PONG_BUFFER_MAX_SIZE_BYTES-gus_LenDebugBuff);


   //if available length in internal buffer is more than the number of bytes to be written
   if(s_avail_length >= us_num_bytes)
   {
      //Available length in Ping is less than number of bytes to be written and DMA of Pong is not yet over => error
      if((!l_Ping_DMA_once) && (s_avail_length_ping <= us_num_bytes) && (GetXdmaAccessStatus()== 0))
      {
         gul_DMA_Busy_Error++; //increment the DMA busy error count
         return(INTERNAL_BUFFER_FULL);
      }


      //copy the bytes to be written to internal buffer
      memcpy(&gpuca_debugBuff[gus_LenDebugBuff],puc_base_addr,us_num_bytes);
      //increment the index to next index to be written
      gus_LenDebugBuff+=us_num_bytes;

      //if ping buffer is full then initiate the DMA
      if(!l_Ping_DMA_once && (gus_LenDebugBuff>=DEBUG_INTERNAL_PING_OR_PONG_BUFFER_MAX_SIZE_BYTES))
      {
         // set the flag => DMA of ping is initiated once
         l_Ping_DMA_once=1;

      //set the properties of  DMA structure to initialize DMA of Ping
      RequestSwap_DebugBuff_ShowTime_Tomem((int32 *)(void *)&gusa_DebugBuf,DEBUG_INTERNAL_PING_OR_PONG_BUFFER_MAX_SIZE_BYTES);
      }
   return(NO_DEBUG_BUFFER_ERROR);
   }
   else
   {
      //check if DMA of ping completed
      if(GetXdmaAccessStatus()!= 0)
      {


         //copy available length data to pong buffer. Pong would be full after this memcpy
         memcpy(&gpuca_debugBuff[gus_LenDebugBuff],puc_base_addr,s_avail_length);

         //set the flag to zero so to write the ping now.
         l_Ping_DMA_once=0;

         //set the pointer length to zero
         gus_LenDebugBuff=0;

     //set the properties of  DMA structure to inittialize DMA of Pong
   RequestSwap_DebugBuff_ShowTime_Tomem((int32 *)(void *)&gusa_DebugBuf[(DEBUG_INTERNAL_PING_OR_PONG_BUFFER_MAX_SIZE_BYTES>>1)],DEBUG_INTERNAL_PING_OR_PONG_BUFFER_MAX_SIZE_BYTES);

         //copy the remaining bytes to be written to ping buffer
         memcpy(&gpuca_debugBuff[gus_LenDebugBuff],&puc_base_addr[s_avail_length],(us_num_bytes-s_avail_length));

         //increment the bufer index to point the next index to be written
         gus_LenDebugBuff+=(us_num_bytes-s_avail_length);


      return(NO_DEBUG_BUFFER_ERROR);
      }
      else
   {
         gul_DMA_Busy_Error++;
      return(INTERNAL_BUFFER_FULL);
   }
   }
}
#endif // #ifdef DEBUG_TRACES


#ifdef DEBUG_STREAMING
static int32 DSH_IncQueueIndex(int32 l_Index)
{
   if (++l_Index >= DSH_QUEUE_DEPTH)
   {
      return(0);
   }
   else
   {
      return(l_Index);
   }
}


static int32 DSH_DecQueueIndex(int32 l_Index)
{
   if (--l_Index < 0 )
   {
      return(DSH_QUEUE_DEPTH - 1);
   }
   else
   {
      return(l_Index);
   }
}

static int32 DSH_WriteQueue(DSH_Statics_t * pt_Statics, uint16 us_StreamID, uint16 us_BlockSize, FlagT ft_DoCrc, void * p_Data, void (*Func)(P_DSH_Statics, P_DSH_QueueEntry))
{
   uint16 us_Crc16 = 0;
   int32 l_ReturnCode=-1;
   uint32 ul_int_status;
   uint8 *pc_c_Data;

   // as the host controler can't handle debug streams with an odd number of bytes
   // the block size has to be padded with a byte. To restore the data correctly
   // the odd flag in the status word is set in case the block size is odd.
   FlagT ft_IsOdd = (FlagT)(us_BlockSize & 0x01);
   us_BlockSize = (uint16)(((us_BlockSize + 1) >> 1)<<1);
   pc_c_Data = (uint8 *)p_Data;
   // calculate CRC
   if(ft_DoCrc)
   {
      for(uint16 j=0; j < us_BlockSize;j++)
      {
         us_Crc16 = DSH_calcCrc16(us_Crc16,*pc_c_Data++);
      }
   }
   disable1_save((int32 *)(void *)&ul_int_status);

   int32 l_Idx = pt_Statics->c_QueueWrIdx;

   if( pt_Statics->c_QueueFill < DSH_QUEUE_DEPTH)
   {
      DSH_QueueEntry_t * pt_Entry = &pt_Statics->ta_Queue[l_Idx];
      pt_Entry->us_Id        = (uint16)(us_StreamID | DSH_PROTOCOL_VERSION);
      pt_Entry->us_Size      = us_BlockSize;
      pt_Entry->us_Length    = us_BlockSize;
      pt_Entry->us_Status    = (uint16)(pt_Statics->c_StreamNum++ | DSH_FIRST_FRAGMENT);
      pt_Entry->us_Status    |= DSH_STREAM_SOURCE_CPE;

      if(ft_IsOdd)
      {
         pt_Entry->us_Status |= DSH_STREAM_SIZE_IS_ODD;
      }
      pt_Entry->us_Crc       = us_Crc16;
      pt_Entry->ul_SymCount  = gul_DSH_SymCount;
      pt_Entry->p_Data       = p_Data;
      pt_Entry->Func         = Func;

      pt_Statics->c_QueueFill += 1;
      pt_Statics->c_QueueWrIdx = (int8)DSH_IncQueueIndex(l_Idx);

      l_ReturnCode = pt_Statics->c_StreamNum;
   }
   else
   {
      //@todo AH: Consider to send a message which indicates a Buffer Overflow
      static uint16 us_MailboxCode, us_Word0, us_Word1, us_Word2, us_Word3, us_Word4;
      //static AutoMsgPayLoadEntry_t *AutoMsgBufferOverflow;

      us_MailboxCode = AUTONOMOUS_MSG_NOTIFICATION;
      us_Word0 = AUTO_MSG_INF117_HEADER_0;
      us_Word1 = AUTO_MSG_INF117_HEADER_1;
      us_Word2 = AUTO_MSG_INF117_HEADER_2;
      us_Word3 = AUTO_MSG_INF117_HEADER_3;
      us_Word4 = AUTO_MSG_INF117_HEADER_4;

      //SubmitAutoMsg(pt_PayLoadToSend, us_MailboxCode, us_Word0, us_Word1, us_Word2, us_Word3, us_Word4);

      /*
      T_HMH_AutoMsg Msg;
      Msg.MessageId   = HMH_EVT_DBG_BUFFER_OVERFLOW;
      Msg.Length      = 4;
      Msg.Payload[0]  = StreamID;
      Msg.Payload[1]  = BlockSize;
      HMH_SendAutoMsg(&Msg);
      */
      l_Idx = DSH_DecQueueIndex(l_Idx);
      pt_Statics->ta_Queue[l_Idx].us_Status |= DSH_OVERFLOW_INDICATION;
      pt_Statics->ta_Queue[l_Idx].us_Status |= DSH_STREAM_SOURCE_CPE;
      l_ReturnCode = -1;
   }
   restore1_save(ul_int_status);        // Restore interrupts

   // XDSLRTFW-2701 Make sure we leave sufficient space in BGTaskFifo and do not flood it with debug stream tasks to avoid E_CODE_BKGD_FIFO_ADD_EXCEPTION
   // 8 was the BGTaskFifio size before debug streams were implemented, so make sure this space is always reserved
   if (gp_BGTaskFifo->PendingFunctionCount < (NUM_BG_BUFFERS-BUFFER_RESERVE))
   {
      AddFunctionToBkgdFifo((PtrToBkgdFunc)DSH_BgService);
   }
   else
   {
      // we ran out of space, keep track of this event
      gus_Dbg_DSH_skip_BGF_add++;
   }

   return l_ReturnCode;
}


static FlagT DSH_BgDbgStr(DSH_Statics_t * pt_Statics)
{
   uint16 * pus_Data;
   uint16 us_Status, us_Length, us_Chunk;
   uint32 ul_int_status;
   int32 l_Fill, l_Idx;
   DSH_QueueEntry_t * pt_Entry;



   if(NMP_IsDebugMsgPending() != 0)
   {
      // HMH busy? Quit and retry later...
      return TRUE;
   }

   // The call of NMP_SendDebugMsg will add a function to BG stack. Therefore, check if space on BG stack is really available
   // Make sure we leave sufficient space in BGTaskFifo and do not flood it with debug stream tasks
   // 8 was the BGTaskFifio size before debug streams were implemented, so make sure this space is always reserved
   if (gp_BGTaskFifo->PendingFunctionCount >= (NUM_BG_BUFFERS-5))
   {
      // we ran out of space, keep track of this event
      gus_Dbg_DSH_skip_BGF_add++;
      return TRUE;
   }

   // copy current queue entry to local variables
   l_Idx = pt_Statics->c_QueueRdIdx;

   if(pt_Statics->ft_CleanUpLastStream)
   {
      pt_Statics->ft_CleanUpLastStream = FALSE;

      pt_Entry = &pt_Statics->ta_Queue[l_Idx];
      if(pt_Entry->Func != 0)
      {
         pt_Entry->Func(pt_Statics, pt_Entry);
      }


      l_Idx = DSH_IncQueueIndex(l_Idx);
      pt_Statics->c_QueueRdIdx = (int8)l_Idx;

      // we need to block interrupt to prevent race condition
      disable1_save((int32 *)(void *)&ul_int_status);;
      l_Fill = pt_Statics->c_QueueFill;
      l_Fill--;
      pt_Statics->c_QueueFill = (int8)l_Fill;
      restore1_save(ul_int_status);         // Restore interrupts
   }
   else
   {
      disable1_save((int32 *)(void *)&ul_int_status);;
      l_Fill = pt_Statics->c_QueueFill;
      restore1_save(ul_int_status);
   }

   if(l_Fill == 0)
   {
      // no pending stream? Quit.
      return FALSE;
   }

   pt_Entry = &pt_Statics->ta_Queue[l_Idx];

   us_Status  = pt_Entry->us_Status;
   us_Length  = pt_Entry->us_Length;
   pus_Data   = pt_Entry->p_Data;
   us_Chunk   = us_Length;

   // maximum chunk size is MID_MAX_PAYLOAD_SIZE - sizeof(stream id) - sizeof(stream status) - sizeof(sym. count) - sizeof(crc16)
   // Note: AndreasH 20151112: On Avinax the Chunk size was MID_MAX_PAYLOAD_SIZE -10 (see calculation above)
   // on VR9 we need -16 instead of -10 to prevent a Mailbox overflow for large messages. I don't know why, maybe due a larger message headers on VR9
   if(us_Length > (MAX_MAILBOX_PAYLOAD_LENGTH - 16))
   {
      us_Chunk = MAX_MAILBOX_PAYLOAD_LENGTH - 16;
   }
   else
   {
      us_Status |= DSH_LAST_FRAGMENT;
   }

   NMP_SendDebugMsg(pt_Entry->us_Id, pt_Entry->ul_SymCount, pt_Entry->us_Crc, us_Status, us_Chunk, pus_Data);

   if(us_Status & DSH_LAST_FRAGMENT)
   {
      // wait for last chunkt to be sent befor the entry is removed from the queue
      pt_Statics->ft_CleanUpLastStream = TRUE;
      pt_Entry->us_Status = us_Status;
   }
   else
   {
      // prepare next chunk
      us_Length -= us_Chunk;
      pus_Data  += us_Chunk>>1; // watch up!! 16 bit pointer!
      us_Status &= ~DSH_MASK_FRAGMENT;

      pt_Entry->us_Length = us_Length;
      pt_Entry->us_Status = us_Status;
      pt_Entry->p_Data  = pus_Data;
   }

   return TRUE;
}


static void  DSH_ClearBuffer(DSH_Statics_t * pt_Statics, DSH_QueueEntry_t * pEntry)
{
   uint32 ul_int_status;
   disable1_save((int32 *)(void *)&ul_int_status);
   if(pEntry->us_Status & DSH_STREAM_SIZE_IS_ODD)
   {
      pEntry->us_Size--;
   }
   pt_Statics->s_BufferFill -= pEntry->us_Size;
   restore1_save(ul_int_status);
}


static FlagT DSH_BgEvtBuf(DSH_Statics_t * pt_Statics)
{

   int32 l_WrIdx = pt_Statics->s_BufferWrIdx;
   int32 l_RdIdx = pt_Statics->s_BufferRdIdx;

   if(l_WrIdx < l_RdIdx)
   {
       uint16 BlockSize       = (uint16)(DSH_BUFFER_SIZE - l_RdIdx);
       DSH_WriteQueue(pt_Statics, DSH_STATE_TRAIL, BlockSize, FALSE, &pt_Statics->uca_Buffer[l_RdIdx], DSH_ClearBuffer);
       l_RdIdx                  = 0;
       pt_Statics->s_BufferRdIdx = 0;
   }

   uint16 BlockSize = (uint16)(l_WrIdx - l_RdIdx);
   if(pt_Statics->ft_BufferFlush || (BlockSize >= DSH_BUFFER_FILL_THRESHOLD))
   {
       pt_Statics->ft_BufferFlush = FALSE;
       if(BlockSize)
       {
           DSH_WriteQueue(pt_Statics, DSH_STATE_TRAIL, BlockSize, FALSE, &pt_Statics->uca_Buffer[l_RdIdx], DSH_ClearBuffer);
           pt_Statics->s_BufferRdIdx = (int16)(l_RdIdx + BlockSize);
       }
       return FALSE;
   }
   return FALSE;
}

void DSH_FgService( void )
{
   // Get local copy of the global symbol count from persistent memory
   // we need a counter which does not get reset with a new link start for tracee, and we want a counter which
   // gets incremented every symbol (including unused frame and in idle state)
   uint32 ul_SymCnt = gul_DSH_SymCount;

   //if Mailbox is available, try to stream version info
   if(TxMailBoxPending() == FALSE)
   {
      if(guc_DSH_VersionInfoStreamed < 1)
      {
         int32 StreamVersionInfoResult;
         StreamVersionInfoResult = DSH_SendStream(DSH_VERSION_INFO,10,&gusa_FW_version_number[0]); //XDSLRTFW-3364 (Start_End)
         if (StreamVersionInfoResult>0)
         {
            guc_DSH_VersionInfoStreamed++;
         }
      }
   }


   // send event to signal counter wrap around
   if(ul_SymCnt == 0xFFFFFFFF)
   {
      //DEBUG: Avoid new record being created by ul_SymCnt equal to 0. We see this being triggered unexcpectedly. Hence use a new Event for debugging
      //DSH_SendEvent(DSH_EVT_SYM_CNT_ZERO,0,0);
      DSH_SendEvent(DSH_EVT_CPE_SYM_ZERO_DEBUG,0,0);
      // queue events now, because all events/streams later than symCnt == 0
      // have to be received by Tracee after this event
      gt_DshStatics.ft_BufferFlush = TRUE;

      // XDSLRTFW-2701 Make sure we leave sufficient space in BGTaskFifo and do not flood it with debug stream tasks to avoid E_CODE_BKGD_FIFO_ADD_EXCEPTION
      // 8 was the BGTaskFifio size before debug streams were implemented, so make sure this space is always reserved
      if (gp_BGTaskFifo->PendingFunctionCount < (NUM_BG_BUFFERS-BUFFER_RESERVE))
      {
         AddFunctionToBkgdFifo((PtrToBkgdFunc)DSH_BgService);  //we need to call DSH_BgService (and can't call directly DSH_BgEvtBuf) because we can't pass arguments to BGfunctions
      }
      else
      {
         // we ran out of space, keep track of this event
         gus_Dbg_DSH_skip_BGF_add++;
      }
   }

   //Buffer Flush every 4096 symbols (so around every 1 second) to avoid events stay in buffer 'forever' in the case buffer threshold is not hit
   //(e.g. no new debug events was added to fill the buffer)
   if((ul_SymCnt & 0x00000FFF) == 0)
   {
      gt_DshStatics.ft_BufferFlush = TRUE;
      // XDSLRTFW-2701 Make sure we leave sufficient space in BGTaskFifo and do not flood it with debug stream tasks to avoid E_CODE_BKGD_FIFO_ADD_EXCEPTION
      // 8 was the BGTaskFifio size before debug streams were implemented, so make sure this space is always reserved
      if (gp_BGTaskFifo->PendingFunctionCount < (NUM_BG_BUFFERS-BUFFER_RESERVE))
      {
         AddFunctionToBkgdFifo((PtrToBkgdFunc)DSH_BgService);  //we need to call DSH_BgService (and can't call directly DSH_BgEvtBuf) because we can't pass arguments to BGfunctions
      }
      else
      {
         // we ran out of space, keep track of this event
         gus_Dbg_DSH_skip_BGF_add++;
      }
   }

   // Update the persistent symbol count (and also the local copy)
   ul_SymCnt++;
   gul_DSH_SymCount = ul_SymCnt;

   // XDSLRTFW-2831
   // stream showtime SNR every ~2sec
   if(STATArray[STAT_MacroState] == STAT_ShowTimeTCSyncState)
   {
      if((gul_DSH_SymCount & 0x00001FFF) == 0)
      {
         DSH_SendStream(DSH_SHOWTIME_SNR,gs_RxNumTones*sizeof(int16),(void*)gpsa_MeasuredSnrBuf);
      }
      else if ((gul_DSH_SymCount & 0x00001FFF) == 0x1000)
      {
         DSH_SendStream(DSH_SHOWTIME_BAT,gs_RxNumTones*sizeof(uint8),(void*)guca_RxBat);
      }
   }
}

// DSH background handler, called by event system
void DSH_BgService( void )
{
   DSH_Statics_t *pt_Statics = &gt_DshStatics;

   FlagT ft_Res = FALSE;
   ft_Res |= DSH_BgEvtBuf(pt_Statics);
   ft_Res |= DSH_BgDbgStr(pt_Statics);

   if (ft_Res)
   {
      //In case Background tasks want to be executed again (e.g. because the mailbox was busy) queue BG Task in again
      gus_Dbg_RequeueCnt++;
      gus_RequeueBGFunction |= DSH_REQUEUE_DSH_BG_SERVICE;
   }
}

int32 DSH_SendEvent(uint8 uc_EventId, uint8 uc_BlockSize, void * p_Data)
{

   if(((gt_debugStreamControl.Parameter0 & CMV_INFO115_CONTROL_MASK) == 0)
   ||(gt_debugStreamConfigure.Mask[0] & (1 << DSH_STATE_TRAIL)) == 0)
   {
      //debug streams or state trail stream have been deactivated, return
      return 0;
   }


   uint32 ul_int_status;
   DSH_Statics_t * pt_Statics = &gt_DshStatics;

   int32 l_Result      = 0;
   uint8 * puc_FillGap  = 0;
   uint8 * puc_FillData = 0;

   disable1_save((int32 *)(void *)&ul_int_status);

   int32 l_GapSize = DSH_BUFFER_SIZE - pt_Statics->s_BufferWrIdx;
   if(l_GapSize > (int32)(uc_BlockSize + DSH_EVENT_HEADER_SIZE))
   {
      l_GapSize = 0;
   }

   int32 l_WrIdx = pt_Statics->s_BufferWrIdx;
   int32 l_Fill  = pt_Statics->s_BufferFill;
   if((int32)(l_Fill + uc_BlockSize + DSH_EVENT_HEADER_SIZE + l_GapSize) >= DSH_BUFFER_SIZE)
   {
      pt_Statics->ft_Overflow = TRUE;
      l_Result = -1;
      goto WriteToBuffer_End;
   }

   if(l_GapSize)
   {
      puc_FillGap = &pt_Statics->uca_Buffer[l_WrIdx];
      l_WrIdx = 0;
   }

   puc_FillData = &pt_Statics->uca_Buffer[l_WrIdx];

   l_WrIdx += uc_BlockSize + DSH_EVENT_HEADER_SIZE;
   l_Fill  += uc_BlockSize + DSH_EVENT_HEADER_SIZE + l_GapSize;

   if(l_WrIdx >= DSH_BUFFER_SIZE)
   {
      l_WrIdx = 0;
   }

   pt_Statics->s_BufferWrIdx = (int16)l_WrIdx;
   pt_Statics->s_BufferFill  = (int16)l_Fill;

WriteToBuffer_End:
   restore1_save(ul_int_status);
   if(puc_FillGap)
   {
      memset(puc_FillGap, -1, l_GapSize);
   }

   if(puc_FillData)
   {
      if (pt_Statics->ft_Overflow)
      {
         pt_Statics->ft_Overflow = FALSE;
         uc_EventId |= 0x80;
      }

      uint32 ul_SymCount = gul_DSH_SymCount;
      *puc_FillData++ = uc_EventId;
      *puc_FillData++ = (uint8)(ul_SymCount & 0xFF);
      *puc_FillData++ = (uint8)((ul_SymCount >>= 8) & 0xFF);
      *puc_FillData++ = (uint8)((ul_SymCount >>= 8) & 0xFF);
      *puc_FillData++ = (uint8)((ul_SymCount >>= 8) & 0xFF);
      *puc_FillData++ = uc_BlockSize;
      memcpy(puc_FillData, p_Data, uc_BlockSize);
   }


   // XDSLRTFW-2701 Make sure we leave sufficient space in BGTaskFifo and do not flood it with debug stream tasks to avoid E_CODE_BKGD_FIFO_ADD_EXCEPTION
   // 8 was the BGTaskFifio size before debug streams were implemented, so make sure this space is always reserved
   if (gp_BGTaskFifo->PendingFunctionCount < (NUM_BG_BUFFERS-BUFFER_RESERVE))
   {
      AddFunctionToBkgdFifo((PtrToBkgdFunc)DSH_BgService);
   }
   else
   {
      // we ran out of space, keep track of this event
      gus_Dbg_DSH_skip_BGF_add++;
   }
   return l_Result;
}

int32 DSH_SendStream(uint16 us_StreamID, uint16 us_BlockSize, void * p_Data)
{
   DSH_Statics_t * pt_Statics = &gt_DshStatics;
   uint16 us_MaskIdx, us_MaskBit;

   if((gt_debugStreamControl.Parameter0 & CMV_INFO115_CONTROL_MASK) == 0)
   {
      // return if debug streams are disabled globally
      return 0;
   }

   // compute the offset into Debug Stream Config Mask
   us_MaskIdx = (uint16)(us_StreamID / (sizeof(DSH_MaskEntry_t) << 3));
   us_MaskBit = (uint16)(us_StreamID % (sizeof(DSH_MaskEntry_t) << 3));

   // check if the ID is within the allowed limit and if the particular debug stream is configured via message
   // OR if the ID is equal to DSH_VERSION_INFO, which is always sent, regardless of the configuration
   // OR if the ID is equal the special IPTC Debug Stream forwarding us_Id
   if (((us_StreamID < DSH_CTRL_MAX_NUM) && ((gt_debugStreamConfigure.Mask[us_MaskIdx]  & (1 << us_MaskBit)) != 0)) || (us_StreamID == DSH_VERSION_INFO))
   {
      if(us_StreamID == DSH_FSM_STATE)
      {
         //As the FSM state is used to put all other events into a timely context it
         //is important to make sure all that all events before the change have been sent.
         pt_Statics->ft_BufferFlush = TRUE;
         DSH_BgEvtBuf(pt_Statics);
      }
      //Note: Disabled CRC calculation for debug streams due to MIPS overrun on large Debug Stream messages
      //return DSH_WriteQueue(pt_Statics, us_StreamID, us_BlockSize, TRUE, p_Data, 0);
      return DSH_WriteQueue(pt_Statics, us_StreamID, us_BlockSize, FALSE, p_Data, 0);
   }

   return 0;
}

/**
    calcCrc16 does a byte(octet)-wise CRC calculation for
    an initial CRC and the generator polynomial of 0x8408.

    @note calcCrc16 has been optimized for maximum speed, therefore
    it uses a hard-coded generator polynom. There is no looping involved
    as in other implementations where the generator polynom is selectable
    as a run-time parameter.
    An even faster code can be made available when a 256byte
    look-up table is used.

    @param      c octet to be processed
    @param      crc current CRC16

    @return     updated CRC16 for the input octet
*/
uint16 DSH_calcCrc16(uint16 crc, uint8 c)
{
   // CCITT 16 bit (X^16 + X^12 + X^5 + x^0).
   crc ^= c;
   crc ^= ((crc & 0x000f) << 4);
   crc  = (uint16) (((crc >> 8) & 0x00ff) | (crc << 8));
   crc ^= (crc >> 12);
   crc ^= ((crc & 0xff00) >> 5);
   return (crc);
}

#endif
