/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2002 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*************************************************************************
* Aware DMT Technology. Proprietary and Confidential.
*
* ADDRESS:         40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
* TELEPHONE:       781.276.4000
* FAX:             781.276.4001
* WEB:             http://www.aware.com
*
* FILENAME:        acc40.c
*
* DESCRIPTION:     functions for the 40-bit accumulator.
*
**************************************************************************/
#include "typedef.h"
#include "round_acc48.h"
#include "norm_acc48.h"
#include "acc40.h"

/*****************************************************************************
;  Subroutine Name: round_acc40
;
;   Rounds a 40-bit accumulator A at "bit_position" and then shifts the result
;   right by "bit_position".  The accumulator will always be rounded away from 0;
;   e.g. if A=3 and bit_position=1, round_acc40 will return 2, and if
;   A=-3 and bit_position=1, round_acc48 will return -2.
;
;   The final result must fit within a 32-bit integer.  If the final result
;   cannot fit into a 32-bit integer a saturated result is returned; i.e.
;   if the final result is positive and greater than MAX_32 (the maximum 32-bit integer)
;   the function will return MAX_32, and if the final result is negative and
;   less than MIN_32 (the minimum 32-bit integer), the function will return MIN_32.
;
;  Prototype:
;       int32 round_acc40(int32 l_AccH, int32 l_AccL, int16 s_bit_position)
;
;  Input Arguments:
;     l_AccH         - The HI part of the 40 bits accum.
;     l_AccL         - The LO part of the 40 bits accum.
;       s_bit_position  - The bit position where rounding bit 1 is added
;                         if A>=0 and is subtracted if A<0.
;
;  Output Arguments:
;
;  Return:
;     Returns the rounded accum.
;
;  Global Variables:
;     none
;
*******************************************************************************/
int32 round_acc40(int32 l_AccH, int32 l_AccL, int16 s_bit_position) {
   // the 40-bit accumulator has the same representation as the 48-bit accumulator
   return(round_acc48(l_AccH, l_AccL, s_bit_position));
}

/*****************************************************************************
;  Subroutine Name: accum40
;
;  Adds a 32-bit integer to a 40-bit accumulator.  If the resulting accumulation
;   cannot fit within 40 bits, the accumulator will saturate.
;
;  Prototype:
;       void accum40(int32 *pl_Acc40H, int32 *pl_Acc40L, int32 l_Acc);
;
;  Input Arguments:
;     pl_Acc40H      - The HI part of the 40 bits accum.
;     pl_Acc40L      - The LO part of the 40 bits accum.
;       l_Acc           - the 32-bit integer to be added to the accumulator
;
;  Output Arguments:
;     pl_Acc40H      - The HI part of the 40 bits accum.
;     pl_Acc40L      - The LO part of the 40 bits accum.
;
;  Return:
;
;  Global Variables:
;     none
;
*******************************************************************************/
void accum40(int32 *pl_Acc40H, int32 *pl_Acc40L, int32 l_Acc) {
   // use the 48-bit accumulator as underlying representation
   *pl_Acc40L += l_Acc & 0xFFFF;
   *pl_Acc40H += l_Acc >> 16;

   // check for 40-bit saturation
   norm_acc48(pl_Acc40H, pl_Acc40L);
   if( *pl_Acc40H >= ((int32) 1 << ((40 - 1) - 16)) ) {  // acc40 >= (1<<(40-1)); positive saturation
      *pl_Acc40H = ((int32) 1 << ((40 - 1) - 16)) - 1;
      *pl_Acc40L = 0x0000FFFF;
   }
   else if( *pl_Acc40H < ((int32) (-1) << ((40-1) - 16)) ) {   // acc40 < -(1<<(40-1)); negative saturation
      *pl_Acc40H = ((int32) (-1) << ((40-1) - 16));
      *pl_Acc40L = 0;
   }
}

/*****************************************************************************
;  Subroutine Name: addAcc40ToAcc48
;
;  Adds a 40-bit integer to a 48-bit accumulator.
;
;  Prototype:
;       void addAcc40ToAcc48(int32 *pl_Acc48H, int32 *pl_Acc48L, int32 l_Acc40H, int32 l_Acc40L);
;
;  Input Arguments:
;     pl_Acc48H      - The HI part of the 48 bits accum.
;     pl_Acc48L      - The LO part of the 48 bits accum.
;       l_Acc40H        - The HI part of the 40 bits accum.
;       l_Acc40L        - The LO part of the 40 bits accum.
;
;  Output Arguments:
;     pl_Acc48H      - The HI part of the 48 bits accum.
;     pl_Acc48L      - The LO part of the 48 bits accum.
;
;  Return:
;
;  Global Variables:
;     none
;
*******************************************************************************/
void addAcc40ToAcc48(int32 *pl_Acc48H, int32 *pl_Acc48L, int32 l_Acc40H, int32 l_Acc40L) {
   *pl_Acc48H = (*pl_Acc48H + l_Acc40H) + (*pl_Acc48L >> 16) + (l_Acc40L >> 16);
   *pl_Acc48L = (*pl_Acc48L & 0xFFFF) + (l_Acc40L & 0xFFFF);
}
