/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2002 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*************************************************************************
* Aware DMT Technology. Proprietary and Confidential.
*
* ADDRESS:         40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
* TELEPHONE:       781.276.4000
* FAX:             781.276.4001
* WEB:             http://www.aware.com
*
* FILENAME:        cmplxvecmult.c
*
* DESCRIPTION:     Function for computing product of two complex arrays
*
**************************************************************************/

#include "typedef.h"

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : CmplxVectorMultiply
 *
 *  Description:
 *    This routine multiplies and shifts, two complex arrays to form a single
 *    complex array, which is of the same size as the input arrays.
 *
 * Memory Allocation:
 *    The three arrays in X memory can be at the same location but otherwise
 *    they must not overlap. The two arrays in Y memory can be at the same
 *    location but otherwise they must not overlap. The sizes of each of the
 *    cmplx input and output arrays is "2*s_length" words. The sizes of each
 *    of the cmplx scratch pads is "4*s_length" words.
 *
 * Algorithm:
 *             Z = X * Y
 *          (Zr + j*Zi) = (Xr + j*Xi) * (Yr + j*Yi)
 *    To efficiently implement complex multiplies using the dual MAC unit,
 *    we store the complex data , X = Xr + j*Xi, in X memory as:
 *       ------------
 *       | Xi | Xr  |   address n in 32-bit XY addressing mode
 *       | Xr | -Xi |   address n+1 in 32-bit XY addressing mode
 *       ------------
 *
 *    and Y = Yr + j*Yi, in Y memory as:
 *       -----------
 *       | Yr | Yr |    address m in 32-bit XY addressing mode
 *       | Yi | Yi |    address m+1 in 32-bit XY addressing mode
 *       -----------
 *
 *    To compute the complex multiply, we simply call two MAC instructions and
 *    copy the result in the accumulator to a register %r:
 *       mul 0, %x0_u, %y0_u     (assume that %x0_u=n and %y0_u = m)
 *       mac %r, %x0_u, %y0_u
 *    The result in %r is the complex product P:
 *       | Pi | Pr |
 *
 *  Prototype:
 *    void CmplxVectorMultiply(int16 *psa_destbuf, int16 *psa_srcbuf1, int16 *psa_srcbuf2, int16 s_resultShift, int16 s_length, int16 *psa_CmplxMultScratchPadX, int16 *psa_CmplxMultScratchPadY)
 *
 *  Input Arguments:
 *    *psa_srcbuf1         - Pointer to complex input vector 1 in X memory
 *    *psa_srcbuf2         - Pointer to complex input vector 2 in Y memory
 *    ft_conjugate_src1    - 1 Implies Conjugate complex input vector 1
 *    s_resultShift        - Scalar shift to be applied to result
 *    s_length          - Number of complex elements in array
 *    *psa_CmplxMultScratchPadX  - Scratch Pad in X memory of size 2*sizeof(input_vector1)
 *    *psa_CmplxMultScratchPadY  - Scratch Pad in Y memory of size 2*sizeof(input_vector2)
 *
 *  Output Arguments:
 *    *psa_destbuf         - Pointer to complex output vector in X memory
 *
 *  Return:
 *
 *  Notes:
 *    -  cmplx_vector_D = (cmplx_vector_A * cmplx_vector_B) >> shift if ft_conjugate_src1 == FALSE
 *    -  cmplx_vector_D = (conj(cmplx_vector_A) * cmplx_vector_B) >> shift if ft_conjugate_src1 == TRUE
 *    - shift must be within the following range
 *       0 <= x <= 16
 *    - Elements of all the other vectors are assumed to be signed words
 *       ie 0x8000 <= x <=0x7fff
 *    - No saturation checks are performed
 *------------------------------------------------------------------------
 *^^^
 */
void CmplxVectorMultiply(int16 *psa_destbuf, int16 *psa_srcbuf1, int16 *psa_srcbuf2, FlagT ft_conjugate_src1, int16 s_resultShift, int16 s_length, int16 *psa_CmplxMultScratchPadX, int16 *psa_CmplxMultScratchPadY)
//void CmplxVectorMultiply(void)
{
   int16 i;
   int16 s_x_r, s_x_i, s_y_r, s_y_i;
   int32 l_err_corr_real, l_err_corr_imag;

   for (i= 0; i < s_length; i++)
   {
      s_x_r = *psa_srcbuf1++;
      s_x_i = *psa_srcbuf1++;
      s_y_r = *psa_srcbuf2++;
      s_y_i = *psa_srcbuf2++;

      if(ft_conjugate_src1) s_x_i = -s_x_i;

   // compute real part [x_r*y_r - x_i*y_i]
      l_err_corr_real = (int32)s_x_r * (int32)s_y_r;
      l_err_corr_real -= ((int32)s_x_i * (int32)s_y_i);

   // compute imag part [x_i*y_r + x_r*y_i]
      l_err_corr_imag = (int32)s_x_i * (int32)s_y_r;
      l_err_corr_imag += ((int32)s_x_r * (int32)s_y_i);

   // Right shift to position most significant 16 bits in low order word.
      *psa_destbuf++ = (int16)(l_err_corr_real >> 13);
      *psa_destbuf++ = (int16)(l_err_corr_imag >> 13);
   }
}
