/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-1998 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;  Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
;  Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;  File Name: med2rev.c
;
;  This file contains the noiseacc function used in SNR calculation
;
;***************************************************************************/

#include "common.h"
#include "medley.h"
#include "gdata.h"
#include "GenMedly.h"
#include "rcmedly2.h"
#include "const.h"
#include "cmv.h"
#include "gdata_bis.h"

int16 CalculatePhaseDif(int16 s_ar, int16 s_ai, int16 s_br, int16 sbi);

CONSTANT int16 PhaseDifTable[4][4] = { {   0,  90, 270, 180},
                           { 270,   0, 180,  90},
                           {  90, 180,   0, 270},
                           { 180, 270,  90,   0}};


/********************************************************************************************
;  Subroutine Name: BgMedleyToReverbAccum
;
;  Description:
;     This routine first applies phase shift to a MEDLEY input to
;     make it a REVERB signal, then accumulates it.
;
;     This routine is called during MEDLEY FDQ calculation.
;     The offset parameter is used to control the PN sequence shift during MEDLEY.
;
;
;  Prototype:
;     void BgMedleyToReverbAccum(void)
;
;  Input Arguments:
;     gsa_recv_tones -- pointer to the received DMT tones
;     gsa_ref_tones  -- pointer to the reference DMT tones
;     gs_MedleyOffset      -- offset used to generate Medley signal
;     gs_RxFirstChannel   -- first channel
;     gs_RxLastChannel    -- last channel
;
;  Output Arguments:
;     gla_NoisePower -- pointer to accumulators.
;                    Note that this pointer should point to the address
;                    for the first noise value.  Noise values will get written to
;                 gla_NoisePower[0] through gla_NoisePower[gs_RxLastChannel-gs_RxFirstChannel],
;                 rather than gla_NoisePower[gs_RxFirstChannel] through gla_NoisePower[gs_RxLastChannel]
;
;  Return Value:
;     none
;
;  Global Variables:
;     none
;
;****************************************************************************/
void BgMedleyToReverbAccum(void)
{
   int16 i;
   int16 *psa_rec, *psa_medley, *psa_reverb;
   int32 *pla_out;
   int16 sa_snr_ref_tones[RX_NUM_TONES*2];
   int16 s_Rr, s_Ri, s_Mr, s_Mi, s_Cr, s_Ci;
   int16 s_fft_length = 2 * gs_RxNumTones;

   /* ================================================== */
   /* Set up address pointers                   */
   /* ================================================== */

   //  if(gft_ModemType == G_DMT)
//  {

   if (gft_G992_5_Medley14thOrderPRBS == TRUE)
   {
      /* For 14th order PRBS medley sequence in g992_5, the Reverb signal can not be used as Medley Reference for any frames */
      GenerateReferenceSignal(gsa_ref_tones, sa_snr_ref_tones, gs_MedleyOffset, gs_MedleyModulo, s_fft_length);
      psa_medley = sa_snr_ref_tones + 2*gs_RxFirstChannel; /* reference medley signal */
   }
   else
   {
      /* if gs_MedleyOffset > 0 then use it to index into reference sequence  */
      if (gs_MedleyOffset > 0)
      {
         GenerateReferenceSignal(gsa_ref_tones, sa_snr_ref_tones, gs_MedleyOffset, gs_MedleyModulo, s_fft_length);
         psa_medley = sa_snr_ref_tones + 2*gs_RxFirstChannel; /* reference medley signal */
      }
      else
      {
         psa_medley = gsa_ref_tones + 2*gs_RxFirstChannel; /* reference medley signal */
      }
   }

   psa_reverb = gsa_ref_tones + 2*gs_RxFirstChannel;  /* reference reverb signal */
   psa_rec = gsa_recv_tones + 2*gs_RxFirstChannel; /* received tones */
   pla_out = gla_NoisePower;                    /* accumulator */

   /* ================================================== */
   /* Compute noise power and add it to the total power  */
   /* ================================================== */

   for(i = gs_RxFirstChannel; i <= gs_RxLastChannel; i++)
   {
      s_Rr = (*psa_reverb++);    /* real part of reverb */
      s_Ri = (*psa_reverb++);    /* imag part of reverb */
      s_Mr = (*psa_medley++);    /* real part of medley */
      s_Mi = (*psa_medley++);    /* imag part of medley */
      s_Cr = (*psa_rec++);    /* real part of received symbol */
      s_Ci = (*psa_rec++);    /* imag part of received symbol */

      switch(CalculatePhaseDif(s_Rr,s_Ri,s_Mr,s_Mi)) {

      case 0:
         (*pla_out++) += (int32)  s_Cr;
         (*pla_out++) += (int32)  s_Ci;
         break;

      case 90:
         (*pla_out++) += (int32) -s_Ci;
         (*pla_out++) += (int32)  s_Cr;
         break;

      case 180:
         (*pla_out++) += (int32) -s_Cr;
         (*pla_out++) += (int32) -s_Ci;
         break;

      case 270:
         (*pla_out++) += (int32)  s_Ci;
         (*pla_out++) += (int32) -s_Cr;
         break;

      }
   }

   /* ================================================== */
   /* Accumulation is done. */
   /* ================================================== */
   guc_AccumState = TRAINING_DONE;

   /* Turn on transfer of Rx Tone buffer */
   gft_CopyRxBuffer = TRUE;
}

/********************************************************************************************
;  Subroutine Name: CalculatePhaseDif
;
;  Description:
;     This routine calculates the phase difference, i.e.
;
;     phase(A) - phase(B)
;
;     where A = s_ar + i*s_ai, and B = s_br + s_bi.
;
;     Note that phase differences are allowed to be 0, 90, 180, or 270 degrees and
;     possible phase difference combinations are pre-stored in PhaseDifTable
;     as shown below.
;
;             (--) (-+) (+-) (++)
;         A\B| 00 | 01 | 10 | 11 |                    ^
;        ----+----+----+----+----+              (-+)  |  (++)
;    (--) 00 |   0|  90| 270| 180|                1   |   3
;        ----+----+----+----+----+                    |
;    (-+) 01 | 270|   0| 180|  90|          <---------+--------->
;        ----+----+----+----+----+                    |
;    (+-) 10 |  90| 180|   0| 270|                0   |   2
;        ----+----+----+----+----+              (--)  |  (+-)
;    (++) 11 | 180| 270|  90|   0|                    v
;        ----+----+----+----+----+
;
;  Prototype:
;     int16 CalculatePhaseDif(int16 s_ar, int16 s_ai, int16 s_br, int16 s_bi)
;
/********************************************************************************************/
#define IS_POSITIVE(x) ((x>=0) ? (1) : (0))

int16 CalculatePhaseDif(int16 s_ar, int16 s_ai, int16 s_br, int16 s_bi)
{

   return (PhaseDifTable[2*IS_POSITIVE(s_ar)+IS_POSITIVE(s_ai)] [2*IS_POSITIVE(s_br)+IS_POSITIVE(s_bi)]);

}
