/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2002 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;  Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
;  Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;  File Name: noiseacc.c
;
;  This file contains the noiseacc function used in SNR calculation
;
;***************************************************************************/
#include "common.h"
#include "medley.h"
#include "GenMedly.h"
#include "noiseacc.h"
#include "rcmedly2.h"
#include "gdata.h"
#include "rt_state_bis.h"
#include "gdata_bis.h"

/********************************************************************************************
;  Subroutine Name: NoiseAcc
;
;  Description:
;     This routine performs the following operations:
;
;       For each tone i,
;     1) Compute the noise power
;        pow[i] = (0.5*(ref_tone[i*2]-recv_tone[i*2]))^2
;                    + (0.5*(ref_tone[i*2+1]-recv_tone[i*2+1]))^2
;
;     2) Add pow[i] to the accumulated noise power array.
;
;       The scaling by 0.5 before the squaring is needed to avoid overflowing
;       the 32 bits variable that holds the square. Thus the final
;       accumulated power is the real power dividied by 4.
;
;     The code uses a 48-bit accumulator implemented in 2 32-bit counters.  The details
;     of the implementation of the 48-bit accumulator are not important so long as
;     the RoundNoiseAccum routine can correctly round and shift the 48-bit accumulator
;     into a single 32-bit counter.
;
;     This routine is called both during REVERB and MEDLEY SNR calculations.
;     The offset parameter is used to control the PN sequence shift during
;     MEDLEY SNR calculations.
;
;
;  Prototype:
;     void NoiseAcc(int32 *pla_NoisePower, int16 s_doffset, int16 *psa_recv_tones, int16 s_offset, int16 *psa_ref_tones,
;              int16 s_first_chan, int16 s_last_chan);
;
;  Input Arguments:
;     psa_recv_tones -- pointer to the received DMT tones
;     psa_ref_tones  -- pointer to the reference DMT tones
;     s_offset    -- offset used to generate Medley signal
;     s_first_chan   -- first channel
;     s_last_chan    -- last channel
;
;  Output Arguments:
;     pla_NoisePower -- pointer to accumulators for storing noise power.
;     s_doffset      -- offset for pla_Noisepower to point to the first noise value.
;                 Noise values will get written to pla_NoisePower[s_doffset] through
;                 pla_NoisePower[s_doffset+s_last_chan-s_first_chan], rather than
;                 pla_NoisePower[s_first_chan] through pla_NoisePower[s_last_chan]
;
;  Return Value:
;     none
;
;  Global Variables:
;     none
;
;****************************************************************************/
void NoiseAcc(int32 *pla_NoisePower, int16 s_doffset, int16 *psa_recv_tones, int16 s_offset,
            int16 s_modulo, int16 s_fft_length, int16 *psa_ref_tones,int16 s_first_chan, int16 s_last_chan)
{
   int16 i;
   int32 l_Acc, l_Acc1;
   int16 *psa_rec, *psa_ref;
   int32 *pla_out;
   int16 sa_snr_ref_tones[RX_NUM_TONES*2];


   /* ================================================== */
   /* Set up address pointers                   */
   /* ================================================== */

   pla_NoisePower += s_doffset;

   /* if s_offset > 0 then use it to index into reference sequence   */

// if((gft_ModemType == G_DMT_BIS) && (gs_RxState == R_C_MEDLEY_RX_BIS))  /* For BIS, the Reverb signal can not be used as Medley Reference for any frames */
// {
//    GenerateReferenceSignal(psa_ref_tones, sa_snr_ref_tones, s_offset, s_modulo, s_fft_length);
// psa_ref = sa_snr_ref_tones+2*s_first_chan;
// }
// else
// {

   if ((gft_G992_5_Medley14thOrderPRBS == TRUE) && (gs_RxState == R_C_MEDLEY_RX_BIS))
   {
      /* For 14th order PRBS medley sequence in g992_5, the Reverb signal can not be used as Medley Reference for any frames */
      GenerateReferenceSignal(psa_ref_tones, sa_snr_ref_tones, s_offset, s_modulo, s_fft_length);
      psa_ref = sa_snr_ref_tones+2*s_first_chan;
   }
   else
   {
      if (s_offset > 0)
      {
         GenerateReferenceSignal(psa_ref_tones, sa_snr_ref_tones, s_offset, s_modulo, s_fft_length);
         psa_ref = sa_snr_ref_tones+2*s_first_chan;
      }
      else
      {
         psa_ref = psa_ref_tones+2*s_first_chan;
      }
   }

// }


   psa_rec = psa_recv_tones+2*s_first_chan;
   pla_out = pla_NoisePower;

   /* ================================================== */
   /* Compute noise power and add it to the total power  */
   /* ================================================== */
   for(i = s_first_chan; i <= s_last_chan; i++) {

      /* Compute real_error^2 */
      l_Acc = (int32)(*psa_ref++) - (int32)(*psa_rec++);
      l_Acc >>= 1;
      l_Acc = l_Acc * l_Acc;

      /* Compute imag_error^2 */
      l_Acc1 = (int32)(*psa_ref++) - (int32)(*psa_rec++);
      l_Acc1 >>= 1;
      l_Acc1 = l_Acc1 * l_Acc1;

      /* Add errors to accumulator  */
      mAddInt32ToNoiseAccumulator(l_Acc, pla_out, pla_out+1);
      mAddInt32ToNoiseAccumulator(l_Acc1, pla_out, pla_out+1);
      pla_out += 2;
   }
}


#ifdef NOISEACC_COMB2

/********************************************************************************************
;  Subroutine Name: NoiseAccComb2
;
;  Description:
;     This routine performs the following operations:
;
;       For each tone i,
;     1) Compute the noise power
;        pow[i] = (0.5*(ref_tone[i*2]-recv_tone[i*2]))^2
;                    + (0.5*(ref_tone[i*2+1]-recv_tone[i*2+1]))^2
;
;     2) Add pow[i] to the accumulated noise power array.
;
;       The scaling by 0.5 before the squaring is needed to avoid overflowing
;       the 32 bits variable that holds the square. Thus the final
;       accumulated power is the real power dividied by 4.
;
;     The code uses a 48-bit accumulator implemented in 2 32-bit counters.  The details
;     of the implementation of the 48-bit accumulator are not important so long as
;     the RoundNoiseAccum routine can correctly round and shift the 48-bit accumulator
;     into a single 32-bit counter.
;
;     This routine is called both during REVERB and MEDLEY SNR calculations.
;     The offset parameter is used to control the PN sequence shift during
;     MEDLEY SNR calculations.
;
;
;  Prototype:
;  void NoiseAccComb2(int32 *pla_NoisePower, int16 *psa_recv_tones,  int16 *psa_ref_tones)
;
;
;  Input Arguments:
;     psa_recv_tones -- pointer to the received DMT tones
;     psa_ref_tones  -- pointer to the reference DMT tones
;     s_offset    -- offset used to generate Medley signal
;     s_first_chan   -- first channel
;     s_last_chan    -- last channel
;
;  Output Arguments:
;     pla_NoisePower -- pointer to accumulators for storing noise power.
;     s_doffset      -- offset for pla_Noisepower to point to the first noise value.
;                 Noise values will get written to pla_NoisePower[s_doffset] through
;                 pla_NoisePower[s_doffset+s_last_chan-s_first_chan], rather than
;                 pla_NoisePower[s_first_chan] through pla_NoisePower[s_last_chan]
;
;  Return Value:
;     none
;
;  Global Variables:
;     none
;
;****************************************************************************/
void NoiseAccComb2(int32 *pla_NoisePower, int16 *psa_recv_tones,  int16 *psa_ref_tones)
{
   int16 s_tone_index, s_num_tones;
   int i;
   int32 l_Acc, l_Acc1;
   int16 *psa_rec, *psa_ref;
   int32 *pla_out;


   /* ================================================== */
   /* Set up address pointers                   */
   /* ================================================== */

   pla_out = pla_NoisePower;

   /* ================================================== */
   /* Compute noise power and add it to the total power  */
   /* ================================================== */
   s_num_tones = C_COMB_INDEX_CNT_G992_5 + 20;
   for(i = 0; i < s_num_tones; i++)
   {
        if (i <= gs_C_COMB_LASTCHAN_IDX) s_tone_index = gsa_C_COMB_index[i];
      else s_tone_index = gsa_tref1_candidates[i- gs_C_COMB_LASTCHAN_IDX - 1];

      psa_ref = psa_ref_tones+2*s_tone_index;
        psa_rec = psa_recv_tones+2*s_tone_index;

      /* Compute real_error^2 */
      l_Acc = (int32)(*psa_ref++) - (int32)(*psa_rec++);
      l_Acc >>= 1;
      l_Acc = l_Acc * l_Acc;

      /* Compute imag_error^2 */
      l_Acc1 = (int32)(*psa_ref++) - (int32)(*psa_rec++);
      l_Acc1 >>= 1;
      l_Acc1 = l_Acc1 * l_Acc1;

      /* Add errors to accumulator  */
      mAddInt32ToNoiseAccumulator(l_Acc, pla_out, pla_out+1);
      mAddInt32ToNoiseAccumulator(l_Acc1, pla_out, pla_out+1);
      pla_out += 2;
   }
}

#endif // NOISEACC_COMB2
