/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/***************************************************************************************
;
;  TestHSdecode.c
;
;  File used for the standalone decoding of G.hs messages (its executables are also used
;  from the Cocomo TestHSdecode test)
;
;  USE:     TestHSdecode [<input_file>] ["TX"/"RX"/"CRC"]
;        - The (optional) first argument contains the file containing the Ghs octets to
;        be processed.  If not <input_file> is given, then the program will try to open
;        DbgInput.txt
;        - The (optional) second argument can contain either a CRC option (if "CRC"
;        the utility will calculate the FCS bytes for the input), or determine whether
;        Tx or Rx octets   are going to be decoded (Tx/Rx octets are differentiated by
;        the delimiters AAAE and EEEC, respectively, at teh beginning of each logged
;        segment that the FW captures using DEBUG_LOG_GHSMESSAGES debug trail option).
;        If no third argument is present the utility will attempt to decode Rx octets.
;
;  EXAMPLES:
;        - Used for decoding (without the CRC argument):
;              e.g. TestHSdecode INP_SocCLR_DualLP.txt RX
;        The input file has to contain the _raw_ octets of the message (as transmitted or
;        received), with the octet transparency additional octets included, together with
;        the 2 FCS bytes and a FLAG (0x007E) that indicates the end of a segment.
;        Furthermore, for combatibility with the debug log that the FW generates, each
;        segment is preceded by a delimiter indicating Tx (0xAAAE) or Rx (0xEEEC) segment,
;        and a hex value indicating time that the segment transmission/reception started.
;        The user can always grab the raw octets (in the format that the utility expects,
;        i.e. with starting segment delimiter and time) from gsa_statestrail, by setting
;        the debug log type (INFO 86 0) to 0x4000 (DEBUG_LOG_GHSMESSAGES). Tx and Rx segments
;        may be alternating in the log (they are stored in a common space, according to time
;        order).
;           Note, that if the final message  was not captured in its whole (because of space
;        limitations), it will not be output (even partially) from this routine.
;
;        The utility considers as a segment all continuous lines (after the first two
;        delimiter/time words) that start with a hex value (= 1 octet), until a line
;        starting with 0x007E (FLAG) is encountered.  After the hex value, a line can
;        contain comments etc - the program will ignore them.
;        It considers as successive segments of the same message all successive segments
;        for the same 'direction' (TX/RX, as indicated by the first segment entry) until
;        either a non-octet starting line is encountered or the message is decoded as
;        complete.
;        See sandbox\test\bis\TestHSdecode\Input_TestHSdecode.txt for a real example of
;        an input file.
;
;        - Use for CRC calculation
;              e.g. TestHSdecode INP_Segment.txt CRC
;        The input file contains a single segment of octets (format of each line is as
;        shown above), _before_ octet transparency.  There should be no delimiters
;        in this  input segment !
;        The two FCS bytes are calculated and output in the file MsgOutput.txt
;
;
;  Notes:
;        There are two configurations: Debug (for non-ISDN builds), that creates the
;        executable Debug\TestHSdecode.exe, and Debug_AnnexB   (for ISDN builds), that
;        creates the executable Debug_AnnexB\Test_Hsdecode_B.exe.
;        - Their use (input argument format, output file name etc) is the same.
;        - Each exe outputs only the information that it decodes (i.e. the ISDN build
;        will not output ADSL2 AnxA/I/M information etc).
;
;****************************************************************************************
*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "typedef.h"
#include "gdata.h"
#include "compiler.h"
#include "cmv.h"
#include "const.h"
#include "hs_mesg.h"
#include "hs_misc.h"
#include "hndshk_Data.h"
#include "gpersistent.h"
#include "gdata_bis.h"
#include "gdata_bis_diag.h"

extern void AllocateHSrelatedBuffers(void) ;
extern void InitGlobalVariables2_BisInfoTx(Adsl2AllInfo_t *pt_BisInfo, int16 s_annex);
extern void InitGlobalVariables2_BisInfoRx(Adsl2AllInfo_t *pt_BisInfo, int16 s_annex);
extern void InitGlobalVariables2_BisPlusInfoTx(Adsl2AllInfo_t *pt_BisPlusInfo, int16 s_annex);
extern void InitGlobalVariables2_BisPlusInfoRx(Adsl2AllInfo_t *pt_BisPlusInfo, int16 s_annex);

void PrintInfoField_Wrapper(FILE *trail_fid);
void PrintInfoField_Bis(FILE *trail_fid, Adsl2AllInfo_t *pt_BisInfo, uint8 uc_InfoType, uint8 uc_multiplier);

#define RX_SIDE         0
#define TX_SIDE         1

main(int argc, char *argv[])
{

   int16 i ;
   char inp_line[80];
   FlagT ft_EOF = FALSE ;

   uint8 *puca_temp;
   int16 s_MsgCnt, s_SegmentLen, s_SegmentCnt;
   uint16 us_temp, us_SegmentStartTime;
   uint16 us_FCS; // For the part where we calculate CRC for a test vector ourselves

#ifdef ISDN
   int16 sa_Preferred_PSDMask_G9923x[NUM_G992_3_ANNEX_SUPPORTED] = {DEFAULT_G992_3B_PSDMASK, DEFAULT_G992_3J_PSDMASK, DEFAULT_G992_3M_PSDMASK};
   int16 sa_Preferred_PSDMask_G9925x[NUM_G992_5_ANNEX_SUPPORTED] = {DEFAULT_G992_5B_PSDMASK, DEFAULT_G992_5J_PSDMASK, DEFAULT_G992_5M_PSDMASK};
#else
   int16 sa_Preferred_PSDMask_G9923x[NUM_G992_3_ANNEX_SUPPORTED] = {DEFAULT_G992_3A_PSDMASK, DEFAULT_G992_3I_PSDMASK, DEFAULT_G992_3M_PSDMASK};
   int16 sa_Preferred_PSDMask_G9925x[NUM_G992_5_ANNEX_SUPPORTED] = {DEFAULT_G992_5A_PSDMASK, DEFAULT_G992_5I_PSDMASK, DEFAULT_G992_5M_PSDMASK};
#endif

   FILE *msg_fid;
   FILE *trail_fid;
   FILE *inp_fid;


   // Default is decoding Rx octets
   FlagT ft_ToDecode = RX_SIDE ;
   FlagT ft_CalcCRC = FALSE ;
   FlagT ft_IgnoreSegment = FALSE ;
   uint16 gus_IgnoreNextOctet = 2 ;    // At beginning ignore delimiter, symbol count



   //
   // Initialize
   //
   AllocateHSrelatedBuffers() ;
   for (i = 0; i < NUM_G992_3_ANNEX_SUPPORTED; i++)
   {
      gs_Preferred_PSDMask_G9923x[i] = sa_Preferred_PSDMask_G9923x[i];
      InitGlobalVariables2_BisInfoTx(gpt_RxInfo->pta_G9923xInfo[i], i);
      InitGlobalVariables2_BisInfoRx(gpt_RxInfo->pta_G9923xInfo[i], i);
   }
   for (i = 0; i < NUM_G992_5_ANNEX_SUPPORTED; i++)
   {
      gs_Preferred_PSDMask_G9925x[i] = sa_Preferred_PSDMask_G9925x[i];
      InitGlobalVariables2_BisPlusInfoTx(gpt_RxInfo->pta_G9925xInfo[i], i);
      InitGlobalVariables2_BisPlusInfoRx(gpt_RxInfo->pta_G9925xInfo[i], i);
   }
   gt_GHS_DSTssiInfo.s_MaxNumBrkpt = MAX_NUM_DS_TSSI_VALUES;
   gt_GHS_USTssiInfo.s_MaxNumBrkpt = MAX_NUM_US_TSSI_VALUES;
   //OPTNArray[OPTN_GhsControl] = OPTN_DisableFailOnRxOvfl ;         // No memory overflow issues in HS decode test



   if (argc==3) {
      if (strcmp(argv[2],"TX")==0) {
         ft_ToDecode = TX_SIDE ;
      }
      else if (strcmp(argv[2],"CRC")==0) {
         ft_CalcCRC = TRUE ;
         ft_IgnoreSegment = TRUE ;
         gus_IgnoreNextOctet = 0 ;
      }
   }

   if (argc>=2) {
      inp_fid = fopen(argv[1],"r");
   }
   else {
      inp_fid = fopen("DbgInput.txt","r");
   }
   msg_fid = fopen("MsgOutput.txt","w");
   s_MsgCnt = 0 ;


   if (inp_fid==NULL) {
      fprintf(msg_fid, "\nInvalid input file name specified \n" ) ;
      fprintf(stdout, "\nError: Invalid input filename specified \n" ) ;
      fprintf(stdout, "\nUse: TestHSdecode <input filename> <RX/TX/CRC>\n" ) ;
      fprintf(stdout, "\nExample: TestHSdecode DbgInput.txt TX\n" ) ;
      return(-1) ;
   }


   trail_fid = fopen("DbgOutput.txt","w");

   while(ft_EOF==FALSE)
   {
      guc_RxHSMsg_SegmentCnt=0;
      gs_DecodeBufCnt = 0;
      gs_MaxMsgLen = MAX_MESSAGE_LEN;
      gft_GHS_S0min_Negotiated = FALSE;
      gft_DisableExtendedFrame = FALSE;

      puca_temp = gpuca_RxMsgBuf;

      ft_IgnoreSegment = FALSE ;
      s_SegmentCnt = 0;
      s_SegmentLen = 0;

      us_FCS = 0xFFFF;

      //
      // The following while(1) loop should process one message only !
      // guc_RxHSMsg_SegmentCnt, gs_DecodeBufCnt, etc should be reinitialized at beginning
      // of each message by breakign out of the while(1) loop
      //
      while(1) {     // Keep reading from Input file, until EOF or end of message
         if (fgets(inp_line, 80, inp_fid)==NULL) {
            ft_EOF = TRUE ;
            break ;
         }
         else {
            i=sscanf(inp_line, "%x", &us_temp);
            if (i!=1) {
               // Non-numerical entry means end of message
               break ;
            }
            else {
               if (ft_CalcCRC) {
                  us_FCS = CalcCRC_Byte((uint8) us_temp, us_FCS, FCS_GEN_POLY);
                  continue ;
               }

               if (gus_IgnoreNextOctet>0) {
                  gus_IgnoreNextOctet-- ;
                  if (gus_IgnoreNextOctet==1) {
                     if ( ((us_temp==0xAAAE)&&(ft_ToDecode==RX_SIDE)) || ((us_temp==0xEEEC)&&(ft_ToDecode==TX_SIDE)) ) {
                        // AAAE = Tx segment, EEEC = Rx segment
                        ft_IgnoreSegment = TRUE ;
                     }
                     else {
                        ft_IgnoreSegment = FALSE ;
                     }
                  }
                  else {
                     // if gus_IgnoreNextOctet==0 -> SymbolCount
                     us_SegmentStartTime = us_temp ;
                  }
                  continue ;
               }

               // Octet read
               if (ft_IgnoreSegment==FALSE) {
                  *puca_temp++ = (uint8) us_temp ;
                  s_SegmentLen++;
               }

               if (us_temp==0x7E){     // If end of segment
                  gus_IgnoreNextOctet = 2 ;
                  if (ft_IgnoreSegment==FALSE) {
                     s_SegmentCnt++;


                     // msg_fid was used to output additional info during the utility verification
                     fprintf(msg_fid, "\nTrying to decode segment# %d (started at %d) of message# %d with following octets:\n", s_SegmentCnt, us_SegmentStartTime, s_MsgCnt+1);
                     for (i=0 ; i<s_SegmentLen ; i++ ) {
                        fprintf(msg_fid, "gpuca_RxMsgBuf at locations %d is: 0x%2x \n", i, gpuca_RxMsgBuf[i]);
                     }

                     fprintf(trail_fid, "\n// \n");
                     fprintf(trail_fid, "//Trying to decode segment# %d (started at RxSymbolCount %d) of message# %d \n", s_SegmentCnt, us_SegmentStartTime, s_MsgCnt+1);
                     DecodeMessage();

                     // msg_fid was used to output additional info during the utility verification
                     fprintf(msg_fid, "\n\n") ;
                     for (i=0 ; i<gs_DecodeBufCnt ; i++ ) {
                        fprintf(msg_fid, "gpuca_DecodBufCnt at locations %d is: 0x%2x \n", i, gpuca_DecodeBuf[i]);
                     }

                     fprintf(trail_fid, "//Decode result: ");
                     switch (gt_hsc.s_DecodeResult) {
                        case DECODE_SUCCESS:
                           fprintf(trail_fid, "\t DECODE_SUCCESS\n") ;
                           break ;
                        case INVALID_MESSAGE:
                           fprintf(trail_fid, "\t INVALID_MESSAGE\n") ;
                           break ;
                        case FCS_ERROR:
                           fprintf(trail_fid, "\t FCS_ERROR\n") ;
                           break ;
                        case INCOMPLETE_STANDARD:
                           fprintf(trail_fid, "\t INCOMPLETE_STANDARD\n") ;
                           break ;
                        case INCOMPLETE_NON_STANDARD:
                           fprintf(trail_fid, "\t INCOMPLETE_NON_STANDARD\n") ;
                           break ;
                        case TSSI_BreakPoint_OVERFLOW:
                           fprintf(trail_fid, "\t TSSI_BreakPoint_OVERFLOW\n");
                           break;
                        default:
                           fprintf(trail_fid, "\t Other code: %d\n", gt_hsc.s_DecodeResult) ;
                           break ;
                     }
                     fprintf(trail_fid, "// \n");

                     if (gt_hsc.s_DecodeResult == DECODE_SUCCESS) {
                        // Successful decoding (complete message) means end of message;
                        // break out of inner while(1) loop that corresponds to a single message
                        s_MsgCnt++ ;

                        fprintf(trail_fid, "\n// \n");
                        fprintf(trail_fid, "//Parsing message# %d\n",   s_MsgCnt);
                        fprintf(trail_fid, "// \n");
                        PrintInfoField_Wrapper(trail_fid) ;
                        fprintf(trail_fid, "// \n");
                        fprintf(trail_fid, "//End of message# %d\n", s_MsgCnt);
                        fprintf(trail_fid, "// \n\n\n");
                        break ;
                     }
                     else {      // Incomplete message - prepare for next segment
                        s_SegmentLen = 0;
                        puca_temp = gpuca_RxMsgBuf ;
                     }
                  }// process segment
               }// end of segment
            }
         }
      } // while (1)


      if (ft_CalcCRC==TRUE) {
         us_FCS = ~us_FCS;
         fprintf(msg_fid, "CRC byte 1 is: 0x%x \n", (uint8) us_FCS);
         fprintf(msg_fid, "CRC byte 2 is: 0x%x \n", (uint8) (us_FCS >> 8));
      }

   } // EOF

   // If EOF was reached and we didn't decode succesfully the last message, then
   // output the intermediate decode result here for debugging purposes
   if (gt_hsc.s_DecodeResult != DECODE_SUCCESS) {
      fprintf(trail_fid, "\n// \n");
      fprintf(trail_fid, "//Parsing contents of RxInfo struct after (not successfully decoded) last message \n");
      fprintf(trail_fid, "// \n");
      PrintInfoField_Wrapper(trail_fid) ;
      fprintf(trail_fid, "// \n");
      fprintf(trail_fid, "// \n\n\n");
   }

   fclose(trail_fid) ;
   fclose(msg_fid) ;
   return(0);
}

void PrintInfoField_Wrapper(FILE *trail_fid)
{
   //
   // Parse out gt_RxInfo
   //
    fprintf(trail_fid, "Message Type:\t 0x%02X", gpt_RxInfo->uc_Type);
   switch (gpt_RxInfo->uc_Type) {
      case M_MS:
         fprintf(trail_fid, "\t (MS)\n") ;
         break ;
      case M_MR:
         fprintf(trail_fid, "\t (MR)\n") ;
         break ;
      case M_CL:
         fprintf(trail_fid, "\t (CL)\n") ;
         break ;
      case M_CLR:
         fprintf(trail_fid, "\t (CLR)\n") ;
         break ;
      case M_MP:
         fprintf(trail_fid, "\t (MP)\n") ;
         break ;
      case M_ACK1:
         fprintf(trail_fid, "\t (ACK1)\n") ;
         break ;
      case M_ACK2:
         fprintf(trail_fid, "\t (ACK2)\n") ;
         break ;
      case M_NAK_EF:
         fprintf(trail_fid, "\t (NAK_EF)\n") ;
         break ;
      case M_NAK_NR:
         fprintf(trail_fid, "\t (NAK_NR)\n") ;
         break ;
      case M_NAK_NS:
         fprintf(trail_fid, "\t (NAK_NS)\n") ;
         break ;
      case M_NAK_CD:
         fprintf(trail_fid, "\t (NAK_CD)\n") ;
         break ;
      case M_REQ_MS:
         fprintf(trail_fid, "\t (REQ_MS)\n") ;
         break ;
      case M_REQ_MR:
         fprintf(trail_fid, "\t (REQ_MR)\n") ;
         break ;
      case M_REQ_CLR:
         fprintf(trail_fid, "\t (REQ_CLR)\n") ;
         break ;
      case M_REQ_RTX:
         fprintf(trail_fid, "\t (REQ_RTX)\n") ;
         break ;
      default:
         fprintf(trail_fid, "\t (Unknown type)\n") ;
         break ;
   }
    fprintf(trail_fid, "Revision Number: \t%d\n", gpt_RxInfo->uc_Rev);

    /* ---- Vendor ID (required only for CL and CLR messages) ---- */
    switch (gpt_RxInfo->uc_Type)
   {
    case M_CL  :
    case M_CLR :
        fprintf(trail_fid, "Country Code:\t 0x%04X\n", gpt_RxInfo->us_Country);
        fprintf(trail_fid, "Provider ID:\t 0x%08X\n", gpt_RxInfo->ul_Provider);
        fprintf(trail_fid, "Vendor Specific Info:\t 0x%04X\n", gpt_RxInfo->us_VendorInfo);

    }   /*  switch (vendor ID) */

    /* ==== Service and Channel parameters (for CL, CLR, MS only) ==== */
    switch (gpt_RxInfo->uc_Type)
   {
    case M_CL :
    case M_CLR :
    case M_MS :
      fprintf(trail_fid, "\n-------------------------\n" ) ;
      fprintf(trail_fid, "ID Information Field: \n" ) ;
      fprintf(trail_fid, "-------------------------\n" ) ;

        fprintf(trail_fid, "ID_NPar1 :\t 0x%02X ( ,NS block, , ;; )\n", gpt_RxInfo->uc_ID_NPar1);
      fprintf(trail_fid, "ID_SPar1, octet 1 :\t 0x%02X ( , ,ATU-C,R splitter ;; DataFlow DS,US, Net DR DS,US)\n",  gpt_RxInfo->uc_ID_SPar1[0]);
      fprintf(trail_fid, "ID_SPar1, octet 2 :\t 0x%02X ( , ,Rel power C43 DS,US ;; B43 DS,US, A43 BS,US)\n",  gpt_RxInfo->uc_ID_SPar1[1]);
      fprintf(trail_fid, "ID_SPar1, octet 3 :\t 0x%02X ( , ,Rel power J43 DS,US, Bonding ;; A43c DS,US, A4 DS,US)\n",  gpt_RxInfo->uc_ID_SPar1[2]);
      fprintf(trail_fid, "ID_SPar1, octet 4 :\t 0x%02X ( ;; , ,Rel power B43c DS,US)\n",  gpt_RxInfo->uc_ID_SPar1[3]);
      fprintf(trail_fid, "ID SPar1 bits in unknown octets:\t %d \n\n",  gpt_RxInfo->uc_ID_SPar1_ExtraBits);

        /* ---- ID_NPar2 ---- */
      // All NDR/Data Flow info is left as received in struct (no actual kbps value) and not used by code
        fprintf(trail_fid, "Maximum Net Data Rate Upstream:\t %d kbps\n",
         (gpt_RxInfo->uc_UpMaxNDR & LARGE_NDR) ? ((gpt_RxInfo->uc_UpMaxNDR & NDR_BITS)*2000) : ((gpt_RxInfo->uc_UpMaxNDR & NDR_BITS)*64));

        fprintf(trail_fid, "Minimum Net Data Rate Upstream:\t %d kbps\n",
         (gpt_RxInfo->uc_UpMinNDR & LARGE_NDR) ? ((gpt_RxInfo->uc_UpMinNDR & NDR_BITS)*2000) : ((gpt_RxInfo->uc_UpMinNDR & NDR_BITS)*64));

        fprintf(trail_fid, "Average Net Data Rate Upstream:\t %d kbps\n",
         (gpt_RxInfo->uc_UpAvgNDR & LARGE_NDR) ? ((gpt_RxInfo->uc_UpAvgNDR & NDR_BITS)*2000) : ((gpt_RxInfo->uc_UpAvgNDR & NDR_BITS)*64));

        fprintf(trail_fid, "Maximum Net Data Rate Downstream:\t %d kbps\n",
         (gpt_RxInfo->uc_DnMaxNDR & LARGE_NDR) ? ((gpt_RxInfo->uc_DnMaxNDR & NDR_BITS)*2000) : ((gpt_RxInfo->uc_DnMaxNDR & NDR_BITS)*64));

        fprintf(trail_fid, "Minimum Net Data Rate Downstream:\t %d kbps\n",
         (gpt_RxInfo->uc_DnMinNDR & LARGE_NDR) ? ((gpt_RxInfo->uc_DnMinNDR & NDR_BITS)*2000) : ((gpt_RxInfo->uc_DnMinNDR & NDR_BITS)*64));

        fprintf(trail_fid, "Average Net Data Rate Downstream:\t %d kbps\n",
         (gpt_RxInfo->uc_DnAvgNDR & LARGE_NDR) ? ((gpt_RxInfo->uc_DnAvgNDR & NDR_BITS)*2000) : ((gpt_RxInfo->uc_DnAvgNDR & NDR_BITS)*64));

        fprintf(trail_fid, "Maximum Latency Upstream:\t %d ms\n",
         (gpt_RxInfo->uc_UpMaxLat & LARGE_LATENCY) ? ((4+(gpt_RxInfo->uc_UpMaxLat & LATENCY_BITS))*10) : (gpt_RxInfo->uc_UpMaxLat & LATENCY_BITS));
        fprintf(trail_fid, "Average Latency Upstream:\t %d ms\n",
         (gpt_RxInfo->uc_UpAvgLat & LARGE_LATENCY) ? ((4+(gpt_RxInfo->uc_UpAvgLat & LATENCY_BITS))*10) : (gpt_RxInfo->uc_UpAvgLat & LATENCY_BITS));

        fprintf(trail_fid, "Maximum Latency Downstream:\t %d ms\n",
         (gpt_RxInfo->uc_DnMaxLat & LARGE_LATENCY) ? ((4+(gpt_RxInfo->uc_DnMaxLat & LATENCY_BITS))*10) : (gpt_RxInfo->uc_DnMaxLat & LATENCY_BITS));
        fprintf(trail_fid, "Average Latency Downstream:\t %d ms\n",
         (gpt_RxInfo->uc_DnAvgLat & LARGE_LATENCY) ? ((4+(gpt_RxInfo->uc_DnAvgLat & LATENCY_BITS))*10) : (gpt_RxInfo->uc_DnAvgLat & LATENCY_BITS));

        fprintf(trail_fid, "ATU-R Splitter Information:\t 0x%02X\n", gpt_RxInfo->uc_ID_NPar2_O11);

        fprintf(trail_fid, "ATU-C Splitter Information:\t 0x%02X\n", gpt_RxInfo->uc_ID_NPar2_O12);


      // Only DnA43atten is used by the code - why ???
      // DnB43, DnJ43 atten is not paying attention to new (AMD2) octet 2 - hence value only up to 31.5 dB; not used by code anyway.
      //if ( (gpt_RxInfo->uc_ID_SPar1[1] & (0x1 << 0)) != 0 )
         fprintf(trail_fid, "Atten in G.994.1 xmit pwr rel. to max pwr for US carrier set A43:\t %fdB\n", (gpt_RxInfo->uc_UpA43Attenuation*0.5));
      //if ( (gpt_RxInfo->uc_ID_SPar1[1] & (0x1 << 1)) != 0 )
         fprintf(trail_fid, "Atten in G.994.1 xmit pwr rel. to max pwr for DS carrier set A43:\t %fdB\n", (gpt_RxInfo->uc_DnA43Attenuation*0.5));
      //if ( (gpt_RxInfo->uc_ID_SPar1[1] & (0x1 << 2)) != 0 )
         fprintf(trail_fid, "Atten in G.994.1 xmit pwr rel. to max pwr for US carrier set B43:\t %fdB\n", (gpt_RxInfo->uc_UpB43Attenuation*0.5));
      //if ( (gpt_RxInfo->uc_ID_SPar1[1] & (0x1 << 3)) != 0 )
         fprintf(trail_fid, "Atten (clipped) in G.994.1 xmit pwr rel. to max pwr for DS carrier set B43:\t %fdB\n", (gpt_RxInfo->uc_DnB43Attenuation*0.5));
      //if ( (gpt_RxInfo->uc_ID_SPar1[1] & (0x1 << 4)) != 0 )
         fprintf(trail_fid, "Atten in G.994.1 xmit pwr rel. to max pwr for US carrier set C43:\t %fdB\n", (gpt_RxInfo->uc_UpC43Attenuation*0.5));
      //if ( (gpt_RxInfo->uc_ID_SPar1[1] & (0x1 << 5)) != 0 )
         fprintf(trail_fid, "Atten in G.994.1 xmit pwr rel. to max pwr for DS carrier set C43:\t %fdB\n", (gpt_RxInfo->uc_DnC43Attenuation*0.5));
      //if ( (gpt_RxInfo->uc_ID_SPar1[2] & (0x1 << 0)) != 0 )
         fprintf(trail_fid, "Atten in G.994.1 xmit pwr rel. to max pwr for US carrier set A4:\t %fdB\n", (gpt_RxInfo->uc_UpA4Attenuation*0.5));
      //if ( (gpt_RxInfo->uc_ID_SPar1[2] & (0x1 << 1)) != 0 )
         fprintf(trail_fid, "Atten in G.994.1 xmit pwr rel. to max pwr for DS carrier set A4:\t %fdB\n", (gpt_RxInfo->uc_DnA4Attenuation*0.5));
      //if ( (gpt_RxInfo->uc_ID_SPar1[2] & (0x1 << 2)) != 0 )
         fprintf(trail_fid, "Atten in G.994.1 xmit pwr rel. to max pwr for US carrier set A43c:\t %fdB\n", (gpt_RxInfo->uc_UpA43cAttenuation*0.5));
      //if ( (gpt_RxInfo->uc_ID_SPar1[2] & (0x1 << 3)) != 0 )
         fprintf(trail_fid, "Atten in G.994.1 xmit pwr rel. to max pwr for DS carrier set A43c:\t %fdB\n", (gpt_RxInfo->uc_DnA43cAttenuation*0.5));
      //if ( (gpt_RxInfo->uc_ID_SPar1[2] & (0x1 << 5)) != 0 )
         fprintf(trail_fid, "Atten in G.994.1 xmit pwr rel. to max pwr for US carrier set J43:\t %fdB\n", (gpt_RxInfo->uc_UpJ43Attenuation*0.5));
      //if ( (gpt_RxInfo->uc_ID_SPar1[2] & (0x1 << 6)) != 0 )
         fprintf(trail_fid, "Atten (clipped) in G.994.1 xmit pwr rel. to max pwr for DS carrier set J43:\t %fdB\n", (gpt_RxInfo->uc_DnJ43Attenuation*0.5));
      //if ( (gpt_RxInfo->uc_ID_SPar1[3] & (0x1 << 0)) != 0 )
         fprintf(trail_fid, "Atten in G.994.1 xmit pwr rel. to max pwr for US carrier set B43c:\t %fdB\n", (gpt_RxInfo->uc_UpB43cAttenuation*0.5));
      //if ( (gpt_RxInfo->uc_ID_SPar1[3] & (0x1 << 1)) != 0 )
         fprintf(trail_fid, "Atten in G.994.1 xmit pwr rel. to max pwr for DS carrier set B43c:\t %fdB\n", (gpt_RxInfo->uc_DnB43cAttenuation*0.5));

    }   /*  switch (parameter field) */


    /* ==== Standard information (for CL, CLR, MS only) ==== */
    switch (gpt_RxInfo->uc_Type)
   {
    case M_CL :
    case M_CLR :
    case M_MS :
      fprintf(trail_fid, "\n-------------------------\n" ) ;
      fprintf(trail_fid, "Standard Information Field:\n") ;
      fprintf(trail_fid, "-------------------------\n" ) ;

        fprintf(trail_fid, "Standard Information NPar1 :\t 0x%02X ( ;; G997, Silent, V8bis, V8)\n", gpt_RxInfo->uc_SI_NPar1);

      fprintf(trail_fid, "Standard Information SPar1 octet 1 :\t 0x%02X ( , 1I, 1H, 2C ;; 2AB, 1C, 1B, 1A)\n", gpt_RxInfo->uc_SI_SPar1[0]);
      fprintf(trail_fid, "Standard Information SPar1 octet 2 :\t 0x%02X (~VDSL/SHDSL)\n", gpt_RxInfo->uc_SI_SPar1[1]);
      fprintf(trail_fid, "Standard Information SPar1 octet 3 :\t 0x%02X ( , , 4I, 4A ;; 3J, 3I, 3B, 3A)\n", gpt_RxInfo->uc_SI_SPar1[2]);
      fprintf(trail_fid, "Standard Information SPar1 octet 4 :\t 0x%02X ( , , , 5J ;; 3M, 5I, 5B, 5A)\n", gpt_RxInfo->uc_SI_SPar1[3]);
      fprintf(trail_fid, "Standard Information SPar1 octet 5 :\t 0x%02X ( ;; , , , 5M)\n", gpt_RxInfo->uc_SI_SPar1[4]);
      fprintf(trail_fid, "Standard Information SPar1 bits in unknown octets:\t %d \n", gpt_RxInfo->uc_SI_SPar1_ExtraBits);


      if (gpt_RxInfo->uc_SI_SPar1[0] & G992_1_A) {
         fprintf(trail_fid, "\n-------------------------\n" );
         fprintf(trail_fid, "For Annex 1A: \n" );
         fprintf(trail_fid, "-------------------------\n" );
            fprintf(trail_fid, "Standard Information NPar2 :\t 0x%02X ( , , Clear EOC, ATM ;; STM, , R-ACK2, R-ACK1)\n", gpt_RxInfo->uc_SI_1A_NPar2);
            fprintf(trail_fid, "Standard Information SPar2 :\t 0x%02X ( ;;  , Spectr Freq DS, Spectr Freq US, Sub-Channel)\n", gpt_RxInfo->uc_SI_1A_SPar2);

            /* ---- SI_NPar3 ---- */
         if (gpt_RxInfo->uc_SI_1A_SPar2 & 0x01) {
            fprintf(trail_fid, "Sub-Channel info1 :\t 0x%02X ( , , , LS0 DS ;; AS3 DS, AS2 DS, AS1 DS, AS0 DS)\n",   gpt_RxInfo->uc_1A_SubChan1);
            fprintf(trail_fid, "Sub-Channel info2 :\t 0x%02X ( , , , LS2 US ;; LS1 US, LS0 US, LS2 DS, LS1 DS)\n", gpt_RxInfo->uc_1A_SubChan2);
            }
         // Min/Max spectrum calculated inside the code
         if (gpt_RxInfo->uc_SI_1A_SPar2 & 0x02) {
            fprintf(trail_fid, "Minimum Spectrum Upstream:\t %d\n", gpt_RxInfo->uc_1A_UpMinSpect);
            fprintf(trail_fid, "Maximum Spectrum Upstream:\t %d\n", gpt_RxInfo->uc_1A_UpMaxSpect);
         }
         if (gpt_RxInfo->uc_SI_1A_SPar2 & 0x04) {
            fprintf(trail_fid, "Minimum Spectrum Downstream:\t %d\n", gpt_RxInfo->uc_1A_DnMinSpect);
            fprintf(trail_fid, "Maximum Spectrum Downstream:\t %d\n", gpt_RxInfo->uc_1A_DnMaxSpect);
         }
        }   // 1A


      if (gpt_RxInfo->uc_SI_SPar1[0] & G992_1_B) {
         fprintf(trail_fid, "\n-------------------------\n" );
         fprintf(trail_fid, "For Annex 1B: \n" );
         fprintf(trail_fid, "-------------------------\n" );
            fprintf(trail_fid, "Standard Information NPar2 :\t 0x%02X ( , , Clear EOC, ATM ;; STM, US1_32, R-ACK2, R-ACK1)\n", gpt_RxInfo->uc_SI_1B_NPar2);
            fprintf(trail_fid, "Standard Information SPar2 :\t 0x%02X ( ;;  , Spectr Freq DS, Spectr Freq US, Sub-Channel)\n", gpt_RxInfo->uc_SI_1B_SPar2);

            /* ---- SI_NPar3 ---- */
         if (gpt_RxInfo->uc_SI_1B_SPar2 & 0x01) {
            fprintf(trail_fid, "Sub-Channel info1 :\t 0x%02X ( , , , LS0 DS ;; AS3 DS, AS2 DS, AS1 DS, AS0 DS)\n",   gpt_RxInfo->uc_1B_SubChan1);
            fprintf(trail_fid, "Sub-Channel info2 :\t 0x%02X ( , , , LS2 US ;; LS1 US, LS0 US, LS2 DS, LS1 DS)\n", gpt_RxInfo->uc_1B_SubChan2);
            }
         if (gpt_RxInfo->uc_SI_1B_SPar2 & 0x02) {
            fprintf(trail_fid, "Minimum Spectrum Upstream:\t %d\n", gpt_RxInfo->uc_1B_UpMinSpect);
            fprintf(trail_fid, "Maximum Spectrum Upstream:\t %d\n", gpt_RxInfo->uc_1B_UpMaxSpect);
         }
         if (gpt_RxInfo->uc_SI_1B_SPar2 & 0x04) {
            fprintf(trail_fid, "Minimum Spectrum Downstream:\t %d\n", gpt_RxInfo->uc_1B_DnMinSpect);
            fprintf(trail_fid, "Maximum Spectrum Downstream:\t %d\n", gpt_RxInfo->uc_1B_DnMaxSpect);
         }
        }   // 1B


      if (gpt_RxInfo->uc_SI_SPar1[0] & G992_2_AB) {
         fprintf(trail_fid, "\n-------------------------\n" );
         fprintf(trail_fid, "For Annex 2AB: \n" );
         fprintf(trail_fid, "-------------------------\n" );
            fprintf(trail_fid, "Standard Information NPar2 :\t 0x%02X ( , , Clear EOC, RS16 ;; FR, , R-ACK2, R-ACK1)\n", gpt_RxInfo->uc_SI_2AB_NPar2);
            fprintf(trail_fid, "Standard Information SPar2 :\t 0x%02X ( ;;  , Spectr Freq DS, Spectr Freq US, )\n", gpt_RxInfo->uc_SI_2AB_SPar2);

            /* ---- SI_NPar3 ---- */
         if (gpt_RxInfo->uc_SI_2AB_SPar2 & 0x02) {
            fprintf(trail_fid, "Minimum Spectrum Upstream:\t %d\n", gpt_RxInfo->uc_2AB_UpMinSpect);
            fprintf(trail_fid, "Maximum Spectrum Upstream:\t %d\n", gpt_RxInfo->uc_2AB_UpMaxSpect);
         }
         if (gpt_RxInfo->uc_SI_2AB_SPar2 & 0x04) {
            fprintf(trail_fid, "Minimum Spectrum Downstream:\t %d\n", gpt_RxInfo->uc_2AB_DnMinSpect);
            fprintf(trail_fid, "Maximum Spectrum Downstream:\t %d\n", gpt_RxInfo->uc_2AB_DnMaxSpect);
         }
        }   // 2AB


#ifndef ISDN
      if (gpt_RxInfo->uc_SI_SPar1[2] & (G992_3_A >> 16)) {
         fprintf(trail_fid, "\n-------------------------\n" );
         fprintf(trail_fid, "For Annex 3A: \n" );
         fprintf(trail_fid, "-------------------------\n" );
         PrintInfoField_Bis(trail_fid, gpt_RxInfo->pta_G9923xInfo[G992_3_ANNEX_A], gpt_RxInfo->uc_Type, 1);
        }
      if (gpt_RxInfo->uc_SI_SPar1[2] & (G992_3_I >> 16)) {
         fprintf(trail_fid, "\n-------------------------\n" );
         fprintf(trail_fid, "For Annex 3I: \n" );
         fprintf(trail_fid, "-------------------------\n" );
         PrintInfoField_Bis(trail_fid, gpt_RxInfo->pta_G9923xInfo[G992_3_ANNEX_I], gpt_RxInfo->uc_Type, 1);
        }
#else
      if (gpt_RxInfo->uc_SI_SPar1[2] & (G992_3_B >> 16)) {
         fprintf(trail_fid, "\n-------------------------\n" );
         fprintf(trail_fid, "For Annex 3B: \n" );
         fprintf(trail_fid, "-------------------------\n" );
         PrintInfoField_Bis(trail_fid, gpt_RxInfo->pta_G9923xInfo[G992_3_ANNEX_B], gpt_RxInfo->uc_Type, 1);
        }
      if (gpt_RxInfo->uc_SI_SPar1[2] & (G992_3_J >> 16)) {
         fprintf(trail_fid, "\n-------------------------\n" );
         fprintf(trail_fid, "For Annex 3J: \n" );
         fprintf(trail_fid, "-------------------------\n" );
         PrintInfoField_Bis(trail_fid, gpt_RxInfo->pta_G9923xInfo[G992_3_ANNEX_J], gpt_RxInfo->uc_Type, 1);
        }
#endif
      if (gpt_RxInfo->uc_SI_SPar1[3] & (G992_3_M >> 24)) {
         fprintf(trail_fid, "\n-------------------------\n" );
         fprintf(trail_fid, "For Annex 3M: \n" );
         fprintf(trail_fid, "-------------------------\n" );
         PrintInfoField_Bis(trail_fid, gpt_RxInfo->pta_G9923xInfo[G992_3_ANNEX_M], gpt_RxInfo->uc_Type, 1);
        }


#ifndef ISDN
      if (gpt_RxInfo->uc_SI_SPar1[3] & (G992_5_A >> 24)) {
         fprintf(trail_fid, "\n-------------------------\n" );
         fprintf(trail_fid, "For Annex 5A: \n" );
         fprintf(trail_fid, "-------------------------\n" );
         PrintInfoField_Bis(trail_fid, gpt_RxInfo->pta_G9925xInfo[G992_5_ANNEX_A], gpt_RxInfo->uc_Type, 2);
      }
      if (gpt_RxInfo->uc_SI_SPar1[3] & (G992_5_I >> 24)) {
         fprintf(trail_fid, "\n-------------------------\n" );
         fprintf(trail_fid, "For Annex 5I: \n" );
         fprintf(trail_fid, "-------------------------\n" );
         PrintInfoField_Bis(trail_fid, gpt_RxInfo->pta_G9925xInfo[G992_5_ANNEX_I], gpt_RxInfo->uc_Type, 2);
      }
#else
      if (gpt_RxInfo->uc_SI_SPar1[3] & (G992_5_B >> 24)) {
         fprintf(trail_fid, "\n-------------------------\n" );
         fprintf(trail_fid, "For Annex 5B: \n" );
         fprintf(trail_fid, "-------------------------\n" );
         PrintInfoField_Bis(trail_fid, gpt_RxInfo->pta_G9925xInfo[G992_5_ANNEX_B], gpt_RxInfo->uc_Type, 2);
      }
      if (gpt_RxInfo->uc_SI_SPar1[3] & (G992_5_J >> 24)) {
         fprintf(trail_fid, "\n-------------------------\n" );
         fprintf(trail_fid, "For Annex 5J: \n" );
         fprintf(trail_fid, "-------------------------\n" );
         PrintInfoField_Bis(trail_fid, gpt_RxInfo->pta_G9925xInfo[G992_5_ANNEX_J], gpt_RxInfo->uc_Type, 2);
      }
#endif
      if (gpt_RxInfo->uc_SI_SPar1[4] & (G992_5_M)) {
         fprintf(trail_fid, "\n-------------------------\n" );
         fprintf(trail_fid, "For Annex 5M: \n" );
         fprintf(trail_fid, "-------------------------\n" );
         PrintInfoField_Bis(trail_fid, gpt_RxInfo->pta_G9925xInfo[G992_5_ANNEX_M], gpt_RxInfo->uc_Type, 2);
      }

    }   /*  switch (Standard Info) */
}



void PrintInfoField_Bis(TRAILFILE trail_fid, Adsl2AllInfo_t *pt_BisInfo, uint8 uc_InfoType, uint8 uc_multiplier )
{
   int16 i, j, s_shift;
   uint8 uc_Spar2, uc_Type, *puca_BreakFreq;
   uint8 tempValue;
   double temp;

   char *p_SPar2_descriptor[NUM_SI_SPAR2_OCTETS] = {" , , L, Nyq ;; tssi/bounds DS, tssi/bnds US ",
      " ;; MaxNum TPS-TC US, DS, Ovhd US, DS",
      "  , , PTM TPS-TC #0 US, DS ;; ATM TPS-TC #0 US, DS, STM TPS-TC #0 US, DS",
      " ;; , , PMS-TC LP0 US, DS",
      "  , , PTM TPS-TC #1 US, DS ;; ATM TPS-TC #1 US, DS, STM TPS-TC #1 US, DS",
      " ;; , , PMS-TC LP1 US, DS",
      "  , , PTM TPS-TC #2 US, DS ;; ATM TPS-TC #2 US, DS, STM TPS-TC #2 US, DS",
      " ;; , , PMS-TC LP2 US, DS",
      "  , , PTM TPS-TC #3 US, DS ;; ATM TPS-TC #3 US, DS, STM TPS-TC #3 US, DS",
      " ;; , , PMS-TC LP3 US, DS" };

   char *p_PSD_3L_descriptor[4] = {"US Mask1 (Wide)" , "US Mask2 (Narrow)", "DS Non-Overlapped" , "DS Overlapped"} ;
   char *p_PSD_M_descriptor[9] = {"EU-32" , "EU-36" , "EU-40" , "EU-44" , "EU-48" , "EU-52" , "EU-56" , "EU-60" , "EU-64" } ;
   char *p_PSD_J_descriptor[9] = {"ADLU-32" , "ADLU-36" , "ADLU-40" , "ADLU-44" , "ADLU-48" , "ADLU-52" , "ADLU-56" , "ADLU-60" , "ADLU-64" } ;

   fprintf(trail_fid, "Standard Information NPar2:\t 0x%02X ( ;; , Diag, ShortInit, NTR)\n", pt_BisInfo->uc_SI_NPar2);
   for(i=0; i< NUM_SI_SPAR2_OCTETS; i++)
      fprintf(trail_fid, "Standard Information SPar2 octet %d :\t 0x%02X (%s)\n", (i+1), pt_BisInfo->uc_SI_SPar2[i], p_SPar2_descriptor[i]);

   /* pmd info */
   if ( (pt_BisInfo->uc_SI_SPar2[0] & SPECTRUM_BOUNDS_UP) != 0 )
   {
      /* spectrum bounds upstream info */
      temp = (double)(pt_BisInfo->t_PMD_NPar3_US.s_NOMPSD)/10.0;
      fprintf(trail_fid, "\nNominal PSD Upstream (relative to -38dBm/Hz):\t %+2.1lfdBm/Hz\n", temp);

      temp = (double)pt_BisInfo->t_PMD_NPar3_US.s_MAXNOMPSD/10.0;
      fprintf(trail_fid, "Max. Nominal PSD Upstream (relative to -38dBm/Hz):\t %+2.1lfdBm/Hz\n", temp);

      temp = (double)pt_BisInfo->t_PMD_NPar3_US.s_MAXNOMATP/10.0;
      fprintf(trail_fid, "Max. Nominal Aggregate Transmit Power Upstream (relative to 12.5dBm):\t %+2.1lfdBm\n", temp);
   }

   if ( (pt_BisInfo->uc_SI_SPar2[0] & SPECTRUM_BOUNDS_DN) != 0 )
   {
      /* spectrum bounds downstream info */
      temp = (double)pt_BisInfo->t_PMD_NPar3_DS.s_NOMPSD/10.0;
      fprintf(trail_fid, "\nNominal PSD Downstream (relative to -40dBm/Hz):\t %+2.1lfdBm/Hz\n", temp);

      temp = (double)pt_BisInfo->t_PMD_NPar3_DS.s_MAXNOMPSD/10.0;
      fprintf(trail_fid, "Max. Nominal PSD Downstream (relative to -40dBm/Hz):\t %+2.1lfdBm/Hz\n", temp);

      temp = (double)pt_BisInfo->t_PMD_NPar3_DS.s_MAXNOMATP/10.0;
      fprintf(trail_fid, "Max. Nominal Aggregate Transmit Power Downstream (relative to 19.9dBm):\t %2.1lfdBm\n", temp);
   }

   /* tssi info */
   /* upstream spectrum shaping info */
   if ( (pt_BisInfo->uc_SI_SPar2[0] & SPECTRUM_SHAPE_UP) != 0 )
   {
      int16 s_num_US_Tssi_brkpt;

      fprintf(trail_fid, "\nUpstream spectrum shaping breakpoints: ");

      fprintf(trail_fid, "we decoded %d breakpoints, max we can keep is %d.\n",
                        gsa_num_US_Tssi_brkpt[uc_multiplier-1], gt_GHS_USTssiInfo.s_MaxNumBrkpt);
      s_num_US_Tssi_brkpt = gsa_num_US_Tssi_brkpt_kept[uc_multiplier-1];

      fprintf(trail_fid, "Subcarrier Indices: ");
      puca_BreakFreq = (uint8*)pt_BisInfo->t_PMD_NPar3_US.pus_BreakFreq;
      for (i = 0; i < s_num_US_Tssi_brkpt; i++)
         fprintf(trail_fid, "%d\t", puca_BreakFreq[i]);;

      fprintf(trail_fid, "\nIndication bit:     ");

      for (i = 0; i < s_num_US_Tssi_brkpt; i++)
      {
         s_shift = i - ((i>>3)<<3);
         fprintf(trail_fid, "%d\t", ((pt_BisInfo->t_PMD_NPar3_US.pf_InSprtSet[i>>3])>>s_shift)&0x1);
      }
      fprintf(trail_fid, "\nlog_tssi:           ");

      for (i = 0; i < s_num_US_Tssi_brkpt; i++)
         fprintf(trail_fid, "%d\t", pt_BisInfo->t_PMD_NPar3_US.puc_tssi[i]);

      fprintf(trail_fid, "\n");
   }

   if ( (pt_BisInfo->uc_SI_SPar2[0] & SPECTRUM_SHAPE_DN) != 0 )
   {
      fprintf(trail_fid, "\nDownstream spectrum shaping breakpoints: ");

      fprintf(trail_fid, "we decoded %d breakpoints, max we can keep is %d.\n",
                        gsa_num_DS_Tssi_brkpt[uc_multiplier-1], gt_GHS_DSTssiInfo.s_MaxNumBrkpt);

      fprintf(trail_fid, "Subcarrier Indices: ");
      for (i = 0; i < gsa_num_DS_Tssi_brkpt_kept[uc_multiplier-1]; i++)
         fprintf(trail_fid, "%d\t", pt_BisInfo->t_PMD_NPar3_DS.pus_BreakFreq[i]);;

      fprintf(trail_fid, "\nIndication bit:     ");

      for (i = 0; i < gsa_num_DS_Tssi_brkpt_kept[uc_multiplier-1]; i++)
      {
         s_shift = i - ((i>>3)<<3);
         fprintf(trail_fid, "%d\t", ((pt_BisInfo->t_PMD_NPar3_DS.pf_InSprtSet[i>>3])>>s_shift)&0x1);
      }

      fprintf(trail_fid, "\nlog_tssi:           ");

      for (i = 0; i < gsa_num_DS_Tssi_brkpt_kept[uc_multiplier-1]; i++)
         fprintf(trail_fid, "%d\t", pt_BisInfo->t_PMD_NPar3_DS.puc_tssi[i]);

      fprintf(trail_fid, "\n");
   }


   // TxSignalNyq info is left as received in struct and not used by code
   if ( (pt_BisInfo->uc_SI_SPar2[0] & TX_IMAGE_NYQ) != 0 ) {
      /* transmit signal images above nyquist info */
      if((pt_BisInfo->uc_TxSignalNyq & 0x3C) == 0)
         fprintf(trail_fid, "IDFT size N:\t is not a power of 2\n");
      else
         fprintf(trail_fid, "IDFT size N:\t %d\n", (1<<((pt_BisInfo->uc_TxSignalNyq &0x3C)>>2)));

      if((pt_BisInfo->uc_TxSignalNyq & 0x3) == 1)
         fprintf(trail_fid, "IFFT fill:\t complex conjugate of the base-band signal\n");
      else if((pt_BisInfo->uc_TxSignalNyq & 0x3) == 2)
         fprintf(trail_fid, "IFFT fill:\t zero-filled\n");
      else if ((pt_BisInfo->uc_TxSignalNyq & 0x3) == 0)
         fprintf(trail_fid, "IFFT fill:\t Neither zero-filled, nor complex conjugate of the base-band signal\n");

      fprintf(trail_fid, "\n");
   }


   if (pt_BisInfo->uc_SI_SPar2[0] & 0x20) {  // PSD bit in SI_Spar2 octet 1
#ifndef ISDN
      if (pt_BisInfo==gpt_RxInfo->pta_G9923xInfo[G992_3_ANNEX_A])
      {  // Annex3L
         fprintf(trail_fid, "US PSD Masks Supported: \n") ;
         for (i=0 ; i<2 ; i++) {
            if (pt_BisInfo->uca_PSD_Mask[0] & (1<<i)) {
               fprintf(trail_fid, "%s \n", p_PSD_3L_descriptor[i]);
            }
         }
         fprintf(trail_fid, "DS PSD Masks Supported: \n") ;
         for (i=0 ; i<2 ; i++) {
            if (pt_BisInfo->uca_PSD_Mask[1] & (1<<i)) {
               fprintf(trail_fid, "%s \n", p_PSD_3L_descriptor[i+2]);
            }
         }
      }
#else
      if ((pt_BisInfo==gpt_RxInfo->pta_G9923xInfo[G992_3_ANNEX_J]) || (pt_BisInfo==gpt_RxInfo->pta_G9925xInfo[G992_5_ANNEX_J]))
      {  // AnnexJ
         fprintf(trail_fid, "Submode PSD Masks Supported: \n") ;
         for (i=0 ; i<6 ; i++) {
            if (pt_BisInfo->uca_PSD_Mask[0] & (1<<i)) {
               fprintf(trail_fid, "%s \n", p_PSD_J_descriptor[i]);
            }
         }
         for (i=0 ; i<3 ; i++) {
            if (pt_BisInfo->uca_PSD_Mask[1] & (1<<i)) {
               fprintf(trail_fid, "%s \n", p_PSD_J_descriptor[6+i]);
            }
         }
      }
#endif


      if ((pt_BisInfo==gpt_RxInfo->pta_G9923xInfo[G992_3_ANNEX_M]) || (pt_BisInfo==gpt_RxInfo->pta_G9925xInfo[G992_5_ANNEX_M]))
      {  // AnnexJ
         fprintf(trail_fid, "Submode PSD Masks Supported: \n") ;
         for (i=0 ; i<6 ; i++) {
            if (pt_BisInfo->uca_PSD_Mask[0] & (1<<i)) {
               fprintf(trail_fid, "%s \n", p_PSD_M_descriptor[i]);
            }
         }
         for (i=0 ; i<3 ; i++) {
            if (pt_BisInfo->uca_PSD_Mask[1] & (1<<i)) {
               fprintf(trail_fid, "%s \n", p_PSD_M_descriptor[6+i]);
            }
         }
      }
   }// PSD bit


   fprintf(trail_fid, "\nMinimum DS MSG overhead data rate:\t %d Kbps\n", pt_BisInfo->t_Overhead.s_DSMSGOvhdMinRate);

   fprintf(trail_fid, "\nMinimum US MSG overhead data rate:\t %d Kbps\n\n", pt_BisInfo->t_Overhead.s_USMSGOvhdMinRate);

   fprintf(trail_fid, "Maximum num of DS STM TPS-TC functions:\t %d\n", pt_BisInfo->uc_DS_Max_Num_STM_TPS_TC);
   fprintf(trail_fid, "Maximum num of DS ATM TPS-TC functions:\t %d\n", pt_BisInfo->uc_DS_Max_Num_ATM_TPS_TC);
   fprintf(trail_fid, "Maximum num of DS PTM TPS-TC functions:\t %d\n\n", pt_BisInfo->uc_DS_Max_Num_PTM_TPS_TC);

   fprintf(trail_fid, "Maximum num of US STM TPS-TC functions:\t %d\n", pt_BisInfo->uc_US_Max_Num_STM_TPS_TC);
   fprintf(trail_fid, "Maximum num of US ATM TPS-TC functions:\t %d\n", pt_BisInfo->uc_US_Max_Num_ATM_TPS_TC);
   fprintf(trail_fid, "Maximum num of US PTM TPS-TC functions:\t %d\n\n", pt_BisInfo->uc_US_Max_Num_PTM_TPS_TC);

   /* Go through BC's / LP's in order that they appear on CL/CLR */
   for(i=0; i<4; i++)
   {
      /* DS TPS-TC info */
      uc_Spar2 = pt_BisInfo->uc_SI_SPar2[2*(i+1)];
      for(j=0; j<MAX_TPS_TC_TYPE; j++)
      {
         /* the assumption below is that STM, ATM and PTM codepoints for all TPS-TC functions are the same (atleast it is so in the current g994.1 standard */
         /* so we are using tps-tc function 0's STM, ATM and PTM constants ie. constant TPSTC_FUNC0_STM_DN, TPSTC_FUNC0_ATM_DN, TPSTC_FUNC0_PTM_DN */
         /* already printed STM info. So, set the STM bit corresponding to Spar2 to zero */
         switch(j)
         {
         case 0:
            uc_Type = TPSTC_FUNC_STM_DN;
            break;

         case 1:
            uc_Type = TPSTC_FUNC_ATM_DN;
            break;

         case 2:
            uc_Type = TPSTC_FUNC_PTM_DN;
            break;
         }

         if(uc_Spar2 & uc_Type)
         {
            /* print for only those tps tc types that are supported */
            fprintf(trail_fid, "\nFor DS TPS-TC Function#%d type %d:\n", i, j);
            fprintf(trail_fid, "Minimum net data rate =  %d (times %d bps)\n", pt_BisInfo->ta_DSBearerChannel[i].ta_TpsTcType[j].s_MinNetDataRate, uc_multiplier*4000);
            fprintf(trail_fid, "Maximum net data rate =  %d (times %d bps)\n", pt_BisInfo->ta_DSBearerChannel[i].ta_TpsTcType[j].s_MaxNetDataRate, uc_multiplier*4000);
            fprintf(trail_fid, "Minimum reserved net data rate =  %d (times %d bps)\n", pt_BisInfo->ta_DSBearerChannel[i].ta_TpsTcType[j].s_MinResNetDataRate, uc_multiplier*4000);
            if ((pt_BisInfo->ta_DSBearerChannel[i].ta_TpsTcType[j].s_MaxLatency == 1024) || (pt_BisInfo->ta_DSBearerChannel[i].ta_TpsTcType[j].s_MaxLatency == 0))
               fprintf(trail_fid, "Maximum latency = (no delay boundary)\n");
            else
               fprintf(trail_fid, "Maximum latency =  %d ms \n", (pt_BisInfo->ta_DSBearerChannel[i].ta_TpsTcType[j].s_MaxLatency));
            if((pt_BisInfo->ta_DSBearerChannel[i].ta_TpsTcType[j].uc_MaxBER) == 0)
               fprintf(trail_fid, "Maximum BER =  10^(-3)\n");
            else if((pt_BisInfo->ta_DSBearerChannel[i].ta_TpsTcType[j].uc_MaxBER) == 0x1)
               fprintf(trail_fid, "Maximum BER =  10^(-5)\n");
            else if((pt_BisInfo->ta_DSBearerChannel[i].ta_TpsTcType[j].uc_MaxBER) == 0x2)
               fprintf(trail_fid, "Maximum BER =  10^(-7)\n");

            tempValue = pt_BisInfo->ta_DSBearerChannel[i].ta_TpsTcType[j].s_IMAxINP & 0xFF;
            if (tempValue == 0)
               fprintf(trail_fid, "Minimum INP = 0\n");
            else if (tempValue == 1)
               fprintf(trail_fid, "Minimum INP = 1/2\n");
            else if (tempValue == 2)
               fprintf(trail_fid, "Minimum INP = 1\n");
            else if (tempValue == 3)
               fprintf(trail_fid, "Minimum INP = 2\n");
            else if (tempValue == 0x37)
               fprintf(trail_fid, "Minimum INP = 3\n");
            else if (tempValue == 0x7)
               fprintf(trail_fid, "Minimum INP = 4\n");
            else if (tempValue == 0x5B)
               fprintf(trail_fid, "Minimum INP = 5\n");
            else if (tempValue == 0x6B)
               fprintf(trail_fid, "Minimum INP = 6\n");
            else if (tempValue == 0x7B)
               fprintf(trail_fid, "Minimum INP = 7\n");
            else if (tempValue == 0xB)
               fprintf(trail_fid, "Minimum INP = 8\n");
            else if (tempValue == 0x9F)
               fprintf(trail_fid, "Minimum INP = 9\n");
            else if (tempValue == 0xAF)
               fprintf(trail_fid, "Minimum INP = 10\n");
            else if (tempValue == 0xBF)
               fprintf(trail_fid, "Minimum INP = 11\n");
            else if (tempValue == 0xCF)
               fprintf(trail_fid, "Minimum INP = 12\n");
            else if (tempValue == 0xDF)
               fprintf(trail_fid, "Minimum INP = 13\n");
            else if (tempValue == 0xEF)
               fprintf(trail_fid, "Minimum INP = 14\n");
            else if (tempValue == 0xFF)
               fprintf(trail_fid, "Minimum INP = 15\n");
            else if (tempValue == 15)
               fprintf(trail_fid, "Minimum INP = 16\n");


            if (uc_Type == TPSTC_FUNC_ATM_DN)
            {
               tempValue = (pt_BisInfo->ta_DSBearerChannel[i].ta_TpsTcType[j].s_IMAxINP >>8) & 0x01;
               fprintf(trail_fid, "IMA Flag = %d\n", tempValue);
            }
         }
      } /* for(j=0; j<MAX_TPS_TC_TYPE; j++) */


      /* US TPS-TC info */
      uc_Spar2 = pt_BisInfo->uc_SI_SPar2[2*(i+1)];
      for(j=0; j<MAX_TPS_TC_TYPE; j++)
      {
         /* the assumption below is that STM, ATM and PTM codepoints for all TPS-TC functions are the same (atleast it is so in the current g994.1 standard */
         /* so we are using tps-tc function 0's STM, ATM and PTM constants ie. constant TPSTC_FUNC_STM_US, TPSTC_FUNC_ATM_US, TPSTC_FUNC_PTM_US */
         /* already printed STM info. So, set the STM bit corresponding to Spar2 to zero */
         switch(j)
         {
         case 0:
            uc_Type = TPSTC_FUNC_STM_US;
            break;
         case 1:
            uc_Type = TPSTC_FUNC_ATM_US;
            break;
         case 2:
            uc_Type = TPSTC_FUNC_PTM_US;
            break;

         }
         if(uc_Spar2 & uc_Type)
         {
            /* print for only those tps tc types that are supported */
            fprintf(trail_fid, "\nFor US TPS-TC Function#%d type %d:\n", i, j);
            fprintf(trail_fid, "Minimum net data rate =  %d (times %d bps)\n", pt_BisInfo->ta_USBearerChannel[i].ta_TpsTcType[j].s_MinNetDataRate, uc_multiplier*4000);
            fprintf(trail_fid, "Maximum net data rate =  %d (times %d bps)\n", pt_BisInfo->ta_USBearerChannel[i].ta_TpsTcType[j].s_MaxNetDataRate, uc_multiplier*4000);
            fprintf(trail_fid, "Minimum reserved net data rate =  %d (times %d bps)\n", pt_BisInfo->ta_USBearerChannel[i].ta_TpsTcType[j].s_MinResNetDataRate, uc_multiplier*4000);

            if ((pt_BisInfo->ta_USBearerChannel[i].ta_TpsTcType[j].s_MaxLatency == 1024) || (pt_BisInfo->ta_USBearerChannel[i].ta_TpsTcType[j].s_MaxLatency == 0))
               fprintf(trail_fid, "Maximum latency = (no delay boundary)\n");
            else
               fprintf(trail_fid, "Maximum latency =  %d ms \n", (pt_BisInfo->ta_USBearerChannel[i].ta_TpsTcType[j].s_MaxLatency));
            if((pt_BisInfo->ta_USBearerChannel[i].ta_TpsTcType[j].uc_MaxBER) == 0)
               fprintf(trail_fid, "Maximum BER =  10^(-3)\n");
            else if((pt_BisInfo->ta_USBearerChannel[i].ta_TpsTcType[j].uc_MaxBER) == 0x1)
               fprintf(trail_fid, "Maximum BER =  10^(-5)\n");
            else if((pt_BisInfo->ta_USBearerChannel[i].ta_TpsTcType[j].uc_MaxBER) == 0x2)
               fprintf(trail_fid, "Maximum BER =  10^(-7)\n");

            tempValue = pt_BisInfo->ta_USBearerChannel[i].ta_TpsTcType[j].s_IMAxINP & 0xFF;
            if (tempValue == 0)
               fprintf(trail_fid, "Minimum INP = 0\n");
            else if (tempValue == 1)
               fprintf(trail_fid, "Minimum INP = 1/2\n");
            else if (tempValue == 2)
               fprintf(trail_fid, "Minimum INP = 1\n");
            else if (tempValue == 3)
               fprintf(trail_fid, "Minimum INP = 2\n");
            else if (tempValue == 7)
               fprintf(trail_fid, "Minimum INP = 4\n");
            else if (tempValue == 11)
               fprintf(trail_fid, "Minimum INP = 8\n");
            else if (tempValue == 15)
               fprintf(trail_fid, "Minimum INP = 16\n");


            if (uc_Type == TPSTC_FUNC_ATM_US)
            {
               tempValue = (pt_BisInfo->ta_USBearerChannel[i].ta_TpsTcType[j].s_IMAxINP >>8) & 0x01;
               fprintf(trail_fid, "IMA Flag = %d\n", tempValue);
            }
         }
      } /* for(j=0; j<MAX_TPS_TC_TYPE; j++) */

      /* DS PMS-TC info */
      if(pt_BisInfo->uc_SI_SPar2[2*(i+1)+1] & LATENCY_PATH_DN)
      {
         fprintf(trail_fid, "\nFor DS Latency Path #%d\n", i);
         fprintf(trail_fid, "Maximum data rate = %d (times %d bps)\n", pt_BisInfo->ta_DSLatencyPath[i].s_MaxDataRate, uc_multiplier*4000);
         if (i==0) {    //Extended framing
            fprintf(trail_fid, "Min S = 1/%d \n", pt_BisInfo->ta_DSLatencyPath[i].s_OneOverS0min);
            fprintf(trail_fid, "Higher D support (bit[13]=511, [12]=480,...[8]=352,...[4]=224,...[0]=96) = 0x%x\n", pt_BisInfo->ta_DSLatencyPath[i].s_D0Mask);
         }
         else {         //Non-mandatory R/D for fast LP's
            fprintf(trail_fid, "Maximum R = %d\n", 2*(pt_BisInfo->ta_DSLatencyPath[i].s_MaxRandD >> 8));
            fprintf(trail_fid, "Maximum D = %d\n", (1<<(pt_BisInfo->ta_DSLatencyPath[i].s_MaxRandD & 0xFF)));
         }
      }

      /* US PMS-TC info */
      if(pt_BisInfo->uc_SI_SPar2[2*(i+1)+1] & LATENCY_PATH_UP)
      {
         fprintf(trail_fid, "\nFor US Latency Path #%d\n", i);
         fprintf(trail_fid, "Maximum data rate = %d (times %d bps)\n", pt_BisInfo->ta_USLatencyPath[i].s_MaxDataRate, uc_multiplier*4000);
         if (i>0) {
            fprintf(trail_fid, "Maximum R = %d\n", 2*(pt_BisInfo->ta_USLatencyPath[i].s_MaxRandD >> 8));
            fprintf(trail_fid, "Maximum D = %d\n", (1<<(pt_BisInfo->ta_USLatencyPath[i].s_MaxRandD & 0xFF)));
         }
      }
   } /* for (i=0; i<4; i++) // BC's/ LP's */
}
