/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2003 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*****************************************************************************
;
;   Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
;   Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;   decimal_gain.c
;
;  dB-to-linear conversion
;
;****************************************************************************/

#include "common.h"
#include "snr.h"
#include "dsp_op.h"
#include "bitload_const.h"
#include "mul.h"

/*^^^
*-------------------------------------------------------------------
*
*  int16 DecimalGain(int16 s_gain_db);
*
*  Description:
*      Computes the decimal equivalent of the fine gain in 3.9 format,
*     left-justified, i.e. least significant 4 bits are zeroes.
*      Assumes the the input gain_db is in 8.8 format and is within the
*      range [-14.0 dB, +6 dB].
*      Returns 10^(gain_db/20) in 3.13 format (i.e. 8192 is eqv to 1.0)
*
*      Algorithm:
*          1) Map input x to range [256, 768] (=[+1.0 dB, 3.0 dB])
*
*          2) Approximates 10^(x/20) in the desired range by
*             p(x) = 0.00104309497*x^2 + 3.57736185790*x + 8208.07332753051
*
*          3) Adjust approximation if original input x was not in
*             range [256, 768]
*
*  Input Arguments:
*     s_gain_db -- file gain in dB (Q8.8 format)
*
*  Return:
*     decimal gain (in Q3.13 format)
*
*-------------------------------------------------------------------
*^^^
*/
C_SCOPE int16 DecimalGain(int16 s_gain_db)
{
   int32 s_adj_factor = (1 << 14);   // nominal adjustment factor
   int32 l_Acc, l_temp;
   int16 s_DecimalValue;

   if (s_gain_db == NEG_INFINITY_DB)
      l_Acc = 0;
    // Check that input is within allowed range of [-14.0dB, +6dB].

   else if (s_gain_db < -14<<8)
   { // if x < -14.0 dB
      l_Acc = (int32) 1634;                  // return 0.1995*8192
   }
   else if (s_gain_db > 6<<8)
   {
      l_Acc = (int32) 16345;                 // return 1.9952*8192

   }
   else
   {

      if (s_gain_db > 3<<8)
      {
         while (s_gain_db > 3<<8)
         {
            s_gain_db -= 2<<8;
            // for every 2dB, we adjust the final result by 10^(2/20) = (20626/16384)
            MULS16(l_temp, 20626, s_adj_factor);
            s_adj_factor = (int16)(l_temp >> 14);
         }
      }

      else if (s_gain_db < 256)
      {
         // Map input to range [+1, +3], where the approximation is optimized for
         while (s_gain_db < 256) {
            s_gain_db += 2<<8;
            // for every 2dB, we adjust the final result by 10^(-2/20) = (13014/16384)
            MULS16(l_temp, 13014, s_adj_factor);
            s_adj_factor = (int16)(l_temp >> 14);
         }
      }

      // The coefficients c0, c1 and c2 are represented as mantissa
      // plus exponent pairs to get the maximum dynamic range. Hence
      // the rightshifts after the multiplications. c0 is a constant and
      // can be greater than 16 bits.

      MULS16(l_Acc, s_gain_db, s_gain_db);

      l_Acc  = l_Acc  >> 5;

      MULS32x16(l_temp, l_Acc, 17500);
      l_Acc  = l_temp >> 6;            // c2*x*x
      MULS16(  l_temp, 29306,s_gain_db);          // c2*x*x + c1*x
      l_Acc += l_temp;

      l_Acc +=  67240537;                         // c2*x*x + c1*x + c0
      l_Acc  = (l_Acc + (1 << 12)) >> 13;         // round result

      // Adjust and round in case original input was not in range [+1, +3]
      MULS32x16(l_temp, l_Acc, s_adj_factor);
      l_Acc  = (l_temp + (1 << 13)) >> 14;

   }

   s_DecimalValue = (int16) l_Acc;

   return (s_DecimalValue);
}

