/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 2003 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   minmaxmargin.c
*
*   Routines for finding tone with smallest, largest SNR margin.
*
*-------------------------------------------------------------------------
*/

// ******************************************************************
// minmaxmargin.c
//
// History
//
// 05/07/2012 Ram: Merged "ReTx" Segmented Bitswap Request code from ARX ADSL code base.
//   To cope with sweeping and fixed RFI, the following changes are done.
//   1. Below changes #2,#4,#5,#6 are under cmv control INFO 103 23 bitmask(0x0040). This bit is one (enabled) by default.
//   2. Modify the maximum number of bits for bitswap (gs_REDUCE_LP_DELTA_BITSWAP) from 20 to 120.
//   3. Increase Tx HDLC buffer size (MAX_TX_HDLC_MSG_SIZE, MAX_TX_HDLC_BUFFER_SIZE) from 700 to 1000.
//   4. Increase the maximum tone number (gs_MaxTonePerOvhdMsg) in one HDLC segment from 160 to 240.
//   5. Enable segmented bitswap request.
//   6. Change gs_DD_RxChannelsPerSymbol from 16 to 32 to improve SNR updating frequency.
//   7. Once marign on tones were ever dropped to below -6 dB, then bitswap is triggered when minimum margin is below 0 dB.
//      The threshold (-6dB by default) can be changed with cmv INFO 119 0.
//   8. Do not move bits to the tones whose margin were ever dropped to below -6 dB since these tones were very likely
//      to be affected by RFI before, and will be affected by RFI again. This mechanism is under cmv control
//      INFO 103 23 bitmask(0x0080). This bit is one (enabled) by default.
//   9. If bitswap is failed, checking of #7 condition will be ignored. Then bitswap will be tried again.
//   10. Include fix SMS01320806 IOP_A_BisPlus_CNXT_ImprovedBitSwap.
//   Grep for "XDSLRTFW-443: Feature_DS_BisPlus_ALL_SegmentedBitswapRequest"
// 08/11/2012 Anantha:  1. Changes to search tones,which have margin between min margin and min maring+delta,
//                   in descending order
//                2. Removing margin fudge while calculating margin to serch min or max margin
//             grep for XDSLRTFW-571:Enh_DS_ALL_ALL_BitSwap
// 20/02/2013 Mahesh:  Removed Ananta's bitswap enhancement changes for L2 & SRA
// ******************************************************************

#include "common.h"
#include "gdata.h"
#include "gdata_bis.h"
#include "bitload_support.h"
#include "bitload_const.h"
#include "snr.h"
#include "states.h"
#include "cmv.h"

/*^^^
*-----------------------------------------------------------------------------
*   Prototype: FlagT FindToneWithExtremeMargin  (int16 s_SearchType, uint8 *puca_RxBat, int16 *psa_RxFineGains,
*                          int16 *psa_SNRBuffer, RxToneFlags p_ActiveTones,
*                          int16 s_MinBitsPerTone, int16 s_MaxBitsPerTone, int16 *p_ch, int16 *ps_Margin)
*
*   Description:
*     Find the tone with the smallest or largest SNR margin.  There are two modes:
*
*        1) LARGEST_MARGIN - Search for the tone which, after adding a bit (or adding two in the case of a bi=0
*              tone for g.992.1/.2) has the smallest resulting SNR margin.
*
*        2) SMALLEST_MARGIN - Search for the tone which has the smallest SNR margin.  This tone is a good candidate
*              for bi reduction.
*
*     A one-bit-per-tone array mask indicates which tones will be considered in the comparison.  The tones considered
*     can also be restricted by specifying min and max bi values.
*
*     SNR margin is calculated using the input SNR and fine gains buffers.  Coding gain is also added to this margin.
*
*   Input Parameters:
*     s_SearchType -- specifies whether the search should be for the tone with the smallest or largest margin.
*              recognized values are LARGEST_MARGIN or SMALLEST_MARGIN (default).
*     puca_RxBat -- pointer to an Rx bit allocation table
*     psa_RxFineGains -- pointer to an Rx fine gain table
*     psa_SNRBuffer -- measured SNR buffer, not adjusted for coding gain or anything else.
*     p_ActiveTones -- One-bit-per-tone array indicating which tones in the BAT should be tested for margin.
*     s_MinBitsPerTone -- Only tones with current bi >= s_MinBitsPerTone will be tested for margin.
*     s_MaxBitsPerTone -- Only tones with current bi <= s_MaxBitsPerTone will be tested for margin.

*  Output Parameters:
*     p_ch -- Tone index of the tone with the smallest/largest margin.
*     ps_Margin -- Smallest/largest margin for this tone.
*
*  Return value:
*           SUCCEED: success (no error)
*           FAIL:    only possible if the set of tones to consider (according to p_ActiveTones, s_MinBitsPerTone and
*                    s_MaxBitsPerTone) is empty.
  *-----------------------------------------------------------------------------
^^^*/
C_SCOPE FlagT FindToneWithExtremeMargin   (int16 s_SearchType, uint8 *puca_RxBat, int16 *psa_RxFineGains,
                           int16 *psa_SNRBuffer, RxToneFlags p_ActiveTones,
                           int16 s_MinBitsPerTone, int16 s_MaxBitsPerTone, int16 *p_ch, int16 *ps_Margin)
{
   int16 i, ch, s_bi, s_bi_delta, s_Margin, s_ExtremeMargin;
   int16 s_bi_delta_init ;
   int16 s_ToneIndex = -1;
   int16 s_CodingGain;
   FlagT ft_FoundBetterChoice;

   if (s_SearchType == LARGEST_MARGIN) {
      s_ExtremeMargin = (int16) 0x8000;   // Largest negative number.
      // For this mode, bi_delta is the smallest allowed increase to bi.
      s_bi_delta_init = 1;
   }
   else { // SMALLEST_MARGIN
      s_ExtremeMargin = (int16) 0x7fff;   // Largest positive number.
      s_bi_delta_init = 0;
   }

   for (i=0 ; i < gs_RxNumTones ; i++) {
      ch = gsa_RxToneOrder[i];

      s_bi = puca_RxBat[ch];
      s_bi_delta = s_bi_delta_init ;

      // To use this function with g.992.1, we should define a common global
      // variable gs_RxMinBitsPerTone that is 2 for g.992.1 and 1 for ADSL2.
      if ((s_SearchType == LARGEST_MARGIN) && (s_bi == 0) && ((( gl_SelectedMode & (MODE_ADSL2)  ))== 0) )
         s_bi_delta = (int16)RX_MIN_BITS_PER_TONE;
      //XDSLRTFW-571:Enh_DS_ALL_ALL_BitSwap(start_end)
      //Changes to search tones,which have margin between min margin and min maring+delta,
      // in descending order
      if (IS_TONEFLAGSET(p_ActiveTones, ch) &&
         (s_bi >= s_MinBitsPerTone) &&
         (s_bi <= s_MaxBitsPerTone) &&
         (!gs_IncBitInBinReduced || (gs_IncBitInBinReduced && IS_TONEFLAGSET(guca_RxBitswapTones,ch)))){

         //XDSLRTFW-443: Feature_DS_BisPlus_ALL_SegmentedBitswapRequest (Start)
         //Do not move bits to the tones whose margin were ever dropped to below -6 dB since
         //these tones were very likely to be affected by RFI before, and will be affected
         //by RFI again. guca_TimeVaryNoise_Ind_sav is the array to indicate the tones whose
         //margin were ever dropped to below -6 dB.
         if ((gl_SelectedMode & (MODE_ADSL2)) &&
             (STATArray[STAT_OLRStatus_DS] == STAT_OLR_BITSWAP) &&
             (s_SearchType == LARGEST_MARGIN) &&
             (IS_TONEFLAGSET(guca_TimeVaryNoise_Ind_sav,ch)))
            continue;
         //XDSLRTFW-443: Feature_DS_BisPlus_ALL_SegmentedBitswapRequest (End)

         if (gft_ModemType == G_DMT) {
            if(i <= gs_MaxToneForFast)
               s_CodingGain = gsa_TotalCodingGain[FAST_DATA_PATH];
            else
               s_CodingGain = gsa_TotalCodingGain[INTERLEAVE_DATA_PATH];
         }
         else { // for g.992.3/g.992.5
            // Currently only supports single latency.
               s_CodingGain = gsa_TotalCodingGain[LP0_DATA_PATH];
         }

         s_Margin = psa_SNRBuffer[ch];

         // If there's any existing fine gain in reserve it could be used here to increase the
         // margin of a newly loaded tone, but that gets complicated.
         if (s_bi!=0)
            // For bi=0 tones, its fine gain either is negative infinity db or 0db, so ignore its fine gain here
            // avoid the possible overflow problem (for negative inifinity) and also fine for 0db case.
            s_Margin += psa_RxFineGains[ch];
         //XDSLRTFW-571:Enh_DS_ALL_ALL_BitSwap(start)
         //removing margin fudge
         if(!TESTArray[Test_DisableBitSwapImprvmnt] &&
            ((STATArray[STAT_MacroState] == STAT_ShowTimeTCSyncState) || (STATArray[STAT_MacroState] ==
STAT_ShowTimeState)) &&
            (gs_OlrReq != L2_REQ) && (gs_OlrReq != SRA_REQ))
         {
            s_Margin += (s_CodingGain - gsa_ConstellationSNR[s_bi+s_bi_delta]);
         }
         else
         {
            s_Margin += (s_CodingGain + OPTNArray[OPTN_MarginDelta] - gsa_ConstellationSNR[s_bi+s_bi_delta]);
         }
         //XDSLRTFW-571:Enh_DS_ALL_ALL_BitSwap(end)

         if (s_SearchType == LARGEST_MARGIN)
            ft_FoundBetterChoice = (s_Margin > s_ExtremeMargin);
         else // finding smallest margin
            ft_FoundBetterChoice = (s_Margin < s_ExtremeMargin);

         if (ft_FoundBetterChoice) {
            s_ExtremeMargin = s_Margin;
            s_ToneIndex = ch;
         }
      }
   }

   *p_ch = s_ToneIndex;
   *ps_Margin = s_ExtremeMargin;

   if (s_ToneIndex == -1)
      return (FAIL);
   else
      return (SUCCEED);

}

//XDSLRTFW-571:Enh_DS_ALL_ALL_BitSwap(start)
/*^^^
*-----------------------------------------------------------------------------
*   Prototype: void getBinsHavingMargnBtwenMinAndPlusDelata(uint8 *puca_RxBat,
                              int16 *psa_RxFineGains,
                              int16 *psa_SNRBuffer,
                              RxToneFlags p_ActiveTones,
                              int16 s_MinMarginBeforeSwap,
                              int16 s_Bitswap_Accept_Delta,
                              int16 s_MinBitsPerTone,
                              int16 s_MaxBitsPerTone )
*
*   Description:
*     Marks the tones which have margin between smallest margin and smallest margin + delat  There are two modes:
*     SNR margin is calculated using the input SNR and fine gains buffers.  Coding gain is also added to this margin.
*
*   Input Parameters:
*     puca_RxBat -- pointer to an Rx bit allocation table
*     psa_RxFineGains -- pointer to an Rx fine gain table
*     psa_SNRBuffer -- measured SNR buffer, not adjusted for coding gain or anything else.
*     p_ActiveTones -- One-bit-per-tone array indicating which tones in the BAT should be tested for margin.
*     s_MinBitsPerTone -- Only tones with current bi >= s_MinBitsPerTone will be tested for margin.
*     s_MaxBitsPerTone -- Only tones with current bi <= s_MaxBitsPerTone will be tested for margin.

*  Output Parameters:
*     Tones which have margin between minimum margin and minimum margin + delta are marked in
*       array p_ActiveTones
*
*  Return value: void
*-----------------------------------------------------------------------------
^^^*/

void getBinsHavingMargnBtwenMinAndPlusDelta(uint8 *puca_RxBat,
                              int16 *psa_RxFineGains,
                              int16 *psa_SNRBuffer,
                              RxToneFlags p_ActiveTones,
                              int16 s_MinMarginBeforeSwap,
                              int16 s_Bitswap_Accept_Delta,
                              int16 s_MinBitsPerTone,
                              int16 s_MaxBitsPerTone )
{
   int16 i, ch, s_bi, s_Margin;
   int16 s_ToneIndex = -1;
   int16 s_CodingGain;
   for (i=0 ; i < gs_RxNumTones ; i++) {
      ch = gsa_RxToneOrder[i];

      s_bi = puca_RxBat[ch];

      // To use this function with g.992.1, we should define a common global
      // variable gs_RxMinBitsPerTone that is 2 for g.992.1 and 1 for ADSL2.
      if (IS_TONEFLAGSET(p_ActiveTones, ch) &&
         (s_bi >= s_MinBitsPerTone) &&
         (s_bi <= s_MaxBitsPerTone)
         ) {

         //XDSLRTFW-443: Feature_DS_BisPlus_ALL_SegmentedBitswapRequest (Start)
         //Do not move bits to the tones whose margin were ever dropped to below -6 dB since
         //these tones were very likely to be affected by RFI before, and will be affected
         //by RFI again. guca_TimeVaryNoise_Ind_sav is the array to indicate the tones whose
         //margin were ever dropped to below -6 dB.
         if ((gl_SelectedMode & (MODE_ADSL2)) &&
             (IS_TONEFLAGSET(guca_TimeVaryNoise_Ind_sav,ch)))
            continue;
         //XDSLRTFW-443: Feature_DS_BisPlus_ALL_SegmentedBitswapRequest (End)

         if (gft_ModemType == G_DMT) {
            if(i <= gs_MaxToneForFast)
               s_CodingGain = gsa_TotalCodingGain[FAST_DATA_PATH];
            else
               s_CodingGain = gsa_TotalCodingGain[INTERLEAVE_DATA_PATH];
         }
         else { // for g.992.3/g.992.5
            // Currently only supports single latency.
               s_CodingGain = gsa_TotalCodingGain[LP0_DATA_PATH];
         }

         s_Margin = psa_SNRBuffer[ch];

         // If there's any existing fine gain in reserve it could be used here to increase the
         // margin of a newly loaded tone, but that gets complicated.
         if (s_bi!=0)
            // For bi=0 tones, its fine gain either is negative infinity db or 0db, so ignore its fine gain here
            // avoid the possible overflow problem (for negative inifinity) and also fine for 0db case.
            s_Margin += psa_RxFineGains[ch];
         s_Margin += (s_CodingGain - gsa_ConstellationSNR[s_bi]);
         if((s_Margin <= s_MinMarginBeforeSwap + s_Bitswap_Accept_Delta)) {
            SETTONEFLAG(guca_RxBitswapTones, ch);
         }
      }
   }
}

extern int16 gsa_dsPrevSNRMargnPerBand[];
extern int16 gsa_dsPresentSNRMargnPerBand[];
extern int16 gs_lastLoadedBin;
void getPerBandAvgMargin(uint8 *puca_RxBat,
                  int16 *psa_RxFineGains,
                  int16 *psa_SNRBuffer,
                  RxToneFlags p_ActiveTones)
{
   int16 i, s_bi, s_Margin;
   int16 s_CodingGain;
   int16 s_index;
   int32 l_MargnSum;
   int16 s_RxFirstChannel, s_RxLastChannel, s_numTones;

   s_RxFirstChannel = gs_RxFirstChannel;
   s_index = 0;
   while(1) {
      s_RxLastChannel = s_RxFirstChannel + gs_DD_RxChannelsPerSymbol - 1;
      if(s_RxLastChannel>gs_RxLastChannel) {
         s_RxLastChannel = gs_RxLastChannel;
      }
      s_numTones = 0;
      l_MargnSum = 0;
      for(i=s_RxFirstChannel; i <= s_RxLastChannel; i++) {
         if (gft_ModemType == G_DMT) {
            if(i <= gs_MaxToneForFast)
               s_CodingGain = gsa_TotalCodingGain[FAST_DATA_PATH];
            else
               s_CodingGain = gsa_TotalCodingGain[INTERLEAVE_DATA_PATH];
         } else { // for g.992.3/g.992.5
            // Currently only supports single latency.
            s_CodingGain = gsa_TotalCodingGain[LP0_DATA_PATH];
         }
         s_bi = puca_RxBat[i];
         if (IS_TONEFLAGSET(p_ActiveTones, i) && (s_bi)) {
            gs_lastLoadedBin = i;
            s_numTones++;
            s_Margin = psa_SNRBuffer[i];
            s_Margin += psa_RxFineGains[i];
            s_Margin += (s_CodingGain - gsa_ConstellationSNR[s_bi]);
            l_MargnSum += s_Margin;
         }
      }
      gsa_dsPresentSNRMargnPerBand[s_index++]=l_MargnSum/s_numTones;
      s_RxFirstChannel += gs_DD_RxChannelsPerSymbol; // reset channels
      if(s_RxFirstChannel > gs_RxLastChannel) {
         break;
      }
   }
}
//XDSLRTFW-571:Enh_DS_ALL_ALL_BitSwap(end)
