/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2003 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   dec_gain.c
 *
 *   Function that computes the combined decoder gain for each channel.
 *
 *------------------------------------------------------------------------
 */

#include "common.h"
#include "dsp_op2.h"
#include "gdata.h"
#include "gdata_bis.h"
#include "snr.h"
#include "decimalgain.h"
#include "bitload_const.h"


/*^^^
*-------------------------------------------------------------------
*
*   Prototype:
*               void CalcDecodGain(uint8 *psa_BAT, int16 *psa_DivisorGains, int16 *psa_QuotientGains, int16 s_ch, int16 s_NumTones, FlagT ft_CalcFineGain, RxToneFlags p_ToneMask)
*
*       Description:
*               This function is used in two different ways:
*                       1) If the ft_CalcFineGain flag is 0, it is used to calculate the Rx combined gains from the given fine gains and BAT:
*                               Set inputs to DivisorGain = FineGain, QuotientGain = CombinedGain and result is:
*
*                                       CombinedGain[i] = ConstellationGain[i] / FineGain[i]
*
*                       2) If the ft_CalcFineGain flag is 1, it is used to calculate the Rx fine gains from the given combined gains and BAT:
*                               Set inputs to DivisorGain = CombinedGain, QuotientGain = FineGain and result is:
*
*                                       FineGain[i] = ConstellationGain[i] / CombinedGain[i]
*
*               For use 1), the input fine gain should be in 8.8 dB format.  The output combined gain will be in 3.13 linear format.  The input values
*                       are converted to linear format then are rounded to 12-bit values before performing the division.
*               For use 2), the input combined gain should be in 3.13 linear format.  The output fine gain will also be in 3.13 linear format.
*
*       Input Arguments:
*               psa_BAT -- pointer to the bit allocation table for DS channel
*               psa_FineGains -- pointer to the fine gain table for DS channel. Format is 8.8 dB
*               ft_CalcFineGain -- Flag which indicates whether fine gains or combined gains are being calculated, i.e. use 1) or 2) above.
*               p_ToneMask -- Array of flags, one per tone, indicating for which bits the calculation should be done.  If NULL, the operation is performed
*                       for all tones.
*
*       Output Arguments:
*               psa_RxCombinedGains-- pointer to combined gains (fine gain * constellation decode gain)
*                                                       Format is 3.13 linear (0x2000 = 1.0)
*       Return:
*               none
*
*       NOTES:
*               1) The mode where ft_CalcFineGain == 1 is used in two cases for ADSL2: L2->L0 transition and after an aborted DS OLR.  In both these
*               2) In the case where ft_CalcFineGain == 0 and the combined gains are being updated because of an OLR, it is important to use the
*                       p_ToneMask to only update the combined gains table for those tones whose b[i] or g[i] are changing and therefore will have b[i] and
*                       g[i] sent to the far end as part of the OLR request.  Because of a loss of accuracy in the conversion from linear to dB and back to linear
*                       again, updating the combined gain for a tone that is not being changed by the OLR can result in a combined gain value that differs slightly
*                       from the far end's value.  For all other uses it is functionally OK to set p_ToneMask to NULL, although it could be used to eliminate unnecessary
*                       computations.
*
*       Global Variables:
*               gsa_QAMDecGainTab[] -- (I) QAM decoder constellation gain table
*               gsa_RxCombinedGains[] -- (O) Combined decoder gains
*
*-------------------------------------------------------------------
*^^^
*/
C_SCOPE void CalcDecodGain(uint8 *psa_BAT, int16 *psa_DivisorGains, int16 *psa_QuotientGains, int16 s_ch, int16 s_NumTones, FlagT ft_CalcFineGain, RxToneFlags p_ToneMask)
{
   int16 s_qc;
   int16 s_num_exp, s_rshift;
   int16 s_Divisor, s_Quotient, s_lshift_ConstGain, s_lshift_Divisor;
   int16 s_MinBitsPerTone = (int16) RX_MIN_BITS_PER_TONE;
   //XDSLRTFW-1727
   uint16 us_Saturate = FG_MINUS_2_5_dB;

   if (gft_Amd4_L2_Entry)
      us_Saturate = FG_MIN_VALUE;// 3643;

   if (gft_ModemType == G_DMT_BIS)
      s_MinBitsPerTone = gs_RxMinBitsPerTone_BIS_TCM;

   while(s_NumTones >0)
   {
       if ((p_ToneMask == NULL) || IS_TONEFLAGSET(p_ToneMask, s_ch))
       {
           s_Divisor = *psa_DivisorGains;
           s_Quotient = *psa_QuotientGains;

           /* Get the contellation size (bit loading) for this channel */
           s_qc = *psa_BAT;

           //=========================================================
           //Compute the combined gain = QAMDecGain/RxFineGain
           //=========================================================

           // Divide_16bit routine requires a normalized
           // 16-bit mantissa format, so left shift numerator and denominator as needed.

           s_lshift_ConstGain = norm_16bit(gsa_QAMDecGainTab[s_qc]);
           if (!ft_CalcFineGain)
           {
               // We are calculating combined gains from the 8.8 dB format fine gains.
               s_Divisor = DecimalGain(s_Divisor);

               // limit the gain value to -2.5 dB (.7498 * 8192) , except when it is '0', which is legal
            //XDSLRTFW-1727
               //@note:range of finegains linear -1543 to 10925 ,ie -14.5dB to +2.5dB
               if ((s_Divisor !=0) && (s_Divisor < us_Saturate))
                  s_Divisor = (int16)us_Saturate;

               // Use the same 12-bit fine gain value that will be sent to the far end.
               s_Divisor = (s_Divisor + (1<<3)) & 0xFFF0;

           }
           s_lshift_Divisor = norm_16bit(s_Divisor);

           if (s_Divisor == 0)
              s_Quotient = 0;
           else
           {
              Divide_16bit((int16)(gsa_QAMDecGainTab[s_qc] << s_lshift_ConstGain),//Numerator mantissa
                             (int16) -(13+s_lshift_ConstGain), // Numerator exponent
                             (int16) (s_Divisor << s_lshift_Divisor), // Denominator mantissa
                             (int16) -(13+s_lshift_Divisor),// Denominator exponent
                             &s_Quotient, &s_num_exp); //Quotient mantissa,exponent

               // If necessary, adjust RxCombinedGains value to have exponent of -13, i.e.
               // be in Q3.13 format.
               // It is an error here if s_num_exp+13 is > 0.
               s_rshift = -(s_num_exp+13);

               if (s_rshift > 0)
               // Round off before rightshifting.
                  s_Quotient = (s_Quotient + (1<<(s_rshift-1)) ) >> s_rshift;
            //XDSLRTFW-1727
               // Extend the inverse gain to full range
               if(s_rshift < 0)
                  s_Quotient = s_Quotient << (13+s_num_exp);

               if (ft_CalcFineGain)
               {
                  // Round and mask the output fine gain to most significant 12 bits.
                  s_Quotient = (s_Quotient + (1<<3)) & 0xFFF0;
               }
           }
           *psa_QuotientGains = s_Quotient;
       }
       psa_DivisorGains++;
       psa_QuotientGains++;
       psa_BAT++;
       s_ch++;
       s_NumTones--;
   }
}
