/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-1998 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;  Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
;  Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;  fdq_adap.c
;
;  This file contains routine of FDQ adaptation.
;
;***************************************************************************/

#include "common.h"
#include "fdq_init.h"
#include "gdata.h"
#include "dsp_op2.h"
#include "fdq_adap.h"


/*****************************************************************************
;  Prototype:
;     void FdqUpdate(int16 *psa_FDQ_coef, int8 *puc_FDQ_exp,
;                 int16 *psa_new_FDQ_coef, int16 s_new_FDQ_exp);
;
;   Description:
;     This subroutine updates the FDQ coefficient for a single channel
;     according to the following steps:
;
;     1) Compute the new FDQ coefficient as the product of the existing coefficient
;     and newly computed coefficient

;  Input Arguments:
;     (defined in the structure pt_FdqAdapt)
;     psa_FDQ_coef -- pointer to the mantissa of the existing FDQ coefficient (complex number)
;     puc_FDQ_exp -- pointer to the exponent of the existing FDQ coefficient
;     psa_new FDQ_coef -- pointer to the newly computed FDQ coefficient mantissa (complex number)
;     s_new_FDQ_exp -- the newly computed FDQ coefficient exponent
;
      Each FDQ_coef[i] is represented Wr[i], Wi[i] and Ws[i] in the form of (Wr[i] + jWi[i]) << Ws[i]
;     where Wr[i] and Wi[i] are mantissas and Ws[i] is their exponent.
;     The format of the FDQ coefficients Wr[i] and Wi[i] is
;           Q(FDQ_MANTISSA_WORDLENGTH - FDQ_MANTISSA_FRAC_BITS,FDQ_MANTISSA_FRAC_BITS).
;
;  Output Arguments:
;     psa_FDQ_coef -- pointer to the FDQ coefficient after updated
;     psa_FDQ_exp -- pointer to the FDQ exponent after updated
;
;  Return:
;     none
;
;  Global Variables:
;     none
;
;****************************************************************************/
C_SCOPE void FdqUpdate(int16 *psa_FDQ_coef, uint8 *puc_FDQ_exp, int16 *psa_new_FDQ_coef, int16 s_new_FDQ_exp)
{
   int16 s_Xr, s_Xi, s_Yr, s_Yi;
   int32 l_Acc_r, l_Acc_i;
   int16 s_shift;
   int16 s_Y_exp, s_X_exp;

   /* ========================================================================================= */
   /* First compute the product of existing FDQ coefficient and newly computed FDQ coefficient */
   /* ========================================================================================= */

   /* Get the real and imaginary part of the existing FDQ coefficient */
   s_Xr = psa_FDQ_coef[0];
   s_Xi = psa_FDQ_coef[1];
   s_X_exp = (int16)*puc_FDQ_exp;

   /* Get the real and imaginary part of the newly computed FDQ coefficient */
   s_Yr = psa_new_FDQ_coef[0];
   s_Yi = psa_new_FDQ_coef[1];

   /* Compute the true new FDQ coefficient = product of existing FDQ and newly computed FDQ */
   /* Format of l_Acc_r, l_Acc_i is Q(2*(FDQ_MANTISSA_WORDLENGTH - FDQ_MANTISSA_FRAC_BITS),2*FDQ_MANTISSA_FRAC_BITS) */
   /* where the exponent is (s_X_exp + s_new_FDQ_exp - FDQ_MANTISSA_FRAC_BITS) */
   l_Acc_r = (int32)s_Xr*s_Yr - (int32)s_Xi*s_Yi;     /* real part of the product */
   l_Acc_i = (int32)s_Xr*s_Yi + (int32)s_Xi*s_Yr;     /* imaginary part of the product */

   /* Update exponent of Yr and Yi */
   s_Y_exp = (s_X_exp + s_new_FDQ_exp - FDQ_MANTISSA_FRAC_BITS);

   // Now put final result into FDQ coefficient format.

   // Find smallest right shift value, s_shift, that guarantees mantissas fit in 16 lsbs,
   // and rightshift l_Acc_r, l_Acc_i by that value. This shifting increases the exponent
   // by s_shift.
   RightNormalizeSignedPair(&l_Acc_r, &l_Acc_i, FDQ_MANTISSA_WORDLENGTH, &s_shift);

   psa_FDQ_coef[0] = (int16) l_Acc_r;
   psa_FDQ_coef[1] = (int16) l_Acc_i;

   /* Update exponent of Yr and Yi */
   s_Y_exp += s_shift;

   if (s_Y_exp < 0)
   {
      /* If negative, force to zero and shift mantissas accordingly. */
      psa_FDQ_coef[0] >>= -s_Y_exp;
      psa_FDQ_coef[1] >>= -s_Y_exp;
      s_Y_exp = 0;
   }
   else if (s_Y_exp > FDQ_EXPONENT_MAX)
   {
      /* Saturate if necessary. */
      s_Y_exp = FDQ_EXPONENT_MAX;
   }

   /* Update exponent */
   *puc_FDQ_exp = (uint8)s_Y_exp;
}

