/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-1998 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;  Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
;  Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;  pll.c
;
;  This file contains routines to implement the phase lock loop (PLL).
;
;***************************************************************************/
// *********************************************************************************************************
// pll_misc.c
//
// History
//
//  2/12/2015 Sriram Shastry : Observed Quartz instabilities with  AVM  fritzbox[7490]. Added Detection algorithm for frequency drift and  phase drift
//  such that PLL adapt's to  fast setting  instead of  showtime setting .It helps to avoid showtime link drops in field
//
// Grep for : XDSLRTFW-2464
//
// *********************************************************************************************************
#include "common.h"
#include "dsp_op.h"
#include "pll.h"
#include "gdata.h"
#include "gdata_bis.h"
#include "rt_state.h"
#include "arctan.h"
#include "dsp_op2.h"
#include "data_alloc.h"
#include <stdlib.h>

/**********************************************************************
 * VARIABLES
 **********************************************************************/

// Pll Scaling Coefficients
int16 gs_Kp;               /* Kp parameter in the loop filter */
int16 gs_Ki;               /* Ki parameter in the loop filter */                          /* Exponent in Ki parameter */
int16 gs_Kp_Fast = 0x4000;    /* Fast Kp parameter in the loop filter */
int16 gs_Ki_Fast = 0x200;     /* Fast Ki parameter in the loop filter */
int16 gs_Kp_Slow = 0x1000;    /* Slow Kp parameter in the loop filter */
int16 gs_Ki_Slow = 0x20;        /* Slow Ki parameter in the loop filter */
int16 gs_Kp_Showtime = 0x800;
int16 gs_Ki_Showtime = 0x8;
/* XDSLRTFW-2464 */
int16 gs_Kp_Showtime_default = 0x800;
int16 gs_Ki_Showtime_default = 0x8;
int16 gs_MaxPhaseErrorThreshold_default = 8192;
int16 gs_Kp_Showtime_FallBack = 0x4000;
int16 gs_Ki_Showtime_FallBack = 0x200;
int16 gs_MaxPhaseErrorThreshold_Fallback = 3431;      // PI/8 - 24 deg
/* XDSLRTFW-2464 */
int16 gs_Kp_After_CECT = 0x1800;
int16 gs_Ki_After_CECT = 0;


// XDSLRTFW-2579 (Start)
int16 gs_PhaseErrorThresholdDuringMI = 28; // 0.2 degree
int16 gs_PhaseErrorThresholdDuringExtNoise = 100; // 0.7 degree
int16 gs_PhaseErrorThreshold = 286;   // 2 degree
// XDSLRTFW-2579 (End)


/* G992x specific */
uint8 guc_CommandedPllState = PLL_DISABLE;            /* if == PLL_DISABLE, PLL should not be run in GetRxTones() */
                                          /* if == PLL_RESET_REFERENCE_TONE, PLL reference tone should be reset */
                                          /* if == PLL_PERFORM_PHASE_LOCK, PLL should be run in GetRxTones() */

DATA_MAP_deILV2_BIS
int16 gsa_RotationX[8] = {32767, 23169, 0,    -23169, -32767, -23169, 0,      23169};
int16 gsa_RotationY[8] = {0,     23169, 32767, 23169, 0,      -23169, -32767, -23169};
DATA_MAP_END

/*****************************************************************************
;  Prototype:
;     void ResetPLL(int16 s_Kp, int16 s_Ki, s_PhaseErrThreshold);
;
;  Description:
;     This subroutine resets the loop filter coefficients of PLL.
;
;  Input Arguments:
;     s_Kp -- new value which Kp variable to be set to
;     s_Ki -- new value which Ki variable to be set to
;     s_PhaseErrThreshold -- new value for max phase error.
;
;  Output Arguments: none
;
;  Global Variables: none
;
;*************************************************************************************/

void ResetPLL(int16 s_Kp, int16 s_Ki, int16 s_PhaseErrThreshold)
{
   gs_Kp = s_Kp;
   gs_Ki = s_Ki;
   gs_MaxPhaseErrorThreshold = s_PhaseErrThreshold;

   //  XDSLRTFW-2464 PLL OPTIMIZSATION (StartEnd)
   gs_MaxPhaseErrorThreshold_default  = gs_MaxPhaseErrorThreshold;
}

/*****************************************************************************
;  Prototype:
;     void ScalePLL(int16 s_scaling);
;
;  Description:
;     This subroutine scales the loop filter coefficients of PLL.
;
;  Input Arguments:
;     s_scaling -- scaling factor in 2.14 format
;
;  Output Arguments: none
;
;  Global Variables: none
;
;*************************************************************************************/

void ScalePLL(int16 s_scaling)
{
   int32 l_Kp, l_Ki;

   l_Kp = (int32)s_scaling*gs_Kp;
   l_Kp >>= 14;
   l_Ki = (int32)s_scaling*gs_Ki;
   l_Ki >>= 14;

   if((l_Kp > 0) && (l_Kp < 0x7fff) &&
      (l_Ki > 0) && (l_Ki < 0x7fff)) {

      gs_Kp = (int16) l_Kp;
      gs_Ki = (int16) l_Ki;
   }
}

/*****************************************************************************
;  Prototype:
;     void ResetPllRefTone(int16 s_InTone_Real, int16 s_InTone_Imag);
;
;  Description:
;     This subroutine resets the reference tone for PLL
;
;  Input Arguments:
;     s_InTone_Real -- real part of the input reference tone
;     s_InTone_Imag -- imaginary part of the input reference tone
;
;  Output Arguments: none
;
;  Global Variables: none
;
;*************************************************************************************/

void ResetPllRefToneUtil(int16 s_InTone_Real, int16 s_InTone_Imag, int16 *ps_pllRefX, int16 *ps_pllRefY)
{
   int32 l_magnitude, l_temp;
    int q, i;

   l_temp       = abs((int32)s_InTone_Real);
   l_magnitude  = abs((int32)s_InTone_Imag);

    /* Approx. magnitude by max(|I|, |Q|) + (11/32)*min(|I|, |Q|) */
   if (l_magnitude > l_temp) {
        l_magnitude = l_magnitude + (11*l_temp >> 5);
   } else {
      l_magnitude = l_temp + (11*l_magnitude >> 5);
   }

   /* Long division: */
   q = 0;
   l_temp = 1;
   for (i=0; i<15; i++) {
      q <<= 1;
      l_temp <<= 1;
      if (l_temp >= l_magnitude) {
         l_temp -= l_magnitude;
         q += 1;
      }
   }

   // if magnitude is very high, q result can be 0.
   // in this case, set q to 1.
   if (q == 0){
      q=1;
   }
   /* Set reference tone to the input tone */
   *ps_pllRefX = (int16)(s_InTone_Real*q);
   *ps_pllRefY = (int16)(s_InTone_Imag*q);

}

void ResetPllRefTone(int16 s_InTone_Real, int16 s_InTone_Imag)
{
   ResetPllRefToneUtil(s_InTone_Real, s_InTone_Imag, &gsa_PllRefTone[0], &gsa_PllRefTone[1]);
}

void PllScalingCalc(int16 s_pilotTone, int16 *ps_scaling)
{
   int16 s_denom, s_denom_exp, s_num_exp, s_exp;
   int32 l_num;

   /* Compute scaling factor for Pll parameters for auxiliary pilot tone. */
   /* scaling factor = C_PILOT_TONE_64/gs_AuxPilotToneIdx                    */
   /* Note that for 32 < gs_AuxPilotToneIdx < 256 --> 1/4 < scaling factor < 2. */
   l_num = (int32)C_PILOT_TONE_64;
   s_num_exp = -norm_l(l_num);
   l_num <<= -s_num_exp;
   s_denom = s_pilotTone;
   s_denom_exp = -norm_l((int32) s_denom)+16;
   s_denom <<= -s_denom_exp;

   Divide_32by16bit(l_num, s_num_exp, s_denom, s_denom_exp, ps_scaling, &s_exp);

   /* Convert scaling to (2.14) format */
   if(s_exp < -14 )
      *ps_scaling >>= -s_exp-14;
   else
      *ps_scaling <<= s_exp+14;
}

/* For software stand-alone unit testing only: */
/* ------------------------------------------------------------------------ */
#ifdef COCOMO_UNIT_TEST

FILE *plltest_out_file=NULL;
#define PI                 (3.1415926535897932384626434)

void OpenFiles(int argc, char *argv[]) {

   /* Check number of arguments */
   if(argc != 2) {
      printf("Usage: plltest output_file \n\n");
      exit(1);
   }

   /* Create new output file */
   plltest_out_file = fopen(argv[1], "w");
   if(plltest_out_file == NULL) {
      printf("File Creation Error: %s.\n", argv[1]);
      exit(1);
   }
}

void CloseFiles(void) {

   fclose(plltest_out_file);
}

void TestPLL(void) {

   int16  i, t, s_PhaseError[100];
   double f, f_min, f_max, phase, prev_phase;
   double real, imag;
   int16 s_co_pilot[512];
   int16 gsa_PilotTone[2];
   int32 l_Acc0, l_Acc1;

   f_min = (8388608.0 / (4194304.0 + 32767.0))*(32.0/512.0);
   f_max = (8388608.0 / (4194304.0 - 32767.0))*(32.0/512.0);
   gs_Ki = 256*16;
   gs_Kp = 1024*16;


   gsa_PllRefTone[0] = -2000;
   gsa_PllRefTone[1] = 650;
    gs_MaxPhaseErrorThreshold = 25000;

   f  = 64.0/512.0;
   for (t = 0; t < 512; t++) {
      phase = 2.0*PI*f*(double)t;
      s_co_pilot[t] = (int16)(1024.0*cos(phase));
   }

   prev_phase = 0.0;
   for (i = 0; i < 100; i++) {
      real = 0.0;
      imag = 0.0;
      for (t = 0; t < 512; t++) {
         f  = 64.0/512.0;
         f += 0.5*(f_max-f_min)*(-gs_vcxo_input/(32768.0));
         phase = prev_phase+2.0*PI*f*(double)t;
         real += (double)s_co_pilot[t]*cos(phase);
         imag += (double)s_co_pilot[t]*sin(phase);
      }
      prev_phase = phase - 2.0*PI;
      gsa_PilotTone[0] = (int16)(real/64.0);
      gsa_PilotTone[1] = (int16)(imag/64.0);


      // RxGoertzel(gsa_PilotTone, frame, 256);

      /* Compute real part of (InPilotTone * complex conjugate of PllRefTone) */
      l_Acc0 = gsa_PllRefTone[0]*gsa_PilotTone[0];
      l_Acc0 += gsa_PllRefTone[1]*gsa_PilotTone[1];

      /* Compute imaginary part of (InPilotTone * complex conjugate of PllRefTone) */
      l_Acc1 = gsa_PilotTone[1]*gsa_PllRefTone[0];
      l_Acc1 -= gsa_PilotTone[0]*gsa_PllRefTone[1];

      gs_PhaseError[i] = FastAtan((int16)(l_Acc1 >> 16), (int16)(l_Acc0 >> 16));

      PLL(&(gsa_PilotTone[0]));

   }
}

void main(int argc, char *argv[]) {

   // ResetPllRefTone(2053, 701);
   // OpenFiles(argc, argv);
   TestPLL();
   // CloseFiles();
}

#endif
