/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-1998 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   bitload.c
*
*   Source code of the DSL Lite bitloading algorithms.
*
*-------------------------------------------------------------------------
*/
// ******************************************************************
// bitload.c
//
// History
//
//
//20/08/2010  Sriram Shastry:  With BRCM CO, 24HDSL noise and >=7Kft, cheat the coding gain a little
//                     grep for PERF_DS_BisPlus_BDCM_CGainCheating24HDSL
//
//20/08/2010  Sriram Shastry: The DS rate on midrange loops with 5T1 noise failes
//                    AT&T requirement against BRCM-based Alcatel EBLT-E
//                    DSLAMs in ADSL2+ mode. Adjust the coding gain
//                    in order to pass the rate requirement.
//                    grep for SMS00801609 PERF_DS_Plus_BDCM_CGainCheating5T1
//
// 22/04/2010 Nihar: Low DS performance on 3000m FB loop ILV in ADSL2plus mode vs. all DSLAMs
//                    margin verification tests shows that we are to pessimistic here
//                    - increase max coding gain by 0.5dB to 8dB in case of FBnoise
//                    Grep for PERF_DS_BIS_PLUS_ALL_LowDSPerfFBNoiseLongLoops
//
// 15/11/2011 ChihWen/Bhadra: There is a special requirement that CPE should sync up in 2km loop with -90 dBm/Hz AWGN (normally DS rate is 600 Kbps),
//                and should be able to reach max capped rate of 6 Mbps by upshift SRA when noise power is decreased to -140 dBm/Hz AWGN.
//            With this special requirement, we need to load more bits not only on already loaded tones, but also on monitored tones,
//                in order to reach the max capped rate. A problem behind this requirement is that CNXT CO does not send signal on unloaded tones.
//                So the workaround to fulfill the requirement for all CO are as below.
//                1. For all CO, changing the threshold for setting monitored tones from 4 dB to 6 dB (SNR_THRESH_LOAD_MIN_BITS).
//                2. For all CO, not limiting number of monitored tones to 20 if SRA is enabled on CO site.
//                3. For all CO, making OLR (SRA and bitswap) only occurring on loaded/monitored tones.
//                4. For CNXT CO, no updating FDQ cofficients on unloaded tones.
//                5. For CNXT CO, measuring noise on monitored tones to get an estimation of SNR.
//                6. For CNXT CO, SNR on monitored tones are capped to 2-bits required SNR, in order to load two bits and get better SNR measurement.
//
//                7. When DS rate capped low, it took bitloading too long time to
//               find gs_deltaLp_max and failed the training. This is the root cause
//               of no-link with 3.5M capped profiles.
//               The solution is to use a larger searching step to find a valid
//               gs_deltaLp_max faster.
//
//                8. For CNXT CO, decreasing the Tx buffer size for overhead message by 2 if the tone number in the last segment of SRA message was 14.
//                   This was because CRC/link drop was always seen when there was 14 tones in the last segment of SRA message.
//
//            Grep for ADSLRTFW-1413 ENH_DS_BisPlus_All_SRA_LargerStep_MonitoredTones

//
//15/03/2012 Kannan: 16/04/2009 Bhadra: Added code for bitload related changes in Universal Retransmission (G.INP)
//                   Added required variables for bitload.
//                     Grep for SMS01435874:FEATURE_ALL_BisPlus_ALL_Univ_Retx_BitLoad to see changes.
// 25/04/2012 Kannan:
//          1. ADSL DS ReTx feature implementation
//             Grep for "XDSLRTFW-443 Feature_DS_BisPlus_All_ReTx"
//
// 04/07/2012 Ram: Merged "ReTx" Bitswap code from ARX ADSL code base.
//                 Grep for "XDSLRTFW-443 FEATURE_DS_BisPlus_ALL_BitSwapReTx"
//
// 19/09/2013 Prashant: Merged fix for XDSLRTFW-1165 (clone of ADSLRTFW-1699)
//            Author of code changes:  Sriram Shastry
//       ADSL1 link drops at  2900M fixed rate case.It has been observed that CO calculates the Rxpower PSD  level  +  10log(gi^2)  dB+ 10log(ncdown2) 2nPCB dBm.
//            The aggregate  Txpower aggregate  transmit  power  exceeds 3.65  +  10log(ncdown2) 2nPCB  dBm,  where  ncdown2  is  the  number of these subcarriers
//            (i.e. with bi = 0 and gi > 0).Monitor tones energy (Gi>0 is where Bi =0 & Bi>0 ) used for calculating  total power. RmsGi limit is violated.
//               Grep for XDSLRTFW-1165 / ADSLRTFW-1699: IOP_ALL_ADSL1_BRCM_DS_LinkDrop_FR_1504_320Kbps
//
// ********************************************************************************************************************************************************
#include <string.h>
#include "common.h"
#include "dsp_op.h"
#include "dsp_op2.h"
#include "gdata.h"
#include "gdata_dmt.h"
#include "gdata_bis.h"
#include "bitload.h"
#include "bitload2.h"
#include "mp.h"
#include "cmv.h"
#include "exchdata.h"
#include "SelectDsRateOption.h"
#include "snr.h"
#include "cmv.h"
#include "bitload_support.h"
#include "bitload_const.h"
#include "pll.h"
#include "memsetbf.h"
#include "hndshk_Data.h"


/*^^^
*-----------------------------------------------------------------------------
*   Prototype:
*     void AddCodingGainToSnr(int16 *psa_SnrInBuf, int16 s_CodingGain, int16 *psa_SnrOutBuf)
*
*   Description:
*       Adds coding gain and noise separation SNR adjustment to input SNR.
*     In BIS mode, if the tones isnt in the DS MEDLEYset, we zero out the snr
*     so that we do not bitload on these tones
*
*   Input Parameters:
*     psa_SnrInBuf -- pointer to input SNR buffer
*     psa_MedleySnrInBuf -- pointer to input medley SNR buffer
*     psa_ReverbSnrInBuf -- pointer to input medley SNR buffer
*     s_CodingGain -- input coding gain
*
*  Output Parameters:
*     psa_SnrOutBuf -- pointer to output SNR buffer
*
*   Returns:
*     none
*
*  Global variables:
*
*-----------------------------------------------------------------------------
^^^*/

C_SCOPE void AddCodingGainToSnr(int16 *psa_SnrInBuf, int16 s_CodingGain, int16 *psa_SnrOutBuf, int16 s_path)
{

    int16 i;
    int32 l_Acc;

    for(i = gs_RxBitLoadFirstChannel; i<= gs_RxBitLoadLastChannel; i++)
    {
        // In BIS mode, if the tones isnt in the DS MEDLEYset we zero
        // out the snr so that we do not bitload on these tones
        if ((( gl_SelectedMode & (MODE_ADSL2)  )) &&
                (!IS_TONEFLAGSET(p_MEDLEYset_DS, i)))
        {
            psa_SnrOutBuf[i] = 0;
            continue;
        }

        /* only if we have dual latency support and we are working for LP0 path,
        assume that LP1 is already set and skip over the LP1 tones */
        if (  ((STATArray[STAT_Mode] & STAT_ConfigMode_ADSL2_ALL) != 0)&&(gt_rx_config.s_Nlp==2)&&(s_path == LP0_DATA_PATH) && (IS_TONEFLAGSET(p_TonesAllocatedtoLP1_DS,i)))
        {
            // if Trellis is ON, add TCM coding gain to these tones, fast path  R=0, D=1
            l_Acc = psa_SnrInBuf[i] + GetCodingGain_BIS(0, 1) ;
        }
        else

        l_Acc = psa_SnrInBuf[i] + s_CodingGain;

        psa_SnrOutBuf[i] = sature16(l_Acc);

    }

    /* In All Modes, set SNR for pilot tone to 0 so that we do not bitload unless we have Aux/DD Pilot On */
    /* In DMT Mode, always set SNR for pilot tone 64 to 0 so that we do not bitload*/
    if (gft_AuxPilot == AUX_PILOT_DISABLE ||
            !(( gl_SelectedMode & (MODE_ADSL2)  )))
    {
        psa_SnrOutBuf[gs_CPilotTone] = 0;
    }
}


/*^^^
*-----------------------------------------------------------------------------
*
*   Prototype:
*       FlagT AllocFixedRates(int16 s_GainsEnabled, int16 s_BitsToAllocate, int16 s_MaxBits, int16 *psa_RxBat, int16 s_RxExtraBits, int16 *pus_ncloaded);
*
*   Description:
*       This subroutine allocates exact "BitsToAllocate" number of bits to the channel
*     by trimming out the extra bits from the maximum bit allocaiton table.
*
*   Input Parameters:
*     s_GainsEnable -- indicator if fine gain is in use (1) or not (0)
*     s_BitsToAllocate -- number of bits to be allocated
*     s_MaxBits -- maximum number of bits that channel can support
*
*   Returns:
*       SUCCEED or FAIL
*
*  Global Variables:
*     gpsa_MeasuredSnrBuf[]   -- (I) pointer to measured buffer
*     gsa_SNRRequired[]    -- (I) required SNR per constellation
*     gs_RxExtraBits       -- (I) no. of extra bits due to use of fine gains
*     gsa_RxBat[]          --   (I/O) bit allocation table
*     gt_RMsgRA            --    (I) R_MSG_RA structure
*-----------------------------------------------------------------------------
^^^*/


/*^^^
*-----------------------------------------------------------------------------
*
*   Prototype:
*       int16 FindChWithSmallestMargin(int16 s_min_bits_per_tone);
*
*   Description:
*       Scans through those channels, which have bit-allocation greater or equal to
*     the specified minimum bits per tone, and find the channel with the
*       smallest margin.
*
*   Input Parameters:
*     s_min_bits_per_tone -- the min. bits per tone of the channels being searched
*
*   Returns:
*     Channel with the smallest SNR
*     or
*     -1: no channel is found
*
*  Global Variables:
*     gsa_RxBat[]    -- (I) bit allocation table
*     gpsa_MeasuredSnrBuf -- (I) pointer to computed SNR buffer
*     gsa_SNRRequired[] -- (O) Required SNR for the given gs_RxDesiredMargin
*-----------------------------------------------------------------------------
^^^*/



/*^^^
*-----------------------------------------------------------------------------
*
*   Prototype:
*       int16 FindChWithLargestMargin(int16 s_min_bits_per_tone);
*
*   Description:
*       Scans through those channels, which have bit-allocation greater or equal to
*     the specified minimum bits per tone, and find the channel with the
*       largest margin.
*
*   Input Parameters:
*     s_min_bits_per_tone -- the min. bits per tone of the channels being searched
*
*   Returns:
*     Channel with the largest SNR
*     or
*     -1: no channel is found
*
*  Global Variables:
*     gsa_RxBat[]    -- (I) bit allocation table
*     gpsa_MeasuredSnrBuf -- (I) pointer to computed SNR buffer
*     gsa_SNRRequired[] -- (O) Required SNR for the given gs_RxDesiredMargin
*
*-----------------------------------------------------------------------------
^^^*/


/*^^^
*-----------------------------------------------------------------------------
*
*   Prototype:
*       int16 CalcAvgSnrMargin(int16 s_FirstTone, int16 s_LastTone, int16 *psa_SnrBuf
*                       int16 *psa_RxBat)
*
*   Description:
*       Compute average SNR margin for given frequency range
*
*   Input Parameters:
*     s_FirstTone -- the first tone number
*     s_LastTone -- the last tone number
*     psa_SnrBuf -- pointer to "first tone" entry of SNR buffer
*
*   Returns:
*     the average SNR margin in Q8.8 format
*
*  Global Variables:
*     gsa_ConstellationSNR[]  -- (I) buffer storing the required SNR
*
*-----------------------------------------------------------------------------
^^^*/

C_SCOPE void CalcAvgSnrMargin(int16 s_FirstTone, int16 s_LastTone, int16 *psa_SnrBuf, uint8 *psa_RxBat, int16 *psa_RxFineGains, int16 *ps_RxAvMargin, int16 *ps_RxMinMargin)
{
    int16 i, s_ch, s_Margin, s_RxChannelsAllocated, s_AvMargin;
    int32 l_TotalMargin;
    //XDSLRTFW-1564 / XDSLRTFW-1665 (start)
    int32 l_TotalMarginCeil = 0;
    int16 s_RxChannelsAllocatedCeil = 0;
    int32 l_TotalMargin_15bits = 0;
    int16 s_RxChannelsAllocated_15bits = 0;
    int16 s_AvMargin_15bits;
    //XDSLRTFW-1564 / XDSLRTFW-1665 (end)

    /*Compute the average margin (SNR[ch] - gsa_ConstellationSNR[ch])*/
    s_RxChannelsAllocated = 0;
    l_TotalMargin = 0;
    int16 s_CodingGain; //XDSLRTFW-443: FEATURE_DS_BisPlus_ALL_BitSwapReTx (Start-End)

    for(s_ch = s_FirstTone, i=0; s_ch <= s_LastTone; s_ch++,i++)
    {
        /* Skip for the DD pilot tone */
        if (s_ch == gs_AuxPilotToneIdx || s_ch == gs_CPilotTone)
        continue;

        //XDSLRTFW-443: FEATURE_DS_BisPlus_ALL_BitSwapReTx (Start)
        if((gft_Bitswap_UreTX == 1) &&
           ((STATArray[STAT_MacroState] == STAT_ShowTimeState) ||
            (STATArray[STAT_MacroState] ==STAT_ShowTimeTCSyncState)))
           s_CodingGain = gsa_TotalCodingGain[LP1_DATA_PATH];
        else   // Coding gain already considered during training.
           s_CodingGain = 0;

        s_Margin = ComputeMarginPerTone((psa_SnrBuf[i] + s_CodingGain),
                                        psa_RxBat[s_ch],
                                        psa_RxFineGains[s_ch],
                                        &s_RxChannelsAllocated);
        //XDSLRTFW-1564 / XDSLRTFW-1665 (start)
        if((psa_RxBat[s_ch] < RX_MAX_BITS_PER_TONE) &&(psa_RxBat[s_ch] > 0 ))
        {
           s_RxChannelsAllocatedCeil++;
           l_TotalMarginCeil += (int32)(s_Margin);
        }
        else if((psa_RxBat[s_ch] == RX_MAX_BITS_PER_TONE))
        {
           s_RxChannelsAllocated_15bits++;
           l_TotalMargin_15bits += (int32)(s_Margin);

        }
        //XDSLRTFW-1564 / XDSLRTFW-1665 (end)
        //XDSLRTFW-443: FEATURE_DS_BisPlus_ALL_BitSwapReTx (Start)

        /* Record the margin per tone during bitloading, as well as the min margin value and corresponding tone */
        /* SEF detection during showtime also reuses this function -- hence we need this check */
        if (STATArray[STAT_MacroState] != STAT_ShowTimeState)
        {
            if ((guca_RxBat[s_ch] > 0) && (s_Margin < *ps_RxMinMargin))
            {
                *ps_RxMinMargin = s_Margin; /* Least margin for a loaded tone */
                gs_RxMinMarginTone = s_ch;  /* Loaded tone with the least margin */
            }
        }

        l_TotalMargin += (int32)(s_Margin);
    }

    /*gs_RxAvMargin = TotalMargin/RxChannelsAllocated in Q8.8 */
    s_AvMargin = QuickAverage(l_TotalMargin, s_RxChannelsAllocated);
    //XDSLRTFW-1564 / XDSLRTFW-1665 (start)
    gs_RxAvMarginWithCap = QuickAverage(l_TotalMarginCeil, s_RxChannelsAllocatedCeil);
    s_AvMargin_15bits = QuickAverage(l_TotalMargin_15bits, s_RxChannelsAllocated_15bits);
    if(s_AvMargin_15bits < gs_RxAvMarginWithCap)
    {
      gs_RxAvMarginWithCap =  (s_AvMargin_15bits + gs_RxAvMarginWithCap)>>1;
    }
    //XDSLRTFW-1564 / XDSLRTFW-1665 (end)
    *ps_RxAvMargin = s_AvMargin;
}


/*^^^
*-----------------------------------------------------------------------------
*
*   Prototype:
*       int16 CalcMarginsAndGains(int16 s_GainsEnabled);
*
*   Description:
*       Compute actual margin and fine gains.
*
*   Input Parameters:
*     s_GainsEnabled -- indicator if fine gain is in use (1) or not (0)
*
*   Returns:
*     none
*
*  Global Variables:
*     gpsa_MeasuredSnrBuf[]   -- (I) pointer to computed SNR buffer
*     gsa_RxBat[]       -- (I) Bit Allocation Table (bits per tone)
*     gsa_RxFineGains[] -- (O) actual fine gains (non-dB value in Q3.13 format)
*     gs_RxMinMargin    -- (O) Minimum SNR margin (in Q8.8)
*     gs_RxAvMargin     -- (O) Average Margin (in Q8.8)
*     gs_RxAvFineGain      -- (O) Average fine gain (in Q8.8)
*
*-----------------------------------------------------------------------------
^^^*/
C_SCOPE void CalcMarginsAndGains(int16 s_GainsEnabled, uint8 *psa_RxBat, int16 *psa_RxFineGains, int16 *ps_RxAvMargin, int16 *ps_RxAvFineGain, int16 *ps_RxMinMargin) {

    int16 s_ch;
    int32 l_TotalMargin = 0;
    int32 l_TotalFineGain = 0 ;
    int16 s_RxChannelsAllocated = 0;
    int16 s_MinBitsPerTone = (int16) RX_MIN_BITS_PER_TONE;
    int16 s_SNR;
    int16 s_SNR_Thresh = (int16) SNR_THRESH_LOAD_MIN_BITS;
    int16   s_count_monitored_tones = 0;
   //XDSLRTFW-443 Feature_DS_BisPlus_All_ReTx  (START)
   //SMS01435874:FEATURE_ALL_BisPlus_ALL_Univ_Retx_BitLoad (Start)
   int16 s_numb_monitored_tones = NUM_MONITORED_TONES;
   if(gt_ReTxConfigInfo.ft_ReTxOn == 1)
        s_numb_monitored_tones = URETX_NUM_MONITORED_TONES;
   //SMS01435874:FEATURE_ALL_BisPlus_ALL_Univ_Retx_BitLoad (End)
   //XDSLRTFW-443 Feature_DS_BisPlus_All_ReTx  (END)
    if (( gl_SelectedMode & (MODE_ADSL2)  ))
    s_MinBitsPerTone = (int16) gs_RxMinBitsPerTone_BIS_TCM;


    /* Clear other unused channels */
    for (s_ch = 0; s_ch < gs_RxBitLoadFirstChannel; s_ch++)
    {
        psa_RxBat[s_ch] = 0;
        psa_RxFineGains[s_ch] = NEG_INFINITY_DB;
    }
    for(s_ch = gs_RxBitLoadLastChannel+1; s_ch < gs_RxNumTones; s_ch++)
    {
        psa_RxBat[s_ch] = 0;
        psa_RxFineGains[s_ch] = NEG_INFINITY_DB;
    }


    for(s_ch = gs_RxBitLoadFirstChannel; s_ch <= gs_RxBitLoadLastChannel; s_ch++)
    {
        if (psa_RxBat[s_ch] == 0)

        {
            /* Skip for the DD pilot tone */
            if (s_ch == gs_AuxPilotToneIdx || s_ch == gs_CPilotTone)
            continue;

            s_SNR =  *(gpsa_MeasuredSnrBuf + s_ch);

            // decrease fine gain to -infinity for the tones if there SNR is below a threshold
            // from being bitloadable

            if (s_SNR < gsa_SNRRequired[s_MinBitsPerTone] - s_SNR_Thresh)
            psa_RxFineGains[s_ch] = NEG_INFINITY_DB;

            // for low fixed rate, we may have lots of tones with good SNRs but bi =0;
            // keep a fixed number of tones for monitoring; and put fine gains to be - infinity
            // for the rest

            else
            {
               // In Auto SRA cases (UpShift SRA) conditions we may need more monitor tones
               // For examples where modems are trained with -90 dBm and noise levels are
               // gradually decreased to -140 dBm.
               //ADSLRTFW-1413 ENH_DS_BisPlus_All_SRA_LargerStep_MonitoredTones (START)
            // Not limiting number of monitored tones to 20 if SRA is enabled on CO side
         // XDSLRTFW-1165 / ADSLRTFW-1699: IOP_ALL_ADSL1_BRCM_DS_LinkDrop_FR_1504_320Kbps (Start_End)
               #ifndef ISDN
            if ((gft_AutoSRA_ErrorCondition == TRUE) || (gl_SelectedMode & (MODE_ADSL1)))
               #endif
               {
               //XDSLRTFW-443 Feature_DS_BisPlus_All_ReTx  (START_END)
               //SMS01435874:FEATURE_ALL_BisPlus_ALL_Univ_Retx_BitLoad (Start End)
               if (s_count_monitored_tones < s_numb_monitored_tones)
               s_count_monitored_tones ++;
               else
               psa_RxFineGains[s_ch] = NEG_INFINITY_DB;
               }
               //ADSLRTFW-1413 ENH_DS_BisPlus_All_SRA_LargerStep_MonitoredTones (END)

            }

        }
    } //s_ch


    /* compute the average margin and min margin */

    CalcAvgSnrMargin(gs_RxBitLoadFirstChannel, gs_RxBitLoadLastChannel, gpsa_MeasuredSnrBuf + gs_RxBitLoadFirstChannel, psa_RxBat, psa_RxFineGains, ps_RxAvMargin, ps_RxMinMargin);

    return;
}



/*^^^
*-----------------------------------------------------------------------------
*   Prototype:
*     itn16 GetCodingGain(int16 s_R)
*
*   Description:
*       Adds a coding gain to input SNR
*
*   Input Parameters:
*     s_R -- No. of check bytes per codeword
*
*  Output Parameters:
*
*   Returns:
*     Compute coding gain (in Q15.1 format)
*
*  Global variables:
*     gft_TcmFlag/gft_TcmFlag_bis_DS -- = TRUE: TCM is used, otherwise not used
*-----------------------------------------------------------------------------
^^^*/

C_SCOPE int16 GetCodingGain(int16 s_R)
{
    FlagT ft_TcmFlag;
    int16 s_CodingGain;

    if (( gl_SelectedMode & (MODE_ADSL2)  ))
    ft_TcmFlag =  gft_TcmFlag_bis_DS;
    else ft_TcmFlag = gft_TcmFlag;

    //Use the maximum coding gain (in Q15.1 format) in computing channel capacity
    s_CodingGain = guca_RSCodingGain[(s_R>>1)];

    //Add the coding gain from TCM in Q15.1 format
#if (TCM_OPTION == 1)
    if(ft_TcmFlag == TRUE)
    {

        // The rate options provided by certain CO for ilv path may give R=0, which is equivalent to fast path.
        // If R=0 is picked for ilv path, the trellis coding gain should be the same as fast path (5.5dB).
        if (s_R==0) gs_tcm_coding_gain=TCM_CODING_GAIN;
        s_CodingGain += (int16)gs_tcm_coding_gain;

        //Set the upper limit to the maximum coding gain
        if(s_CodingGain > gs_max_coding_gain)
        s_CodingGain = gs_max_coding_gain;

    } //if(ft_TcmFlag == TRUE)
#endif

    return(s_CodingGain);
}


/*^^^
*-----------------------------------------------------------------------------
*   Prototype:
*     int16 GetCodingGain_BIS(int16 s_R, int16 s_D)
*
*   Description:
*       Adds a coding gain to input SNR
*
*   Input Parameters:
*     s_R -- No. of check bytes per codeword
*     s_D -- Interleaver Depth

*  Output Parameters:
*
*   Returns:
*     Compute coding gain (in Q8.8 format)
*
*  Global variables:
*     gft_TcmFlag_bis_DS -- = TRUE: TCM is used, otherwise not used
*-----------------------------------------------------------------------------
^^^*/
C_SCOPE int16 GetCodingGain_BIS(int16 s_R, int16 s_D)
{
    FlagT ft_TcmFlag;
    int16 s_CodingGain;
    uint8 s_RbyTwo;

#ifndef ISDN   // Only for Anx-A
   // R3_P1: AR8_TF: PERF_DS_BisPlus_BDCM_CGainCheating24HDSL
   int16 s_Adjust_CodingGain = 0;
#endif // ifndef ISDN
   /* Reed Solomon Coding Gain BIS*/
   //used local array instead of guca_CodingGain_TCMOFF
   uint8 uca_CodingGain_TCMOFF[9] = {    0,
                              24,
                              48,
                              56,
                              64,
                              72,
                              80,
                              80,
                              88 }; /* in Q4.4 format */// R4:CodeCleanup - End


   uint8 uca_CodingGain_TCMON_adjust[9] = { 0, 20, 32, 32, 32, 32, 32, 32, 32 };

    if ((gl_SelectedMode & (MODE_ADSL2)))
    {
        ft_TcmFlag =  gft_TcmFlag_bis_DS;
        // PERF_DS_BIS_PLUS_ALL_LowDSPerfFBNoiseLongLoops (Start)
#ifdef ISDN
        if(OPTNArray[OPTN_AlgControl3] & CMV_TO_ENABLE_DTAG_US_PERF_DS_PERF)
        {
            //Loop      hsk_tone_power_dB
            //2600m     28ab
            //2700m     2730
            //2800m     2594
            //2900m     23fa
            if ((gs_hsk_tone_power_dB <= MINIMUM_GHS_TONE_PWR_AT_2700M) //Loops >=2750M
                    && (STATArray[STAT_Performance] & STAT_FBNoiseLowThreshold))
            gs_max_coding_gain = MAX_CODING_GAIN_80dB;


        }
        // PERF_DS_BIS_PLUS_ALL_LowDSPerfFBNoiseLongLoops (End)
#endif
    }
    else
    ft_TcmFlag = gft_TcmFlag;

    s_RbyTwo = s_R>>1;

    // Coding Gain should be a function of D as well, it may need to change

    //Use the maximum coding gain (in Q4.4 format) in computing channel capacity
    s_CodingGain = (int16)uca_CodingGain_TCMOFF[s_RbyTwo];  // R4:CodeCleanup


    //Add the coding gain from TCM in Q15.1 format
#if (TCM_OPTION == 1)
    if(ft_TcmFlag == TRUE)
    {
        // Assuming single latency only, need to readdress for dual latency case.
        if (gt_HandshakeBis.sa_DS_MaxLatencyBC[0] <= 2)
        {
            s_CodingGain -= (int16)uca_CodingGain_TCMON_adjust[s_RbyTwo];
            //if D<=16, decrease the extra coding gain beyond TCM further by 0.25dB
            if ((s_D <= 16) && (s_CodingGain >=4)) s_CodingGain -= 4;
        }

        s_CodingGain += (int16)(gs_tcm_coding_gain<< 3);
   //XDSLRTFW-443 Feature_DS_BisPlus_All_ReTx  (START)
      //SMS01435874:FEATURE_ALL_BisPlus_ALL_Univ_Retx_BitLoad (Start)
      if(gt_ReTxConfigInfo.ft_ReTxOn == 1) // ReTx; 3=Interleaved + ReTx: In that case botch Coding Gains sum up
      {
         s_CodingGain += (int16)gt_ReTxConfigInfo.us_ReTxCodingGain; // Take care of the format!
         //XDSLRTFW-2066 (start)
         if((gt_INFX_CMV.us_SRA_IOP_Bits & CMV_FT_ReTx_CG_Performance)
            &&(gt_ReTxConfigInfo.us_INP_min == 0 )
            &&(gt_ReTxConfigInfo.us_SHINEratio == 0)
            &&(gt_ReTxConfigInfo.us_INP_min_rein == 0)
            && (gs_hsk_tone_power_dB < 10500)
            && (STATArray[STAT_Performance] & STAT_M140WhiteNoise)
           )
               s_CodingGain += 8;  //0.5 dB extra for mid to long loops
            //XDSLRTFW-2066 (end)
      }
      //SMS01435874:FEATURE_ALL_BisPlus_ALL_Univ_Retx_BitLoad (End)
   //XDSLRTFW-443 Feature_DS_BisPlus_All_ReTx  (END)
        //Set the upper limit to the maximum coding gain
        if(s_CodingGain > (gs_max_coding_gain<<3))
        s_CodingGain = (gs_max_coding_gain << 3);

    } //if(ft_TcmFlag == TRUE)
#endif

   // R3_P1: AR8_TF: PERF_DS_BisPlus_BDCM_CGainCheating24HDSL (Start)
#ifndef ISDN   // Only for Anx-A
   // If the CO is Broadcom, there is 24-HDSL noise, and the loop is > 6Kft
   // we will cheat our coding gain by 0.75 dB
   if ((gs_CurrentCoChipset == BDCM_CO_CHIPSET) && (gft_24HDSLNoise) &&
       (gs_PGAforLoopLengthEstimation > PGA_THRESHOLD_6_5_KFT1))
   {
      s_Adjust_CodingGain = 0xC0;
   }
   //SMS00801609 PERF_DS_Plus_BDCM_CGainCheating5T1 (START)
   // If the CO is Broadcom, there is 5T1 noise, and the loop range is
   // from 3kft to 5kft, we will cheat our coding gain by 1.25 dB to
   // pass ATT test requirement 4-111e
   // This is applied to ADSL2+ with 1ms latency to limit the impact
   if ((gs_CurrentCoChipset == BDCM_CO_CHIPSET) && (gft_5T1Noise) &&
       (gl_SelectedMode & (MODE_G992_5)) &&
       (gt_HandshakeBis.sa_DS_MaxLatencyBC[LP0_DATA_PATH] == 1) &&
       (gs_PGAforLoopLengthEstimation >= PGA_THRESHOLD_3_KFT) &&
       (gs_PGAforLoopLengthEstimation <= PGA_THRESHOLD_5_KFT))
   {
      s_Adjust_CodingGain = 0x0140;
   }
   //SMS00801609 PERF_DS_Plus_BDCM_CGainCheating5T1 (END)
   return ((s_CodingGain << 4) + s_Adjust_CodingGain);
#else // Only for Anx-B
    return(s_CodingGain<<4);
#endif // ifndef ISDN
   // R3_P1: AR8_TF: PERF_DS_BisPlus_BDCM_CGainCheating24HDSL (End)
}


