/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-1998 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;  Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413
;  Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;
;  File Name: rx_aoc.c
;
;  This file contains functions for RX side of AOC operations.
;
;***************************************************************************/

#include <string.h>
#include "common.h"
#include "aoc.h"
#include "tx_aoc.h"
#include "rx_aoc.h"
#include "gdata.h"
#include "mp.h"
#include "cmv.h"
#include "bitswap.h"
#include "prf_man.h"
#include "rx_ib.h"
#include "trail.h"

/*****************************************************************************
;  Subroutine Name: RxMsgDecod(...)
;
;  This subroutine decodes the message from received AOC bytes.
;
;  Prototype:
;     uint8 RxMsgDecod(uint8 uc_byte, AocMsgStruc_t *pt_RxMsg)
;
;  Input Arguments:
;     uc_byte -- current AOC input byte
;
;  Output Arguments:
;     pt_RxMsg -- decoded message, with the following parameter set:
;        uc_msg_length -- (O) message length
;        uc_msg_body[] -- (O) message data
;
;  Return:
;     message ID
;
;   Global Variables:
;     guca_AOCRxBuffer
;     gpuc_AOCRxGetPtr
;     gpuc_AOCRxPutPtr
;     gs_AOCRxDecoderState
;     gs_AOCRxMsgId
;     gs_AOCRxMsgLength
;     gs_AOCRxBlockLength
;
*******************************************************************************/

FlagT gft_StartCountStuffByte = FALSE;  // Flag to indicate whether we should start counting stuffing byte
FlagT gft_EndMsgBlock = FALSE;         // Flag to indicate whether we have reached the msg block end
int16 gs_AOCStuffByteCnt = 0;       // Stuffing byte counter

uint8 RxMsgDecod(uint8 uc_byte, AocMsgStruc_t *pt_RxMsg)
{
   uint8 uc_msg_id, *puc_idx;

   int16 i;

   uc_msg_id = (uint8)AOC_MSG_ID_UNDEFINED;

   /* Write new byte to AOC Rx buffer. */
   *gpuc_AOCRxPutPtr++ = uc_byte;

   // count stuffing byte
   if (gft_StartCountStuffByte)
   {
      if (uc_byte == AOC_STUFFING_BYTE)
         gs_AOCStuffByteCnt++;
      else // there is possible we have 0x00 in the msg, if in this case, we should reset counter.
         gs_AOCStuffByteCnt = 0;
   }
   // stop count stuffing byte once we get consecutive 20 stuffing bytes and claim we have reached the end of msg block
   if (gs_AOCStuffByteCnt >= 20)
      gft_EndMsgBlock = TRUE;

   switch(gs_AOCRxDecoderState) {

      case AOC_SEARCH_FOR_MESSAGE:

         /* Search data in buffer until a valid message is found. */
         for(; gpuc_AOCRxGetPtr < gpuc_AOCRxPutPtr-1; gpuc_AOCRxGetPtr++) {

            /* Identify a message based on the first two bytes. */
            AocMsgId(*gpuc_AOCRxGetPtr, *(gpuc_AOCRxGetPtr+1), &gs_AOCRxMsgId, &gs_AOCRxMsgLength);

            /* Check if valid message was found. */
            if(gs_AOCRxMsgId != (int16) AOC_MSG_ID_UNDEFINED) {

               // Trigger stuffing byte count once we get a new msg
               gft_StartCountStuffByte = TRUE;

               /* Set block length equal to five times the message length. */
               gs_AOCRxBlockLength = (gs_AOCRxMsgLength << 2) + gs_AOCRxMsgLength;

               /* Go to next state. */
               gs_AOCRxDecoderState = AOC_COMPARE_MESSAGES;
               break;
            }
         }

         /* Realign data in buffer (so that it starts at the beginning of the buffer) */
         if(gpuc_AOCRxGetPtr != guca_AOCRxBuffer) {

            puc_idx=guca_AOCRxBuffer;
            while(gpuc_AOCRxGetPtr < gpuc_AOCRxPutPtr)
               *puc_idx++ = *gpuc_AOCRxGetPtr++;
            gpuc_AOCRxPutPtr = puc_idx;
            gpuc_AOCRxGetPtr = guca_AOCRxBuffer;
         }
         break;

      case AOC_COMPARE_MESSAGES:

         // Wait until we have collected a data block equal in length to five
         // times the message length
         if((gpuc_AOCRxPutPtr-gpuc_AOCRxGetPtr >= gs_AOCRxBlockLength) || (gft_EndMsgBlock)) {

            gft_EndMsgBlock = FALSE;

            // turn off stuffing byte counting once we claim we reach the end of msg block
            gft_StartCountStuffByte = FALSE;

            gs_AOCStuffByteCnt = 0;

            /* Check if the message was transmitted the required number of times. */
            if(AocMsgMatch(guca_AOCRxBuffer, (uint8) gs_AOCRxMsgLength) == SUCCEED) {  /* match found */

               /* Copy the current message information to the output structure. */
               pt_RxMsg->uc_msg_length = (uint8) gs_AOCRxMsgLength;
               memcpy(pt_RxMsg->uca_msg_body, guca_AOCRxBuffer, gs_AOCRxMsgLength);

               /* Return message ID. */
               uc_msg_id = (uint8) gs_AOCRxMsgId;

               /* Skip block of five messages. */
               gpuc_AOCRxGetPtr = gpuc_AOCRxPutPtr;

               // clean the AOCMsgBuffer for the next msg
               for (i = 0; i < (int16) (gpuc_AOCRxPutPtr-gpuc_AOCRxGetPtr); i++)
                  guca_AOCRxBuffer[i] = 0;

#ifdef COLLECT_SHOWTIME_EVENTS_INFO
   if ((gs_RxShowtimeEventsInfoEnable & 0x2) != 0)
   {
      if (gs_RxShowtimeEventsInfoBufferIndex < (2*MAX_STATES - 20))
      {
         gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) 0xBBBB;          /* delimiter indicating aoc information */
         gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) gl_RxSymbolCount;   /* rx symbol count */
         gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) gs_AOCRxMsgId;      /* aoc message id */
         gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) gs_AOCRxMsgLength;     /* aoc message length */
         for (i=0; i<gs_AOCRxMsgLength; i++)
            gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) guca_AOCRxBuffer[i];   /* aoc message, variable length */
      }
      else
      {
         gs_RxShowtimeEventsInfoBufferIndex = gs_InitRxShowtimeEventsInfoBufferIndex;
      }
      if (gs_febe_cnt > gs_RxShowtimeEventsError_threshold)
         gs_InitRxShowtimeEventsInfoBufferIndex = 2*MAX_STATES;
   }
#endif //#ifdef COLLECT_SHOWTIME_EVENTS_INFO

            }
            else
               /* Move to next AOC byte in buffer.*/
               gpuc_AOCRxGetPtr++;

            /* Go back to searching for a valid message. */
            gs_AOCRxDecoderState = AOC_SEARCH_FOR_MESSAGE;
         }
         break;
   }

   return(uc_msg_id);
}

/*****************************************************************************
;  Subroutine Name: AocMsgMatch(...)
;
;  This subroutine checks to see if there are two other messages
;  match to the first message in the input message buffer. It returns
;  SUCCEED or FAIL depending on whether match is found or not.
;
;  Prototype:
;     uint8 AocMsgMatch(uint8 *puca_msg_buf, uint8 uc_msg_length)
;
;  Input Arguments:
;     puca_msg_buf   -- pointer to the input message buffer
;     uc_msg_length  -- length of the first message in the input buffer
;
;  Output Arguments:
;     none
;
;  Return:
;     SUCCEED  -- if three messages including the first message are the same
;     FAIL  -- if the above is not true
;
;  Global Variables:
;
*******************************************************************************/

uint8 AocMsgMatch(uint8 *puca_msg_buf, uint8 uc_msg_length)
{

   int16 i, j, s_num_of_matches, s_count;

   s_num_of_matches = 0;

   /* Check to see if there are other two messages in the last four messages which */
   /* match the first message */
   for(j=uc_msg_length; j<AOC_MSG_REPEAT_CNT*uc_msg_length; j += uc_msg_length) {

      s_count = 0;
      for(i=0; i<uc_msg_length; i++) {
         if(puca_msg_buf[i] == puca_msg_buf[j+i])
            s_count++;
         else
            break;
      }

      /* if match is found, increase the count */
      if(s_count == uc_msg_length) {
         s_num_of_matches++;

         if(s_num_of_matches == 2)
            return(SUCCEED);
      }
   } /* for(j= */

   return(FAIL);


}

/*****************************************************************************
;  Subroutine Name: AocMsgId(...)
;
;  This subroutine uses the message header and command to decide the
;  type of the message represented by a message ID.
;  If the message command doesn't match the message header, it
;  return AOC_MSG_ID_UNDEFINED.
;
;  Prototype:
;     void AocMsgId(uint8 uc_msg_header, uint8 uc_msg_cmd, int16 *ps_msg_id, int16 *ps_msg_length)
;
;  Input Arguments:
;     uc_msg_header  -- header byte
;     uc_msg_cmd     -- command byte (follows header byte)
;
;  Output Arguments:
;     *ps_msg_id     -- message ID or an ID for undefined messages
;     *ps_msg_length -- message length when a valid message is found
;
;  Return:
;     N/A
;
;  Global Variables:
;
*******************************************************************************/
void AocMsgId(uint8 uc_msg_header, uint8 uc_msg_cmd, int16 *ps_msg_id, int16 *ps_msg_length)
{
   switch(uc_msg_header) {

      case AOC_HEADER_UTC:

         *ps_msg_id = (int16) AOC_MSG_ID_UTC;
         *ps_msg_length = (int16) AOC_MSG_UTC_LEN;
         break;

      case AOC_HEADER_PMR:

         if(uc_msg_cmd == AOC_CMD_PMR) {
            *ps_msg_id = (int16) AOC_MSG_ID_PMR;
            *ps_msg_length = (int16) AOC_MSG_PMR_LEN;
         }
         else if(uc_msg_cmd == AOC_CMD_PMA) {
            *ps_msg_id = (int16) AOC_MSG_ID_PMA;
            *ps_msg_length = (int16) AOC_MSG_PMA_LEN;
         }
         else
            *ps_msg_id = (int16) AOC_MSG_ID_UNDEFINED;
         break;

      case AOC_HEADER_BSR:

         if(uc_msg_cmd == AOC_CMD_BSA) {
            *ps_msg_id = (int16) AOC_MSG_ID_BSA;
            *ps_msg_length = (int16) AOC_MSG_BSA_LEN;
         }
         else if(uc_msg_cmd <= AOC_CMD_PWR_DEC2) {
            *ps_msg_id = (int16) AOC_MSG_ID_BSR;
            *ps_msg_length = (int16) AOC_MSG_BSR_LEN;
         }
         else
            *ps_msg_id = (int16) AOC_MSG_ID_UNDEFINED;
         break;

      case AOC_HEADER_EBSR:

         if(uc_msg_cmd <= AOC_CMD_PWR_DEC2) {
            *ps_msg_id = (int16) AOC_MSG_ID_EBSR;
            *ps_msg_length = (int16) AOC_MSG_EBSR_LEN;
         }
         else
            *ps_msg_id = (int16) AOC_MSG_ID_UNDEFINED;
         break;

      default:

         *ps_msg_id = (int16) AOC_MSG_ID_UNDEFINED;
         break;
   }
}

/*****************************************************************************
*   Subroutine Name: RxAocHandler
*
*   Description:
*     This subroutine takes the current AOC input byte and packs it into
*     the 16-bit global variable gus_RxAocRegister with the NEW_AOC_MESSAGE
*     bit that is checked in RxAocProcessor()
*
*     This function is part of the SW Engine layer.
*
*
*   Prototype:
*       void RxAocHandler(uint8 uc_sb);
*
*   Input Arguments:
*       uc_sb           - current AOC input byte
*
*   Output Arguments: none
*
*   Return: none
*
*   Global Variables:
*       gus_RxAocRegister - Global AOC variable
*
*
*******************************************************************************/
void RxAocHandler(uint8 uc_sb)
{
   gus_RxAocRegister = (uint16) uc_sb;
   //set and pack NEW_AOC_MESSAGE bit with overhead byte
   gus_RxAocRegister |= NEW_AOC_MESSAGE;
}

/*****************************************************************************
;  Subroutine Name: RxAocProcessor()
;
;  This subroutine processes the received AOC byte.
;
;  Prototype:
;     uint8 RxAocProcessor()
;
;  Input Arguments:
;     none
;
;  Output Arguments:
;     none
;
;  Return:
;     message ID
;
;  Global Variables:
;
;  gt_TxAocHander -- the TX AOC handler structure, the following variables are modified:
;     uc_msg_repeat_count              -- (O) TX sending message repeat count
;     uc_outstanding_msg_count         -- (I/O) TX outstanding message count
;     pt_current_msg->uc_msg_flag         -- (O) TX current message status flag
;     pt_current_msg->uc_reason_of_fail   -- (O) TX message fail reason
;     pt_current_msg->uca_msg_body[]      -- (I) TX message data
;
;  gt_RxAocHandler -- the RX AOC handler structure, the following variables are modified:
;     uc_outstanding_msg_count   -- (I/O) RX outstanding message count
;     pt_current_msg          -- (I/O) pointer to the current RX outstanding message
;
;  gt_RxMsgBSA -- the structure of the message BSA, the following variables are modified:
;     uc_msg_length           -- (O) message length in bytes
;     uc_msg_body[]           -- (O) message data
;
;  gt_RxMsgUTC                -- (I) the structure containing the UTC message
;  gft_SendAocMsg_Flag           -- (I/O) status of sending BSR message
;  gft_TxBitSwapFlag          -- (O) indicate whether to perform TX bit swap
;  gft_TxBitSwapFlag          -- (O) indicate whether to perform RX bit swap
;  gt_TxBitSwapInfo           -- (O) structure to store TX bit swap information
;  gt_RxBitSwapInfo           -- (O) structure to store RX bit swap information
;
*******************************************************************************/

#define uc_rx_outstanding_msg_count gt_RxAocHandler.uc_outstanding_msg_count
#define pt_rx_current_msg        gt_RxAocHandler.pt_current_msg

uint8 RxAocProcessor()
{
   AocMsgStruc_t t_RcvdMsg;

   uint8 uc_byte, uc_msg_id = AOC_MSG_ID_UNDEFINED;

   if (gus_RxAocRegister & NEW_AOC_MESSAGE) {
      /* ====================================================================== */
      /* Decode the message */
      /* ====================================================================== */
      uc_byte = (gus_RxAocRegister & 0xff);  /* only 8 bits */



      uc_msg_id = RxMsgDecod(uc_byte, &t_RcvdMsg);
      gus_RxAocRegister = 0;  /* set to 0 to avoid processing same byte again */

      /* ====================================================================== */
      /* Take actions based on the message type */
      /* ====================================================================== */
      switch(uc_msg_id) {

         /* ================================================================== */
         /* Process Uable To Comply */
         /* ================================================================== */
         case AOC_MSG_ID_UTC:

            /* Clear the current TX outstanding message if there is one */
            if(gt_TxAocHandler.uc_outstanding_msg_count > 0)
            {

#ifdef INCLUDE_FASTRETRAIN_CODE
               if (((STATArray[STAT_Mode] & STAT_ConfigMode_G992_2_AB) == 0) && ((STATArray[STAT_Mode] & STAT_ConfigMode_G992_2_C) == 0) )
               {  /* if G992_1 */
                        /* also valid for G.992.2AB in the absence of fast retrain*/
#endif

                  /* If the TX outstanding message expects an acknowledgement, */
                  /* set the following parameters */
                  if((gt_TxAocHandler.pt_outstanding_msg->uc_msg_id == AOC_MSG_ID_BSR) ||
                     (gt_TxAocHandler.pt_outstanding_msg->uc_msg_id == AOC_MSG_ID_EBSR)) {

                     //Increment the bitswap UTC msg counter
                     gt_DSOLRPMCnt.t_BitswapCnt.us_OLRPMUTCResp_Cnt++;

                     /* Set the TX message flag */
                     gt_TxAocHandler.pt_outstanding_msg->uc_msg_flag = FAIL;

                     /* Set the reason of TX message fail to UTC */
                     gt_TxAocHandler.pt_outstanding_msg->uc_reason_of_fail = AOC_FAIL_OF_UTC;

                     /* Set TX outputstanding message count to 0 (so other message can be sent) */
                     gt_TxAocHandler.uc_outstanding_msg_count = 0;

                     if(gt_TxAocHandler.uc_nonoutstanding_msg_count == 0)
                        /* Set TX message repeat count to 0 to stop transmit the current TX outstanding message */
                        gt_TxAocHandler.uc_msg_repeat_count = 0;

                     /* If BSR has been sent, don't try to resent it any more */
                     if(gft_SendAocMsg_Flag == SUCCEED)
                        gft_SendAocMsg_Flag = IDLE;
                  }

#ifdef INCLUDE_FASTRETRAIN_CODE

               }
               else
               {  /* else, g992_2 */
                  if((gt_TxAocHandler.pt_outstanding_msg->uca_msg_body[0] == AOC_HEADER_PMR) ||
                     (gt_TxAocHandler.pt_outstanding_msg->uca_msg_body[1] == AOC_CMD_PMR)) {

                     //Increment the bitswap UTC msg counter
                     gt_DSOLRPMCnt.t_BitswapCnt.us_OLRPMUTCResp_Cnt++;

                     /* Set the TX message flag */
                     gt_TxAocHandler.pt_outstanding_msg->uc_msg_flag = FAIL;

                     /* Set the reason of TX message fail to UTC */
                     gt_TxAocHandler.pt_outstanding_msg->uc_reason_of_fail = AOC_FAIL_OF_UTC;

                     /* Set TX outputstanding message count to 0 (so other message can be sent) */
                     gt_TxAocHandler.uc_outstanding_msg_count = 0;

                     if(gt_TxAocHandler.uc_nonoutstanding_msg_count == 0)
                        /* Set TX message repeat count to 0 to stop transmit the current TX outstanding message */
                        gt_TxAocHandler.uc_msg_repeat_count = 0;

                     /* If PMR has been sent, don't try to resent it any more */
                     if(gft_SendAocMsg_Flag == SUCCEED)
                        gft_SendAocMsg_Flag = IDLE;

                  }
               }
#endif

               // Log Bitswap info
               if (gs_BitSwapLogIndex < (2*MAX_STATES - 20))
               {
                  gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) 0xBBBB;        /* delimiter indicating DS Bitswap UTC information */
                  gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) gus_ShowtimeSNRUpdateCount;   /* rx symbol count */
                  gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) uc_msg_id;  /* aoc message id */
               }
            }
            break;

         /* ====================================================================== */
         /* Process Bit Swap Acknowledgement */
         /* ====================================================================== */
         case AOC_MSG_ID_BSA:

            /* if there is no a TX outstanding message, ignore this message */
            if(gt_TxAocHandler.uc_outstanding_msg_count <= 0)
               break;

            /* If the TX outstanding message is not BSR or EBSR, ignore this message */
            if((gt_TxAocHandler.pt_outstanding_msg->uc_msg_id != AOC_MSG_ID_BSR ) &&
               (gt_TxAocHandler.pt_outstanding_msg->uc_msg_id != AOC_MSG_ID_EBSR )) {
               break;
            }

            //Increase the bitswap actual performed counter
            gt_DSOLRPMCnt.t_BitswapCnt.us_ActualOLRPMPerform_Cnt++;

            /* Store the RX bit-swap information      */
            StoreRxBitSwapInfo(gt_TxAocHandler.pt_outstanding_msg, &t_RcvdMsg);

            /* Set RX bit swap flag so the bit-swap will take place at right time */
            gft_RxBitSwapFlag = TRUE;

            /* Clean message flags */
            gt_TxAocHandler.pt_outstanding_msg->uc_msg_flag = SUCCEED;

            /* Set TX outputstanding message count to 0 (so other message can be sent) */
            gt_TxAocHandler.uc_outstanding_msg_count = 0;

            if(gt_TxAocHandler.uc_nonoutstanding_msg_count == 0)
               /* Set TX message repeat count to 0 (to stop transmit the current TX outstanding message) */
               gt_TxAocHandler.uc_msg_repeat_count = 0;

            break;

         /* ==================================================================== */
         /* Process Bit Swap Request or Extended Bit Swap Request */
         /* ==================================================================== */
         case AOC_MSG_ID_BSR: case AOC_MSG_ID_EBSR:

            /* if there is no RX outstanding message */
            if(uc_rx_outstanding_msg_count == 0) {

               // Increment the US bitswap request counters
               if (uc_msg_id == AOC_MSG_ID_BSR)
                  gt_USOLRPMCnt.t_BitswapCnt.us_SimpleOLRPMReq_Cnt++;
               else
                  gt_USOLRPMCnt.t_BitswapCnt.us_ExtendOLRPMReq_Cnt++;

               /* If the previous TX bit swap has not finished, return UTC */
               /* If Upstream bitswap is disabled, return UTC */
               if(OPTNArray[OPTN_AlgControl] & OPTN_USBitSwapDisable)
               {
                  gs_DMT_USBitSwapErrorCode = REQ_BEFORE_FINISH_PREV_BITSWAP;

                  pt_rx_current_msg = &gt_RxMsgUTC;

                  // Increment the US bitswap UTC response counters
                  gt_USOLRPMCnt.t_BitswapCnt.us_OLRPMUTCResp_Cnt++;
               }

               /* Process BSR or EBSR message */
               else {
                  // Discard any previous bit-swap requests
                  gft_TxBitSwapFlag = FALSE;

                  if(StoreTxBitSwapInfo(&t_RcvdMsg) == SUCCEED) {

                     /* Set TX bit swap flag so the bit-swap will take place at right time */
                     gft_TxBitSwapFlag = TRUE;

                     /* Form a BSA message */
                     TxAocMsgBSA(gt_TxBitSwapInfo.uc_SuperframeCount, &gt_RxAocMsgBSA);

                     /* Set the current message pointer to PMA message */
                     pt_rx_current_msg = &gt_RxAocMsgBSA;

                     //Increment the actual performed US bitswap counter
                     gt_USOLRPMCnt.t_BitswapCnt.us_ActualOLRPMPerform_Cnt++;
                  }
                  else {

                     /* Set the current message pointer to UTC message */
                     pt_rx_current_msg = &gt_RxMsgUTC;

                     // Increment the US bitswap UTC response counters
                     gt_USOLRPMCnt.t_BitswapCnt.us_OLRPMUTCResp_Cnt++;
                  }
               }
               uc_rx_outstanding_msg_count = 1;
            }

            // Log Bitswap info
            if ((gs_BitSwapLogIndex < (2*MAX_STATES - 20)) && (pt_rx_current_msg->uc_msg_id == AOC_MSG_ID_UTC))
            {
               gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) 0xDDDD;        /* delimiter indicating US Bitswap UTC information */
               gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) gus_ShowtimeSNRUpdateCount;   /* rx symbol count */
               gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) AOC_MSG_ID_UTC;   /* aoc message id */
               gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) gs_DMT_USBitSwapErrorCode;/* Bitswap Error Code(internal) */
            }

            /* else drop the response message since only one outstanding message is allowed  */
            /* as specified in G.992.1 */
            break;

         /* ====================================================================== */
         /* Process Profile Management Acknowledgement */
         /* ====================================================================== */
         case AOC_MSG_ID_PMA:

            /* if there is no a TX outstanding message, ignore this message */
            if(gt_TxAocHandler.uc_outstanding_msg_count <= 0)
               break;

            /* If the TX outstanding message is not PMR, ignore this message */
            if((gt_TxAocHandler.pt_outstanding_msg->uca_msg_body[0] != AOC_HEADER_PMR) ||
               (gt_TxAocHandler.pt_outstanding_msg->uca_msg_body[1] != AOC_CMD_PMR)) {
               break;
            }

#ifdef INCLUDE_FASTRETRAIN_CODE
            /* Verify the PMA message and store Rx Profile     */
            if(StoreRxProfile(gt_TxAocHandler.pt_outstanding_msg, &t_RcvdMsg) == SUCCEED) {

               gt_TxAocHandler.pt_outstanding_msg->uc_msg_flag = SUCCEED;

               /* Figure out the maximum profile number that both sides can support */
               i = NUM_PROFILES;
               if( guc_remote_max_profiles < NUM_PROFILES )
                  i = guc_remote_max_profiles;

               /* Update the next rx profile number (wrap around it if it reaches the maximum) */
               guc_next_rx_profile_number++;
               if(guc_next_rx_profile_number >= i)
                  guc_next_rx_profile_number = 0;

               /* The PMR message sent during showtime has been acknowledged, */
               /* so clear SendPMR_Flag */
               if(gft_SendAocMsg_Flag == SUCCEED)
                  gft_SendAocMsg_Flag = IDLE;
            }
            else { /* Fail */
               gt_TxAocHandler.pt_outstanding_msg->uc_msg_flag = FAIL;
               gt_TxAocHandler.pt_outstanding_msg->uc_reason_of_fail = AOC_FAIL_OF_UNMATCH;

               /* The PMR message sent during showtime has not been acknowledged correctly, */
               /* try to resent it by setting SendPMR_Flag to 1 */
               if(gft_SendAocMsg_Flag == SUCCEED)
                  gft_SendAocMsg_Flag = OUTSTANDING;
            }

            /* Set TX outputstanding message count to 0 (so other message can be sent) */
            gt_TxAocHandler.uc_outstanding_msg_count = 0;

            if(gt_TxAocHandler.uc_nonoutstanding_msg_count == 0)
               /* Set TX message repeat count to 0 (to stop transmit the current TX outstanding message) */
               gt_TxAocHandler.uc_msg_repeat_count = 0;

            break;

#else
            gt_TxAocHandler.pt_outstanding_msg->uc_msg_flag = SUCCEED;
            gft_SendAocMsg_Flag = IDLE;
            break;
#endif

         /* ==================================================================== */
         /* Process Profile Management Request */
         /* ==================================================================== */
         case AOC_MSG_ID_PMR:

            /* if there is no RX outstanding message */
            if(uc_rx_outstanding_msg_count == 0) {

#ifdef INCLUDE_FASTRETRAIN_CODE

               if(t_RcvdMsg.uca_msg_body[2] < NUM_PROFILES) {

                  /* Store this profile */
                  StoreTxProfile(t_RcvdMsg.uca_msg_body[2]);

                  /* Form a PMA message */
                  TxAocMsgPMA(t_RcvdMsg.uca_msg_body[2], NUM_PROFILES, &gt_RxMsgPMA);

                  /* Set the current message pointer to PMA message */
                  pt_rx_current_msg = &gt_RxMsgPMA;
               }
               else {

                  /* Set the current message pointer to UTC message */
                  pt_rx_current_msg = &gt_RxMsgUTC;
               }

#else
               /* Set the current message pointer to UTC message */
               pt_rx_current_msg = &gt_RxMsgUTC;
#endif

               uc_rx_outstanding_msg_count = 1;
            }

            /* else drop the response message since only one outstanding message is allowed  */
            /* as specified in G.992.2 */


            break;


         /* ======================================================================= */
         /* Ignore all others messages */
         /* ======================================================================= */
         default:

            /* Ignore it for now since this software don't support bit swap operation */
            break;
      }
   }

   /* ====================================================================== */
   /* Try to send out the RX outstanding message */
   /* ====================================================================== */
   if(uc_rx_outstanding_msg_count > 0) {
      if(TxAocMsgPut(pt_rx_current_msg) == SUCCEED)
         uc_rx_outstanding_msg_count = 0;
   }

   return(uc_msg_id);

}

#undef uc_rx_outstanding_msg_count
#undef pt_rx_current_msg

