/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-1998 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;  Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413
;  Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;
;  File Name: tx_aoc.c
;
;  This file contains functions for TX side of AOC operations.
;
;***************************************************************************/

#include "config.h"
#include "common.h"
#include "aoc.h"
#include "fifo.h"
#include "gdata.h"
#include "common.h"
#include "rx_aoc.h"
#include "trail.h"

#define uc_msg_repeat_count         gt_TxAocHandler.uc_msg_repeat_count
#define uc_next_byte          gt_TxAocHandler.uc_next_byte
#define uc_stuffing_byte_count      gt_TxAocHandler.uc_stuffing_byte_count
#define uc_tx_outstanding_msg_count gt_TxAocHandler.uc_outstanding_msg_count
#define uc_tx_nonoutstanding_msg_count gt_TxAocHandler.uc_nonoutstanding_msg_count
#define uca_AOC_outbuf           gt_TxAocHandler.uca_AOC_outbuf
#define pt_tx_nonoutstanding_msg    gt_TxAocHandler.pt_nonoutstanding_msg
#define pt_tx_outstanding_msg       gt_TxAocHandler.pt_outstanding_msg

/*****************************************************************************
;  Subroutine Name: TxAocHandler(...)
;
;  This subroutine decides which AOC byte is sent in the current synch byte.
;
;  Prototype:
;     uint8 TxAocHandler(int16 s_frame_number)
;
;  Input Arguments:
;     s_frame_count  -- current frame count
;
;  Output Arguments:
;     none
;
;  Return:
;     AOC byte for this frame.
;
;  Global Variables:
;
;  gt_TxAocHander -- TX AOC handler structure with the following variables used:
;     uc_msg_repeat_count     -- (I/O) count for times of repeating the same message
;     uc_next_byte         -- (I/O) index of next byte to be sent
;     uc_stuffing_byte_count  -- (I/O) count for times of sending stuffing bytes between messages
;     uc_tx_outstanding_msg_count -- (I/O) number of TX outstanding messages
;     uc_tx_nonoutstanding_msg_count -- (I/O) number of TX nonoutstanding messages
;     uca_AOC_outbuf[]     -- (I) AOC TX output buffer
;     pt_tx_nonoutstanding_msg      -- (I) pointer to the nonoutstanding message structure
;
;  gs_pre_frame_number -- the frame count when this function is called previously
*******************************************************************************/

uint8 TxAocHandler(int16 s_frame_number)
{

   uint8 uc_byte, uc_tx_msg_length;
   int16 i;
   uint16 us_TxAocRegister = 0;

   /* ================================================================================= */
   /* Transmit at least 20 stuffing patterns between two consecutive groups of messages */
   /* ================================================================================= */
   if(uc_stuffing_byte_count > 0) {
      uc_byte = AOC_STUFFING_BYTE;
      uc_stuffing_byte_count--;
   }

   /* ================================================================================= */
   /* Transmit each message for five times consecutively */
   /* ================================================================================= */
   else if(uc_msg_repeat_count > 0) {

      /* Transmit next byte in output buffer */
      uc_byte = uca_AOC_outbuf[uc_next_byte++];

      if(uc_tx_nonoutstanding_msg_count == 1)
         uc_tx_msg_length = pt_tx_nonoutstanding_msg->uc_msg_length;
      else
         uc_tx_msg_length = pt_tx_outstanding_msg->uc_msg_length;

      /* Update output buffer parameters */
      if(uc_next_byte == uc_tx_msg_length) {
         uc_next_byte = 0;
         uc_msg_repeat_count--;

         if(uc_msg_repeat_count == 0) {

            /* set stuffing byte count */
            uc_stuffing_byte_count = AOC_MIN_NUM_STUFFING_BYTES;

         }
      }

   }
   /* ================================================================================= */
   /* Transmit stuffing bytes indefinitely until a new message is ready */
   /* ================================================================================= */
   else {

      uc_byte = AOC_STUFFING_BYTE;
   }

   //=================================================================================
   //Clear up uc_tx_nonoutstanding_msg_count if the message doesn't need acknowledgement
   //=================================================================================
   if((uc_tx_nonoutstanding_msg_count == 1) && (uc_msg_repeat_count == 0)) {
      pt_tx_nonoutstanding_msg->uc_msg_flag = SUCCEED;
      uc_tx_nonoutstanding_msg_count = 0;

      if(uc_tx_outstanding_msg_count == 1) //copy outstanding TX request to output buffer
         for(i=0; i<(pt_tx_outstanding_msg->uc_msg_length); i++)
            uca_AOC_outbuf[i] = pt_tx_outstanding_msg->uca_msg_body[i];
   }

   /* ================================================================================= */
   /* Update the current outstanding message timer once per DMT symbol */
   /* ================================================================================= */
   if((uc_tx_outstanding_msg_count == 1) &&
      ((uc_msg_repeat_count == 0) || (uc_tx_nonoutstanding_msg_count == 1))) {
      /* ========= check timer ========= */

      /* Compute the number of frame passed since last call of this function */
      i= s_frame_number - gs_pre_frame_number;
      if(i<0)
         i += TX_SYMBOLS_PER_SFRAME;

      /* Update the timer */
      pt_tx_outstanding_msg->s_msg_timer -= i;

      /* if the timeout occurs and done sending possible nonoutstanding message */
      if((pt_tx_outstanding_msg->s_msg_timer <= 0) && (uc_tx_nonoutstanding_msg_count == 0)) {

         // for DS bitswap request, we will resend it infinitely until get an ACK.
         if ((pt_tx_outstanding_msg->uc_msg_id == AOC_MSG_ID_BSR) ||
            (pt_tx_outstanding_msg->uc_msg_id == AOC_MSG_ID_EBSR))
         {
            // log time out count
            gt_DSOLRPMCnt.t_BitswapCnt.us_RespTimeOut_Cnt++;

            // Log Bitswap info
            if (gs_BitSwapLogIndex < (2*MAX_STATES - 20))
            {
               gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) 0xEEEE;        /* delimiter indicating US Bitswap Ack information */
               gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) gus_ShowtimeSNRUpdateCount;   /* rx symbol count */
               gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) gt_DSOLRPMCnt.t_BitswapCnt.us_SimpleOLRPMReq_Cnt+gt_DSOLRPMCnt.t_BitswapCnt.us_ExtendOLRPMReq_Cnt; /* aoc message id */
               gsa_StatesTrail[gs_BitSwapLogIndex++] = (int16) AOC_FAIL_OF_TIMEOUT;
            }
            //XDSLRTFW-2144 (start)
            if (  (gs_CurrentCoChipset ==  ANCDA_CO_CHIPSET) )
            {
               (pt_tx_outstanding_msg->uc_msg_retry_count)--;
            }
            //XDSLRTFW-2144 (end)
         }
         else
            /* reduce retry count */
            (pt_tx_outstanding_msg->uc_msg_retry_count)--;

         /* if retry count > 0, send out the same message again for five times */
         if(pt_tx_outstanding_msg->uc_msg_retry_count > 0) {

            /* reset timer */
            pt_tx_outstanding_msg->s_msg_timer =
               AOC_TX_TIMEOUT+(pt_tx_outstanding_msg->uc_msg_length*AOC_MSG_REPEAT_CNT);

            /* set the TX AOC Handler structure parameters */
            uc_msg_repeat_count = AOC_MSG_REPEAT_CNT;
            uc_next_byte = 0;
         }
         else {   /* Declare the current message transmission failed */
            pt_tx_outstanding_msg->uc_msg_flag = FAIL;
            pt_tx_outstanding_msg->uc_reason_of_fail = AOC_FAIL_OF_TIMEOUT;
            uc_tx_outstanding_msg_count = 0;

            /* Set additional parameters */
            gft_SendAocMsg_Flag = IDLE;
         }
      } /* ========= check timer ========= */
   }

   /* Store current frame number */
   gs_pre_frame_number = s_frame_number;

   us_TxAocRegister = (uint16) uc_byte;

   /* put the AOC message onto the TX AOC fifo */
   AddMessageToFifo(&gt_TxAocFifo, us_TxAocRegister);

   /* =============================================================================== */
   /* Return current AOC byte */
   /* =============================================================================== */
   return(uc_byte);

}



/*****************************************************************************
;  Subroutine Name: TxAocMsgPut(...)
;
;  This subroutine copies a new message from a given buffer into the TX AOC
;  output buffer and return SUCCEED if the outstanding message count is 0.
;  if the outstanding message count is greater than 0, then it doesn't
;  copy and will return (FAIL).
;
;  Prototype:
;     uint8 TxAocMsgPut(AocMsgStruc_t *pt_msg)
;
;  Input Arguments:
;     pt_msg -- (I) poitner to a structure containing the following data:
;        uc_msg_length     -- message length in bytes
;        uca_msg_body[];      -- message buffer
;
;  Output Arguments:
;     pt_msg -- (0) poitner to a structure containing the following data:
;        s_msg_timer    -- message timer in DMT symbols (when it is 0, time expired)
;        uc_msg_retry_count   -- no. of retries should be done for this message
;        uc_msg_flag       -- indicate the status of this message
;  Return:
;     SUCCEED -- message has been put into the TX AOC output buffer
;     FAIL  -- message has not put into the TX AOC output buffer
;              becasue there is a TX outstanding message.
;
;  Global Variables:
;  gt_TxAocHander -- (I/O) the structure containing the following variables:
;     uc_msg_repeat_count        -- sending message repeat count
;     uc_next_byte            -- index of next byte to be sent
;     uc_stuffing_byte_count     -- sending stuffing byte count
;     uc_tx_outstanding_msg_count   -- TX outputstanding message count
;     uc_tx_nonoutstanding_msg_count   -- TX nonoutstanding message count
;     uca_AOC_outbuf[]        -- AOC TX output buffer
;
;
*******************************************************************************/

uint8 TxAocMsgPut(AocMsgStruc_t *pt_msg)
{
   int16 i, j;
   uint8 uc_msg_id;
   FlagT ft_nonoutstanding_msg_flag;

   //find message type
   AocMsgId(pt_msg->uca_msg_body[0], pt_msg->uca_msg_body[1], &i, &j);
   uc_msg_id = (uint8) i;
   ft_nonoutstanding_msg_flag = ((uc_msg_id == AOC_MSG_ID_UTC) || (uc_msg_id == AOC_MSG_ID_PMA) || (uc_msg_id == AOC_MSG_ID_BSA));

   if((((!ft_nonoutstanding_msg_flag) && (uc_tx_outstanding_msg_count == 0)) || ft_nonoutstanding_msg_flag) &&
      (uc_msg_repeat_count == 0)){ //send acknowledgement even with pending outstanding TX message to avoid deadlock

      //There is no current message in the TX AOC output buffer
      /* copy the new message to that buffer */
      for(i=0; i<(pt_msg->uc_msg_length); i++)
         uca_AOC_outbuf[i] = pt_msg->uca_msg_body[i];

      /* set the TX AOC Handler structure parameters */
      uc_msg_repeat_count = AOC_MSG_REPEAT_CNT;
      uc_next_byte = 0;

      //set the message status flag
      pt_msg->uc_msg_flag        = OUTSTANDING;

      //If the message is UTC, or any acknowledge messages
      //Set nonoutstanding msg pointer, set retry count to 1 and timer to infinity (don't expect acknowledgement)
      if(ft_nonoutstanding_msg_flag) {
         pt_tx_nonoutstanding_msg = pt_msg;
         uc_tx_nonoutstanding_msg_count = 1;
         pt_msg->s_msg_timer  = AOC_INFINITY;
         pt_msg->uc_msg_retry_count = 1;
      }
      //Set message pointer and timer and repeating count
      else {
         pt_tx_outstanding_msg = pt_msg;
         uc_tx_outstanding_msg_count = 1;
         pt_msg->s_msg_timer        = AOC_TX_TIMEOUT+(pt_msg->uc_msg_length*AOC_MSG_REPEAT_CNT);
         pt_msg->uc_msg_retry_count = guc_AocTxMaxRetries;
      }
      return(SUCCEED);
   }
   else
      return(FAIL);


}

#undef uc_msg_repeat_count
#undef uc_next_byte
#undef uc_stuffing_byte_count
#undef uc_tx_outstanding_msg_count
#undef uc_tx_nonoutstanding_msg_count
#undef uca_AOC_outbuf
#undef pt_tx_nonoutstanding_msg
#undef pt_tx_outstanding_msg


/*****************************************************************************
;  Subroutine Name: TxAocMsgUTC(...)
;
;  This subroutine forms a Unable To Comply Message.
;
;  Prototype:
;     void TxAocMsgUTC(uint8 *puca_outbuf)
;
;  Input Arguments:
;
;
;  Output Arguments:
;     pt_msg   -- pointer to the output message structure
;
;  Return:
;
;  Global Variables:
;
;
*******************************************************************************/
void TxAocMsgUTC(AocMsgStruc_t *pt_msg)
{
   /* set message length */
   pt_msg->uc_msg_length = (uint8)AOC_MSG_UTC_LEN;

   /* set message body */
   pt_msg->uca_msg_body[0] = (uint8)AOC_HEADER_UTC;

   /* set message identifier */
   pt_msg->uc_msg_id = AOC_MSG_ID_UTC;
}

