/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-1998 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   rx_bert.c
 *
 *   Receive routines for implemeting a software Bit Error Rate Test (BERT).
 *
 *------------------------------------------------------------------------
 */

#include "common.h"
#include "bert.h"
#include "pn_tab.h"
#include "xgdata.h"
#include "gdata.h"
#include "gdata_bis.h"
#include "data_alloc.h"

#ifndef DISABLE_BERT
/* =============================================== */
/* static function prototypes */
/* =============================================== */
void Rx_BERT_GetSync(RXBERTDEF *t_RxBertStat,uint8 *puca_rx_input, int16 s_NumRxBytes);

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: Init_Rx_BERT
 *
 *  Description: Initializes the structure defininf the RX BERT
 *
 *  Prototype: void Init_Rx_Bert(RXBERTDEF *t_RxBertStat, int16 s_pattern_num);
 *
 *  Input Arguments:
 *      s_pattern_num       indicaties which PN sequence to use, options include
 *                          the following:
 *                            FIREBIRD_511   for a Fireberd 511 PN sequence
 *                            PN512       for a PN512 PN sequence
 *                            PN64        for a PN64 PN sequence
 *
 *  Output Arguments:
 *      t_RxBertStat        RX BERT structure
 *
 *  Return: none
 *
 *  Global Variables Used:
 *      us_PN512Tab[32]     bit pattern table for PN512 sequence
 *      us_PN64Tab[4]       bit pattern table for PN64 sequence
 *
 *------------------------------------------------------------------------
 *^^^
 */
void Init_Rx_Bert(RXBERTDEF *t_RxBertStat, int16 s_pattern_num) {

    t_RxBertStat->s_patt_cnt = 0;
    t_RxBertStat->ul_num_sframes = 0;
    t_RxBertStat->s_num_frames = 0;
    t_RxBertStat->ul_num_bit_errs = 0;
    t_RxBertStat->s_num_bytes = 0;
    t_RxBertStat->ul_candidate = 0;
    t_RxBertStat->s_num_err_bytes = 0;

    switch (s_pattern_num) {

    case FIREBIRD_511:
        t_RxBertStat->us_patt_ptr = gusa_Fireberd511;
        t_RxBertStat->s_patt_len = 511;
        break;

    case PN512:
        t_RxBertStat->us_patt_ptr = gusa_PN512Tab;
        t_RxBertStat->s_patt_len = 511;
        break;

    case PN64:
        t_RxBertStat->us_patt_ptr = gusa_PN_US_Tab;
        t_RxBertStat->s_patt_len = gs_PN_US_LEN;
        break;
    }
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: Rx_BERT_GetSync
 *
 *  Description:
 *    Declare pattern sync if BERT_SYNC_BITS consecutive bits
 *    are received which match the current BERT pattern
 *
 *  Prototype:
 *
 *  Arguments:
 *
 *  Return: none
 *
 *  Global Variables Used:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void Rx_BERT_GetSync(RXBERTDEF *t_RxBertStat,uint8 *puca_rx_input, int16 s_NumRxBytes) {

    uint16  *pus_patternptr;
    uint32  ul_patt_template, ul_candidate;
    uint8   uc_inbyte, uc_nextbit;
    int16   i, j;

    /*  point to BERT sequence */
    pus_patternptr = t_RxBertStat->us_patt_ptr;

    /*  create template */
    ul_patt_template = (uint32)( ((uint32)pus_patternptr[1]<<16) | (uint32)pus_patternptr[0]);

    /*  get candidate */
    ul_candidate = t_RxBertStat->ul_candidate;

    for (i = 0; i < s_NumRxBytes; i++) {
        uc_inbyte = *puca_rx_input++;
        for (j = 0; j < 8; j++) {
            /*  get next bit */
            uc_nextbit = uc_inbyte & 0x01;
            uc_inbyte >>= 1;

            /*  shift bit into sync candidate */
            ul_candidate >>= 1;
            ul_candidate |= (int32)uc_nextbit << 31;

            /*  compare candidate w/ template */
            if ( (ul_patt_template ^ ul_candidate) == 0) {  /*  sync found */
                /*  sync has been found */
                t_RxBertStat->s_pattern_sync = 1;

                /*  for now assume rest of bits received in this frame match */
                /*  adjust bit counter for next frame and then return */
                t_RxBertStat->s_patt_cnt = 32 +         /*  bits in template */
                    ( s_NumRxBytes*8 - (i*8 + j + 1) ); /*  total bits - bits tested */
                return;
            }

        } /*  j */
    } /*  i */
    t_RxBertStat->ul_candidate = ul_candidate;
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : Rx_BERT_PN
 *
 *  Description:
 *    Analyzes the received data for bit errors.  Using the
 *    sequence pointed to by t_RxBertStat->us_patt_ptr as the expected data,
 *    received bytes are compared to this sequence and bit errors are
 *    accumulated.
 *
 *  Prototype: void Rx_BERT_PN(RXBERTDEF *t_RxBertStat, uint8 *puca_rx_input, int16 s_NumRxBytes);
 *
 *  Input Arguments:
 *      t_RxBertStat    RX BERT structure
 *      puca_rx_input   received data
 *      s_NumRxBytes    number of bytes to load from BERT sequence
 *
 *  Output Arguments:
 *      t_RxBertStat    RX BERT structure
 *
 *  Return: none
 *
 *  Global Variables Used:
 *      int16 gsa_BERT_ErrorTable[256]    - look-up table for counting bit errors
 *      int16 s_CRC_err_count          - total received CRC errors
 *
 *------------------------------------------------------------------------
 *^^^
 */
void Rx_BERT_PN(RXBERTDEF *t_RxBertStat, uint8 *puca_rx_input, int16 s_NumRxBytes) {

    uint16  *pus_patternptr;
    uint8   uc_rxtestbyte, uc_hibits;
    int16    i, s_pattern_cnt;
    int16    s_errors = 0;
    int16    indx;
    int16    s_bitcnt, s_hibitcnt;

    if (t_RxBertStat->s_pattern_sync == 0) {
        Rx_BERT_GetSync(t_RxBertStat, puca_rx_input, s_NumRxBytes);
        return;
    }

    /* ==================================================================== */
    /*  update counters if sync frame (s_NumRxBytes == 0) */
    /* ==================================================================== */
    if (s_NumRxBytes == 0) {
        if (t_RxBertStat->s_num_frames >= RX_SYMBOLS_PER_SFRAME) {
            t_RxBertStat->ul_num_sframes++;
            t_RxBertStat->s_num_frames -= RX_SYMBOLS_PER_SFRAME;
        }

        return;
    }

    /* ==================================================================== */
    /*  set up pointer to current pattern */
    /* ==================================================================== */
    /*  point to BERT sequence */
    pus_patternptr = t_RxBertStat->us_patt_ptr;

    /*  get bit offset into sequence; s_pattern_cnt is limmited by the size of pattern array */
   if ( t_RxBertStat->s_patt_cnt <= 511 )
      s_pattern_cnt = t_RxBertStat->s_patt_cnt;
   else
      s_pattern_cnt = 511;

    /*  convert to 16 bit offset and update pointer */
    pus_patternptr += (s_pattern_cnt>>4);

    /*  set bit pointer within 16 bit word */
    s_bitcnt = s_pattern_cnt & 0x000F;

    for (i = 0; i < s_NumRxBytes; i++) {

        /*  get next test byte from current pattern word,  */
        if (s_bitcnt <= 8) {
            /*  next test byte is contained within current pattern word */
            uc_rxtestbyte = (uint8)((((uint16) *pus_patternptr) >> s_bitcnt) & 0xFF);
            s_pattern_cnt += 8;
            s_bitcnt += 8;

            if (s_bitcnt == 16) {
                s_bitcnt = 0;
                pus_patternptr++;
            }

            if (s_pattern_cnt > t_RxBertStat->s_patt_len) {
                s_pattern_cnt = 1;      /* last bit of pattern == first bit of pattern */
                                        /*  do not retest this bit */
                s_bitcnt = 1;
                pus_patternptr = t_RxBertStat->us_patt_ptr;
            }
        }

        else {
            /*  next rx test byte straddles current and next pattern words */

            /*  get lower bits of pattern */
            uc_rxtestbyte = (uint8)((((uint16) *pus_patternptr++) >> s_bitcnt) & 0xFF);
            s_pattern_cnt += 16 - s_bitcnt;
            s_hibitcnt = s_bitcnt - 8;
            s_bitcnt = 0;

            /*  check for end of pattern */
            if (s_pattern_cnt > t_RxBertStat->s_patt_len) {
                s_pattern_cnt = 1;      /* last bit of pattern == first bit of pattern */
                                        /*  do not retest this bit */
                s_bitcnt = 1;
                pus_patternptr = t_RxBertStat->us_patt_ptr;
            }

            /*  get upper bit of pattern */
            uc_hibits = (uint8)(((((uint16) *pus_patternptr)>>s_bitcnt) & guca_ByteMaskTable[s_hibitcnt]) << (8-s_hibitcnt));
            uc_rxtestbyte |= uc_hibits;
            s_bitcnt += s_hibitcnt;
            s_pattern_cnt += s_hibitcnt;
        }

        /* ================================================================ */
        /*  get number of errors in received byte */
        /* ================================================================ */
        indx = puca_rx_input[i] ^ uc_rxtestbyte;
        s_errors += gsa_BERT_ErrorTable[indx];
        if (gsa_BERT_ErrorTable[indx] != 0)  {             /*  check for pattern sync loss */
            t_RxBertStat->s_num_err_bytes++;
            if (t_RxBertStat->s_num_err_bytes >= 16) {   /*  declase pattern sync loss */
                t_RxBertStat->s_pattern_sync = 0;
                t_RxBertStat->ul_candidate = 0;
                t_RxBertStat->s_num_err_bytes = 0;
                return;
            }
        }
        else {
            t_RxBertStat->s_num_err_bytes = 0;
        }

    } /*  for */

    /* ==================================================================== */
    /*  update statistics  */
    /* ==================================================================== */
    t_RxBertStat->ul_num_bit_errs += (uint32) s_errors;
    t_RxBertStat->s_num_frames++;

    /*  save byte offset */
    t_RxBertStat->s_patt_cnt = s_pattern_cnt;
}

DATA_MAP_SIM
/* Buffers for getting the PN sequence from the RX frame buffer */
uint8 gucaa_RX_BERT_Buffer_A[NUM_BEARER_CHANNELS][MAX_DS_BC_PAYLOAD_BYTES];
uint8 gucaa_RX_BERT_Buffer_B[NUM_BEARER_CHANNELS][MAX_DS_BC_PAYLOAD_BYTES];

int16 gsa_RX_BERT_Buffers_Size_A[NUM_BEARER_CHANNELS];
int16 gsa_RX_BERT_Buffers_Size_B[NUM_BEARER_CHANNELS];

int16 gsa_RX_BERT_BCRunCnt[NUM_BEARER_CHANNELS];
uint8 guc_RX_BERT_Ready = 0;
DATA_MAP_END

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: RxBERTSetup
 *
 *  Description:
 *    Rx BERT Setup tasks to be done in TC A block (once every PMD symbol)
 *
 *  Prototype:
 *
 *  Arguments:
 *
 *  Return: none
 *
 *  Global Variables Used:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void RxBERTSetup(void)
{
   int16 i, s_BC_LPath;

   /* Select the appropriate RX BERT Buffer */
   guc_BERT_RX_Select = !guc_BERT_RX_Select;
   if(guc_BERT_RX_Select)  /* If 1, RX BERT Buffer A to be used */
   {
      for(i = 0; i < NUM_BEARER_CHANNELS; i++)
         gsa_RX_BERT_Buffers_Size_A[i] = 0;
   }
   else              /* If 0, RX BERT Buffer B to be used */
   {
      for(i = 0; i < NUM_BEARER_CHANNELS; i++)
         gsa_RX_BERT_Buffers_Size_B[i] = 0;
   }

   /* Compute the BC Run Count */
   for(i = 0; i < NUM_BEARER_CHANNELS; i++)
   {
      if(   (gt_RxShowTimeVars.t_BCParms[i].sa_BC_Bytes) || /* If BC enabled */
         (gt_RxShowTimeVars.t_BCParms[i].sa_BC_CChan) )
      {
         /* Get latency path corresponding to the bearer channel */
         s_BC_LPath = gt_RxShowTimeVars.t_BCParms[i].sa_BC_LPath;

         gsa_RX_BERT_BCRunCnt[i] = 1;
      }
      else
      {
         gsa_RX_BERT_BCRunCnt[i] = 0;
      }
   }
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: RxPNSequenceSetup
 *
 *  Description:
 *    Get one frame of PN sequence from the RX frame buffer
 *
 *  Prototype:
 *
 *  Arguments:
 *
 *  Return: none
 *
 *  Global Variables Used:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void RxPNSequenceSetup(void)
{
   int16 i;

   if(!guc_RX_BERT_Ready) guc_RX_BERT_Ready++;

   for(i = 0; i < NUM_BEARER_CHANNELS; i++)
   {
      if(gsa_RX_BERT_BCRunCnt[i])
      {
         gsa_RX_BERT_BCRunCnt[i]--;
         if(guc_BERT_RX_Select)  /* If 1, write to RX BERT Buffer A */
            GetRxFrameBuffer(i, gucaa_RX_BERT_Buffer_A[i],&gsa_RX_BERT_Buffers_Size_A[i]);
         else              /* If 0, write to RX BERT Buffer B */
            GetRxFrameBuffer(i, gucaa_RX_BERT_Buffer_B[i],&gsa_RX_BERT_Buffers_Size_B[i]);
      }
   }
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: RxPNSequence
 *
 *  Description:
 *    BERT code for the receive path. Originally in the Exec layer.
 *
 *  Prototype:
 *
 *  Arguments:
 *
 *  Return: none
 *
 *  Global Variables Used:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void RxPNSequence(void)
{
   int16 i, s_LSXASX;

   if (!guc_RX_BERT_Ready)
      return;              /* RX frame buffer is not ready yet */

    for (i = 0; i < NUM_BEARER_CHANNELS; i++)
   {
      /* Get the mapping for the bearer channel to the corresponding LSx/ASx channel */
      s_LSXASX = gt_RxShowTimeVars.t_BCParms[i].sa_BC_LSXASX;

      if(guc_BERT_RX_Select && gsa_RX_BERT_Buffers_Size_B[i])        /* If 1, process RX BERT Buffer B */
      {
         ReadRxBERTDataToFile(i, gucaa_RX_BERT_Buffer_B[i],gsa_RX_BERT_Buffers_Size_B[i]);
         Rx_BERT_PN(&gta_RxBertStat[s_LSXASX], gucaa_RX_BERT_Buffer_B[i],gsa_RX_BERT_Buffers_Size_B[i]);
      }
      else if(!guc_BERT_RX_Select && gsa_RX_BERT_Buffers_Size_A[i])  /* If 0, process RX BERT Buffer A */
      {
         ReadRxBERTDataToFile(i, gucaa_RX_BERT_Buffer_A[i],gsa_RX_BERT_Buffers_Size_A[i]);
         Rx_BERT_PN(&gta_RxBertStat[s_LSXASX], gucaa_RX_BERT_Buffer_A[i],gsa_RX_BERT_Buffers_Size_A[i]);
      }
   }
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: ReadRxBERTDataToFile
 *
 *  Description:
 *    BIS Debug BERT code for the receive path.
 *
 *  Prototype:
 *
 *  Arguments:
 *
 *  Return: none
 *
 *  Global Variables Used:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void ReadRxBERTDataToFile(int16 s_bc, uint8 *Buffer, int16 n)
{
#if DEBUG_BIS

   static FILE *fpt[2] = {NULL,NULL};
   static uint32 numBytesOut[2] = {0,0};
   int16 i;
   char filename[20];

   numBytesOut[s_bc] += n;

   sprintf(filename, "rx_bert%d.bin", s_bc);
   if(!fpt[s_bc]) fpt[s_bc] = fopen(filename, "w");
   else fpt[s_bc] = fopen(filename, "a");

   for(i=0;i<n;i++)
   {
      fprintf(fpt[s_bc], "%c", Buffer[i]);
   }

   fclose(fpt[s_bc]);

#endif
}

#endif
