/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-1998 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   tx_bert.c
 *
 *   Transmitter routines for implemeting a software Bit Error Rate Test
 *   (BERT).
 *
 *------------------------------------------------------------------------
 */

#include "common.h"
#include "bert.h"
#include "pn_tab.h"
#include "gdata.h"
#include "gdata_bis.h"
#include "data_alloc.h"


int16 gs_do_bert = 0;
int16 gs_bert_sf_cnt = 0;

#ifndef  DISABLE_BERT
/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: Init_Tx_BERT
 *
 *  Description:
 *    Initializes the structure defining the TX BERT
 *
 *  Prototype:
 *    void Init_Tx_Bert(TXBERTDEF *t_TxBertStat, int16 s_pattern_num);
 *
 *  Input Arguments:
 *      s_pattern_num       indicaties which PN sequence to use
 *                          FIREBIRD_511  for PN sequence Fireberd 511
 *                          PN512         for PN sequence PN512
 *                          PN64       for PN sequence PN64
 *
 *  Output Arguments:
 *      t_TxBertStat        TX BERT structure
 *
 *  Return: none
 *
 *  Global Variables Used:
 *      us_PN512Tab[32]     bit pattern table for PN512 sequence
 *      us_PN64Tab[4]       bit pattern table for PN64 sequence
 *
 *------------------------------------------------------------------------
 *^^^
 */
void Init_Tx_Bert(TXBERTDEF *t_TxBertStat, int16 s_pattern_num, uint32 ul_rate) {

    t_TxBertStat->s_patt_cnt = 0;           /*  set pattern offset to 0 */
    t_TxBertStat->ul_bit_err_rt = ul_rate;  /*  set error insert */
    t_TxBertStat->ul_bit_err_cnt = 0;       /*  set bits from prev. error to 0 */

    switch (s_pattern_num) {

    /*  set up to use Fireberd 511 sequence */
    case FIREBIRD_511:
        t_TxBertStat->us_patt_ptr = gusa_Fireberd511;
        t_TxBertStat->s_patt_len = 511;
        break;

    /*  set up to use PN512 sequence */
    case PN512:
        t_TxBertStat->us_patt_ptr = gusa_PN512Tab;
        t_TxBertStat->s_patt_len = 511;
        break;

    /*  set up to use PN64 sequence */
    case PN64:
        t_TxBertStat->us_patt_ptr = gusa_PN_US_Tab;
        t_TxBertStat->s_patt_len = gs_PN_US_LEN;
        break;
    }
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: Tx_BERT_PN
 *
 *  Description:
 *    Loads the tx data buffer with the BERT sequence.  The
 *    data buffer, puca_tx_input, will be loaded with the next s_NumTxBytes from
 *    the BERT sequence.  The BERT sequence will be sent in a repetative manner.
 *    The sequence will restart from the first byte at each superframe, indicated
 *    by s_NumTxByte == 0
 *
 *  Prototype:  Tx_BERT_PN(TXBERTDEF *t_TxBertStat,
 *                         uint8 *puca_tx_input,
 *                         int16 s_NumTxBytes)
 *
 *  Input Arguments:
 *      t_TxBertStat    TX BERT structure
 *      s_NumTxBytes    number of bytes to load from BERT sequence
 *
 *  Output Arguments:
 *      t_TxBertStat    TX BERT structure
 *      puca_tx_input   transmit data
 *
 *  Return:
 *    none
 *
 *  Global Variables Used:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void Tx_BERT_PN(TXBERTDEF *t_TxBertStat,uint8 *puca_tx_input, int16 s_NumTxBytes) {

    uint16  *pus_patternptr;
    uint8   uc_txbyte, uc_hibits;
    int16    i, s_pattern_cnt, s_bitcnt, s_hibitcnt;

    /* ==================================================================== */
    /*  set up pointer to current pattern */
    /* ==================================================================== */

    /*  point to BERT sequence */
    pus_patternptr = t_TxBertStat->us_patt_ptr;

    /*  get bit offset into sequence */
    s_pattern_cnt = t_TxBertStat->s_patt_cnt;

    /*  convert to 16 bit offset and update pointer */
    pus_patternptr += (s_pattern_cnt>>4);

    /*  set bit pointer within 16 bit word */
    s_bitcnt = s_pattern_cnt & 0x000F;

    for (i = 0; i < s_NumTxBytes; i++) {

        /*  get next byte from current pattern word,  */
        if (s_bitcnt <= 8) {
            /*  next tx byte is contained within current pattern word */
            uc_txbyte = (uint8)((((uint16) *pus_patternptr) >> s_bitcnt) & 0xFF);
            s_pattern_cnt += 8;
            s_bitcnt += 8;

            if (s_bitcnt == 16) {
                s_bitcnt = 0;
                pus_patternptr++;
            }

            if (s_pattern_cnt > t_TxBertStat->s_patt_len) {
                s_pattern_cnt = 1;      /* last bit of pattern == first bit of pattern */
                                        /*  do not resend this bit */
                s_bitcnt = 1;
                pus_patternptr = t_TxBertStat->us_patt_ptr;
            }

        }
        else {
            /*  next tx byte straddles current and next pattern words */

            /*  get lower bits of pattern */
            uc_txbyte = (uint8)((((uint16) *pus_patternptr++) >> s_bitcnt) & 0xFF);
            s_pattern_cnt += 16 - s_bitcnt;
            s_hibitcnt = s_bitcnt - 8;
            s_bitcnt = 0;

            /*  check for end of pattern */
            if (s_pattern_cnt > t_TxBertStat->s_patt_len) {
                s_pattern_cnt = 1;      /* last bit of pattern == first bit of pattern */
                                        /*  do not resend this bit */
                s_bitcnt = 1;
                pus_patternptr = t_TxBertStat->us_patt_ptr;
            }

            /*  get upper bit of pattern */
            uc_hibits = (uint8)(((((uint16) *pus_patternptr)>>s_bitcnt) & guca_ByteMaskTable[s_hibitcnt]) << (8-s_hibitcnt));
            uc_txbyte |= uc_hibits;
            s_bitcnt += s_hibitcnt;
            s_pattern_cnt += s_hibitcnt;

        }

        puca_tx_input[i] = uc_txbyte;   /*  put byte into tx buffer */

        /*  check for error insertion */
        if (t_TxBertStat->ul_bit_err_rt != 0) {
            t_TxBertStat->ul_bit_err_cnt += 8;
            if (t_TxBertStat->ul_bit_err_cnt > t_TxBertStat->ul_bit_err_rt) {
                puca_tx_input[i] ^= 0x01;   /*  insert error */
                t_TxBertStat->ul_bit_err_cnt -= t_TxBertStat->ul_bit_err_rt;
            }
        }
    }

    /*  save bit offset */
    t_TxBertStat->s_patt_cnt = s_pattern_cnt;
}

DATA_MAP_SIM
/* Buffers for loading the PN sequence into the TX frame buffer */
uint8 gucaa_TX_BERT_Buffer[NUM_BEARER_CHANNELS][TX_BERT_BUFFER_SIZE];
int16 gsa_TX_BERT_Buffer_ReadAddr[NUM_BEARER_CHANNELS] = {0, 0};
int16 gsa_TX_BERT_Buffer_WriteAddr[NUM_BEARER_CHANNELS] = {0, 0};
uint8 guca_TX_BERT_Buffer_Full[NUM_BEARER_CHANNELS] = {0, 0};

int16 gsa_TX_BERT_BCRunCnt[NUM_BEARER_CHANNELS];
DATA_MAP_END

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: TxBERTSetup
 *
 *  Description:
 *    Tx BERT Setup tasks to be done in TC A block (once every PMD symbol)
 *
 *  Prototype:
 *
 *  Arguments:
 *
 *  Return: none
 *
 *  Global Variables Used:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void TxBERTSetup(void)
{
   int16 i, s_BC_LPath;

   /* Compute the BC Run Count */
   for(i = 0; i < NUM_BEARER_CHANNELS; i++)
   {
      if(   (gt_TxShowTimeVars.t_BCParms[i].sa_BC_Bytes) || /* If BC enabled */
         (gt_TxShowTimeVars.t_BCParms[i].sa_BC_CChan) )
      {
         /* Get latency path corresponding to the bearer channel */
         s_BC_LPath = gt_TxShowTimeVars.t_BCParms[i].sa_BC_LPath;

         gsa_TX_BERT_BCRunCnt[i] = 1;
      }
      else
      {
         gsa_TX_BERT_BCRunCnt[i] = 0;
      }
   }
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: TxPNSequenceSetup
 *
 *  Description:
 *    Prepare one frame of PN sequence for the TX frame buffer.
 *    TxPNSequence() will load it into the TX frame buffer.
 *    Originally done in the Exec layer.
 *
 *  Prototype:
 *
 *  Arguments:
 *
 *  Return: none
 *
 *  Global Variables Used:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void TxPNSequenceSetup(void)
{
   int16 i, s_bytes_generate, s_LSX;
    uint8 *puc_txdata;

   for (i = 0; i  < NUM_BEARER_CHANNELS; i++)
    {
      if(   (gt_TxShowTimeVars.t_BCParms[i].sa_BC_Bytes) || /* If BC enabled */
         (gt_TxShowTimeVars.t_BCParms[i].sa_BC_CChan) )
      {
         /* If TX Bert Buffer for BC is full,don't call Tx_BERT_PN until   */
         /* atleast a byte has been loaded into the TX frame buffer.    */
         if (guca_TX_BERT_Buffer_Full[i])
            continue;

         /* Get the mapping for the bearer channel to the corresponding LSx channel */
         s_LSX = gt_TxShowTimeVars.t_BCParms[i].sa_BC_LSX;

         // Get amount of free entries in Tx BERT Buffer (in num of bytes)
         s_bytes_generate = gsa_TX_BERT_Buffer_ReadAddr[i] - gsa_TX_BERT_Buffer_WriteAddr[i];
         /*  Generate BERT data */
         if(s_bytes_generate <= 0)
         {
            s_bytes_generate =  TX_BERT_BUFFER_SIZE - gsa_TX_BERT_Buffer_WriteAddr[i];
            puc_txdata = &gucaa_TX_BERT_Buffer[i][gsa_TX_BERT_Buffer_WriteAddr[i]];
            Tx_BERT_PN(&gta_TxBertStat[s_LSX], puc_txdata, s_bytes_generate);
            /*  Generate pattern loss */
            /*if ( (gl_TxSymbolCount > 99 + guc_BERT_DELAY) && (gl_TxSymbolCount < 124 + guc_BERT_DELAY) )
            {
               for (j = 0; j < s_bytes_generate; j++)
                  *puc_txdata++ = 0x00;
            }*/

            s_bytes_generate =  gsa_TX_BERT_Buffer_ReadAddr[i];
            puc_txdata = &gucaa_TX_BERT_Buffer[i][0];
            Tx_BERT_PN(&gta_TxBertStat[s_LSX], puc_txdata, s_bytes_generate);
            /*  Generate pattern loss */
            /*if ( (gl_TxSymbolCount > 99 + guc_BERT_DELAY) && (gl_TxSymbolCount < 124 + guc_BERT_DELAY) )
            {
               for (j = 0; j < s_bytes_generate; j++)
                  *puc_txdata++ = 0x00;
            }*/
         }
         else
         {
            puc_txdata = &gucaa_TX_BERT_Buffer[i][gsa_TX_BERT_Buffer_WriteAddr[i]];
            Tx_BERT_PN(&gta_TxBertStat[s_LSX], puc_txdata, s_bytes_generate);

            /*  Generate pattern loss */
            /*if ( (gl_TxSymbolCount > 99 + guc_BERT_DELAY) && (gl_TxSymbolCount < 124 + guc_BERT_DELAY) )
            {
               for (j = 0; j < s_bytes_generate; j++)
                  *puc_txdata++ = 0x00;
            }*/
         }

         gsa_TX_BERT_Buffer_WriteAddr[i] = gsa_TX_BERT_Buffer_ReadAddr[i];
         guca_TX_BERT_Buffer_Full[i] = 1;
      }
   }

}


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: TxPNSequence
 *
 *  Description:
 *    Load one frame of PN sequence into the TX frame buffer
 *
 *  Prototype:
 *
 *  Arguments:
 *
 *  Return: none
 *
 *  Global Variables Used:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void TxPNSequence(void)
{
   int16 i, s_bytes_loaded;

   for (i = 0; i  < NUM_BEARER_CHANNELS; i++)
    {
      if(gsa_TX_BERT_BCRunCnt[i])
      {
         gsa_TX_BERT_BCRunCnt[i]--;

         /* If TX Bert Buffer for BC is not full, populate it now */
         if (!guca_TX_BERT_Buffer_Full[i])
            TxPNSequenceSetup(); /* If PN sequence is not generated yet, generate it now. */

         LoadTxFrameBuffer(i, gucaa_TX_BERT_Buffer[i],
                        gsa_TX_BERT_Buffer_ReadAddr[i], &s_bytes_loaded);

         if(s_bytes_loaded)
         {
            gsa_TX_BERT_Buffer_ReadAddr[i] += s_bytes_loaded;
            if(gsa_TX_BERT_Buffer_ReadAddr[i] >= TX_BERT_BUFFER_SIZE)
               gsa_TX_BERT_Buffer_ReadAddr[i] -= TX_BERT_BUFFER_SIZE;
            guca_TX_BERT_Buffer_Full[i] = 0;
         }
      }
   }
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name :  WriteTxBERTDataToFile
 *
 *  Description: BIS Debug BERT code for the transmit path.
 *
 *  Prototype: void WriteTxBERTDataToFile
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Return: None
 *
 *  Global Variables Used:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void WriteTxBERTDataToFile(int16 s_bc, uint8 *Buffer, int16 n)
{
#if DEBUG_BIS

#endif
}

#endif
