/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-1998 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
*  Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
*  Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*  rx_eoc.c
*
*  Functions and variables for receiver Embedded Operations Channel (EOC)
*   operations.
*
****************************************************************************/
//******************************************************************
//    rx_eoc.c
//
// History
//
// 19/10/2010 AdeelJ/Nihar: Bug fix for
//                 0 dB DS SNR Margin reported at Alcatel IKNS NVLT-C DSLAM
//                 There is a problem in our EOC state machine as we reach a point where we process the
//                 wrong message in the EOC_DATA_READ_ODD_STATE_RX state.
//                 The EOC message processing starts when (consecutive_message counter == 3).
//                 This message is then supposed to be processed this once, no matter how much it is repeatedly
//                 transmitted from the CO. The STATE machine is progressed within this processing only
//                       In the code point where EOC messages are processed, the following messages were
//                 processed twice consecutively
//                         0x13F3 --> "Request parameter update" OPCODE_REQTPU
//                         0x4xF3 --> "Read parameter" message.
//                 This means that in this case the consecutive counter is being reset when >3 consecutive
//                 messages are received. It doesnt matter if the counter is reset with 3 consecutive
//                 "Request parameter update" messages since the state remains the same (idle).
//                 but after receiving 3 consecutive "Read parameter" messages, the state machine moves to
//                 data transfer state EOC_DATA_READ_ODD_STATE_RX. then onwards we are just supposed to echo
//                 the subsequent copies of the received 0x4xF3 messages. The bug here is that the consecutive
//                 counter is reset as well. and if the CO sends a 6th copy of the message (which it does in
//                 the NVLT-c Case), we will come across another instance of consecutive counter == 3 in state
//                 EOC_DATA_READ_ODD_STATE_RX.
//                   In EOC_DATA_READ_ODD_STATE_RX state we are expecting READ_ODD, and READ_EVEN messages
//                 only and in alternating order. If we receive any other message in this state we are supposed
//                 to respond with UTC whilst resetting the state machine to idle.
//                   The messages READ_ODD, and READ_EVEN are also sent by the CO many consecutive times
//                 each but here our state machine is working fine since the consecutive counter is not reset.
//                 We just process on the 3rd consecutive message and then just repeat the tx data till the message
//                 changes. when the message changes the consecutive counter is reset.
//                       This NVLT-C CO always sends 6 consecutive "Request parameter Update" messages and then
//                 6 consecutive "Read parameter" messages. According to standard this shouldnt be a problem,
//                 but this bug in our state machine caused it to process the wrong message in the
//                 EOC_DATA_READ_ODD_STATE_RX state. This message was supposed to be dealt with in the way
//                 duplicate messages continuing from the previous state are dealt with.
//                       In the Fix for this bug, the consecutive count reset is removed from the entry of
//                 Idle state handler and distributed into the cases where needed. Not many changes were needed
//                 to be made since many opcode handlers had an inbuilt counter reset. This is a bug fix and has
//                 no negative impact on other CO Chipsets.
//             Grep for IOP_US_ADSL1_IKNS_EOC_MSGS
//
// 19/10/2010 Nihar/Anantha Ramu: Link drops observed during  stress tests in DMT mode with BRCM CO.
//                          It was found that the changes under IOP_US_ADSL1_IKNS_EOC_MSGS
//                          was causing link drop. The changes of "IOP_US_ADSL1_IKNS_EOC_MSGS"
//                          was disbled only for HUWEI BRCM CO.
//                          Grep for IOP_US_ADSL1_IKNS_EOC_MSGS_ForBRCMversion
//
// 27/03/14 Hanyu: Ported SMS00821888: Added code to work around the CO bug on EOC margin data parsing
//                 with LS FS+ CTLM/0278 which parsed SNR margin data of 0x0E dB
//                 as EOD (End Of Data command) cuasing 0-dB DS margin reporting on DSLAM.
//                 Grep for XDSLRTFW-1681 IOP_DS_ADSL1_CTLM_EOC_MARGIN
//
// **************************************************************************

#include "typedef.h"
#include "common.h"
#include "rt_state.h"
#include "gdata.h"
#include "eoc.h"
#include "rx_eoc.h"
#include "tx_eoc.h"
#include "fifo.h"
#include "DSLEngin.h"
#include "cmv.h"
#include "rx_ib.h"
#include "rxceoc_com.h"
#include "trail.h"

#ifdef TARGET_HW
#include "mei_iof.h"
#endif
#include "gdata_dmt.h"
FlagT gft_repeat_echo;
/* =============================================== */
/* static function prototypes */
/* =============================================== */
int32 HDLC_Deframer(uint8 uc_byte);


/* =============================================== */
/* constants used by this file and eoc_ini.c file */
/* =============================================== */
/* define constants for clarifying code */
#define s_tx_eoc_msg_put_ptr            gpt_TxEocHandler->s_tx_eoc_msg_put_ptr
#define s_tx_eoc_msg_get_ptr            gpt_TxEocHandler->s_tx_eoc_msg_get_ptr

#define us_tx_eoc_msg                   gpt_TxEocHandler->us_tx_eoc_msg
#define puc_tx_eoc_buf                  gpt_TxEocHandler->puc_tx_eoc_buf
#define us_rx_eoc_msg_count             gpt_RxEocHandler->us_rx_eoc_msg_count
#define us_rx_eoc_msg                   gpt_RxEocHandler->us_rx_eoc_msg
#define us_pre_eoc_msg                  gpt_RxEocHandler->us_pre_eoc_msg
#define us_RxEocState                   gpt_RxEocHandler->us_RxEocState
#define uc_rx_info_buf_index            gpt_RxEocHandler->uc_rx_info_buf_index
#define us_msg_len                      gpt_RxEocHandler->us_msg_len
#define puc_rx_eoc_buf                  gpt_RxEocHandler->puc_rx_eoc_buf
#define us_vendor_id                    gpt_EocRxReadBuf->us_vendor_id
#define us_revision_num                 gpt_EocRxReadBuf->us_revision_num
#define us_serial_num                   gpt_EocRxReadBuf->us_serial_num
#define uc_line_attenuation             gpt_EocRxReadBuf->uc_line_attenuation
#define c_snr_margin                    gpt_EocRxReadBuf->c_snr_margin
#define uc_atu_r_configuration          gpt_EocRxReadBuf->uc_atu_r_configuration
#define uc_vendor_discr                 gpt_EocRxReadBuf->uc_vendor_discr
#define uc_clear_eoc_payload            gpt_EocRxReadBuf->uc_clear_eoc_payload
#define uc_Address                      gpt_RxEocDecod->uc_Address
#define uc_Parity                       gpt_RxEocDecod->uc_Parity
#define uc_DataOrOpcode                 gpt_RxEocDecod->uc_DataOrOpcode
#define uc_AutonomousMsg                gpt_RxEocDecod->uc_AutonomousMsg
#define uc_Information                  gpt_RxEocDecod->uc_Information

/*****************************************************************************
*   Subroutine Name: RxEocProcessor
*
*   Description:
*     This subroutine processes the received EOC message.  If the
*  EOC Fifo is empty, then the function simply returns
*  If the Fifo contains data, then the message is processed.
*
*     This function is designed as a NTC task.
*
*   Prototype:
*       void RxEocProcessor(void)
*
*   Input Arguments: none
*
*   Output Arguments: none
*
*   Return: none
*
*   Global Variables:
*   gt_RxEocHandler -- the RX EOC handler structure, the following variables are modified:
*       us_rx_eoc_msg_count - Received eoc message count
*       us_RxEocState       - current state of Rx Eoc state machine
*       us_rx_eoc_msg       - current received eoc message
*       us_pre_eoc_msg      - previous received eoc message
*       *puc_rx_eoc_buf     - pointer to the members of ReadEocBuf
*       us_msg_len          - length of the ATU-R data registers
*
*   gt_RxEocDecod -- the RX EOC Decode structure, the following variables are modified:
*       uc_Address          - Address field
*       uc_DataOrOpcode     - Data/Opcode field
*       uc_Parity           - Byte Parity field
*       uc_AutonomousMsg    - Autonomous message field
*       uc_Information      - Information field
*
*   gt_EocRxReadBuf  -- RX EOC Read structure with the following variables initialized:
*       us_vendor_id[EOC_VENDOR_ID_LEN]                     - ATU-R vendor id
*       us_revision_num[EOC_REVISION_NUM_LEN]               - ATU-R Revision number
*       us_serial_num[EOC_SERIAL_NUM_LEN]                   - ATU-R Serial number
*       uc_line_attenuation[EOC_LINE_ATTENUATION_LEN]       - ATU-R Line Attenuation
*       c_snr_margin[EOC_SNR_MARGIN_LEN]                    - ATU-R snr margin
*       uc_atu_r_configuration[EOC_ATU_R_CONFIGURATION_LEN] - ATU-R Configuration
*       uc_link_state[EOC_LINK_STATE_LEN]                   - ATU-R Link state
*       uc_vendor_discr[EOC_VENDOR_DISCR_LEN]               - vendor discretionary information
*       uc_clear_eoc_payload                                - clear eoc payload byte
*
*******************************************************************************/

#if 0 // debug
#define MAX_CEOC_JUNK 10
uint16 usa_ceoc_buf[MAX_CEOC_JUNK];
int16 s_ceoc_i;
#endif

extern void QueueInterruptRequest(int16 s_InterruptCode);

void RxEocProcessor()
{
    uint16 us_rx_eoc;
    int16 s_temp_ptr;

    // IOP_US_ADSL1_IKNS_EOC_MSGS_ForBRCMversion (START)
    FlagT ft_BRCM_Version = 0;
    if((((guca_fe_G9941_VendorID[6] << 8) | (guca_fe_G9941_VendorID[7])) == (0X6284))&&
            (gs_CurrentCoChipset == BDCM_CO_CHIPSET))
    {
        ft_BRCM_Version = 1;
    }
    // IOP_US_ADSL1_IKNS_EOC_MSGS_ForBRCMversion (END)
    /* =======================================================================================  */
    /* if the eoc fifo is empty, then just return                                   */
    /*                                                                     */
    /* =======================================================================================  */

    if (RemoveMessageFromFifo(&gt_RxEocFifo, &us_rx_eoc) == 0)
    {
        /* not an eoc message, send "No synchronization action" byte. */

        /* When there is data to be transmitted in the FIFO buffer, but a NO_SYNC_ACTION message is    */
        /* received from CO, transmit the data from the FIFO buffer instead of NO_SYNC_ACTION message. */
        s_temp_ptr = s_tx_eoc_msg_get_ptr;
        if(s_temp_ptr > s_tx_eoc_msg_put_ptr)
            s_temp_ptr -= TX_EOC_MSG_BUF_LEN;

        if(s_temp_ptr < s_tx_eoc_msg_put_ptr)
            gft_no_sync_flag = FALSE;   /* do not transmit NO_SYNC_ACTION byte */
        else
        {
            /* no new message to transmit or if the transmit buffer is full, transmit "No Synchronization Action" byte */
            if ((gft_tx_msg_pending == FALSE) || (s_tx_eoc_msg_get_ptr == s_tx_eoc_msg_put_ptr))
                gft_no_sync_flag = TRUE;
        }

        us_rx_eoc_msg = 0; /* clear us_rx_eoc_msg */
        return;
    }

    /* =======================================================================================  */
    /* otherwise, process the eoc message                                        */
    /*                                                                     */
    /* =======================================================================================  */

    /* set global variable us_rx_eoc_msg  */
    us_rx_eoc_msg = us_rx_eoc;

    gft_no_sync_flag = FALSE;

    /*  If bit 5 of EOC msg., is 0, then it is autonomous message */
    if ((us_rx_eoc & MASKBIT6) == 0)
    {

        RxEocMsgDecod(us_rx_eoc, gpt_RxEocDecod);

        if ((((us_rx_eoc >>2) & 0x03) == EOC_ATU_R) && (uc_DataOrOpcode == EOC_DATA)) /*  clear eoc msg. */
        {
            uc_clear_eoc_payload = uc_Information;
            HDLC_Deframer(uc_Information);

#if 0 // debug
            if (s_ceoc_i < MAX_CEOC_JUNK)
            {
                usa_ceoc_buf[s_ceoc_i++] = us_rx_eoc;
            }
#endif
        }

    }
    else   /*  not an autonomous message */
    {
        if (us_pre_eoc_msg != us_rx_eoc)    /*  not receiving the same message consecutively */
        {
            gft_tx_msg_pending = FALSE;
            us_rx_eoc_msg_count = 1;
            gus_discarted_EOC = 0;
            us_pre_eoc_msg = us_rx_eoc;
        }
        else
        {
            us_rx_eoc_msg_count++;  /* count the number of consecutive messages */
            if ((gs_CurrentCoChipset == ADI_CO_CHIPSET) || (gs_CurrentCoChipset ==  ANCDA_CO_CHIPSET))
            {
                if((us_rx_eoc_msg_count == 2)&&(gus_discarted_EOC > 1))
                {
                    us_rx_eoc_msg_count = 3;
                    gus_discarted_EOC = 0;
                    gft_repeat_echo = TRUE;
                }
            }
        }
        /*  Receive properly addressed 3 consecutive msgs before taking action on it  */
        if ((((us_rx_eoc >>2) & 0x03) == EOC_ATU_R) && (us_rx_eoc_msg_count == 3))
        {

            RxEocMsgDecod(us_rx_eoc, gpt_RxEocDecod);   /*  decode the received message */

            switch(us_RxEocState)
            {

                /*  IDLE state */
            case EOC_IDLE_STATE_RX:
                // IOP_US_ADSL1_IKNS_EOC_MSGS(START_END)
                // Removed consecutive counter reset
                // IOP_US_ADSL1_IKNS_EOC_MSGS_ForBRCMversion (START_END)
                if(ft_BRCM_Version) us_rx_eoc_msg_count = 0;
                RxIdleStateHandler();
                break;

                /*  Data Write Odd State */
            case EOC_DATA_WRITE_ODD_STATE_RX:
                us_rx_eoc_msg_count = 0;
                RxWriteStateHandler(EOC_ODD_PARITY, EOC_DATA_WRITE_EVEN_STATE_RX, us_msg_len);
                break;

                /*  Data Write Even State  */
            case EOC_DATA_WRITE_EVEN_STATE_RX:
                us_rx_eoc_msg_count = 0;
                RxWriteStateHandler(EOC_EVEN_PARITY, EOC_DATA_WRITE_ODD_STATE_RX, us_msg_len);
                break;

                /*  Data Read Odd State */
            case EOC_DATA_READ_ODD_STATE_RX:

                RxReadStateHandler(EOC_NEXT_ODD_MSG, EOC_DATA_READ_EVEN_STATE_RX, us_msg_len);
                break;

                /*  Data Read Even State */
            case EOC_DATA_READ_EVEN_STATE_RX:

                RxReadStateHandler(EOC_NEXT_EVEN_MSG, EOC_DATA_READ_ODD_STATE_RX, us_msg_len);
                break;

            } /*  switch(.. */
        }
        if (gft_tx_msg_pending == FALSE)
        {
            us_tx_eoc_msg[s_tx_eoc_msg_put_ptr++] = us_rx_eoc;  // echo the received message to ATU-C
            if ((gs_CurrentCoChipset == ADI_CO_CHIPSET) || (gs_CurrentCoChipset ==  ANCDA_CO_CHIPSET))
            {
                if (gft_repeat_echo)
                {
                    us_tx_eoc_msg[s_tx_eoc_msg_put_ptr++] = us_rx_eoc;
                }
            }
        }
        else
        {
            // write into the buffer the new data to be transmitted
            us_tx_eoc_msg[s_tx_eoc_msg_put_ptr++] = gus_eoc_register_data;
        }
        gft_repeat_echo = FALSE;  //This variable is only used for ADI_CO_CHIPSET || ANCDA_CO_CHIPSET
        if(s_tx_eoc_msg_put_ptr >= TX_EOC_MSG_BUF_LEN)
            s_tx_eoc_msg_put_ptr = 0;

    } /*  else, not autonomous msg. */

    us_rx_eoc_msg = 0; /* clear us_rx_eoc_msg */
}

/*****************************************************************************
*   Subroutine Name: RxIdleStateHandler
*
*   Description:
*     This subroutine handles the messages in the IDLE state
*
*   Prototype:
*       void RxIdleStateHandler(void);
*
*   Input Arguments: none
*
*   Output Arguments: none
*
*   Global Variables:
*   gt_RxEocHandler -- the RX EOC handler structure, the following variables are modified:
*       us_rx_eoc_msg_count - Received eoc message count
*       us_RxEocState       - current state of Rx Eoc state machine
*       us_rx_eoc_msg       - current received eoc message
*       us_pre_eoc_msg      - previous received eoc message
*       *puc_rx_eoc_buf     - pointer to the members of ReadEocBuf
*       us_msg_len          - length of the ATU-R data registers
*
*   gt_RxEocDecod -- the RX EOC Decode structure, the following variables are modified:
*       uc_Address          - Address field
*       uc_DataOrOpcode     - Data/Opcode field
*       uc_Parity           - Byte Parity field
*       uc_AutonomousMsg    - Autonomous message field
*       uc_Information      - Information field
*
*   gt_EocRxReadBuf  -- RX EOC Read structure with the following variables initialized:
*       us_vendor_id[EOC_VENDOR_ID_LEN]                     - ATU-R vendor id
*       us_revision_num[EOC_REVISION_NUM_LEN]               - ATU-R Revision number
*       us_serial_num[EOC_SERIAL_NUM_LEN]                   - ATU-R Serial number
*       uc_line_attenuation[EOC_LINE_ATTENUATION_LEN]       - ATU-R Line Attenuation
*       c_snr_margin[EOC_SNR_MARGIN_LEN]                    - ATU-R snr margin
*       uc_atu_r_configuration[EOC_ATU_R_CONFIGURATION_LEN] - ATU-R Configuration
*       uc_link_state[EOC_LINK_STATE_LEN]                   - ATU-R Link state
*       uc_vendor_discr[EOC_VENDOR_DISCR_LEN]               - vendor discretionary information
*       uc_clear_eoc_payload                                - clear eoc payload byte
*
*******************************************************************************/
void RxIdleStateHandler(void)
{
    // IOP_US_ADSL1_IKNS_EOC_MSGS_ForBRCMversion (START)
    FlagT ft_BRCM_Version=0;
    if((((guca_fe_G9941_VendorID[6] << 8) | (guca_fe_G9941_VendorID[7])) == (0X6284))&&
            (gs_CurrentCoChipset == BDCM_CO_CHIPSET))
    {
        ft_BRCM_Version = 1;
    }
    // IOP_US_ADSL1_IKNS_EOC_MSGS_ForBRCMversion (END)
    if (uc_DataOrOpcode == EOC_OPCODE)   /*  check if opcode */
    {

        switch (uc_Information)
        {

        case OPCODE_HOLD:
            RxResetVariables();
            break;

        case OPCODE_RTN:

            RxResetVariables();
            gft_receive_crc_corrupt_bits_flag  = FALSE;
            gft_send_crc_corrupt_bits_flag  = FALSE;
            break;

        case OPCODE_REQTPU:

            /* UpdateTestParameters();   // is a placeholder function for now */
            RxResetVariables();
            break;

        case OPCODE_SLFTST:

            /* PerformSelfTest();        // is a placeholder function for now */
            RxResetVariables();
            break;

        case OPCODE_WRITE5:
        case OPCODE_WRITE6:

            puc_rx_eoc_buf = uc_vendor_discr;
            RxSetStateVariables(EOC_DATA_WRITE_ODD_STATE_RX, EOC_VENDOR_DISCR_LEN);
            break;

        case OPCODE_READ1:

            puc_tx_eoc_buf = (uint8*) us_vendor_id;
            if(STATArray[STAT_Misc] & STAT_T1413_Signal_Detected)
                RxSetStateVariables(EOC_DATA_READ_ODD_STATE_RX, EOC_T1413_VENDOR_ID_LEN);
            else
                RxSetStateVariables(EOC_DATA_READ_ODD_STATE_RX, EOC_VENDOR_ID_LEN);
            break;

        case OPCODE_READ2:

            puc_tx_eoc_buf = (uint8*) us_revision_num;
            RxSetStateVariables(EOC_DATA_READ_ODD_STATE_RX, EOC_REVISION_NUM_LEN);
            break;

        case OPCODE_READ3:

            puc_tx_eoc_buf = (uint8*) us_serial_num;
            RxSetStateVariables(EOC_DATA_READ_ODD_STATE_RX, EOC_SERIAL_NUM_LEN);
            break;

        case OPCODE_READ4:
        case OPCODE_READ5:
        case OPCODE_READ6:

            puc_tx_eoc_buf = uc_vendor_discr;
            RxSetStateVariables(EOC_DATA_READ_ODD_STATE_RX, EOC_VENDOR_DISCR_LEN);
            break;

        case OPCODE_READ7:

            puc_tx_eoc_buf = uc_line_attenuation;
            RxSetStateVariables(EOC_DATA_READ_ODD_STATE_RX, EOC_LINE_ATTENUATION_LEN);
            break;

        case OPCODE_READ8:
#ifndef ISDN   // Only for Anx-A
          // XDSLRTFW-1681 IOP_DS_ADSL1_CTLM_EOC_MARGIN (START)
          // LS FS+ CTLM/0278 parsed SNR margin data of 0x0E dB
          // as EOD (End Of Data command) cuasing 0-dB DS margin reporting on DSLAM.
          // To work around this issue, report 0x0F dB margin instead of 0x0E.
          if ((c_snr_margin[0] == 0x0E) && (gs_CurrentCoChipset == CTLM_CO_CHIPSET))
           c_snr_margin[0] = 0x0F;
          // XDSLRTFW-1681 IOP_DS_ADSL1_CTLM_EOC_MARGIN (END)
#endif   // ifndef ISDN

            puc_tx_eoc_buf = (uint8 *) c_snr_margin;
            RxSetStateVariables(EOC_DATA_READ_ODD_STATE_RX, EOC_SNR_MARGIN_LEN);
            break;

        case OPCODE_READ9:

            puc_tx_eoc_buf = uc_atu_r_configuration;
            RxSetStateVariables(EOC_DATA_READ_ODD_STATE_RX, EOC_ATU_R_CONFIGURATION_LEN);
            break;

        case OPCODE_NOTCOR:
            // IOP_US_ADSL1_IKNS_EOC_MSGS(START_END)
            // IOP_US_ADSL1_IKNS_EOC_MSGS_ForBRCMversion (START_END)
            if(!(ft_BRCM_Version)) us_rx_eoc_msg_count = 0;
            gft_tx_msg_pending = FALSE;
            gft_receive_crc_corrupt_bits_flag = TRUE;
            break;

        case OPCODE_REQCOR:
            // IOP_US_ADSL1_IKNS_EOC_MSGS(START_END)
            // IOP_US_ADSL1_IKNS_EOC_MSGS_ForBRCMversion (START_END)
            if(!(ft_BRCM_Version)) us_rx_eoc_msg_count = 0;
            gft_tx_msg_pending = FALSE;
            gft_send_crc_corrupt_bits_flag = TRUE;
            break;

        case OPCODE_NOTEND:

            RxResetVariables();
            gft_receive_crc_corrupt_bits_flag  = FALSE;
            break;

        case OPCODE_REQEND:

            RxResetVariables();
            gft_send_crc_corrupt_bits_flag  = FALSE;
            break;

        case OPCODE_GNTPDN:
            gft_response_to_powerdown = GRANTED;
            RxResetVariables();
            break;

        case OPCODE_REJPDN:
            gft_response_to_powerdown = REJECTED;
            RxResetVariables();
            break;

        case OPCODE_WRITE11:
            puc_rx_eoc_buf = gt_EocRxReadBuf.uc_link_state;
            RxSetStateVariables(EOC_DATA_WRITE_ODD_STATE_RX, EOC_LINK_STATE_LEN);
            break;

        case OPCODE_READ11:
            puc_tx_eoc_buf = gt_EocRxReadBuf.uc_link_state;
            RxSetStateVariables(EOC_DATA_READ_ODD_STATE_RX, EOC_LINK_STATE_LEN);
            break;

        case OPCODE_EOD:
            // IOP_US_ADSL1_IKNS_EOC_MSGS(START_END)
            // IOP_US_ADSL1_IKNS_EOC_MSGS_ForBRCMversion (START_END)
            if(!(ft_BRCM_Version)) us_rx_eoc_msg_count = 0;
            gft_tx_msg_pending = FALSE;
            break;

        default:    /*  for any other opcodes received, send "Unable to Comply" message */
            //Note: OPCODE_NEXT is included in this default case. We should send UTC if recieve
            //NEXT message in IDLE state. This will prevent us hanging in IDLE state echoing
            //back NEXT message to CO.
            // IOP_US_ADSL1_IKNS_EOC_MSGS(START_END)
            // IOP_US_ADSL1_IKNS_EOC_MSGS_ForBRCMversion (START_END)
            if(!(ft_BRCM_Version))us_rx_eoc_msg_count = 0;
            gft_tx_msg_pending = TRUE;
            gus_eoc_register_data = EOC_UTC_MSG;

#ifdef COLLECT_SHOWTIME_EVENTS_INFO
            if ((gs_RxShowtimeEventsInfoEnable & 0x1) != 0)
            {
                if (gs_RxShowtimeEventsInfoBufferIndex < (2*MAX_STATES - 20))
                {
                    gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) 0xAAAB;           /* delimiter indicating Eoc information and xmitting UTC */
                    gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) gl_RxSymbolCount; /* rx symbol count */
                }
                else
                {
                    gs_RxShowtimeEventsInfoBufferIndex = gs_InitRxShowtimeEventsInfoBufferIndex;
                }
                if (gs_febe_cnt > gs_RxShowtimeEventsError_threshold)
                    gs_InitRxShowtimeEventsInfoBufferIndex = 2*MAX_STATES;
            }

#endif /* COLLECT_SHOWTIME_EVENTS_INFO */

            break;

        } /*  switch (uc_Information) */

    } /*  if (uc_DataOrOpcode == 1) */
    else    /*  if receiving data in this state, send "unable to comply" message */
    {
        gft_tx_msg_pending = TRUE;
        gus_eoc_register_data  = EOC_UTC_MSG;

#ifdef COLLECT_SHOWTIME_EVENTS_INFO
        if ((gs_RxShowtimeEventsInfoEnable & 0x1) != 0)
        {
            if (gs_RxShowtimeEventsInfoBufferIndex < (2*MAX_STATES - 20))
            {
                gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) 0xAAAC;            /* delimiter indicating Eoc information and xmitting UTC */
                gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) gl_RxSymbolCount;  /* rx symbol count */
            }
            else
            {
                gs_RxShowtimeEventsInfoBufferIndex = gs_InitRxShowtimeEventsInfoBufferIndex;
            }
            if (gs_febe_cnt > gs_RxShowtimeEventsError_threshold)
                gs_InitRxShowtimeEventsInfoBufferIndex = 2*MAX_STATES;
        }

#endif /* COLLECT_SHOWTIME_EVENTS_INFO */

    }
}

/*****************************************************************************
*   Subroutine Name: RxWriteStateHandler
*
*   Prototype:
*       void RxWriteStateHandler(uint8 uc_par, uint16 us_nextstate, uint16 us_msg_length);
*
*   Input Arguments:    none
*
*   Output Arguments:   none
*
*   Global Variables:
*   gt_RxEocHandler -- the RX EOC handler structure, the following variables are modified:
*       us_rx_eoc_msg_count - Received eoc message count
*       us_RxEocState       - current state of Rx Eoc state machine
*       us_rx_eoc_msg       - current received eoc message
*       us_pre_eoc_msg      - previous received eoc message
*       *puc_rx_eoc_buf     - pointer to the members of ReadEocBuf
*       us_msg_len          - length of the ATU-R data registers
*
*   gt_RxEocDecod -- the RX EOC Decode structure, the following variables are modified:
*       uc_Address          - Address field
*       uc_DataOrOpcode     - Data/Opcode field
*       uc_Parity           - Byte Parity field
*       uc_AutonomousMsg    - Autonomous message field
*       uc_Information      - Information field
*
*******************************************************************************/
void RxWriteStateHandler(uint8 uc_par, uint16 us_nextstate, uint16 us_msg_length)
{
    uint8 uca_tx_data[2];

    if ((uc_DataOrOpcode == EOC_OPCODE) && (uc_Information == OPCODE_EOD))
        RxResetVariables();
    else
    {

        /*  if the eoc msg. is a data and of appropriate parity */
        if ((uc_DataOrOpcode == EOC_DATA) && (uc_Parity == uc_par))
        {

            uc_rx_info_buf_index++;
            if (uc_rx_info_buf_index > us_msg_length)
            {

                /*  exceeded data register length and hence send EOD command */
                gft_tx_msg_pending = TRUE;

                TxFormEocMsg(EOC_ATU_R, EOC_OPCODE, EOC_ODD_PARITY, 1, OPCODE_EOD, uca_tx_data);
                gus_eoc_register_data = (uca_tx_data[1] << 8) | uca_tx_data[0];
            }
            else   /*  write data into the data registers     */
            {
                gft_tx_msg_pending = FALSE;
                *puc_rx_eoc_buf++ = uc_Information;

                gus_eoc_register_data = us_rx_eoc_msg;
                us_RxEocState = us_nextstate;
            }
        }
        else   /*  send Unable To Comply msg. */
        {

            gft_tx_msg_pending = TRUE;
            gus_eoc_register_data = EOC_UTC_MSG;

#ifdef COLLECT_SHOWTIME_EVENTS_INFO
            if ((gs_RxShowtimeEventsInfoEnable & 0x1) != 0)
            {
                if (gs_RxShowtimeEventsInfoBufferIndex < (2*MAX_STATES - 20))
                {
                    gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) 0xAAAD;           /* delimiter indicating Eoc information and xmitting UTC */
                    gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) gl_RxSymbolCount; /* rx symbol count */
                }
                else
                {
                    gs_RxShowtimeEventsInfoBufferIndex = gs_InitRxShowtimeEventsInfoBufferIndex;
                }
                if (gs_febe_cnt > gs_RxShowtimeEventsError_threshold)
                    gs_InitRxShowtimeEventsInfoBufferIndex = 2*MAX_STATES;
            }

#endif /* COLLECT_SHOWTIME_EVENTS_INFO */


        }
    }
}

/*****************************************************************************
*   Subroutine Name: RxReadStateHandler
*
*   Prototype:
*       void RxReadStateHandler(uint16 us_rx_acceptable_msg, uint16 us_nextstate, uint16 us_msg_length);
*
*   Input Arguments: none
*
*   Output Arguments: none
*
*   Global Variables:
*   gt_RxEocHandler -- the RX EOC handler structure, the following variables are modified:
*       us_rx_eoc_msg_count - Received eoc message count
*       us_RxEocState       - current state of Rx Eoc state machine
*       us_rx_eoc_msg       - current received eoc message
*       us_pre_eoc_msg      - previous received eoc message
*       *puc_rx_eoc_buf     - pointer to the members of ReadEocBuf
*       us_msg_len          - length of the ATU-R data registers
*
*   gt_RxEocDecod -- the RX EOC Decode structure, the following variables are modified:
*       uc_Address          - Address field
*       uc_DataOrOpcode     - Data/Opcode field
*       uc_Parity           - Byte Parity field
*       uc_AutonomousMsg    - Autonomous message field
*       uc_Information      - Information field
*
*******************************************************************************/
void RxReadStateHandler(uint16 us_rx_acceptable_msg, uint16 us_nextstate, uint16 us_msg_length)
{
    uint8 uca_tx_data[2];

    if (us_rx_eoc_msg == EOC_HOLD_MSG)
    {
        /*  go to idle state without affecting latched conditions */
        RxResetVariables();
    }
    else if (us_rx_eoc_msg == EOC_RTN_MSG)
    {
        /*  release latched conditions and go back to Idle state */
        RxResetVariables();
        gft_receive_crc_corrupt_bits_flag  = FALSE;
        gft_send_crc_corrupt_bits_flag  = FALSE;
    }
    else if (us_rx_eoc_msg == us_rx_acceptable_msg)
    {

        gft_tx_msg_pending = TRUE;

        uc_rx_info_buf_index++;
        if (uc_rx_info_buf_index > us_msg_length)
        {

            /*  exceeded data register length and hence send EOD command */
            TxFormEocMsg(EOC_ATU_R, EOC_OPCODE, EOC_ODD_PARITY, 1, OPCODE_EOD, uca_tx_data);
            gus_eoc_register_data = (uca_tx_data[1] << 8) | uca_tx_data[0];


#ifdef COLLECT_SHOWTIME_EVENTS_INFO
            if ((gs_RxShowtimeEventsInfoEnable & 0x1) != 0)
            {
                if (gs_RxShowtimeEventsInfoBufferIndex < (2*MAX_STATES - 20))
                {
                    gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) 0xAABA;           /* delimiter indicating Eoc information and xmitting UTC */
                    gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) gl_RxSymbolCount; /* rx symbol count */
                }
                else
                {
                    gs_RxShowtimeEventsInfoBufferIndex = gs_InitRxShowtimeEventsInfoBufferIndex;
                }
                if (gs_febe_cnt > gs_RxShowtimeEventsError_threshold)
                    gs_InitRxShowtimeEventsInfoBufferIndex = 2*MAX_STATES;
            }

#endif /* COLLECT_SHOWTIME_EVENTS_INFO */

        }
        else
        {

            /*  read data from the data registers */
            TxFormEocMsg(EOC_ATU_R, EOC_DATA, uc_Parity, 1, *puc_tx_eoc_buf++, uca_tx_data);
            gus_eoc_register_data = (uca_tx_data[1] << 8) | uca_tx_data[0];

            us_RxEocState = us_nextstate;

#ifdef COLLECT_SHOWTIME_EVENTS_INFO
            if ((gs_RxShowtimeEventsInfoEnable & 0x1) != 0)
            {
                if (gs_RxShowtimeEventsInfoBufferIndex < (2*MAX_STATES - 20))
                {
                    gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) 0xAAAF;           /* delimiter indicating Eoc information and xmitting UTC */
                    gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) gl_RxSymbolCount; /* rx symbol count */
                }
                else
                {
                    gs_RxShowtimeEventsInfoBufferIndex = gs_InitRxShowtimeEventsInfoBufferIndex;
                }
                if (gs_febe_cnt > gs_RxShowtimeEventsError_threshold)
                    gs_InitRxShowtimeEventsInfoBufferIndex = 2*MAX_STATES;
            }

#endif /* COLLECT_SHOWTIME_EVENTS_INFO */

        }
    }
    else
    {

        /* send Unable To Comply msg. */
        RxResetVariables();
        gft_tx_msg_pending = TRUE;
        gus_eoc_register_data = EOC_UTC_MSG;

#ifdef COLLECT_SHOWTIME_EVENTS_INFO
        if ((gs_RxShowtimeEventsInfoEnable & 0x1) != 0)
        {
            if (gs_RxShowtimeEventsInfoBufferIndex < (2*MAX_STATES - 20))
            {
                gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) 0xAAAE;            /* delimiter indicating Eoc information and xmitting UTC */
                gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) gl_RxSymbolCount;  /* rx symbol count */
            }
            else
            {
                gs_RxShowtimeEventsInfoBufferIndex = gs_InitRxShowtimeEventsInfoBufferIndex;
            }
            if (gs_febe_cnt > gs_RxShowtimeEventsError_threshold)
                gs_InitRxShowtimeEventsInfoBufferIndex = 2*MAX_STATES;
        }

#endif /* COLLECT_SHOWTIME_EVENTS_INFO */


    }
}

/*****************************************************************************
*   Subroutine Name: RxResetVariables
*
*   Description:
*     This subroutine resets the variables to their corresponding values
*
*   Prototype:
*       void RxResetVariables(void);
*
*   Input Arguments: none
*
*   Output Arguments: none
*
*******************************************************************************/
void RxResetVariables(void)
{
    /*  transition RX to IDLE state    */
    us_RxEocState           = EOC_IDLE_STATE_RX;
    us_rx_eoc_msg_count     = 0;
    gft_tx_msg_pending      = FALSE;
}

/*****************************************************************************
*   Subroutine Name: RxSetStateVariables
*
*   This subroutine sets the variables to their corresponding values depending
*   on the type of EOC msg. received
*
*   Prototype:
*       void RxSetStateVariables(uint16 us_rx_state, uint16 us_msglen)
*
*   Input Arguments:
*       us_msg_length   - length of the ATU-R data registers
*       us_rx_state     - next state of Rx Eoc state machine
*
*   Output Arguments:   none
*
*******************************************************************************/
void RxSetStateVariables(uint16 us_rx_state, uint16 us_msg_length)
{
    gft_tx_msg_pending = FALSE;
    us_RxEocState = us_rx_state;
    us_msg_len    = us_msg_length;
    uc_rx_info_buf_index = 0;
}

/* undefine constants for clarifying code */
#undef uc_Address
#undef uc_DataOrOpcode
#undef uc_Parity
#undef uc_AutonomousMsg
#undef uc_Information

/*****************************************************************************
*   Subroutine Name: RxEocMsgDecod
*
*   Description:
*     This subroutine decodes the message from received EOC bytes.
*
*   Prototype:
*       void RxEocMsgDecod(int16 s_eoc_msg, RxEocDecodStruct_t *pt_DecodMsg);
*
*   Input Arguments:
*       s_eoc_msg   - current 16bit EOC message
*
*   Output Arguments:
*       pt_DecodMsg - pointer to the structure containing RX msg decoder variables
*
*   Return: none
*
*******************************************************************************/

void RxEocMsgDecod(int16 s_EocMsg, RxEocDecodStruct_t *pt_DecodMsg)
{
    int16 s_msg;

    s_msg = (s_EocMsg >> 2);
    pt_DecodMsg -> uc_Address       = s_msg & 0x03; /*  extract address */

    s_msg = (s_msg >> 2);
    pt_DecodMsg -> uc_DataOrOpcode  = s_msg & 0x01; /*  extract data or opcode */

    s_msg = (s_msg >> 1);
    pt_DecodMsg -> uc_Parity        = s_msg & 0x01; /*  extract parity */

    s_msg = (s_msg >> 1);
    pt_DecodMsg -> uc_AutonomousMsg = s_msg & 0x01; /*  extract autonomous msg field */

    s_msg = (s_msg >> 1);
    pt_DecodMsg -> uc_Information   = s_msg & 0x01; /*  extract bit 6 of information field */

    s_msg = (s_msg >> 1);
    pt_DecodMsg -> uc_Information   |= (s_msg & 0xfe);  /*  extract bit 7-13 of information field */


#ifdef COLLECT_SHOWTIME_EVENTS_INFO
    if ((gs_RxShowtimeEventsInfoEnable & 0x1) != 0)
    {
        if (gs_RxShowtimeEventsInfoBufferIndex < (2*MAX_STATES - 20))
        {
            gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) 0xAAAA;          /* delimiter indicating Eoc information */
            gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) gl_RxSymbolCount;   /* rx symbol count */
            gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) pt_DecodMsg -> uc_DataOrOpcode; /* indicates if the next byte is a opcode or a data */
            gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) pt_DecodMsg -> uc_AutonomousMsg;   /* indicates if the next byte is a autonomous message */
            gsa_StatesTrail[gs_RxShowtimeEventsInfoBufferIndex++] = (int16) pt_DecodMsg -> uc_Information;  /* received eoc message */

        }
        else
        {
            gs_RxShowtimeEventsInfoBufferIndex = gs_InitRxShowtimeEventsInfoBufferIndex;
        }
        if (gs_febe_cnt > gs_RxShowtimeEventsError_threshold)
            gs_InitRxShowtimeEventsInfoBufferIndex = 2*MAX_STATES;
    }

#endif //#ifdef COLLECT_SHOWTIME_EVENTS_INFO

}


void Rx_DMT_ClearEOC_msg_handler(void)
{
    uint8 *puc_payload;

    // Per G.997.1, an HDLC message has two header bytes: 0xFF followed by 0x03.

    puc_payload = guca_rx_hdlc_msg_buffer + 2;

    //Right now, because of the codeswap page limit, we put the following function in the ShowBG2 page.
    //So, we have to postpone process until the ShowBG2 page (the last showtime page) is swapped in
    if (((STATArray[STAT_Misc] & STAT_AllCodeSwapsDone) == STAT_AllCodeSwapsDone)
            && (!(gt_RxClearEocBuf_CMV.s_RxClearEocBuf_Disable)))
    {
        // Strip off 2-byte header and 2-byte trailing FCS.
        CopyBufferToRxClearEOC(puc_payload, gus_rx_msg_index-4);

#ifdef TARGET_HW
        QueueInterruptRequest(OMB_CLEAREOC_INTERRUPT_CODE);
#endif
    }
}


/* undefine constants for code clarity*/
#undef s_tx_eoc_msg_put_ptr
#undef s_tx_eoc_msg_get_ptr

#undef us_tx_eoc_msg
#undef puc_tx_eoc_buf
#undef us_rx_eoc_msg_count
#undef us_rx_eoc_msg
#undef us_pre_eoc_msg
#undef us_RxEocState
#undef uc_rx_info_buf_index
#undef us_msg_len
#undef puc_rx_eoc_buf
#undef us_vendor_id
#undef us_revision_num
#undef us_serial_num
#undef uc_line_attenuation
#undef c_snr_margin
#undef uc_atu_r_configuration
#undef uc_vendor_discr
#undef uc_clear_eoc_payload
