/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2002 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *-------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   detect.c
 *
 *   detect tone etc
 *
 *-------------------------------------------------------------------------
 */

#include "common.h"
#include "dsp_op.h"
#include "dsp_op2.h"
#include "gdata.h"
#include "gdata_bis.h"
#include "cmv.h"
#include "vecpwr.h"
#include "detect.h"
#include "mul.h"

/*
 *------------------------------------------------------------------------
 *
 *  Name : DetectMultiTones
 *
 *  Description: DetectMultiTones() - Detect tones in given frequency bins of
 *  the carrier set. s_NumCarRequired tones have to be present to return TRUE,
 *  otherwise return FALSE.
 *
 *  Parameters:
 *     int16 sa_CarrierSet[]       - Frequency bins of the carrier set to be used
 *     int16 s_NumCarriers         - Number of subcarriers in the carrier set
 *     int16 s_NumCarriersRequired - Number of subcarriers in the carrier set that
 *                                   must be detected to return TRUE
 *     int16 s_threshold           - detection scale
 *
 *  Returns:
 *     TRUE                        - tone present in at least s_NumCarriersRequired
 *                                   of the s_NumCarriers bins of sa_CarrierSet
 *     FALSE                       - tone not present in at least s_NumCarriersRequired
 *                                   of the s_NumCarriers bins of sa_CarrierSet
 *
 *  Global Variables Used:
 *      gsa_FFT_InBuf[]            - (O) output of FFT
 *      gsa_RxToneBuf[]            - (O) frequency domains samples
 *
 *------------------------------------------------------------------------
 */
FlagT DetectMultiTones(int16 *psa_inbuf, CONSTANT int16 sa_CarrierSet[], int16 s_NumCarriers, int16 s_NumCarriersRequired, int16 s_threshold) {

   int i;
   int16 s_FirstTone, s_LastTone;
   int16 s_NumCarriersDetected;


   /* ---- detect tones ---- */
   s_NumCarriersDetected = 0;  /*  Keep track of the number of subcarriers in the carrier set that are detected */
   for ( i = 0; i < s_NumCarriers; i++ ) {
      s_FirstTone = sa_CarrierSet[i] - HALF_DETECT_BAND;
      s_LastTone  = sa_CarrierSet[i] + HALF_DETECT_BAND;
      if ( DetectTone(psa_inbuf , sa_CarrierSet[i], s_FirstTone, s_LastTone, s_threshold ) )
         s_NumCarriersDetected++;
   }

   return (FlagT)( s_NumCarriersDetected >= s_NumCarriersRequired );

}   /*  DetectMultiTones */

/*****************************************************************************
;  Subroutine Name: DetectTone(...)
;
;  This subroutine detects if the data in the input buffer contains the
;  tone of interest. The function returns TRUE if the tone is detected,
;  FALSE otherwise. The algorithm used for the tone detection is based on the
;  following equation:
;
;     Power_of_Tone
;   ---------------------- > Detection_Threshold
;     Average_Power
;
;  where Power_of_Tone is the power of the tone to be detected,
;  Average_Power is the average power of tones around the detected tone, excluding
;  the detected tone.
;
;  Under strong echo, the above algorithm does not work effectively since tone power
;  is flooded by surrounding echo power. This occurs in BIS when we need to detect
;  between Reverb and C-TREF w/o an AEC . Under such circumstances, we do a pre-train
;  and measure the average power when RT and CO are both transmitting Reverb, which serves
;  as a detection threshold. The tone detection then is based on the following equation:
;
;                    Rev_Reference_Power
;  Average_Power  >  ----------------------------
;                          2
;  where Average_Power is the average power of tones around the detected tone, excluding
;  the detected tone.
;  Rev_Reference_Power is the averaged Average_power computed durin pre-train
;
;
;  Prototype:
;     int16 DetectTone(int16 *psa_inbuf, int16 s_tone_idx, int16 s_first_tone, int16 s_last_tone, int16 s_threshold)
;
;  Input Arguments:
;     psa_inbuf      - pointer to the input buffer containing received DMT tones
;     s_tone_idx     - index of the tone to be detected;
;     s_first_tone   - the index of the first tone used in the detection
;     s_last_tone    - the index of the last tone used in the detection
;     s_threshold    - detection threshold in Q8.8 format
;
;  Output Arguments:
;     none
;
;  Return:
;     TRUE            - the tone is not detected;
;     FALSE           - the tone is detected.
;
;  Global Variables:
;     none
;
*******************************************************************************/
FlagT DetectTone(int16 *psa_inbuf, int16 s_tone_idx, int16 s_first_tone, int16 s_last_tone, int16 s_threshold)
{
    int16 s_tone_power, s_rx_power;
    int32 l_tone_power, l_rx_power;
   int16  i, s_RightShift;
   FlagT ft_isRevRefPwrDetection = FALSE;
   int16 s_shift = 3;

   /* In BIS, for C-TREF detections we do detection based on a reference threshold power that is obtained */
   /* with a pre-train when RT and CO are both transmitting Reverb */
   if (((gs_TxState == R_REVERB1_TX_BIS || gs_TxState == R_REVERB2_TX_BIS) && gl_TxSymbolCount >= 4) || gs_RxState == R_C_REVERB3_RX_BIS) /* Assure that Tx is sending Reverb, before we use the reference power */
   {
      ft_isRevRefPwrDetection = TRUE;
      s_shift = gs_DetectTone_PwrCalcShifts;
   }

   // Calculate power in tone being detected.
   l_tone_power = VectorPower(psa_inbuf, (int16) (2*s_tone_idx), 2, s_shift);

   // Calculate total power in band of tones from first_tone to last_tone.
   l_rx_power = VectorPower(psa_inbuf, (int16) (2*s_first_tone), (int16)(2*(s_last_tone - s_first_tone+1)), s_shift);

   l_rx_power -= l_tone_power;

   /* In BIS, for C-TREF detections we do detection based on a reference threshold power that is obtained */
   /* with a pre-train when RT and CO are both transmitting Reverb */
   if (ft_isRevRefPwrDetection)  /* Assure that Tx is sending Reverb, before we use the reference power */
   {
      /* ==================================================================== */
      /*  Detection declared if:                                              */
      /*  l_rx_power    < gl_DetectTone_RevRefPwr / 2                         */
      /*  gl_DetectTone_RevRefPwr > 2 * l_rx_power                            */
      /* ==================================================================== */

      return (gl_DetectTone_RevRefPwr > 2 * l_rx_power);
   }
   else
   {
      /* If the total power is less than the minimum power threshold, declare no tone */
      if(l_tone_power <= (int32)gs_DetectMin) {
         return(0);
      }

      // Normalize tone_power, rx_power to 16 bits.
      RightNormalizeSignedPair(&l_tone_power, &l_rx_power, 16, &s_RightShift);
      s_tone_power = (int16) l_tone_power;
      s_rx_power   = (int16) l_rx_power;

      /* ==================================================================== */
      /*  Detection declared if: */
      /*  s_tone_power    > s_threshold * s_avg_power */
      /*                  > s_threshold * (s_rx_power / (s_last_tone-s_first_tone)) */
      /*  (s_last_tone-s_first_tone) * s_tone_power > s_threshold * s_rx_power */
      /* ==================================================================== */

      i = s_last_tone - s_first_tone;

      MULS16(l_tone_power, i, s_tone_power);
      MULS16(l_rx_power, s_threshold, s_rx_power);

      /* Since s_threshold is in Q8.8 format, perform the following rounding */
      /* and then right-shift 8 bits */
      l_rx_power = round(l_rx_power, 8);

      return (l_tone_power > l_rx_power);
   }
}
