/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-1998 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;  Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
;  Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;  File Name: dsp_op.c
;
;  This file contains functions emulating DSP operations.
;
;***************************************************************************/

#include "config.h"
#include "const.h"
#include "typedef.h"
#include "dsp_op.h"

/* =============================================== */
/* Local function declarations */
/* =============================================== */

int16 Divide_32by16bit(int32 s_numer_mantissa, int16 s_numer_exponent,
               int16 s_denom_mantissa, int16 s_denom_exponent,
               int16 *ps_quot_mantissa, int16 *ps_quot_exponent
               );

/* =============================================== */
/* constants used by this file only */
/* =============================================== */
#define DSP_OP_PRN FALSE      /*  specify whether printing is turned on (TRUE) or off (FALSE) */
#if DSP_OP_PRN
#include <stdio.h>
#endif

/* =============================================== */
/* static variable declarations */
/* =============================================== */
/* the following is used only by function sature16() */
FlagT gs_Overflow;         /*  indicates that overflow has occurred */


/****************************************************************************
 ;
 ; Subroutine Name : sature16(L_var1)
 ;
 ;  This function limits the 32 bit input to the range of a 16 bit word
 ; with saturation control.
 ;
 ; Prototype:
 ;    int16 sature16(int32 L_var1)
 ;
 ;  Input Arguments:
 ;
 ;    L_var1 - 32 bit long signed integer whose value falls in the
 ;             range : 0x8000 0000 <= L_var1 <= 0x7fff ffff.
 ;
 ;   Output Arguments:
 ;
 ;    none
 ;
 ;  Return Value:
 ;
 ;    s_var_out - 16 bit short signed integer (int16) whose value falls in the
 ;              range : 0xffff 8000 <= var_out <= 0x0000 7fff.
 *****************************************************************************/
int16 sature16(int32 L_var1)
{
   int16 s_var_out;

   gs_Overflow = TRUE;  // Usually overwritten below

    if (L_var1 > (int32)MAX_16) {
        s_var_out = (int16)MAX_16;
    }
    else {
        if (L_var1 < (int32)0xffff8000) {
            s_var_out = (int16)MIN_16;
        }
        else {
         s_var_out = (int16)L_var1;
            gs_Overflow = FALSE;
      }
    }
    return(s_var_out);
}


/*****************************************************************************
 ;
 ;  Subroutine Name: norm_16bit(s_var1)
 ;
 ;  Description:
 ;  Returns the number of left shifts needed to normalize the 16 bit
 ; signed variable s_var1 to Q1.15 format.
 ;
 ; In order
 ;  to normalize the result, the following operation must be done :
 ;                   norm_s_var1 = s_var1 << norm_16bit(s_var1).
 ;
 ;  Prototype:
 ;    int16 norm_16bit(int16 s_var1)
 ;
 ;  Arguments:
 ;    s_var1 -- 16 bit signed integer
 ;
 ;  Outputs:
 ;    none
 ;
 ;  Return Value:
 ;    var_out-- Left shift value.
 ;
 ******************************************************************************/
int16 norm_16bit(int16 s_var1)
{
    int16 s_LeftShift;

    if (s_var1 == 0) {
        s_LeftShift = (int16)0;
    }
    else if (s_var1 == (int16)0xffff) {
      s_LeftShift = (int16)15;
   }
    else {
      if (s_var1 < 0)
         s_var1 = ~s_var1;

      for(s_LeftShift = (int16)0;s_var1 < (int16)0x4000;s_LeftShift++) {
         s_var1 <<= 1;
      }
   }

    return(s_LeftShift);
}

/*****************************************************************************
 ;
 ;  Subroutine Name: norm_l(L_var1)
 ;
 ;  Description:
 ;  Produces the number of left shift needed to normalize the 32 bit varia-
 ;  ble l_var1 for positive values on the interval with minimum of
 ;  1073741824 and maximum of 2147483647, and for negative values on the in-
 ;  terval with minimum of -2147483648 and maximum of -1073741824; in order
 ;  to normalize the result, the following operation must be done :
 ;                   norm_L_var1 = L_shl(L_var1,norm_l(L_var1)).
 ;
 ;  Prototype:
 ;    int16 norm_l(int32 L_var1)
 ;
 ;  Arguments:
 ;    L_var1 -- 32 bit long signed integer (int32) whose value falls in the
 ;              range : 0x8000 0000 <= var1 <= 0x7fff ffff.
 ;
 ;  Outputs:
 ;    none
 ;
 ;  Return Value:
 ;    var_out-- Left shift value.
 ;
 ******************************************************************************/
int16 norm_l(int32 L_var1)
{
    int16 s_var_out;

    if (L_var1 == 0L) {
        s_var_out = (int16)0;  /* crs test */
    }
    else {
        if (L_var1 == (int32)0xffffffffL) {
            s_var_out = (int16)31;
        }
        else {
            if (L_var1 < 0L) {
                L_var1 = ~L_var1;
            }

            for(s_var_out = (int16)0;L_var1 < (int32)0x40000000L;s_var_out++) {
                L_var1 <<= 1L;
            }
        }
    }

    return(s_var_out);
}

/*****************************************************************************
 ;
 ;   Subroutine Name: Divide_16bit( s_numer_mantissa, s_numer_exp,
 ;                         s_denom_mantissa, s_denom_exp,
 ;                         ps_quot_mantissa, ps_quot_exp)
 ;
 ;   Description:
 ;
 ;    Divides 16-bit numerator by 16-bit denominator to get 16-bit quotient.
 ;
 ;    {s_numer_mantissa,s_numer_exponent} and {s_denom_mantissa,s_denom_exponent}
 ; must both represent *normalized* values where
 ; s_numer and s_denom are mantissas in signed format Q1.15, and s_numer_exponent and
 ; s_denom_exponent are signed 16-bit exponents.
 ;
 ; The output quotient is represented by {s_quot_mantissa,s_quot_exponent}
 ;
 ;   s_numer_mantissa and s_denom_mantissa must both be positive.
 ;
 ;   Prototype:
 ;    int16 Divide_16bit(int16 s_numer_mantissa, int16 s_numer_exponent,
 ;             int16 s_denom_mantissa, int16 s_denom_exponent,
 ;             int16 *ps_quot_mantissa, int16 *ps_quot_exponent
 ;             )
 ;
 ;  Input Arguments:
 ;    int16 s_numer_mantissa -- 16-bit normalized numerator mantissa. Must be positive.
 ;    int16 s_numer_exponent -- 16-bit numerator exponent.
 ;    int16 s_denom_mantissa -- 16-bit normalized denominator mantissa. Must be positive.
 ;    int16 s_denom_exponent -- 16-bit denominator exponent.
 ;
 ;   Output Arguments:
 ;    int16 *ps_quot_mantissa -- 16-bit normalized quotient mantissa.
 ;    int16 *ps_quot_exponent -- 16-bit quotient exponent.
 ;
 ;   Return Value:
 ;    0 if no error
 ;    -1 if error (division by zero or negative arguments)
 ;
 ; NOTE: Does not check for overflow of exponents.
 ;
 ******************************************************************************/

int16 Divide_16bit(int16 s_numer_mantissa, int16 s_numer_exponent,
               int16 s_denom_mantissa, int16 s_denom_exponent,
               int16 *ps_quot_mantissa, int16 *ps_quot_exponent
               )
{
   // Convert 16-bit numerator to 32 bit numerator and call other division routine.

   return(Divide_32by16bit((int32)s_numer_mantissa << 16, (int16)(s_numer_exponent-16),
               s_denom_mantissa,  s_denom_exponent,
               ps_quot_mantissa, ps_quot_exponent));
}

/*****************************************************************************
 ;
 ;   Subroutine Name: Divide_32by16bit(   l_numer_mantissa, s_numer_exp,
 ;                            s_denom_mantissa, s_denom_exp,
 ;                            ps_quot_mantissa, ps_quot_exp)
 ;
 ;   Description:
 ;
 ;    {l_numer_mantissa,s_numer_exponent} and {s_denom_mantissa,s_denom_exponent}
 ; must both represent *normalized* values where
 ; l_numer has signed format Q1.31, s_denom has signed format Q1.15, and s_numer_exponent and
 ; s_denom_exponent are signed 16-bit exponents.
 ;
 ; The output quotient is represented by {s_quot_mantissa,s_quot_exponent}
 ;
 ;   s_numer_mantissa and s_denom_mantissa must be positive
 ;
 ;   Prototype:
 ;    int16 Divide_32by16bit(int32 l_numer_mantissa, int16 s_numer_exponent,
 ;             int16 s_denom_mantissa, int16 s_denom_exponent,
 ;             int16 *ps_quot_mantissa, int16 *ps_quot_exponent
 ;             )
 ;
 ;    int32 s_numer_mantissa -- 32-bit normalized numerator mantissa. Must be positive.
 ;    int16 s_numer_exponent -- 16-bit numerator exponent.
 ;    int16 s_denom_mantissa -- 16-bit normalized denominator mantissa. Must be positive.
 ;    int16 s_denom_exponent -- 16-bit denominator exponent.
 ;
 ;   Output Arguments:
 ;    int16 *ps_quot_mantissa -- 16-bit normalized quotient mantissa.
 ;    int16 *ps_quot_exponent -- 16-bit quotient exponent.
 ;
 ;   Return Value:
 ;    0 if no error
 ;    -1 if error (division by zero or negative arguments)
 ;
 ; NOTE: Does not check for overflow of exponents.
 ;
 ******************************************************************************/
int16 Divide_32by16bit(int32 l_numer_mantissa, int16 s_numer_exponent,
               int16 s_denom_mantissa, int16 s_denom_exponent,
               int16 *ps_quot_mantissa, int16 *ps_quot_exponent
               )
{
    int16 s_quot_mantissa;
    int16  iteration, s_First_1bit_position;
    int32 L_numer;
    int32 L_denom, L_NumerPartial;
   uint16 us_Numer_LowWord;
   uint16 us_bitmask;

    if ((l_numer_mantissa < 0) || (s_denom_mantissa <= (int16)0)) {
#if DSP_OP_PRN
      if ((s_denom_mantissa == (int16)0)
         fprintf(stderr, "Division by 0, Fatal error \n");
      else
         fprintf(stderr, "Input arguments for Division don't meet the requirement\n");
#endif
      return(-1);
    }

   L_numer = (int32) l_numer_mantissa;
   L_denom = (int32) s_denom_mantissa;

   L_NumerPartial = L_numer >> 16;

   if (L_NumerPartial >= L_denom){
      // Leftshift denominator by one so that it's strictly greater than numerator.
      L_denom <<= 1;
      s_denom_exponent--;
   }

   // Do long division loop, calculating one bit of quotient per loop.
   // The quotient mantissa has an implicit exponent of +1.
   // Also make note of position of first '1' bit in quotient for purposes of normalizing
   // quotient.

   us_Numer_LowWord = L_numer & 0xffff;
   s_quot_mantissa = 0;
   us_bitmask = 0xffff;
   s_First_1bit_position = 0;

   for(iteration=(int16)0; iteration< (int16)15;iteration++) {
      s_quot_mantissa <<=1;
      // Add the next bit of the full numerator to L_NumerPartial.
      L_NumerPartial = (L_NumerPartial << 1) | (us_Numer_LowWord >> 15);
      us_Numer_LowWord <<= 1;

      if (L_NumerPartial >= L_denom) {
         L_NumerPartial = L_NumerPartial - L_denom;
         s_quot_mantissa += (int16)1;
         // These two lines save position of first '1' bit for
         // normalization purposes, and they do it without a
         // cycle-consuming conditional statement.
         s_First_1bit_position |= iteration & us_bitmask;
         us_bitmask = 0;
      }
   }


   // round the values
   if ((L_NumerPartial >= (L_denom>>1)) &&  (s_quot_mantissa < 0x7FFF))
   {
      s_quot_mantissa++;
   }
   // Re-adjust s_First_1bit_position if rounding changed the position of the first 1-bit
   // (rounding can only shift the position of the first 1-bit left by at most 1)
   if (s_quot_mantissa & (uint16)(1 << (14 - s_First_1bit_position + 1)))
   {
      s_First_1bit_position--;
   }

   // Normalize quotient
   *ps_quot_mantissa = (s_quot_mantissa <<= s_First_1bit_position);

   if (s_quot_mantissa == 0)
      *ps_quot_exponent = 0;     // Force exponent to zero if quotient is zero.
   else
      // Take into account normalization of quotient when calculating quotient exponent.
      *ps_quot_exponent = 1 + s_numer_exponent - s_denom_exponent - s_First_1bit_position;

   return(0);  // No error.
}
//XDSLRTFW-1727
#if 1//L2_13dB
int16 Divide_32by16U(int32 l_numer_mantissa, int16 s_numer_exponent,
                                   uint16 us_denom_mantissa, int16 s_denom_exponent,
                                   int16 *ps_quot_mantissa, int16 *ps_quot_exponent
                                   )
{
    int16 s_quot_mantissa;
    int16  iteration, s_First_1bit_position;
    int32 L_numer;
    int32 L_denom, L_NumerPartial;
        uint16 us_Numer_LowWord;
        uint16 us_bitmask;

    if ((l_numer_mantissa < 0) || (us_denom_mantissa <= (int16)0)) {
#if DSP_OP_PRN
                if ((us_denom_mantissa == (int16)0)
                        fprintf(stderr, "Division by 0, Fatal error \n");
                else
                        fprintf(stderr, "Input arguments for Division don't meet the requirement\n");
#endif
                return(-1);
    }

        L_numer = (int32) l_numer_mantissa;
        L_denom = (int32) us_denom_mantissa;

        L_NumerPartial = L_numer >> 16;

        if (L_NumerPartial >= L_denom){
                // Leftshift denominator by one so that it's strictly greater than numerator.
                L_denom <<= 1;
                s_denom_exponent--;
        }

        // Do long division loop, calculating one bit of quotient per loop.
        // The quotient mantissa has an implicit exponent of +1.
        // Also make note of position of first '1' bit in quotient for purposes of normalizing
        // quotient.

        us_Numer_LowWord = L_numer & 0xffff;
        s_quot_mantissa = 0;
        us_bitmask = 0xffff;
        s_First_1bit_position = 0;

        for(iteration=(int16)0; iteration< (int16)15;iteration++) {
                s_quot_mantissa <<=1;
                // Add the next bit of the full numerator to L_NumerPartial.
                L_NumerPartial = (L_NumerPartial << 1) | (us_Numer_LowWord >> 15);
                us_Numer_LowWord <<= 1;

                if (L_NumerPartial >= L_denom) {
                        L_NumerPartial = L_NumerPartial - L_denom;
                        s_quot_mantissa += (int16)1;
                        // These two lines save position of first '1' bit for
                        // normalization purposes, and they do it without a
                        // cycle-consuming conditional statement.
                        s_First_1bit_position |= iteration & us_bitmask;
                        us_bitmask = 0;
                }
        }


        // round the values
        if ((L_NumerPartial >= (L_denom>>1)) &&  (s_quot_mantissa < 0x7FFF))
        {
                s_quot_mantissa++;
        }
        // Re-adjust s_First_1bit_position if rounding changed the position of the first 1-bit
        // (rounding can only shift the position of the first 1-bit left by at most 1)
        if (s_quot_mantissa & (uint16)(1 << (14 - s_First_1bit_position + 1)))
        {
                s_First_1bit_position--;
        }

        // Normalize quotient
        *ps_quot_mantissa = (s_quot_mantissa <<= s_First_1bit_position);

        if (s_quot_mantissa == 0)
                *ps_quot_exponent = 0;          // Force exponent to zero if quotient is zero.
        else
                // Take into account normalization of quotient when calculating quotient exponent.
                *ps_quot_exponent = 1 + s_numer_exponent - s_denom_exponent - s_First_1bit_position;

        return(0);      // No error.
}


#endif
/*  undefine constant used by this file only */
#undef DSP_OP_PRN

#ifdef HERCULES_ADSL_CPE

/*
*-------------------------------------------------------------------------------
*
*  Prototype: int16 floor16(int16 s_num, int16 s_den)
*
*  This function returns floor(s_num/s_den).
*
*  Input Arguments:
*     s_num: 16-bit numerator
*     s_den: 16-bit denominator
*
*  Output Arguments:
*
*  Returns:
*     floor(s_num/s_den)
*
*  Global Variables:
*
*-------------------------------------------------------------------------------
*/

int16 floor16(int16 s_num, int16 s_den)
{
   int16 s_ans = 0;

   if (s_den == 1)
      s_ans = s_num;
   else if (s_den == 0)
      s_ans = (int16)0x7FFF;
   else {
      while (s_num >= s_den) {
         s_num -= s_den;
         s_ans++;
      }
   }

   return (int16)s_ans;
}

/*
*-------------------------------------------------------------------------------
*
*  Prototype: int16 ceil16(int16 s_num, int16 s_den)
*
*  This function returns ceil(s_num/s_den).
*
*  Input Arguments:
*     s_num: 16-bit numerator
*     s_den: 16-bit denominator
*
*  Output Arguments:
*
*  Returns:
*     ceil(s_num/s_den)
*
*  Global Variables:
*
*-------------------------------------------------------------------------------
*/

int16 ceil16(int16 s_num, int16 s_den)
{
   int16 s_ans = 0;

   if (s_den == 1)
      s_ans = s_num;
   else if (s_den == 0)
      s_ans = (int16)0x7FFF;
   else {
      while (s_num >= s_den) {
         s_num -= s_den;
         s_ans++;
      }
      if (s_num != 0)
         s_ans++;
   }

   return (int16)s_ans;
}

/*
*-------------------------------------------------------------------------------
*
*  Prototype: int16 mod16(int16 s_num, int16 s_den)
*
*  This function returns s_num mod s_den. Inputs are assumed to be positive,
*  and s_den should not be zero.
*
*  Input Arguments:
*     s_num: 16-bit number
*     s_den: 16-bit number
*
*  Output Arguments:
*
*  Returns:
*     s_num mod s_den
*
*  Global Variables:
*
*-------------------------------------------------------------------------------
*/

int16 mod16(int16 s_num, int16 s_den)
{
   if (s_den == 1)
      return (int16)0;

   if (s_den == 0) // error
      return (int16)(0x7FFF);

   while (s_num >= s_den)
      s_num -= s_den;

   return (int16)s_num;
}

#endif // HERCULES_ADSL_CPE
