/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-1998 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   IEEE_FLT.C
 *
 *   Description:  Fixed point emulation of IEEE 754 Single Precision
 *   Floating poing routines
 *
 *------------------------------------------------------------------------
 */

#include "typedef.h"
#include "ieee_flt.h"
#include "dsp_op.h"
#include "mul.h"

/* =============================================== */
/* constants used only by this file */
/* =============================================== */
#define IEEE_FLT_ROUND //used to enable IEEE 754 "toward the nearest representable value" rounding rules */
/* #undef  IEEE_FLT_ROUND */

#define NUM_ROUND_BITS_ADD 6 /* set #of bits for rounding precision in addf32(). Range: 1-6 for 32-bit computation */
#define NUM_ROUND_BITS_MPY 5 /* set ODD (range: 1,3,5) # of bits for rounding precision in mpyf32(). */



/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : f32toint32
 *
 *  Description:    Converts a variable of type Float32 to type int32.
 *  Input values greater than 0x7FFFFFFF will be saturated to the maximum int16
 *  value of 0x7FFFFFFF.  Input values less than -0xFFFFFFF will be saturated to
 *  the minimum int32 value of -0xFFFFFFF. Rounding may be performed by either
 *  rounding towards the nearest integer or by truncating (rounding towards
 *  zero).
 *
 *  Prototype:      int32 f32toint32(Float32 f_in, int16 s_conv_mode)
 *
 *  Input Arguments:
 *      Float32 f_in        Float32 value to be converted
 *      int16    s_conv_mode 0, round to nearset integer
 *                          1, truncate (round towards 0)
 *
 *  Output Arguments:
 *
 *  Return:
 *      int32 representation of f_in
 *
 *  Notes:  Differs from Microsoft Visual C++ (VC) implementation in the
 *  following manners:
 *      1) VC will always truncate when converting from float to int32
 *      2) VC does not saturate when f_in is beyond int32 bounds.
 *      VC will return (f_in MOD 2^32), with truncation
 *
 *------------------------------------------------------------------------
 *^^^
 */

int32 f32toint32(Float32 f_in, int16 s_conv_mode) {

    int32 l_out;

    int16 s_Exp, s_shift;
    Float32 f_Frac;

    /*  unpack uc_Exponent */
    s_Exp  = (int16)(f_in >> EXP32SHIFT_LOW);
    s_Exp &= MASK8LOW;

    /*  test for |f_in| < 0.5 */
    if (s_Exp < (EXPONENT_BIAS - 1)) {
        l_out = 0;
        return l_out;
    }

    /*  test for |f_in| > 2^31 */
    if (s_Exp >= (EXPONENT_BIAS + 31)) {
        if (f_in & NEG32) l_out = (int32) MIN_32;
        else              l_out = (int32) MAX_32;
        return l_out;
    }

    /*  unpack fraction */
    f_Frac  = f_in & MASK23LOW;
    f_Frac += HIDDEN_BIT;

    /*  get shift count for int32 representation */
    s_shift = EXP32SHIFT_LOW - (s_Exp - EXPONENT_BIAS);

    if (s_conv_mode == 0) {
        f_Frac = round(f_Frac,s_shift);
    }
    else {
        f_Frac >>= s_shift;
    }

    /*  get sign and test for overflow */
    if (f_in & NEG32) {
        if (f_Frac > MAX_32) l_out = (int32)MIN_32;
        else                 l_out = - (int32)(f_Frac);
    }
    else {
        if (f_Frac > MAX_32) l_out = (int32)MAX_32;
        else                 l_out = (int32)(f_Frac);
    }

   return(l_out);
}


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : absf32
 *
 *  Description:    Finds the absolute value of a floating point number
 *
 *  Prototype:  Float32 absf32(Float32 f_in)
 *
 *  Input Arguments:
 *      f_in
 *
 *  Output Arguments:
 *
 *  Return:
 *      | f_in |
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

Float32 absf32(Float32 f_in) {

    Float32 f_sign;

    f_sign = f_in & NEG32;

    if (f_sign == NEG32) {
        f_in ^= NEG32;
    }

    return f_in;

}



/*^^^
 *------------------------------------------------------------------------
 *
 *  Name :      expf32()
 *
 *  Description:    Given Float32 value f_x, returns the largest integer
 *  s_x such that
 *      2^(s_x) <= f_x
 *
 *  Prototype:  int16 expf32(Float32 f_x)
 *
 *  Input Arguments:
 *      f_x
 *
 *  Output Arguments:
 *
 *  Return:
 *     s_x      largest integer uc_Exponent of 2 such that 2^s_x >= f_x
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

int16 expf32(Float32 f_x) {

    static int16 s_x;

    s_x  = (int16) (f_x >> EXP32SHIFT_LOW);
    s_x &= MASK8LOW;
    s_x  = s_x - EXPONENT_BIAS;

    return s_x;

}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name :      pow2f32()
 *
 *  Description:    Given an int16 value, s_x, returns 2^s_x
 *
 *  Prototype:  Float32 pow2f32(int16 s_x)
 *
 *  Input Arguments:
 *      s_x     power of 2 to be computed
 *
 *  Output Arguments:
 *
 *  Return:
 *      2^s_x
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

Float32 pow2f32(int16 s_x) {

    Float32 f_z;

    /*  check limits to see if 2^s_x can be represented as a Float32 */
    if (s_x < -126) return 0;
    if (s_x > 127)  return REAL_INF;

    /*  pack uc_Exponent */
    f_z = ((Float32)s_x + EXPONENT_BIAS)<<EXP32SHIFT_LOW;

    return f_z;

}

#undef IEEE_FLT_ROUND
#undef NUM_ROUND_BITS_ADD
#undef NUM_ROUND_BITS_MPY
