/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-1998 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*****************************************************************************
;
;   Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
;   Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;   FDQcoefPerTone.c
;
;  Routine to perform frequency-domain equalization (FDQ) initialization.
;
;****************************************************************************/

#include "common.h"
#include "dsp_op.h"
#include "dsp_op2.h"
#include "gdata.h"
#include "gdata_bis.h"
#include "rx_ops.h"
#include "bufmisc.h"

/*****************************************************************************
;  Subroutine Name: FDQcoefPerTone(int16 *psa_in_tone, int16 *psa_ref_tone, int16 *psa_FDQ_coef, int16 *ps_FDQ_exp)
;
;  Description:
;     This routine calculates the FDQ coefficient for a single channel using
;     the following method:
;
;     FDQ_coef[i] = reference_tone[i]/input_tone[i] for each channel i, where
;     all of FDQ_coef[i], reference_tone[i] and input_tone[i] are complex numbers.
;     FDQ_coef[i] is represented Wr[i], Wi[i] and Ws[i] in the form of (Wr[i] + jWi[i]) * 2^Ws[i]
;     where Wr[i] and Wi[i] are mantissas and Ws[i] is their exponent.
;     Wr[i], Wi[i] and Ws[i] are all 16-bit values.
;
;     The fixed point format of the FDQ coefficients Wr[i] and Wi[i] is
;           Q(FDQ_MANTISSA_WORDLENGTH - FDQ_MANTISSA_FRAC_BITS,FDQ_MANTISSA_FRAC_BITS).
;
;  Prototype:
;       void FDQcoefPerTone(int16 *psa_in_tone, int16 *psa_ref_tone, int16 *psa_FDQ_coef, int16 *ps_FDQ_exp);
;
;  Input Arguments:
;     psa_in_tone - pointer to representative input tone
;     psa_ref_tone -- pointer to reference tone
;
;  Output Arguments:
;     psa_FDQ_coef -- pointer to the computed mantissa of complex FDQ coefficient
;     ps_FDQ_exp -- pointer to the common exponent of this pair of FDQ coefficients
;
;  Return Value:
;     none
;
;  Global Variables:
;     none
;
;****************************************************************************/

C_SCOPE void FDQcoefPerTone(int16 *psa_in_tone, int16 *psa_ref_tone, int16 *psa_FDQ_coef, uint8 *ps_FDQ_exp)
{
   int16 s_Xr, s_Xi, s_Yr, s_Yi;
   int32 l_Acc, l_term1, l_term2;
   int16 s_exp_den, s_denum, s_rcp_denum, s_exp_rcp;
   int16 s_exp_num_r, s_num_r;
   int16 s_exp_num_i, s_num_i;
   int32 l_mant_r, l_mant_i;
   int16 s_exp_r, s_exp_i, s_exp_delta, s_RightShift;

   /* Get input tone */
   s_Xr = psa_in_tone[0];
   s_Xi = psa_in_tone[1];

   /* Get reference tone */
   s_Yr = psa_ref_tone[0];
   s_Yi = psa_ref_tone[1];

   // ================================================================ */
   //
   // FDQ coef = ref/input = (Yr+jYi)/(Xr+jXi)
   //
   // Multiply numerator and denominator by (Xr-jXi) to make
   // denominator real.
   //
   // ref/input = (Yr+jYi)/(Xr+jXi) * (Xr-jXi)/(Xr-jXi)
   //
   //       = [(Xr*Yr + Xi*Yi) + j(Xr*Yi - Xi*Yr)] / (Xr^2 + Xi^2)
   //
   //
   // ================================================================ */

   // ================================================================
   // Step 1. Calculate (real) denominator
   // ================================================================ */
   l_Acc = (int32)s_Xr * s_Xr + (int32)s_Xi * s_Xi;

   if (l_Acc < 0){
      // Check for overflow. Check is simple since s_Xr*s_Xr and
      // s_Xi*s_Xi are always positive.
      l_Acc = 0x7fffffff;
   }
   else if (l_Acc == 0){
      // if input energy is zero, put FDQ to be maximum
      psa_FDQ_coef[0] = FDQ_MANTISSA_MAX;
      psa_FDQ_coef[1] = 0;
      *ps_FDQ_exp = FDQ_EXPONENT_MAX;
      return;
   }

   /* Get the normalization left shift count for the denominator */
   s_exp_den = -norm_l(l_Acc);

   /* Normalize the denominator and represent it in 16 bit */
   /* and Round it Up */
   s_denum =(int16) (round((l_Acc << -s_exp_den), 16));
   // ==================================================================================
   // Step 2: Compute real part of numerator = (Xr*Yr + Xi*Yi)
   // ==================================================================================

   l_term1 = (int32)s_Xr * s_Yr;
   l_term2 = (int32)s_Xi * s_Yi;

   l_Acc = l_add(l_term1, l_term2); // Add with saturation.

   /* Get the normalization shift count */
   s_exp_num_r = -norm_l(l_Acc);

   /* Normalize and represent it in 16 bit, with rounding up */
   s_num_r =(int16) (round((l_Acc << -s_exp_num_r), 16));
   // =====================================================================================
   // Step 3: Compute imaginary part of numerator = j(Xr*Yi - Xi*Yr)
   // =====================================================================================

   l_Acc =  (int32)s_Xr * s_Yi - (int32)s_Xi * s_Yr;  // Cannot overflow.

   /* Get the normalization shift count */
   s_exp_num_i = -norm_l(l_Acc);

   /* Normalize and represent it in 16 bit, with rounding up */
   s_num_i =(int16) (round((l_Acc << -s_exp_num_i), 16));

   // =====================================================================================
   // Step 4: Compute reciprocal of denominator = 1 / (Xr^2 + Xi^2)
   // =====================================================================================

   // Representation of '1' is mantissa=0x4000, exponent is -14.  Note the
   // divide routine requires normalized Q1.15 format.

   Divide_16bit(0x4000, -14, s_denum, s_exp_den, &s_rcp_denum, &s_exp_rcp  );

   // =====================================================================================
   /* Step 5: Compute real part of FDQ coefficient = (Xr*Yr + Xi*Yi) * [1 / (Xr^2 + Xi^2)] */
   // =====================================================================================

   /* Compute mantissa */
   l_mant_r = (int32)s_num_r * s_rcp_denum;

   /* Get normalization shift count */
   s_exp_r = -norm_l(l_mant_r);

   /* Normalize it */
   l_mant_r = l_mant_r << -s_exp_r;

   /* Compute exponent of real part of FDQ coefficient */
   s_exp_r += s_exp_num_r + s_exp_rcp;

   // =====================================================================================
   /* Step 6: Compute imaginary part of FDQ coefficient = j(Xr*Yi - Xi*Yr) * [1 / (Xr^2 + Xi^2)] */
   // =====================================================================================

   /* Compute mantissa */
   l_mant_i = (int32)s_num_i * s_rcp_denum;

   /* Get normalization shift count */
   s_exp_i = -norm_l(l_mant_i);

   /* Normalize mantissa */
   l_mant_i = l_mant_i << -s_exp_i;

   /* Compute exponent of imaginary part of FDQ coefficient */
   s_exp_i += s_exp_num_i + s_exp_rcp;

   // =====================================================================================
   // Step 7:
   // Renormalize real or imaginary part of FDQ coefficient so both
   // share the same exponent.
   // =====================================================================================

   // Common exponent value is stored in s_exp_r.
   // If either mantissa is zero, don't do any renormalizing.

   if (l_mant_r == 0)
      s_exp_r = s_exp_i;   // Do this to avoid normalizing either component.

   else if (l_mant_i == 0)
      s_exp_i = s_exp_r;   // Do this to avoid normalizing either component.

   s_exp_delta = s_exp_r - s_exp_i;

   if (s_exp_delta >= 0){
      // Adjust imaginary component (or do nothing if s_exp_delta == 0).
      l_mant_i = round(l_mant_i, s_exp_delta);
   }
   else {
      // Adjust real component, with rounding up
      s_exp_delta = -s_exp_delta;
      l_mant_r = round(l_mant_r, s_exp_delta);
      s_exp_r = s_exp_i;
   }

   // Check whether input tone energy was saturated.  If so, reduce FDQ coefficient
   // magnitude by additional factor of 2.  This is done to allow faster adaptation
   // of the coefficient in the downward (smaller magnitude) direction.  Currently the
   // reference signal level is 1/4 of the input tone's saturation level. Therefore the
   // minimum FDQ coefficient that would be generated here has a magnitude
   // on the order of 0.25.  If, for example, the true input signal level is >4x the saturation
   // level, it would take three iterations of adaptation for the FDQ coefficient to reach its
   // proper value.  By decreasing the coefficient magnitude by a factor of 2 for saturated
   // input tones, we reduce the minimum coefficient magnitude to 0.125 and allow adaptation
   // after only 2 iterations.

   if ( ((s_Xr == MAX_TONE_AMPLITUDE) || (s_Xr == MIN_TONE_AMPLITUDE)) &&
      ((s_Xi == MAX_TONE_AMPLITUDE) || (s_Xi == MIN_TONE_AMPLITUDE)) )
   {
      s_exp_r -= 1;
   }

   /* =====================================================================   */
   // Step 8:
   // Format the FDQ coefficients into required fixed point format
   // based on the word length, fractional
   /* format of the mantissa, and exponent.                          */
   /* The resulting format will be                                   */
   /* Q( FDQ_MANTISSA_WORDLENGTH - FDQ_MANTISSA_FRAC_BITS,              */
   /*    FDQ_MANTISSA_FRAC_BITS).                                 */
   /*                                                       */
   /* =====================================================================   */

   /* limit the wordsize if necessary and adjust exponent      */
   s_RightShift = 32 - FDQ_MANTISSA_WORDLENGTH;
   s_exp_r += s_RightShift + FDQ_MANTISSA_FRAC_BITS;
   if (s_exp_r < 0)
   {
      // Exponent must be zero or positive.
      s_RightShift -= s_exp_r;
      s_exp_r = 0;
   }
   else if (s_exp_r > FDQ_EXPONENT_MAX)
   {
      s_exp_r = FDQ_EXPONENT_MAX;
   }

   psa_FDQ_coef[0] = (int16) (round(l_mant_r, s_RightShift));
   psa_FDQ_coef[1] = (int16) (round(l_mant_i, s_RightShift));
   *ps_FDQ_exp = (uint8) s_exp_r;
}

