/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-1998 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*****************************************************************************
;
;   Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413
;   Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;   aec_init.c
;
;  Routine to perform AEC training using LMS algorithm.
******************************************************************************/

#include "common.h"
#include "typedef.h"
#include "DSLEngin.h"
#include "rt_state.h"
#include "common.h"
#include "dsp_op.h"
#include "gdata.h"
#include "aec_init.h"
#include "ieee_flt.h"
#include "compiler.h"
#include "mul.h"

#ifdef FIXED_POINT_AEC

/*local function prototype */
int32 sature23(int32 L_var1);

/****************************************************************************
; Prototype:
; void AecLmsTrain(int16 *psa_LMSDesBuf, int16 *psa_LMSInBuf,
; int16 s_LMSInBufLen, int16 *psa_LMSOutBuf, int16 s_LMSOutBufLen,
; int16 s_log2LMSOutBufLen)
;
;  This routine implements the LMS algorithm for AEC training.
;  The function AecLmsTrain() will be called in R_C_Reverb1 for M,
;  not necessarily consecutive frames. Hence, the total number of LMS
;  updates is M*gs_RxFftLength - s_LMSOutBufLen + 1
;
;  Given:
;  d_n^{(m)}, for n = 0,1,..., s_LMSInBufLen-(s_LMSOutBufLen-1)-1,
;                 mth (fixed during AecLmsTrain) LMS
;                 reference sequence(average received
;                 sequence through hybrid, with pilot).
;
;  i_n, for n = 0,1,..., s_LMSInBufLen -1,
;                 Prepended buffer containing averaged LMS
;                 input sequence through unity AEC path
;                 (average received sequence through AEC path).
;
;     h_l^{(k)}, for l = 0, 1, ..., L-1,
;                 kth LMS update (AEC filter taps)
;                    where L= AEC_ORDER,
;                    and   k = kth AEC filter estimate)
;
;  The LMS algorithm uses the error between kth desired
;  signal sample and kth estimate of desired signal sample to
;  update the AEC taps, where.
;
;     error_k = d_k^{(m)} - \sum_{l=0}^{L-1} h_l^{(k)} * i_{k-l}
;           where k=0: K-1, m is fixed,
;           K = Number of LMS updates = Input length - AEC filter length + 1
;           and d_k^{(m)} = desired sample estimate
;                    = innerproduct of time reversed LMS input
;                       sequence with (k-1)th LMS update.
;
;  The (k+1)th filter update is given by:
;        h_l^{(k+1)} =  h_l^{(k)} +  step * error_k * i_{k-l},    l=0:L-1
;
;  where step is related to the energy of the
;  representative input sequence (normalized LMS).
;  The AEC filter is initialized to zero, h_l^{0}, l=0:L-1.
;
;  Note 1, to estimate 0<gs_pre_aec_h_delay<31, train the AEC with
;  length AEC_ORDER + max(gs_pre_aec_h_delay) = 61.  The AEC filter
;  taps gs_pre_aec_h[AEC_ORDER] are chosen as the max energy
;  window of length AEC_ORDER.
;
;  Note 2, for g992_2, train AEC for length AEC_ORDER/2 and interpolate
;  the AEC taps.
;
;  Arguments:
;
;  psa_LMSDesBuf     (I)     pointer to the desired sequence
;  psa_LMSInBuf      (I)     pointer to the circular input sequence
;  s_LMSInBufLen     (I)   length of input sequence to LMS
;  psa_LMSOutBuf     (I/O) pointer to the LMS updates for AEC filter
;  s_LMSOutBufLen    (I)   AEC filter length
;  s_log2LMSOutBufLen   (I)   approx Log 2 of AEC filter length
;
;  Return Value:
;     none
;       global values:
;     gs_pre_aec_h_exp  AEC output shift
;     gsa_pre_aec_h     AEC filter taps
;****************************************************************************/
void AecLmsTrain(int16 *psa_LMSDesBuf,int16 *psa_LMSInBuf,int16 s_LMSInBufLen,int16 *psa_LMSOutBuf,int16 s_LMSOutBufLen,int16 s_log2LMSOutBufLen) {

   int16 k, l, m, j;
   int32 l_coeff[AEC_ORDER];
   int32 l_acc, l_temp, l_error, l_max;
   int16 s_RS, s_logmu, s_error_exp;


   /*  keep local LMS updates (l_coeff) in S4.19 = S(1+3).(15+4) */
   /* -4 < gs_pre_aec_h_exp <3 ==> add 4 to insure positve shift */

   for (l = 0; l< s_LMSOutBufLen; l++ )
      l_coeff[l]= ((int32) psa_LMSOutBuf[l])<<(gs_pre_aec_h_exp+4);

   /*  |--------------------|*******************************|
   *  | s_LMSOutBufLen-1   |       s_NumLmsUpdates         |
   *  |++++++++++++++++++s_LMSInBufLen+++++++++++++++++++++| */

#ifdef AEC_ANALYSIS_ON
   gl_error = 0;
#endif

   for (k=0; k<s_LMSInBufLen-s_LMSOutBufLen+1; k++ ){

      m = k + s_LMSOutBufLen-1;
      l_acc = 0;

      /* Compute inner product between filter estimate and
      * time-reversed input sequence for s_LMSOutBufLen samples */

      for(l=0; l<s_LMSOutBufLen; l++){

         /*convert l_coeff to S1.15, psa_LMSInbuf in S1.15*/
         // l_temp = (l_coeff[l]>>(gs_pre_aec_h_exp+4)) * psa_LMSInBuf[m-l];
         MULS32x16(l_temp, (l_coeff[l]>>(gs_pre_aec_h_exp+4)), psa_LMSInBuf[m-l]);

         l_temp >>= s_log2LMSOutBufLen;
         l_acc += l_temp;
      }

      /* incorporate AEC exponent into right shift, s_RS.
      * note, a positive AEC exponent is a left shift.  */

      s_RS = (15-s_log2LMSOutBufLen) - gs_pre_aec_h_exp;
      l_acc >>= s_RS;

      /*  compute error used to update AEC taps */
      l_error = ((int32)psa_LMSDesBuf[k]) - l_acc;
      s_error_exp = 1;
      l_error >>=s_error_exp;

      /* incorporate error exponent into LMS step factor, s_logmu */
      s_logmu = 15 - s_error_exp ;              /* include error exponent */

      s_logmu -= 4 + gs_pre_aec_h_exp;          /* shift back to s4.19 */

      /* Find max absolute value of updated AEC taps to determine new AEC exponent */
      l_max = 0;
      j=0;     // index of max absolute value

      /* update filter estimate */
      for(l=0; l<s_LMSOutBufLen; l++){

         // l_temp = l_error * psa_LMSInBuf[m-l];
         MULS32x16(l_temp, l_error, psa_LMSInBuf[m-l]);
         l_temp >>= s_logmu;

         /* check for underflow */
         if (l_temp==(int32)0xFFFFFFFFL)
            l_temp = 0;

         l_temp += l_coeff[l] ;

         /* check for OF
         * S4.19 format: saturate to 23 bits 0xffc00000L < l_coeff < 0x3fffffL */
         l_coeff[l] = sature23(l_temp);

         /* find index of max absolute  */
         if(l_coeff[l] > l_max) {
            l_max = l_coeff[l];
            j = l;
         }
         else if(-l_coeff[l] > l_max) {
            l_max = -l_coeff[l];
            j = l;
         }
      }

      /* update AEC exponent using max absolute value, l_max >= 0
      * if l_max=0, norm_l(l_max) returns 0 */

      if (l_max > 0)
         gs_pre_aec_h_exp =  12 - norm_l(l_coeff[j]);
      else if  (l_max ==0 )
         gs_pre_aec_h_exp = -4;

      /* norml > 16 ==> exp<-4,  fix exp = -4,  Shift = 0
      * norml = 16 ==> exp=-4,  Shift = 4+exp = 0
      * norml = 12 ==> exp= 0,  Shift = 4+exp = 4
      * norml = 9  ==> exp= 3,  Shift = 4+exp = 7  */

      if(gs_pre_aec_h_exp < -4)
         gs_pre_aec_h_exp = -4;

#ifdef AEC_ANALYSIS_ON
      if (l_error>=0) {
         gl_error += l_error;
      }
      else {
         gl_error -= l_error;
      }
#endif

   }

#ifdef AEC_ANALYSIS_ON
   gl_error >>= gs_Log2RxSamplesPerFrame;
#endif

      /* xxxx xxxx xxxx xxxx S--- ---- ---- ----     EXP = -4
      * xxxx xxxx xxxx xS-- ---- ---- ---- -yyy     EXP =  0
      * xxxx xxxx xS-- ---- ---- ---- -yyy yyyy     EXP = +3
      *            |-------- 23 bits ---------|       */

   /* store final filter result in s1.15 */
   for (l = 0; l< s_LMSOutBufLen; l++ )
      psa_LMSOutBuf[l]= (int16) (l_coeff[l]>>(gs_pre_aec_h_exp+4));
}

/****************************************************************************
 ; Subroutine Name : sature23(l_var_in)
 ;
 ;  This function limits the 32 bit input to the range of a 23 bit word
 ;  with saturation control.  The output is stored into 32 bits.
 ;
 ; Prototype:
 ;    int32 sature23(int32 l_var_in)
 ;
 ;  Input Arguments:
 ;
 ;    l_var_in - 32 bit long signed integer whose value falls in the
 ;              range : 0x8000 0000 <= l_var_in <= 0x7fff ffff .
 ;
 ;   Output Arguments:
 ;
 ;    none
 ;
 ;  Return Value:
 ;
 ;    l_var_out - 23 bit long signed integer (int32) in S4.19 format
 ;    whose value falls in the range:
 ;    0xffc0 0000 <= l_var_out <= 0x003f ffff.
 *****************************************************************************/
int32 sature23(int32 l_var_in)
{
   int32 l_var_out;

   if (l_var_in > (int32)0x003fffffL)
         l_var_out = (int32) 0x003fffffL;
      else if (l_var_in < (int32)0xffc00000L)
                  l_var_out = (int32)0xffc00000L;
        else l_var_out = (int32)(l_var_in);

    return(l_var_out);
}

#else  /* FIXED_POINT_AEC */
/**********************************************************************/
void AecLmsTrain(int16 *psa_LMSDesBuf, int16 *psa_LMSInBuf, int16 s_LMSInBufLen,
   Float32 *pfa_LMSOutBuf, int16 s_LMSOutBufLen, Float32 f_InvStep) {

   int16 i, k, l, m;
   Float32 f_InnerProduct, f_error;

   /* number of LMS updates per frame = s_LMSInBufLen */

   for (k = 1; k<= s_LMSInBufLen; k++ ){

      f_InnerProduct = 0;

      /* Compute inner product between filter estimate and
      * time-reversed input sequence for s_LMSOutBufLen samples */

      for(i=-k; i<=s_LMSOutBufLen-1-k; i++){
         l = i+k;
         if (i<0)
            m=-i-1;
         else
            m=s_LMSInBufLen-1-i;

         f_InnerProduct = addf32(f_InnerProduct,mpyf32(pfa_LMSOutBuf[l],int2f32(psa_LMSInBuf[m])));
      }

      f_error = subf32(int2f32(psa_LMSDesBuf[k-1]),f_InnerProduct);

      f_error = divf32(f_error,f_InvStep);

      /* update AEC filter estimate */
      for (i = -k; i<= s_LMSOutBufLen-1-k; i++){
         l = i+k;
         if (i<0)
            m=-i-1;
         else /*i>=0*/
            m=s_LMSInBufLen-1-i;

         pfa_LMSOutBuf[l] = addf32(pfa_LMSOutBuf[l], mpyf32(f_error,int2f32(psa_LMSInBuf[m]))) ;
      }
   }
}


/**********************************************************************/

void Aec(Float32 *pfa_InBuf, int16 s_InBufLen) {

   int16 l, j;
   int32 l_max, l_coeff[AEC_ORDER];

   l_max=0;
   j=0;
   for(l=0; l<s_InBufLen; l++){

      l_coeff[l] = f32toint32(mpyf32(int32toFloat32((int32)1 << 15),pfa_InBuf[l]),0);

      /* find index of max absolute value */
      if(l_coeff[l] > l_max) {
         l_max = l_coeff[l];
         j = l;
      }
      else if(-l_coeff[l] > l_max) {
         l_max = -l_coeff[l];
         j = l;
      }
   }

   if (l_max > 0)
      gs_pre_aec_h_exp =  16 - norm_l(l_coeff[j]);
   else if  (l_max ==0 )
      gs_pre_aec_h_exp = -4;

      /* xxxx xxxx xxxx xxxx xxxx S--- ---- ----     exp = -4
      * xxxx xxxx xxxx xxxx xS-- ---- ---- ----     exp =  0
      * xxxx xxxx xxxx xS-- ---- ---- ---- -yyy     exp = +3  */

      /* norml >= 16 +4 ==> exp=-4,  Shift left by (-exp)
      * norml = 12 +4 ==> exp= 0,   Shift by exp
      * norml =  9 +4 ==> exp= 3,   Shift right by exp   */

   if(gs_pre_aec_h_exp < -4)
      gs_pre_aec_h_exp = -4;

   /* store final filter taps in S1.15 with exponent */
   for (l = 0; l< s_InBufLen; l++ )    {

      if (gs_pre_aec_h_exp >= 0)
         gsa_pre_aec_h[l]= (int16) (l_coeff[l]>>gs_pre_aec_h_exp);
      else
         gsa_pre_aec_h[l]= (int16) (l_coeff[l]<<(-gs_pre_aec_h_exp));
   }
}




#endif /* FIXED_POINT_AEC */
