/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-1998 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   pwr_ctbk.c
 *
 *   Functions for determining and applying downtstream power cutback.
 *
 *------------------------------------------------------------------------
 */

#include "common.h"
#include "dsp_op.h"
#include "snr.h"
#include "rx_ops.h"
#include "gdata.h"
#include "dsp_op2.h"
#include "calibrat.h"

/*^^^
 *------------------------------------------------------------------------
 *
 *  Prototype:
 *    uint16 LoopAttenuation(int32 l_Rx_Total_Power, int16 s_NumTones, int16 s_DSPowerCutback, uint16 us_atten_range)
 *
 *  Description:  Compute the average downstream loop attenuation
 *
 *
 *  Input Arguments:
 *      l_Rx_Total_Power   -- Total RX input power summed over 's_NumTones' tones.
 *    s_NumTones        -- Number of tones over which l_Rx_Total_Power was summed.
 *    s_DSPowerCutback  -- Current power cutback at CO, in units of dB and format Q8.8.
 *      us_atten_range     -- Range to which twice the attenuation is limited. e.g.:
 *                             us_atten_range = 0x3f limits returned value to 32.5dB,
 *                             us_atten_range = 0x7f limits returned value to 63.5dB,
 *
 *  Output Arguments:
 *      none
 *
 *  Return:
 *    Average downstream loop attenuation, units of dB, format Q8.8.
 *
 *  Global Variables Used:
 *    gt_RxPowerCalibration, gs_Atten_PGA_set
 *
 *------------------------------------------------------------------------
 *^^^
 */
C_SCOPE uint16 LoopAttenuation(int32 l_Rx_Total_Power, int16 s_NumTones, int16 s_DSPowerCutback, uint16 us_atten_range)
{
   int16 s_AvLoopAttenuation, s_PowerExp, s_NumTonesExp, s_AvgCOTxTonePowerDBM;
   uint16 us_AvLoopAttenu;

   int16 s_AvgRxTonePowerMantissa, s_AvgRxTonePowerExponent, s_AvgRxTonePowerDBM;
   int16 s_LeftShift;
   int32 l_Product;

   // After normalizing, divide total power by number of tones.

   s_PowerExp = -norm_l(l_Rx_Total_Power);
   l_Rx_Total_Power <<= -s_PowerExp;

   s_NumTonesExp = -norm_16bit(s_NumTones);
   s_NumTones <<= -s_NumTonesExp;

   Divide_32by16bit(l_Rx_Total_Power,s_PowerExp,s_NumTones,s_NumTonesExp,
      &s_AvgRxTonePowerMantissa, &s_AvgRxTonePowerExponent);

   // Scale digital count if number of FFT scalebacks are different. Every extra scaleback
   // scales digital values by 2 and digital power values by 4.

   s_AvgRxTonePowerExponent -= 2*(gs_RxLog2FftScaling - gt_RxPowerCalibration.s_FFT_Scalebacks);

   // Convert units from digital count to milliWatts using calibrated conversion factors.

   l_Product = (int32) s_AvgRxTonePowerMantissa * (gt_RxPowerCalibration.s_RxPowerConvert_Mantissa);
   s_LeftShift = norm_l(l_Product);

   l_Product <<= s_LeftShift;

   s_AvgRxTonePowerMantissa = (int16) (l_Product >> 16);

   s_AvgRxTonePowerExponent += gt_RxPowerCalibration.s_RxPowerConvert_Exponent - s_LeftShift + 16;

   // Convert from milliWatts to dBm.
   // Use relation log(mantissa*2^exp) = log(mantissa) + (exp)*log(2)

   s_AvgRxTonePowerDBM = ConvertToDB(s_AvgRxTonePowerMantissa) + s_AvgRxTonePowerExponent*TWO_DB_SHORT;

   // Finally, correct for any difference between current AFE gain setting and
   // reference AFE gain setting.

   s_AvgRxTonePowerDBM -= gs_CumulAnlgGain;

   // Get CO's transmit power

   s_AvgCOTxTonePowerDBM = (int16) CO_REVERB_TONEPOWER_DBM;

   // Adjust for CO power cutback, if any.

   s_AvgCOTxTonePowerDBM -= s_DSPowerCutback;

   // Loop attenuation = CO's transmit power - RT's received power.

   s_AvLoopAttenuation = s_AvgCOTxTonePowerDBM - s_AvgRxTonePowerDBM;

   /* Represent the loop attenuation in Q7.1 format */
   s_AvLoopAttenuation += (1<<6);
   s_AvLoopAttenuation >>= 7;
   if(s_AvLoopAttenuation > (int16) us_atten_range)
      s_AvLoopAttenuation = (int16) us_atten_range;
   else if(s_AvLoopAttenuation < 0)
      s_AvLoopAttenuation = 0;

   us_AvLoopAttenu = s_AvLoopAttenuation;

   return(us_AvLoopAttenu);
}


