/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C) 1998-2000 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *----------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Condfidential
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413
 *   Phone (781) 276-4000
 *   FAX   (781) 276-4001
 *
 *   IRI_IOf_FFT.C
 *
 *   Iridia CPE Data Transfer Layer.
 *
 *
 *----------------------------------------------------------------------------
 */

#include "iridia.h"
#include "alphaeus_const.h"

#include "const.h"
#include "gdata.h"
#include "error.h"
#include "goertzel.h"
#include "aoc.h"

#include "LL_Iof.h"
#include "memrymap.h"
#include "string.h"
#include "IRI_Iof.h"

#include "socrates_memrymap.h"
#include "dma_iof.h"

extern void WhichFFTBuffers(uint16 *, uint16 *);

/***************************************************************************************
;  Subroutine Name: GetRxTones
;
;  Description:
;     This routine copies the FFT buffer output from Iridia
;
;  Prototype:
;     void GetRxTones(int16 *psa_OutBuf, int16 s_FirstTone, int16 s_NumTones)
;
;  Input Arguments:
;
;  Output Arguments:
;     none
;
;  Return Value:
;     none
;
;  Global Variables:
;     gsa_RxToneBuf[] (O)      - DMT tones output from FDQ
;
;   Substates:
;             None
;
;**********************************************************************************************/

void GetRxTones(int16 *psa_OutBuf, int16 s_FirstTone, int16 s_NumTones)
{
   uint16 RealPart;  /* points to IRI_RAM_RX_FFT_A0 or IRI_RAM_RX_FFT_B0 */
   uint16 ImagPart;  /* points to IRI_RAM_RX_FFT_A1 or IRI_RAM_RX_FFT_B1 */
   uint32 ul_StartAddress;
   int l_EndOffset;

   /* ===========================================================================   */
   /* if gft_CopyRxBuffer is FALSE, then do not load gsa_RxToneBuf               */
   /* ===========================================================================   */

   if (gft_CopyRxBuffer == FALSE)
   {
      return;
   }

   /* ===========================================================================   */
   /* if gft_CopyRxBuffer is TRUE, then load gsa_RxToneBuf with freq domain data */
   /* ===========================================================================   */

   WhichFFTBuffers(&RealPart,&ImagPart);
   mComputeSocOffsetAddress(ul_StartAddress, RealPart);
   ul_StartAddress += s_FirstTone<<2 ;    // Each tone causes 2(Real/Imag word)*2(address offset for each word) = 4 address inc

   // Compute the end offset of the DMA.
   // If we transfer N words from buffer sa_buf
   // then we want to transfer all words from &sa_buf[0]
   // to &sa_buf[N-1], inclusive.
   // &sa_buf[N-1] - &sa_buf[0] = 2N - 2.
   // Here N = s_NumTones << 1.
   // So the end offset is:
   l_EndOffset = (s_NumTones<< 2) - 2;

   SetUpDMATransfer(ul_StartAddress, (ul_StartAddress + l_EndOffset), (uint32) psa_OutBuf);

}

/***************************************************************************************
;  Subroutine Name: MoveBufferUp
;
;  Description:
;     This routine packs the non-zero data in ps_data such that, all non-zero samples sit next to
;       each other, while the second half of the ps_data array is zeroed out.
;
;  Prototype:
;     MoveBufferUp(int16 *ps_data, int16 s_ndata, uint8 uc_RealOrImag)
;
;  Input Arguments:
;     ps_data       - buffer that contains data
;     ndata         - Number of samples to read from the FFT buffer
;     uc_RealorImag - Indicates real or imag data is non-zero.
;
;
;  Output Arguments:
;     ps_data - Pointer to a buffer in which to store the time-domain samples
;
;  Return Value:
;     none
;
;  Global Variables:
;
;   Substates:
;
;**********************************************************************************************/

void MoveBufferUp(int16 *ps_data, int16 s_ndata, uint8 uc_RealOrImag)
{

   int i;

   // Pack the non-zero data in ps_data such that, all non-zero samples sit next to
   // each other, while the second half of the ps_data array is zeroed out.
   for (i = 0; i < s_ndata; i++)
      ps_data[i] = ps_data[(i<<1) + uc_RealOrImag];
   for (i = s_ndata; i < (s_ndata << 1); i++)
      ps_data[i] = 0;

   guc_NonZeroPart = uc_RealOrImag;

}

/***************************************************************************************
;  Subroutine Name: GetRxTime
;
;  Description:
;     This routine copies the time domain signal from FFT buffer.
;
;  Prototype:
;     void GetRxTime(int16 *ps_data, int16 ndata)
;
;  Input Arguments:
;     ndata - Number of samples to read from the FFT buffer
;
;  Output Arguments:
;     ps_data - Pointer to a buffer in which to store the time-domain samples
;
;  Return Value:
;     none
;
;  Global Variables:
;       gs_RxNumTones - Variable used to indicate what sampling rate is used for the TDQ
;                       filtering in Strymon.
;
;   Substates:
;
;**********************************************************************************************/
#define REAL_PART (0)
#define IMAG_PART (1)

void GetRxTime(int16 *ps_data, int16 ndata)
{
   uint16 RealPart;  // points to IRI_RAM_RX_FFT_A0 or IRI_RAM_RX_FFT_B0
   uint16 ImagPart;  // points to IRI_RAM_RX_FFT_A1 or IRI_RAM_RX_FFT_B1
   int i;
   guc_NonZeroPart = REAL_PART;
   uint32 ul_StartAddress, l_NumberOfBytes;
   int16 ft_RealFlag = 0, ft_ImagFlag = 0, s_NumWords;


   // ===========================================================================
   // if gft_CopyRxBuffer is FALSE, then do not load gsa_RxToneBuf
   // ===========================================================================

   if (gft_CopyRxBuffer == FALSE)
   {
      return;
   }

   // ===========================================================================
   // if gft_CopyRxBuffer is TRUE, then load gsa_RxToneBuf with time domain data
   // ===========================================================================

   WhichFFTBuffers(&RealPart,&ImagPart);

   if (ndata == 256)
      s_NumWords = 512;
   else
      s_NumWords = ndata;

   l_NumberOfBytes = (int32) (s_NumWords - 1) << 1;
   mComputeSocOffsetAddress(ul_StartAddress, RealPart);

   SetUpDMATransfer(ul_StartAddress, (ul_StartAddress + l_NumberOfBytes),(uint32) ps_data );

   if (ndata == 256) {
      // Every other sample of received data is a zero, which means either the Real or the
      // Imag FFT buffer is all zero.  Determine which buffer has the non-zero data.
      // Pack the non-zero data in ps_data such that, all non-zero samples sit next to
      // each other, while the second half of the ps_data array is zeroed out.

      ft_RealFlag = 0;
      ft_ImagFlag = 0;

      for (i = 0; i < ndata; i++) {
         if (ps_data[(i << 1)] != 0) {
            ft_RealFlag = 1;
            break;
         }
         if (ps_data[(i << 1) + 1] != 0) {
            ft_ImagFlag = 1;
            break;
         }
      }

      // Pack the non-zero data in ps_data such that, all non-zero samples sit next to
      // each other, while the second half of the ps_data array is zeroed out.
      if (ft_RealFlag == 1)
         MoveBufferUp(ps_data, ndata, REAL_PART);
      else if (ft_ImagFlag == 1)
         MoveBufferUp(ps_data, ndata, IMAG_PART);

   }
}

#undef REAL_PART
#undef IMAG_PART

#ifdef USE_ENGINE_FOR_BG_FFTS
/***************************************************************************************
;  Subroutine Name: LoadFFTInput
;
;  Description:
;     This loading function copies a time domain signal to be transformed from the task layer to
;     Iridia's FFT buffer.
;
;  Prototype:
;     void LoadFFTInput(void)
;
;  Input Arguments:
;     None.
;
;  Output Arguments:
;     None.
;
;  Return Value:
;     none
;
;  Global Variables:
;       gs_EngineFFTSize - Specifies size of FFT. Equal to either 128 or 256.
;     gpsa_EngineFFTInput - Pointer to time-domain real-valued buffer of size 2*gs_FastFFTSize.
;
;   Substates:
;
;**********************************************************************************************/

void LoadEngineFFTInput(void)
{
   uint16 RealPart;  /* points to IRI_RAM_RX_FFT_A0 or IRI_RAM_RX_FFT_B0 */
   uint16 ImagPart;  /* points to IRI_RAM_RX_FFT_A1 or IRI_RAM_RX_FFT_B1 */
   uint32 ul_StartAddress;

   /* ===========================================================================   */
   /* if gft_CopyRxBuffer is TRUE, then load gsa_RxToneBuf with time domain data */
   /* ===========================================================================   */
   WhichFFTBuffers(&RealPart,&ImagPart);
   mComputeSocOffsetAddress(ul_StartAddress, RealPart);

   SetUpDMATransfer((uint32) gpsa_EngineFFTInput,
         (uint32) &gpsa_EngineFFTInput[2*gs_EngineFFTSize-1],ul_StartAddress);

}

/***************************************************************************************
;  Subroutine Name: GetEngineFFTResult
;
;  Description:
;     This time-critical task copies the contents of the current FFT buffer to the
;     output buffer specified for the Fast FFT.  It is similar to GetRxTones() but is
;     especially for use in reading the results of the Fast FFT.
;
;  Prototype:
;     void GetEngineFFTResult(void)
;
;  Input Arguments:
;     None.
;
;  Output Arguments:
;     None.
;
;  Return Value:
;     none
;
;  Global Variables:
;       gs_EngineFFTSize - Specifies size of FFT. Equal to either 128 or 256.
;     gpsa_EngineFFTOutput - Pointer to buffer of size 2*gs_EngineFFTSize words.
;
;   Substates:
;
;**********************************************************************************************/
void GetEngineFFTResult(void)
{
   uint16 RealPart;  /* points to IRI_RAM_RX_FFT_A0 or IRI_RAM_RX_FFT_B0 */
   uint16 ImagPart;  /* points to IRI_RAM_RX_FFT_A1 or IRI_RAM_RX_FFT_B1 */
   uint32 ul_StartAddress, l_NumberOfBytes;

   WhichFFTBuffers(&RealPart,&ImagPart);
   l_NumberOfBytes = (2*gs_EngineFFTSize - 1) << 1;

   mComputeSocOffsetAddress(ul_StartAddress, RealPart);

   SetUpDMATransfer((uint32) ul_StartAddress,
         (uint32) (ul_StartAddress + l_NumberOfBytes),(uint32) gpsa_EngineFFTOutput);

}
#endif

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : CopyTone
 *
 *  Description:  tone data is copied form FFT output
 *
 *  Prototype:  void CopyTone(int16 *ps_PilotToneBuf, int16 s_toneIndex)
 *
 *
 *  Input Arguments: None.
 *    s_toneIndex :  Tone Index
 *
 *  Output Arguments:
 *      ps_PilotToneBuf:  two element array to store the pilot tone
 *
 *  Return:
 *          None
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void CopyTone(int16 *buf, int16 s_toneIndex)
{
    uint16 RealPart; /* points to IRI_RAM_RX_FFT_A0 or IRI_RAM_RX_FFT_B0 */
    uint16 ImagPart; /* points to IRI_RAM_RX_FFT_A1 or IRI_RAM_RX_FFT_B1 */
   uint32 ul_SocAdr;

   WhichFFTBuffers(&RealPart, &ImagPart);
   mComputeSocOffsetAddress(ul_SocAdr, (RealPart + s_toneIndex));
   *((uint32 *) buf) = (uint32) (*((uint32 *) (ul_SocAdr)));
}


/***************************************************************************************
;  Subroutine Name: GetRxCP
;
;  Description:
;     This routine copies the Cyclic Prefix (CP) from Iridia CP buffers
;
;  Prototype:
;     void GetRxCP(void)
;
;  Input Arguments:
;
;  Output Arguments:
;     none
;
;  Return Value:
;     none
;
;  Global Variables:
;     gsa_RxCPBuf[] (O)      - Cyclic Prefix (CP) for ISI analysis
;
;   Substates:
;             None
;
;**********************************************************************************************/

#ifdef ISI_ANALYSIS_ON
void GetRxCP(void)
{
   uint16 RealPart;  /* points to IRI_RAM_RX_CYCLICPREFIX_A0 or IRI_RAM_RX_CYCLICPREFIX_B0 */
   uint16 ImagPart;  /* points to IRI_RAM_RX_CYCLICPREFIX_A1 or IRI_RAM_RX_CYCLICPREFIX_B1 */
   uint32 ul_StartAddress, l_NumberOfBytes;

   /* ===========================================================================   */
   /* if gft_CopyRxBuffer is FALSE, then do not load gsa_RxCPBuf              */
   /* ===========================================================================   */

   if (gft_CopyRxBuffer == FALSE)
      return;

   l_NumberOfBytes = (RX_PREFIX_LENGTH - 1) << 1;

   WhichCPBuffers(&RealPart, &ImagPart);

   mComputeSocOffsetAddress(ul_StartAddress, RealPart);

   SetUpDMATransfer((uint32) ul_StartAddress,
         (uint32) (ul_StartAddress + l_NumberOfBytes),(uint32) gsa_RxCPBuf);

}
#endif


/***************************************************************************/

