/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2000 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;  Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413
;  Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;
;  File Name: LL_IOf.c
;
;  Low-level core interface layer.
;
;  Note: Currently these functions can access memory and
;     registers in Strymon and Iridia only.
*****************************************************************************/

#include "typedef.h"
#include "strymon_cpe.h"
#include "iridia.h"
#include "zephyr.h"
#include "alphaeus.h"
#include "cri_cocomo_interface.h"
#include "byteflip.h"



/****************************************************************************
; Name: ReadCoreReg
;
; Prototype:
;  void ReadCoreReg(uint16 us_addr, uint16 *pus_word)
;
; Description:
;  Function for reading a register in the Strymon CPE BM.
;
; Arguments:
;  us_addr     (I)      physical address of register
;  *pus_word   (O)      16-bit data read from the register
;
;  Return Value:
;  N/A
*****************************************************************************/

void ReadCoreReg(uint16 us_addr, uint16 *pus_word)
{

   int32 ul_longword = 0;

// *pus_word = *((uint16*)((uint32)(us_addr << 1) + ADSL_BASE_OFFSET));    // move us_addr to a 16-bit word boundary

   if(!StrymonCPE_ReadWord(us_addr, pus_word))
      return;
// if(!Electra_ReadWord(us_addr, pus_word))
//    return;
   if(!CRI_ReadWord(us_addr, pus_word))
      return;

   if(!IRI_ReadWord(us_addr, &ul_longword)) {
      *pus_word = (uint16) (ul_longword & 0xFFFF);
      return;
   }
    if(!ZEP_ReadWord(us_addr, &ul_longword)) {
      *pus_word = (uint16) (ul_longword & 0xFFFF);
      return;
   }
   if(!Alphaeus_ReadWord(us_addr, &ul_longword)) {
      *pus_word = (uint16) (ul_longword & 0xFFFF);
      return;
   }
}


/****************************************************************************
; Name: WriteCoreReg
;
; Prototype:
;  void WriteCoreReg(uint16 us_addr, uint16 us_word)
;
; Description:
;  Function for writing to a register in the Strymon CPE BM.
;
; Arguments:
;  us_addr     (I)      physical address of register
;  us_word     (I)      16-bit data to be written to the register
;
;  Return Value:
;  N/A
*****************************************************************************/

void WriteCoreReg(uint16 us_addr, uint16 us_word)
{
   if(!StrymonCPE_WriteWord(us_addr, us_word))
      return;
// if(!Electra_WriteWord(us_addr, us_word))
//    return;
   if(!CRI_WriteWord(us_addr, us_word))
      return;

   if(!IRI_WriteWord(us_addr, (uint32) us_word))
      return;
   if(!ZEP_WriteWord(us_addr, (uint32) us_word))
      return;
   if(!Alphaeus_WriteWord(us_addr, (uint32) us_word))
      return;
}


/****************************************************************************
; Name: SetCoreReg
;
; Prototype:
;  void SetCoreReg(uint16 us_addr, uint16 us_mask)
;
; Description:
;  Function for setting bits in a register in the Strymon CPE
;  BM.
;
; Arguments:
;  us_addr     (I)      physical address of register
;  us_mask     (I)      bits set to 1 cause the corresponding
;                 bits in the register to be set to 1
;
;  Return Value:
;  N/A
*****************************************************************************/
void SetCoreReg(uint16 us_addr, uint16 us_mask)
{
   uint16 tmp;

   ReadCoreReg(us_addr, &tmp);
   tmp |= us_mask;
   WriteCoreReg(us_addr, tmp);

}

/****************************************************************************
; Name: ResetCoreReg
;
; Prototype:
;  void ResetCoreReg(uint16 us_addr, uint16 us_mask)
;
; Description:
;  Function for resetting bits in a register in the Strymon CPE
;  BM.
;
; Arguments:
;  us_addr     (I)      physical address of register
;  us_mask     (I)      bits set to 1 cause the corresponding
;                 bits in the register to be set to 0
;
;  Return Value:
;  N/A
*****************************************************************************/
void ResetCoreReg(uint16 us_addr, uint16 us_mask)
{
   uint16 tmp;

   ReadCoreReg(us_addr, &tmp);
   tmp &= ~us_mask;
   WriteCoreReg(us_addr, tmp);
}

/****************************************************************************
; Name: ReadCoreBuf
;
; Prototype:
;  void ReadCoreBuf(uint16 us_addr, int16 *psa_words, uint16 us_num, uint16 us_DestPitch)
;
; Description:
;  Function for reading a buffer in the ADSL cores.
;
; Arguments:
;  us_addr     (I)      physical address at which to start reading
;  psa_words   (O)      array containing data read from the buffer
;  us_num      (I)      number of 16-bit words to read from the buffer
;  us_DestPitch(I)      Amount to increment destination write pointer after each word is written.
;
;  Return Value:
;  N/A
*****************************************************************************/
void ReadCoreBuf(uint16 us_addr, int16 *psa_words, uint16 us_num, uint16 us_DestPitch)
{
   uint8 uc_Error;
   uint16 us_idx;
   uint16 us_SrcAddr, us_DestOffset;
   uint32 ul_longword;

   us_SrcAddr = us_addr;
   us_DestOffset = 0;
   uc_Error = 0;


   for(us_idx=0; us_idx<us_num; us_idx++, us_SrcAddr++, us_DestOffset += us_DestPitch) {
      if(StrymonCPE_ReadWord(us_SrcAddr, (uint16*) &(psa_words[us_DestOffset]))){
         uc_Error = 1;
         break;
      }
   }
   if (!uc_Error)
      return;

   us_SrcAddr = us_addr;
   us_DestOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++, us_SrcAddr++, us_DestOffset += us_DestPitch) {
      if(IRI_ReadWord(us_SrcAddr, (uint16*) (psa_words+us_DestOffset))){
         uc_Error = 1;
         break;
      }
   }

}

/****************************************************************************
; Name: WriteCoreBuf
;
; Prototype:
;  void WriteCoreBuf(uint16 us_addr, int16 *psa_words, uint16 us_num, uint16 us_SrcPitch)
;
; Description:
;  Function for writing to a buffer in the ADSL cores.
;
; Arguments:
;  us_addr     (I)      physical address at which to start writing
;  psa_words   (I)      array containing data to be written to the buffer
;  us_num      (I)      number of 16-bit words to write to the buffer
;  us_SrcPitch(I)    Amount to increment destination read pointer after each word is read.
;
;
;  Return Value:
;  N/A
*****************************************************************************/
void WriteCoreBuf(uint16 us_addr, int16 *psa_words, uint16 us_num, uint16 us_SrcPitch)
{
   uint8 uc_Error;
   uint16 us_idx;
   uint16 us_SrcOffset, us_DestAddr;
   uint16 us_word;

   us_DestAddr = us_addr;
   us_SrcOffset = 0;
   uc_Error = 0;


   for(us_idx=0; us_idx<us_num; us_idx++, us_SrcOffset += us_SrcPitch, us_DestAddr++) {

      if(StrymonCPE_WriteWord(us_DestAddr, (uint16) psa_words[us_SrcOffset])){
         uc_Error = 1;
         break;
      }
   }

   if (!uc_Error)
      return;

   us_DestAddr = us_addr;
   us_SrcOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++, us_SrcOffset += us_SrcPitch, us_DestAddr++) {
      if(IRI_WriteWord(us_DestAddr, (uint16) *(psa_words+us_SrcOffset))){
         uc_Error = 1;
         break;
      }
   }

}

/****************************************************************************
; Name: WriteCoreBuf32NoDMA
;
; Prototype:
;  void WriteCoreBuf32NoDMA(uint16 us_addr, int16 *psa_words, uint16 us_num)
;
; Description:
;  Function for writing to a buffer in the ADSL cores.
;
; Arguments:
;  us_addr     (I)      physical address at which to start writing
;  psa_words   (I)      array containing data to be written to the buffer
;  us_num      (I)      number of 32-bit words to write to the buffer
;
;
;  Return Value:
;  N/A
*****************************************************************************/
void WriteCoreBuf32NoDMA(uint16 us_addr, int16 *psa_words, uint16 us_num)
{
   uint8 uc_Error;
   uint16 us_idx;
   uint16 us_SrcOffset, us_DestAddr;
   uint32 ul_data1;
   uint32 ul_data2;

   us_DestAddr = us_addr;
   us_SrcOffset = 0;
   uc_Error = 0;



#ifdef HWENGINE_30
   for(us_idx=0; us_idx<us_num; us_idx++){

      /* read ls 16-bits */
      ul_data1= (uint32)(*(psa_words+us_SrcOffset));
      ul_data1 = ul_data1 & 0x0000FFFF;
      us_SrcOffset++;

      /* read mw 16-bits */
      ul_data2 = (uint32)(*(psa_words+us_SrcOffset));
      ul_data2 = (ul_data2<<16) & 0xFFFF0000;
      us_SrcOffset++;

      /* combine into a 32-bit word */
      ul_data1 = ul_data1 | ul_data2;

      /* write data to hardware */
      if(StrymonCPE_WriteLongWord(us_DestAddr, ul_data1)){
         uc_Error = 1;
         break;
      }
      us_DestAddr++;

   }


   if (!uc_Error)
      return;
#endif

   us_DestAddr = us_addr;
   us_SrcOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++){

      /* read ls 16-bits */
      ul_data1= (uint32)(*(psa_words+us_SrcOffset));
      ul_data1 = ul_data1 & 0x0000FFFF;
      us_SrcOffset++;

      /* read mw 16-bits */
      ul_data2 = (uint32)(*(psa_words+us_SrcOffset));
      ul_data2 = (ul_data2<<16) & 0xFFFF0000;
      us_SrcOffset++;

      /* combine into a 32-bit word */
      ul_data1 = ul_data1 | ul_data2;

      /* write data to hardware */
      if(IRI_WriteWord(us_DestAddr, ul_data1)){
         uc_Error = 1;
         break;
      }
      us_DestAddr++;

   }

}

void
/****************************************************************************
; Name: WriteCoreBufBitRevDest
;
; Prototype:
;  void WriteCoreBufBitRevDest(uint16 us_addr, int16 *psa_words, uint16 us_num, uint16 us_SrcPitch, uint16 s_NBitReversal);
;
; Description:
;  Function for writing to a buffer in the ADSL cores. Offsets to the destination
;  starting address are bit-reversed. Intended for loading FFT input data.
;
;
; Arguments:
;  us_addr     (I)      physical address at which to start writing
;  psa_words   (I)      array containing data to be written to the buffer
;  us_num      (I)      number of 16-bit words to write to the buffer. MUST BE <= 2^s_NBitReversal.
;  us_SrcPitch(I)    Amount to increment destination read pointer after each word is read.
;  s_NBitReversal    MUST BE 8 (for now). Is number of LSBs in destination address to bit-reverse.
;
;  Return Value:
;  N/A
*****************************************************************************/
WriteCoreBufBitRevDest(uint16 us_addr, int16 *psa_words, uint16 us_num, uint16 us_SrcPitch, uint16 s_NBitReversal)
{
   uint8 uc_Error;
   uint16 us_idx;
   uint16 us_SrcOffset, us_DestBase, us_DestOffset, us_DestAddr;
   uint32 * ul_Destination;

   us_DestBase = us_addr;
   us_DestOffset = 0;
   us_SrcOffset = 0;
   uc_Error = 0;


   for(us_idx=0; us_idx<us_num; us_idx++, us_SrcOffset += us_SrcPitch, us_DestOffset++) {
      us_DestAddr = us_DestBase + guca_bitreverse[us_DestOffset];
      if(StrymonCPE_WriteWord(us_DestAddr, (uint16) psa_words[us_SrcOffset])){
         uc_Error = 1;
         break;
      }
   }

   if (!uc_Error)
      return;

   us_DestBase = us_addr;
   us_DestOffset = 0;
   us_SrcOffset = 0;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++, us_SrcOffset += us_SrcPitch, us_DestOffset++) {
      us_DestAddr = us_DestBase + guca_bitreverse[us_DestOffset];
      if(IRI_WriteWord(us_DestAddr, (uint16) *(psa_words+us_SrcOffset))){
         uc_Error = 1;
         break;
      }
   }

}


/****************************************************************************
; Name: FillCoreBuf
;
; Prototype:
;  void FillCoreBuf(uint16 us_addr, int16 s_Value, uint16 us_num, uint16 us_DestPitch)
;
; Description:
;  Function for writing a constant to multiple locations in a buffer in the ADSL cores.
;
; Arguments:
;  us_addr     (I)      physical address at which to start writing
;  s-Value     (I)      16-bit value to be written to the buffer
;  us_num      (I)      number of 16-bit locations to write the value to
;  us_DestPitch(I)      Amount to increment destination write pointer after each word is written.
;
;
;  Return Value:
;  N/A
*****************************************************************************/
void FillCoreBuf(uint16 us_addr, int16 s_Value, uint16 us_num, uint16 us_DestPitch)
{
   uint8 uc_Error;
   uint16 us_idx;
   uint16 us_DestAddr;

   us_DestAddr = us_addr;
   uc_Error = 0;


   for(us_idx=0; us_idx<us_num; us_idx++, us_DestAddr += us_DestPitch) {
      if(StrymonCPE_WriteWord(us_DestAddr, s_Value )){
         uc_Error = 1;
         break;
      }
   }

   if (!uc_Error)
      return;

   us_DestAddr = us_addr;
   uc_Error = 0;

   for(us_idx=0; us_idx<us_num; us_idx++, us_DestAddr += us_DestPitch) {
      if(IRI_WriteWord(us_DestAddr, s_Value)){
         uc_Error = 1;
         break;
      }
   }

}

/*****************************************************************************/
