/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2002 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*************************************************************************
* Aware DMT Technology. Proprietary and Confidential.
*
* ADDRESS:         40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
* TELEPHONE:       781.276.4000
* FAX:             781.276.4001
* WEB:             http://www.aware.com
*
* FILENAME:        arcfileio.c
*
* DESCRIPTION:     Socrates implementation of stdio.h file I/O functions
*
**************************************************************************/
#include "typedef.h"
#include "arcfileio.h"
#include "data_alloc.h"
#include <string.h>
#include <stdarg.h>
#include "compiler.h"

#ifdef LEAVE_TRAIL
// the string buffer for the SIMEXT-ARC file I/O interface
DATA_MAP_FileIOString
char FileIOString[MAX_ARC_FILE_STRING_LENGTH];
DATA_MAP_END

uint16 ReadFileIOStatus(void);

/****************************************************************************
;
; Name: fopen
;
; Prototype (should have the same prototype as the ANSI C fopen):
;  FILE *fopen(const char *filename, const char *mode);
;
; Description:
;  Implements ANSI C fopen through SIMEXT.
;
*****************************************************************************/
FILE *fopen(const char *filename, const char *mode) {
   int32  l_strlen;
   uint32 ul_data;

   // write filename to the shared memory buffer
   strcpy(FileIOString, filename);
   l_strlen = strlen(filename);

   // write the file mode to the shared memory buffer
   strcpy(FileIOString + l_strlen + 1, mode);

   // notify SIMEXT of file open request
   *(uint16*)(ARCFILEIO_CMD_REG) = (uint16) ARCFILEIO_OPEN;

   // read out SIMEXT fid
   ul_data = *(uint32 *) ARCFILEIO_FILE_ID_REG;
   return((FILE *) ul_data);
}

/****************************************************************************
;
; Name: fclose
;
; Prototype (should have the same prototype as the ANSI C fclose):
;  int fclose(FILE *stream);
;
; Description:
;  Implements ANSI C fclose through SIMEXT.
;
*****************************************************************************/
int fclose(FILE *stream) {
   uint16 us_status;

   // pass file ID to SIMEXT
   *(uint32 *) ARCFILEIO_FILE_ID_REG = (uint32) stream;

   // notify SIMEXT of file close request
   *(uint16*)(ARCFILEIO_CMD_REG) = (uint16) ARCFILEIO_CLOSE;

   // read out status
   us_status = ReadFileIOStatus();
   if(us_status == ARCFILEIO_EOF)
      return EOF;
   else
      return 0;
}

/****************************************************************************
;
; Name: fprintf
;
; Prototype (should have the same prototype as the ANSI C fprintf):
;  int fprintf(FILE *stream, const char *format, ...);
;
; Description:
;  Implements ANSI C fprintf through SIMEXT.
;
*****************************************************************************/
int fprintf(FILE *stream, const char *format, ...) {
   uint16  us_status;
   va_list args;

   // pass file ID to SIMEXT
   if((stream != stdout) && (stream != stderr))
      *(uint32 *) ARCFILEIO_FILE_ID_REG = (uint32) stream;

   // write print string to the shared memory buffer
   va_start(args, format);
   vsprintf(FileIOString, format, args);  // can't use a simple sprintf because we need to specify the argument list
   va_end(args);

   // notify SIMEXT to process print request
   if(stream == stdout)
      *(uint16*)(ARCFILEIO_CMD_REG) = (uint16) ARCFILEIO_PRINT_STDOUT;
   else if(stream == stderr)
      *(uint16*)(ARCFILEIO_CMD_REG) = (uint16) ARCFILEIO_PRINT_STDERR;
   else
      *(uint16*)(ARCFILEIO_CMD_REG) = (uint16) ARCFILEIO_PRINT;

   // read out status
   us_status = ReadFileIOStatus();
   if(us_status == ARCFILEIO_ERROR)
      return -1;
   else
      return 0;
}

/****************************************************************************
;
; Name: printf
;
; Prototype (should have the same prototype as the ANSI C printf):
;  int printf(const char *format, ...);
;
; Description:
;  Implements ANSI C printf through SIMEXT.
;
*****************************************************************************/
int printf(const char *format, ...) {
   uint16  us_status;
   va_list args;

   // write print string to the shared memory buffer
   va_start(args, format);
   vsprintf(FileIOString, format, args);  // can't use a simple sprintf because we need to specify the argument list
   va_end(args);

   // notify SIMEXT to process print request
   *(uint16*)(ARCFILEIO_CMD_REG) = (uint16) ARCFILEIO_PRINT_STDOUT;

   // read out status
   us_status = ReadFileIOStatus();
   if(us_status == ARCFILEIO_ERROR)
      return -1;
   else
      return 0;
}

/****************************************************************************
;
; Name: fscanf
;
; Prototype (should have the same prototype as the ANSI C fscanf):
;  int fscanf(FILE *stream, const char *format, ...);
;
; Description:
;  Implements ANSI C fscanf through SIMEXT.
;
*****************************************************************************/
int fscanf(FILE *stream, const char *format, ...) {
   uint16  us_status;
   va_list args;
   int     i_valAssigned;

   if(stream == stdin) {
      // notify SIMEXT of request to begin scan
      *(uint16 *)(ARCFILEIO_CMD_REG) = (uint16) ARCFILEIO_SCAN_STDIN;
   }
   else {
      // pass file ID to SIMEXT
      *(uint32 *) ARCFILEIO_FILE_ID_REG = (uint32) stream;

      // notify SIMEXT of request to begin scan
      *(uint16 *)(ARCFILEIO_CMD_REG) = (uint16) ARCFILEIO_SCAN_BEGIN;

      // read out status
      us_status = ReadFileIOStatus();
      if((us_status == ARCFILEIO_ERROR) || (us_status == ARCFILEIO_SCAN_EOF))
         return EOF;

      // scan string from the shared memory buffer
      va_start(args, format);
      i_valAssigned = _vsscanf(FileIOString, format, args); // _vsscanf is a Metaware High C extension to ANSI C
      va_end(args);

      // read out status
      us_status = ReadFileIOStatus();
      if(us_status == ARCFILEIO_ERROR)
         return EOF;

      // notify SIMEXT of request to begin end
      *(uint16 *)(ARCFILEIO_CMD_REG) = (uint16) ARCFILEIO_SCAN_END;
   }

   // read out status
   us_status = ReadFileIOStatus();
   if(us_status == ARCFILEIO_ERROR)
      return EOF;
   else
      return i_valAssigned;
}

/****************************************************************************
;
; Name: fflush
;
; Prototype (should have the same prototype as the ANSI C fflush):
;  int fflush(FILE *stream);
;
; Description:
;  Implements ANSI C fflush through SIMEXT.
;
*****************************************************************************/
int fflush(FILE *stream) {
   uint16 us_status;

   if(stream == stdout)
      // notify SIMEXT of flush request
      *(uint16*)(ARCFILEIO_CMD_REG) = (uint16) ARCFILEIO_FLUSH_STDOUT;
   else if(stream == stderr)
      // notify SIMEXT of flush request
      *(uint16*)(ARCFILEIO_CMD_REG) = (uint16) ARCFILEIO_FLUSH_STDERR;
   else {
      // pass file ID to SIMEXT
      *(uint32 *) ARCFILEIO_FILE_ID_REG = (uint32) stream;

      // notify SIMEXT of flush request
      *(uint16*)(ARCFILEIO_CMD_REG) = (uint16) ARCFILEIO_FLUSH;
   }

   // read out status
   us_status = ReadFileIOStatus();
   if(us_status == ARCFILEIO_EOF)
      return EOF;
   else
      return 0;
}

/****************************************************************************
;
; Name: ReadFileIOStatus
;
; Prototype:
;  uint16 ReadFileIOStatus(void);
;
; Description:
;  Reads the ARCFILEIO_STATUS register.  We need this function because the Metaware compiler
;  constantly optimizes reads to this register.
;
; Returns:
;  the contents of ARCFILEIO_STATUS
;
*****************************************************************************/
uint16 ReadFileIOStatus(void) {
   return(*(uint16*)(ARCFILEIO_STATUS_REG));
}

#endif

