/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2000 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   arctan.c
 *
 *   Fast ArcTangent implementation.
 *------------------------------------------------------------------------
 */

#include "typedef.h"
#include <stdio.h>
#include <stdlib.h>

#ifdef COCOMO_UNIT_TEST
#include <math.h>
#endif

#define abs_atan(x) ((x) < 0 ? -(x) : (x))
/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: FastAtan
 *
 *  Description: Computes 4 quadrant arctan(y/x).
 *
 *  Return: scaled arctan(y/x) with 1 radian = 8192
 *
 *  Notes: The algorithm computes the arctan in the first octant and maps
 *         the answer to the remaining octants as needed. The first octant
 *         is divided into 7 sections and each section is approximated by
 *         1) a(y/x) + b  or
 *         2) (xy/(x^2 + ay^2)) + b
 *
 *  The relative approximation error is no more than -51 dB.
 *
 *  Here is the pseudo code:
 *
 *  if (x > 20*y) {
 *       p = y/x;
 * } else if (x > 8*y) {
 *    p = (x*y)/(x*x + y*y*0.25);
 * } else if (x > 2*y) {
 *    p = (x*y)/(x*x + y*y*0.3125) - 0.0001;
 * } else if (2*x > 3*y) {
 *    p = (x*y)/(x*x + y*y*0.3) - 0.001;
 * } else if (3*x > 4*y) {
 *    p = 0.66925389*(y/x) +  0.14194371;
 * } else if (7*x > 8*y) {
 *    p = 0.60255826*(y/x) +  0.19225833;
 * } else {
 *    p = 0.53246136*(y/x) +  0.25353658;
 * }
 *
 *------------------------------------------------------------------------
 *^^^
 */
int16 FastAtan(int16 y_in, int16 x_in) {

   int16 i, b, q;
   int32 l_num, l_den;
   int32 l_x, l_y;


   /* Handle special cases: */
   if (x_in == 0) {
      if (y_in > 0)      return 12868;  /* pi/2 */
      else if (y_in < 0) return -12868; /* -pi/2 */
      else               return 0;
   }

    /* map x and y into first octant: */
   if (abs_atan(x_in) >= abs_atan(y_in)) {
      l_x = abs_atan((int32)x_in);
      l_y = abs_atan((int32)y_in);
   } else { /* switch the roles of x and y */
      l_x = abs_atan((int32)y_in);
      l_y = abs_atan((int32)x_in);
   }

    /* approximate: */
   if (l_x >= 20*l_y) {
      l_num = (l_y);
      l_den = (l_x);
      b     = 0;
   } else if (l_x >= 8*l_y) {
      l_num = (l_x*l_y) >> 1;
      l_den = ((l_x*l_x) >> 1) + (l_y*l_y >> 3);
      b     = 0;
   } else if (l_x >= 2*l_y) {
      l_num = (l_x*l_y) >> 1;
      l_den = (l_x*l_x >> 1) + ((5*l_y >> 2)*l_y >> 3);
      b     = -1;
   } else if (2*l_x >= 3*l_y) {
      l_num = (l_x*l_y) >> 1;
      l_den = ((l_x*l_x) >> 1) + ((19661*l_y >> 14)*l_y >> 3);
      b     = -8;
   } else if (3*l_x >= 4*l_y) {
      l_num = l_y*10965;
      l_den = l_x*16384;
      b     = 1163;
   } else if (7*l_x >= 8*l_y) {
      l_num = l_y*9872;
      l_den = l_x*16384;
      b     = 1575;
   } else {
      l_num = l_y*8724;
      l_den = l_x*16384;
      b     = 2077;
   }

   /* long division: */
   q = 0;
   for (i=0; i<13; i++) {
      if (l_num >= l_den) {
         l_num -=  l_den;
         q += 1;
      }
      q <<= 1;
      l_num <<= 1;
   }
   /* add constant */
   q += b;

   /* Default answer is in first octant */
   if (x_in >= 0) {
      if (abs_atan(x_in) < abs_atan(y_in)) { /* 2nd octant */
         q = 12868 - q; /* answer = pi/2 - answer */
      }
   } else {
      if (abs_atan(x_in) < abs_atan(y_in)) { /* 3rd octant */
         q = 12868 + q; /* answer = pi/2 + answer */
      } else {                     /* 4th octant */
         q = 25736 - q; /* answer = pi - answer */
      }
   }

   /* mirror the answer to 5th, 6th, 7th and 8th octants: */
   if (y_in >= 0) return q;
   else return -q;
}

/* For unit testing only: */
/* ------------------------------------------------------------------------ */
#if 0

FILE *fastatan_out_file=NULL;

void OpenFiles(int argc, char *argv[]) {

   /* Check number of arguments */
   if(argc != 2) {
      printf("Usage: fastatan output_file \n\n");
      exit(1);
   }

   /* Create new output file */
   fastatan_out_file = fopen(argv[1], "w");
   if(fastatan_out_file == NULL) {
      printf("File Creation Error: %s.\n", argv[1]);
      exit(1);
   }
}

void CloseFiles(void) {


   fclose(fastatan_out_file);
}

void TestAtan(void) {

   int32  k, x, y;
   double p, p3;

   fprintf(fastatan_out_file, "Q        I        Actual angle  Calc. angle  Error        Relative Err (dB)\n");
   fprintf(fastatan_out_file, "---------------------------------------------------------------------------\n");

      for (x = -30000; x <= +32000; x += 2000) {
         for (y = -30000; y <= 32000; y += 2000) {
            p = atan2(y, x);
            p3 = (double)FastAtan((int16)y, (int16)x) / (8192.0);
            fprintf(fastatan_out_file, "I=%5d, Q=%5d: A = %f A' = %f e = %f re = %f\n",
                  x, y, p, p3, fabs(p-p3), 20*log10(fabs(p-p3)/fabs(p)));
         }
         fprintf(fastatan_out_file, "\n");
      }
}


void main(int argc, char *argv[]) {

   OpenFiles(argc, argv);
   TestAtan();
   CloseFiles();
}

#endif
