/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-1998 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;  Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
;  Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;  File Name: dsp_op.c
;
;  This file contains functions emulating DSP operations.
;
;***************************************************************************/

#include "typedef.h"
#include "dsp_op.h"
#include "rx_ops.h"

/****************************************************************************
 ;
 ; Subroutine Name : round(L_var, s_bit_position)
 ;
 ;  This function performs conventional rounding at bit
 ; indicated by "bit_position" and then shift the result
 ; right by "bit_position"
 ;
 ; Prototype:
 ;    int32 round(int32 L_var, int s_bit_position)
 ;
 ;  Input Arguments:
 ;
 ;    L_var - 32 bit long signed integer whose value falls in the
 ;             range : 0x8000 0000 <= L_var1 <= 0x7fff ffff
 ;    s_bit_position - the bit position rounding bit 1 is added
 ;
 ;   Output Arguments:
 ;
 ;    none
 ;
 ;  Return Value:
 ;
 ;    l_var_out - 32 bit long signed integer whose value falls in the
 ;              range : 0x8000 0000 <= L_var1 <= 0x7fff ffff
 *****************************************************************************/
int32 round(int32 L_var, int16 s_bit_position)
{
   int32 l_rnd_bit, l_var_out;

   if(s_bit_position == 0)
      return(L_var);

   /* if the shift is greater than the wordsize, then the result should be 0  */

   if(s_bit_position >= 32)
      return((int32)(0));

   l_rnd_bit = (int32)0x1 <<(s_bit_position-1);

   l_var_out = l_add(L_var, l_rnd_bit);
   l_var_out >>= s_bit_position;
   return(l_var_out);
}

/****************************************************************************
 ;
 ; Subroutine Name: l_add(L_var1, L_var2)
 ;
 ;  Description:
 ;  Adds two 32bit values and tests for overflow/underflow.  If overflow/
 ;  underflow occurs then result if saturated to MAX_32/MIN_32
 ;
 ; Prototype:
 ;    int32 l_add(int32 L_var1, int32 L_var2)
 ;
 ;  Input Arguments:
 ;
 ;    L_var1, L_var2 - 32 bit long signed integers to be summed
 ;
 ;   Output Arguments:
 ;
 ;  Return Value:
 ;
 ;    l_var_out - 32 bit long signed integer whose value falls in the
 ;              range : 0x8000 0000 <= L_var1 <= 0x7fff ffff
 *****************************************************************************/
int32 l_add(int32 L_var1, int32 L_var2)
{
   int32 l_var_out;
    int32 l_sign1, l_sign2;

   l_sign1 = 0x80000000 & L_var1;
   l_sign2 = 0x80000000 & L_var2;

    l_var_out = L_var1 + L_var2;

    if ((l_sign1 ^ l_sign2) == 0) {
        if (l_sign1 == 0) {     /* L_var1, L_var2 positive, check overflow */
            if (l_var_out < L_var1)
                l_var_out = MAX_32;
        }
        else {                  /* L_var1, L_var2 negative, check underflow */
            if (l_var_out > L_var1)
                l_var_out = MIN_32;
        }
    }

    return l_var_out;

}


