/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C) 1998-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *----------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Condfidential
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413
 *   Phone (781) 276-4000
 *   FAX   (781) 276-4001
 *
 *   XdmaDiag_funcs.c
 *
 *   Functions for performing run-time XDMA diagnostic tests.
 *
 *----------------------------------------------------------------------------
 */
#include "common.h"
#include "gdata.h"
#include "states.h"
#include "cmv.h"
#include "aux_regs.h"
#include "dataswap.h"
#include "trail.h"

/* Global variables */
uint32   gl_XdmaTestData[2];
uint32   gl_XdmaStatus;
uint16  gs_BootXdmaStatus;

/****************************************************************************
;   Prototype: void Verify_XdmaAccess (void)
;
;   Description: Function checks whether the XDMA engine can read data from
;  external SDRAM after modem reset. This diagnostic test only checks that
;  the XDMA engine can return the first two longword of non-zero data from
;  the image in SDRAM.
;
;   Arguments:
;       None
;
;   Return Value:
;       None
;
;****************************************************************************/
void Verify_XdmaAccess (void)
{
   gl_XdmaTestData[0] = 0;
   gl_XdmaTestData[1] = 0;

   /* Read the first two longwords from address 0 */
   /* The returned data should be the size field and checksum field of the binary image */
   Xdma_ReadXmem();

   /* Set exception code 208 and go to pause if no data returned */
   if (gl_XdmaStatus != 0 || gl_XdmaTestData[0]==0 || gl_XdmaTestData[1]==0)
   {
      gus_ExceptionCode = E_CODE_XDMA_FAILURE_ON_RESET;
      Pause((E_CODE_XDMA_FAILURE_ON_RESET&0xffff));
   }
}


/****************************************************************************
;   Prototype: void Xdma_ReadXmem (void)
;
;   Description: Function sets up an XDMA transfer to read the first 8 bytes
;  of the modem image in external SDRAM to an internal buffer.
;
;   Arguments:
;       None
;
;   Return Value:
;       None
;
;****************************************************************************/
void Xdma_ReadXmem (void)
{
   int i;

   /* log xdma status on reboot */
   gs_BootXdmaStatus = _lr(ARC_XDMA_STAT);

    /* Load XDMA address registers */
    _sr(0x00, ARC_XDMA_SA_S);    // Read from external SDRAM start at 0
    _sr(0x06, ARC_XDMA_SA_E);    // size = 8 bytes
    _sr((int32)&gl_XdmaTestData[0], ARC_XDMA_DA_S);
    _sr(XDMA_GAP_FAST, ARC_XDMA_GAP);

    /* Start XDMA */
    _sr((XDMA_READ_XMEM | XDMA_CTRL_ENABLED), ARC_XDMA_CTRL);

    /* Wait for XDMA is done, or until it times out */
    for (i=0; i < 5000; i++)
    {
      gl_XdmaStatus = _lr(ARC_XDMA_STAT);

      if ((gl_XdmaStatus & XDMA_STAT_IN_PROC) == 0)
      {
         break;
      }
   }
}


/****************************************************************************
;   Prototype: void Xdma_WriteXmem ()
;
;   Description: Function sets up an XDMA transfer to write data from
;  internal memory to external SDRAM.
;
;   Arguments:
;       int32  l_SrcAddr - Source address of ARC memory
;       int32  l_DstAddr - Destination address of external SDRAM
;       int32  l_ByteCnt - Number of bytes to DMA
;
;   Return Value:
;       None
;
;****************************************************************************/
void Xdma_WriteXmem (int32 l_SrcAddr, int32 l_DstAddr, int32 l_ByteCnt)
{
   int i;
   int32 l_SrcEndAddr = (l_SrcAddr + l_ByteCnt + 3)/4*4 - 2;

    /* Load XDMA address registers */
    _sr(l_SrcAddr, ARC_XDMA_SA_S);        // load source addr
    _sr(l_SrcEndAddr, ARC_XDMA_SA_E);     // load source end addr
    _sr(l_DstAddr, ARC_XDMA_DA_S);        // load destination addr
    _sr(XDMA_GAP_FAST, ARC_XDMA_GAP);

    /* Start XDMA */
    _sr((XDMA_WRITE_XMEM | XDMA_CTRL_ENABLED), ARC_XDMA_CTRL);

    /* Wait for XDMA is done, or until it times out */
    for (i=0; i < 5000; i++)
    {
      gl_XdmaStatus = _lr(ARC_XDMA_STAT);

      if ((gl_XdmaStatus & XDMA_STAT_IN_PROC) == 0)
      {
         break;
      }
   }

}


/****************************************************************************
;   Prototype: void Xdma_ClearXmem (void)
;
;   Description: Function clears external SDRAM to zero. An initialized
;  internal buffer is used to DMA the zero data to SDRAM block by block.
;
;   Arguments:
;       None
;
;   Return Value:
;       None
;
;****************************************************************************/
void Xdma_ClearXmem (void)
{
   int i, TrailBuf_BSize, BlockCnt;

   // Make sure XDMA engine is idle
   if (gs_CodeSwapStatus == CODESWAP_IDLE)
   {
      // Block starting codeswap
      gt_DataSwap.c_State = DATASWAP_IN_PROGRESS;
   }
   else
   {
      return;
   }

   TrailBuf_BSize = DEBUG_TRAIL_SIZE * 2;
   BlockCnt = XDATA_SIZE / TrailBuf_BSize;

   // Clear source buffer
   for (i=0; i < DEBUG_TRAIL_SIZE; i++)
   {
      gsa_TrainStatesTrail[i] = 0;
   }

   // Clear XMEM, one block at a time
   for (i=0; i < BlockCnt; i++)
   {
      Xdma_WriteXmem ((int32)&gsa_TrainStatesTrail[0],
                  (int32)(XDATA_BASE_ADDR + i * TrailBuf_BSize),
                  (int32)TrailBuf_BSize);
   }

   // All done
   gt_DataSwap.c_State = DATASWAP_READY;
}
