/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C) 1998-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *----------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Condfidential
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413
 *   Phone (781) 276-4000
 *   FAX   (781) 276-4001
 *
 *   dataswap_ext.c
 *
 *   An extention to dataswap.c, mainly contain functions that can be run
 *   from cache.
 *
 *----------------------------------------------------------------------------
 */
#include <stdio.h>
#include "common.h"
#include "gdata.h"
#include "fifo.h"
#include "states.h"
#include "cmv.h"
#include "aux_regs.h"
#include "dataswap.h"
#include "trail.h"


/* Global variables */
uint8 guc_STXdmaRequested = 0;         /* Showtime dataswap requested */
int16 gs_ShowtimeTrailBufferIndex = 0; /* Showtime states trail buffer index */
int16 gs_XdmaWordCnt = 0;              /* Xdma transfer count in 16-bit words */


/****************************************************************************
;   Prototype: void Check_ShowtimeDataSwap (void)
;
;   Description: Function checks to see whether the showtime states trail
;           buffer is in use, if so, whether it reaches certain threshold
;           limit, if all true, request a dataswap to XDMA the buffer to
;           external memory.
;
;           If the requestd XDMA is completed, update states trail buffer
;           index and handle extra states trails since last XDMA.
;
;   Arguments:
;       None
;
;   Return Value:
;       None
;
;****************************************************************************/
void Check_ShowtimeDataSwap (void)
{
   int NewTrailCnt;

   /* If XDMA is idle and showtime states trail buffer has been deployed */
    if ((gt_DataSwap.c_State == DATASWAP_READY) && ((int)gsa_StatesTrail == (int)gsa_ShowTimeStatesTrail))
    {
      /* If no pending XDMA request */
      if (guc_STXdmaRequested == 0)
      {
         // Any new entries in the buffer ?
         NewTrailCnt = (gt_debugTrailCollectControl.s_debugTrailLen - gs_ShowtimeTrailBufferIndex) & ~1;
         if (NewTrailCnt < 0)
         {
            NewTrailCnt = gt_debugTrailCollectControl.s_debugTrailSize - gs_ShowtimeTrailBufferIndex;
         }

         // Check to see whether the showtime states trail buffer needs to swap data to external memory
         // and whether there is enough space in XMEM for the swap
         if ((NewTrailCnt > 0) && ((gt_TrailXmemCollectControl.s_debugTrailLen + NewTrailCnt) *
            sizeof(int16)) < gt_TrailXmemCollectControl.s_debugTrailSize)
         {
            // Setup to request XDMA for dataswap
            gs_XdmaWordCnt = NewTrailCnt;
            RequestSwap_STStatesTrailBuf_ToXmem();
            guc_STXdmaRequested = 1;
         }
      }
      else  /* previous dataswap has been completed */
      {
         // Update both local states trail buffer index and XMEM buffer index
         gs_ShowtimeTrailBufferIndex += gs_XdmaWordCnt;
         if (gs_ShowtimeTrailBufferIndex >= gt_debugTrailCollectControl.s_debugTrailSize)
         {
            gs_ShowtimeTrailBufferIndex = 0;
         }

         gt_TrailXmemCollectControl.s_debugTrailLen += gs_XdmaWordCnt;
         guc_STXdmaRequested = 0;
      }
   }
}


/****************************************************************************
;   Prototype: void RequestSwap_STStatesTrailBuf_ToXmem (void)
;
;   Description: Function sets up a request to swap showtime states trail
;           buffer to XMEM.
;
;           Note: XDMA addresses and size must be longword aligned.
;
;   Arguments:
;       None
;
;   Return Value:
;       None
;
;****************************************************************************/
void RequestSwap_STStatesTrailBuf_ToXmem (void)
{
    if (gt_DataSwap.c_State == DATASWAP_READY)
    {
        gt_DataSwap.c_State   = DATASWAP_REQUESTED;
        gt_DataSwap.l_SrcAddr = (int32)&gsa_ShowTimeStatesTrail[0] + gs_ShowtimeTrailBufferIndex * sizeof(int16);
        gt_DataSwap.l_DstAddr = XMEM_STATES_TRAIL_BUF_ADDR +
                                gt_TrailXmemCollectControl.s_debugTrailLen * sizeof(int16);
        gt_DataSwap.s_BufSize = gs_XdmaWordCnt * sizeof(int16);
        gt_DataSwap.c_XferDir = XDMA_WRITE_XMEM;
        gus_DataBuf_InXmem   |= STATES_TRAIL_BUF_IN_XMEM;
    }
}


/****************************************************************************
;   Prototype: int CheckSwappedDataInXmem (int16)
;
;   Description: Function checks to see whether the requested data had been
;                swapped to XMEM;
;   Arguments:
;       int16  CmvAddrField - CMV Address
;
;   Return Value:
;       0 - Data is not in XMEM
;       int32 XMEM data buffer address
;
;****************************************************************************/
int32 CheckSwappedDataInXmem (int16 CmvAddrField)
{
    int32 l_XmemDataAddr = 0;

    switch (CmvAddrField)
    {
        case INFO_ReverbSnrBuf:
            if (gus_DataBuf_InXmem & REVERB_SNR_BUF_IN_XMEM)
            {
                l_XmemDataAddr = XMEM_REVERB_SNR_BUF_ADDR;
            }
            break;

        case INFO_ReverbEchoSnrBuf:
            if (gus_DataBuf_InXmem & REVERB_ECHO_SNR_BUF_IN_XMEM)
            {
                l_XmemDataAddr = XMEM_REVERB_ECHO_SNR_BUF_ADDR;
            }
            break;

        case INFO_MedleySnrBuf:
            if (gus_DataBuf_InXmem & MEMLEY_SNR_BUF_IN_XMEM)
            {
                l_XmemDataAddr = XMEM_MEDLEY_SNR_BUF_ADDR;
            }
            break;

        case INFO_DebugTrail:
            if (gus_DataBuf_InXmem & STATES_TRAIL_BUF_IN_XMEM)
            {
                l_XmemDataAddr = XMEM_STATES_TRAIL_BUF_ADDR;
            }
            break;

        case INFO_RxMarginBuf:
            if (gus_DataBuf_InXmem & RX_MARGIN_BUF_IN_XMEM)
            {
                l_XmemDataAddr = XMEM_RX_MARGIN_BUF_ADDR;
            }
            break;
    }

    return l_XmemDataAddr;
}


/****************************************************************************
;   Prototype: int RequestSwappedDataFromXmem (int32, int32, uint16)
;
;   Description: Function waits until it's ready for data swap then it sets
;                up a request by loading address info into data swap struct.
;   Arguments:
;       int32   l_DstAddr - XDMA destination address
;       int32   l_SrcAddr - XDMA source address
;       uint16  us_Size   - XDMA size in bytes
;
;   Return Value:
;       0      Successful
;     E_CODE_DATASWAP_QUEUE_BUSY_ERROR
;
;****************************************************************************/
int RequestSwappedDataFromXmem (int32 l_DstAddr, int32 l_SrcAddr, uint16 us_Size)
{
   int timeout_cnt = 5000; // prevent hanging

    while (gt_DataSwap.c_State != DATASWAP_READY)
    {
      if (--timeout_cnt == 0)
      {
         return (E_CODE_DATASWAP_QUEUE_BUSY_ERROR);
      }
   }

    gt_DataSwap.c_State   = DATASWAP_REQUESTED;
    gt_DataSwap.l_SrcAddr = l_SrcAddr;
    gt_DataSwap.l_DstAddr = l_DstAddr;
    gt_DataSwap.s_BufSize = us_Size;
    gt_DataSwap.c_XferDir = XDMA_READ_XMEM;

    return (0);
}

