/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
// ******************************************************************
// AdcTrimming.c
//
// History
//
//15/01/2013 Kannan: ADC trimming Exception error indication
//Grep for: "XDSLRTFW-181: VR9_VRX318_AFE_Init_Fail_Indication"
//
//
// 16/05/2013 Kannan: Moved the variables of adc trimming to VR9_Afedata.c
//                    to the code swap page declared there under
//                    "DATA_XILVB_B4_SHOW_INIT_BEGIN"
//                    Grep for "XDSLRTFW-938 Improvement_VR9_VRX318_AdcTrimException"
// 19/05/2013 Kannan: Added a function to do the fast writing into the AFE
//                    which is time critical
//                    Grep for "XDSLRTFW-938 Improvement_VR9_VRX318_AdcTrimException"
// ******************************************************************

/*
** =============================================================================
**                           INCLUDE FILES
** =============================================================================
*/
#include "common.h"

#ifdef ADSL_62
    #include "const.h"
#else
    #include "vdsl_xception.h"
#endif

#include "typedef.h"
#include "cmv.h"
#include "VR9_Afe_const.h"             //SMS00972960
#include "VR9_Afe_typedef.h"           //SMS00972960
#include "AfeRdWrite.h"
#include "vr9_afe_analog_reg.h"
#include "vr9_afe_dig_global_reg.h"
#include "vr9_afe_dig_ch0_reg.h"
#include "vr9_afe_dig_filt_ch0_reg.h"
#include "vr9_afe_glp_slave_reg.h"
#include "VR9_Afedata.h"
#include "VR9_AfeFunc.h"
#include "delay.h"
#include "VRX_AfeCommonData.h"

/*
** =============================================================================
**                           LOCAL DEFINITIONS
** =============================================================================
*/
//Error codes


//Exception codes

#if 0
#define  ADC_TRIM_TIMEOUT                    (1<<14)  //TBD 16384
#else
// ADC Trim timeout is observed with (1<<13) in BedFord, Hence it is increased to
// (1<<15). (2/282Mhz) ~= 7.07 ns for 1 timer count
#define  ADC_TRIM_TIMEOUT                    (1<<15)  //32768 count * 1300(one AFE Rd timer count) * 7.07ns(for 1 timer count) = 300ms
#endif

/*
** =============================================================================
**                           LOCAL TYPES
** =============================================================================
*/
/*
** =============================================================================
**                           LOCAL DATA
** =============================================================================
*/
/*
** =============================================================================
**                           LOCAL FUNCTION PROTOTYPES
** =============================================================================
*/
/*
** =============================================================================
**                           IMPLEMENTATION
** =============================================================================
*/

/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  VR9_ADC_Trim
**
** DESCRIPTION:    This function gets the ADC threshold & Full scale trim
**                 values for 4 different VDSL modes.
**
** PARAMETERS:     None.
**
** RETURN VALUE:
**
**
** NOTES:          Reference: (1) AFE_ADCtrimmings.doc;
**                            SMS00920531
**
** =============================================================================
*/

void VR9_ADC_Trim()
{
   uint16 i, mode;

   //VDSL_30->0, VDSL_17->1, VDSL_12->2, VDSL_5->3
   for(mode = VDSL_30; mode <= VDSL_5; mode++)
   {
      //ADC trim settings are common for VDSL_17 & 12, hence skip for VDSL_12
      if (mode == VDSL_12) continue;

      i = (mode == VDSL_5) ? (mode-1): mode;

      //SMS00972960  - Start
      ADC_Trim_Run(mode, &gt_adc_trim.uca_adc_threshtrim_stg12[i<<5],
         &gt_adc_trim.uca_adc_fullscaletrim_stg12[i<<1]);
      //SMS00972960  - End
   }
}

/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  ADC_Trim_Run
**
** DESCRIPTION:    This function gets the ADC threshold & Full scale trim
**                 values for the given VDSL modes.
**
** PARAMETERS:     us_mode --> VDSl mode;
**                 uca_threshtrim_stg12 --> Threshold trim buffer
**                 uca_fullscaletrim_stg12 --> Full scale trim buffer
**
** RETURN VALUE:
**
**
** NOTES:          Reference: (1) AFE_ADCtrimmings.doc;
**                            (2) VR9_run_ADC_trimming.m
**                            SMS00920527
** =============================================================================
*/

void ADC_Trim_Run (uint16 us_mode, uint8 *uca_threshtrim_stg12,
                   uint8 *uca_fullscaletrim_stg12)
{
   uint16 *pInitValues, i;
   uint16 us_temp;

   //initialize AFE
   //25/11/2009: Kannan; "PAD_CTL"Commented since it needs to be first startup of AFE initialization
   //This is done as per the discussion with Kolhaupt Klaus (IFAT DCV WLC AD 1)
   //and Lusin Thomas (LQAT MSD SDC)
   //PAD_CTL_AFE_SET(0x0010);

   RESET_CTL_AFE_SET(0x00F7);
   //CLOCKING_CTL_AFE_SET(0x31F7);
   //VR9_BRINGUP: Observed the exception code "E_CODE_ADC_TRIM_FSCALE_TIMEOUT_ERROR" while
   // calling ADC Trim function.
   //After discussing with Thomas Lusin, It is required to initialize CLOCKING_CTL
   // with 0x30B3 instead on 0x31F7. Also need to initialize PLL_CTL3 register with
   // value "0"
   CLOCKING_CTL_AFE_SET(0x30B3);
   //Commented since it is not required here as per AFE team, it required for 3GLP
   //TPG_CTL_0_AFE_SET(0x0004);
   PLL_CTL3_AFE_SET(0x0);


   if (us_mode == VDSL_30)
      pInitValues = gusa_AfeAdcTrimRegs_VDSL30;
   else if(us_mode == VDSL_5)
      pInitValues = gusa_AfeAdcTrimRegs_VDSL05;
   else
      //(uc_mode == VDSL12) || (uc_mode == VDSL17)
      pInitValues = gusa_AfeAdcTrimRegs_VDSL17_12;

   for (i = 0; i < NUM_ADC_TRIM_INIT_REG; i++)
   {
      //VR9_AfeWrite(gusa_AfeAdcTrimRegsAddr[i], *(pInitValues + i),
      //   gusa_AfeAdcTrimRegs_Mask[i]);
      //VR9_AfeWrite(Address, Value, Mask);
      VR9_AfeWrite(gusa_AfeAdcTrimRegsAddr[i], *(pInitValues + i), 0xFFFF);
   }

   //25/11/2009: Kannan, There is some delay is required while reset out of the
   //ADC_LNDEC_STG1, hence it is done separately after lookup table
   //initialization. This is done according do the AFE team's recommentation.
   //Kolhaupt Klaus (IFAT DCV WLC AD 1) and Lusin Thomas (LQAT MSD SDC)
   //% Reset ADC
   //VR9(r.ADC_LNDEC_STG1,'0000','0100');
   //VR9_AfeWrite(ADC_LNDEC_STG1, 0x0000, ADC_LNDEC_STG1_ADC_FORCE_RSTQ_MASK);
   ADC_LNDEC_STG1_ADC_FORCE_RSTQ_AFE_SET(0);

   // This delay was arrived by experimental to have 1 micro sec
   delay(20); //20*10 = 200 cycles delay to reset out the ADC,

   //VR9(r.ADC_LNDEC_STG1,'0100','0100');
   //VR9_AfeWrite(ADC_LNDEC_STG1, 0x0100, ADC_LNDEC_STG1_ADC_FORCE_RSTQ_MASK);
   ADC_LNDEC_STG1_ADC_FORCE_RSTQ_AFE_SET(1);

   if(us_mode == VDSL_5)
   {
      //ADC trimming STG1
      //[trim_val_stg1_thresCOMP,trim_val_stg1_thresTOP]=trim_the_ADC(device,1);
      //Stage1: Index 0: 14-->Comp,15-->Top
      ADC_Trim_Do(0, &uca_threshtrim_stg12[0]);//Stg1: Idx 0: 14-->Comp,15-->Top,
      uca_fullscaletrim_stg12[0] = (ADC_COMP_QT_AFE_GET & 0x003F);// 0-->FullThresh
   }
   else
   {
      //(uc_mode == VDSL12) || (uc_mode == VDSL17) || (uc_mode == VDSL30))
      //ADC trimming STG1
      //[trim_val_stg1_thresCOMP,trim_val_stg1_thresTOP]=trim_the_ADC(device,1);
      //ADC trimming STG2
      //[trim_val_stg2_thresCOMP,trim_val_stg2_thresTOP]=trim_the_ADC(device,2);
      //ADC_Trim_Do(1, &uca_threshtrim_stg12[0]);//Stg1:Idx 0 to 14->Comp,15->Top,
      //ADC_Trim_Do(2, &uca_threshtrim_stg12[16]);//Stg2:Idx 16to 30->Comp,31->Top

      for (i = 0; i < NUM_ADC_TRIM_STG; i++)
         ADC_Trim_Do(i, &uca_threshtrim_stg12[i << 4]); //(i*16)

      us_temp = ADC_COMP_QT_AFE_GET;
      uca_fullscaletrim_stg12[0] = (us_temp & 0x003F);      //Stg1 FullThresh
      uca_fullscaletrim_stg12[1] = (us_temp & 0x3F00) >> 8; //Stg2 FullThresh
   }
}

/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  ADC_Trim_Do
**
** DESCRIPTION:    This function Perform the following
**                1. Initialize the trimming
**                2. Perform the threshold trimming of TOP comparator
**                3. Perform the full scale trimming using TOP comparator
**                4. Perform the threshold trimming of comparators from 0-14.
**
**
** PARAMETERS:
**
** RETURN VALUE:
**
**
** NOTES:          Reference: (1) AFE_ADCtrimmings.doc;
**                            (2) trim_the_ADC.m
**                            SMS00920527
** =============================================================================
*/
void ADC_Trim_Do(uint16 us_stage, uint8 *uca_trim_val_thresCOMP)
{
   uint8 uc_trim_val_thresTOP = 0;
   uint16 i, timeout_count, temp_stage = 0;

   //Initialize Trimming process - Start
   if (us_stage == 0)
   {
      //select ADC STAGE1 (default)
      ADCTRIM_0_ADC_STAGE_SEL_AFE_SET(ADC_STAGE_SEL_ADC_STAGE0);
      //set qt counter window to 2^11
      ADCTRIM_2_QT_WINDOW_AFE_SET(QT_WINDOW_ADC_QT_WDW_MAX);
      //reset ADC trimming state machine, and set breakpoint mode
      ADCTRIM_1_QT_SOFT_RESET_BREAKPOINT_AFE_SET(ADCTRIM_1_QT_SOFT_RESET_BREAKPOINT);
      //release reset for ADC trimming state machine
      ADCTRIM_1_BREAKPOINT_AFE_SET(BREAKPOINT_ADC_BREAKP_ON);
   }
   else
   {
      //ADC stage 2 is selected
      ADCTRIM_0_ADC_STAGE_SEL_AFE_SET(ADC_STAGE_SEL_ADC_STAGE1);
   }
   //Initialize Trimming process - End

   //TOP Threshold Trimming - Start
   //25/11/2009 : Kannan The below 2 lines are commented to do the fast writing of the AFE
   //to avoid additional delay
   //TOP Threshold Trimming - Start
   //start_tuning=1, breakpoint=1
   //ADCTRIM_1_START_TUNING_BREAKPOINT_AFE_SET(ADCTRIM_1_START_TUNING_BREAKPOINT);
   //XDSLRTFW-938 Improvement_VR9_VRX318_AdcTrimException   (START)
   VR9_AfeFastWrite(ADCTRIM_1, 0x3);
   //start_tuning=0, breakpoint=1
   //ADCTRIM_1_BREAKPOINT_AFE_SET(BREAKPOINT_ADC_BREAKP_ON);
   VR9_AfeFastWrite(ADCTRIM_1, 0x2);
   //XDSLRTFW-938 Improvement_VR9_VRX318_AdcTrimException   (END)

   //wait_for_continue=0;
   //while(wait_for_continue ~= 1)
   //    wait_for_continue=bitand(HSFPGA_SSC(device,r.ADCTRIM_STAT),
   //hex2dec('0040'))/2^6; %check the "wait" bit
   //end
   timeout_count = 0;
   while ((!ADCTRIM_STAT_WAIT_AFE_GET) && (timeout_count < ADC_TRIM_TIMEOUT))
   {
      timeout_count++;
   }
   if (timeout_count >= ADC_TRIM_TIMEOUT)
   {
      //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (START)
      if (gus_ExceptionCode == E_CODE_NO_ERROR)
         gus_ExceptionCode = E_CODE_ADC_TRIM_TOP_THRESHOLD_TIMEOUT_ERROR;
      //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (END)
   }

   //read and store value of bit<4:0>
   uc_trim_val_thresTOP = ADCTRIM_0_ADC_TRIMVALWR_AFE_GET;
   //TOP Threshold Trimming - End

   //Fullscale Trimming _ start
   // i = 0 & 1 for Full scale trimming and
   // i = 2 to 16 for Threshold trimming
   for (i = 0; i < 2+15; i++)
   {

      //25/11/2009 : Kannan The below 2 lines are commented to do the fast writing of the AFE
      //to avoid additional delay
      //continue=1, breakpoint=1
      //ADCTRIM_1_START_CONTINUE_BREAKPOINT_AFE_SET(ADC_CONTINUE_BREAKPOINT_ON);
      //XDSLRTFW-938 Improvement_VR9_VRX318_AdcTrimException   (START)
      VR9_AfeFastWrite(ADCTRIM_1, 0x6);
      //continue=0, breakpoint=1
      //ADCTRIM_1_BREAKPOINT_AFE_SET(BREAKPOINT_ADC_BREAKP_ON);
      VR9_AfeFastWrite(ADCTRIM_1, 0x2);
      //XDSLRTFW-938 Improvement_VR9_VRX318_AdcTrimException   (END)

      timeout_count = 0;
      while ((!ADCTRIM_STAT_WAIT_AFE_GET) && (timeout_count < ADC_TRIM_TIMEOUT))
      {
         timeout_count++;
      }
      if (timeout_count >= ADC_TRIM_TIMEOUT)
      {
         //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (START)
         if (gus_ExceptionCode == E_CODE_NO_ERROR)
            gus_ExceptionCode = E_CODE_ADC_TRIM_FSCALE_TIMEOUT_ERROR;
         //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (END)

      }
      //calibrating the DAC's in ADC - Start
      if (i == 1)
      {
         ADC_DACCAL_1_AFE_SET(0xA07F); //+1/2LSB
         ADC_DACCAL_2_AFE_SET(0xA07F); //+1/2LSB
         ADC_DACS_STG1_AFE_SET(0x0020);

         //Stage2 settings
         if (us_stage != 0)
            ADC_DACS_STG2_AFE_SET(0xA020);

         ADC_COMP_STG12_AFE_SET(0x0343);
         wait_ms(1);
      }
      //calibrating the DAC's in ADC - End
      //Fullscale Trimming _ End

      //Threshold Trimming - Start
      //Threshold Trimming of Comparators 0...14
      //i= 2 to 16 for threshold trimming
      if (i > 1)
         uca_trim_val_thresCOMP[i-2] = ADCTRIM_0_ADC_TRIMVALWR_AFE_GET;
   } //for

   //Threshold Trimming of thresholdTop 15
   uca_trim_val_thresCOMP[i-2] = uc_trim_val_thresTOP;
   //Threshold Trimming - End

   //Finish trimming process
   //continue=1, breakpoint=1
   //VR9_AfeWrite(ADCTRIM_1, ADCTRIM_1_CONTINUE | ADCTRIM_1_BREAKPOINT,
   //             ADCTRIM_1_CONTINUE | ADCTRIM_1_BREAKPOINT);
   ADCTRIM_1_START_CONTINUE_BREAKPOINT_AFE_SET(ADC_CONTINUE_BREAKPOINT_ON);

   //continue=0, breakpoint=1
   ADCTRIM_1_BREAKPOINT_AFE_SET(BREAKPOINT_ADC_BREAKP_ON);
   //check the "cal_ongoing" bit
   timeout_count = 0;
   while ((ADCTRIM_STAT_CAL_ONGOING_AFE_GET) && (timeout_count < ADC_TRIM_TIMEOUT))
   {
      timeout_count++;
   }
   if (timeout_count >= ADC_TRIM_TIMEOUT)
   {
      //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (START)
      if (gus_ExceptionCode == E_CODE_NO_ERROR)
         gus_ExceptionCode = E_CODE_ADC_TRIM_CAL_TIMEOUT_ERROR;
      //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (END)

   }
   //calibrating the DAC's in ADC - Start
   ADC_DACCAL_1_AFE_SET(0xA87F); //+1/2LSB
   ADC_DACCAL_2_AFE_SET(0xA87F); //+1/2LSB
   ADC_DACS_STG1_AFE_SET(0x0020);

   //Stage2 settings
   if (us_stage != 0)
      ADC_DACS_STG2_AFE_SET(0xA020);

   ADC_COMP_STG12_AFE_SET(0x0343);
   wait_ms(1);
   //calibrating the DAC's in ADC - End
}

/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  ADC_Trim_Comp_Write
**
** DESCRIPTION:    Generic routine which shall be called to write the trim
**                 values in ADC comparators using indirect addressing.
**
** PARAMETERS:
**
** RETURN VALUE:
**
**
** NOTES:          Reference: (1) AFE_ADCtrimmings.doc;
**                            (2) write_comparator.m
**                            SMS00920527
** =============================================================================
*/
void ADC_Trim_Comp_Write(uint16 comp_nr, uint16 stage, uint8 uc_trimvalue)
{
   uint16 value;
   //comparators de-select
   //VR9_AfeWrite(ADCTRIM_0, 0x1F00, 0x3F00); //read-modify-write command
   ADCTRIM_0_ADC_STAGE_COMP_SEL_AFE_SET(ADC_COMP_SEL_ADC_NO_COMP);  //read-modify-write command

   //calculate value for stage and comparator selection
   //value = comp_nr*2^8 + stage*2^13;
   //value_hex = dec2hex(value,4);

   //value = (1 << 8) * comp_nr + (1 << 13)*stage;
   value = ((comp_nr << 8 ) | (stage << 13));

   //select stage and comparator
   VR9_AfeWrite(ADCTRIM_0, value, 0x3F00); //read-modify-write command

   //go into write mode and write trimvalue
   VR9_AfeWrite(ADCTRIM_0, ((1 << 7) | uc_trimvalue), 0x009F);
   //ADCTRIM_0_ADC_TRIMWRITE_EN_TRIMVALWR_AFE_SET((1 << 7) | uc_trimvalue);


   //comparators de-select and enter in read mode
   //comparators de-select
   //VR9_AfeWrite(ADCTRIM_0, 0x1F00, 0x3F80); //read-modify-write command
   ADCTRIM_0_ADC_COMP_DESELECT_EN_TRIMREAD_AFE_SET(ADC_COMP_SEL_ADC_NO_COMP); //read-modify-write command
}



/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  ADC_Trim_Comp_Read ()
**
** DESCRIPTION:    Generic routine which shall be called to read out the
**                 trim values in ADC comparators using indirect addressing.
**
** PARAMETERS:
**
** RETURN VALUE:
**
**
** NOTES:          Reference: (1) AFE_ADCtrimmings.doc;
**                            (2) read_comparator.m
**                            SMS00920527
**
** =============================================================================
*/
uint8 ADC_Trim_Comp_Read(uint16 comp_nr, uint16 stage)
{
   uint16 value;
   uint8  uc_trimval_rd;

   //comparators de-select
   //VR9_AfeWrite(ADCTRIM_0, 0x1F00, 0x3F00); //read-modify-write command
   ADCTRIM_0_ADC_STAGE_COMP_SEL_AFE_SET(ADC_COMP_SEL_ADC_NO_COMP);  //read-modify-write command

   //go into read mode
   //VR9_AfeWrite(ADCTRIM_0, 0x0000, 0x0080); //read-modify-write command
   ADCTRIM_0_ADC_TRIMREAD_AFE_SET(0x0000);//read-modify-write command

   //calculate value for stage and comparator selection
   //value = comp_nr*2^8 + stage*2^13;
   //value_hex = dec2hex(value,4);

   value = (1 << 8) * comp_nr + (1 << 13)*stage;

   //select stage and comparator
   //VR9_AfeWrite(ADCTRIM_0, value, 0x3F00);
   ADCTRIM_0_ADC_STAGE_COMP_SEL_AFE_SET(value);  //read-modify-write command

   //read out the trimvalue
   uc_trimval_rd = (ADCTRIM_STAT_AFE_GET & 0x1F00) >> 8;

   //comparators de-select and enter in read mode
   //comparators de-select
   //VR9_AfeWrite(ADCTRIM_0, 0x1F00, 0x3F80); //read-modify-write command
   ADCTRIM_0_ADC_COMP_DESELECT_EN_TRIMREAD_AFE_SET(ADC_COMP_SEL_ADC_NO_COMP); //read-modify-write command
   return uc_trimval_rd;
}


/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  ADC_Trim_Load()
**
** DESCRIPTION:    This function shall be called to load the trim values
**                 to the ADC comparators for each stage based on the
**                 selected mode of ADC operation.
**
** PARAMETERS:
**
** RETURN VALUE:
**
**
** NOTES:          Reference: (1) AFE_ADCtrimmings.doc;
**                            SMS00920531
**
** =============================================================================
*/
void ADC_Trim_Load (uint16 mode)
{
   uint16 i, j, nstage = 0;
   uint16 Stg12BufIdx = 0;
   uint8 *pThreshTrimStg12;
   uint8 *pFullScaleTrimStg12;
   //VDSL_30->0, VDSL_17->1, VDSL_12->2, VDSL_5->3
   i = ((mode == VDSL_5) || (mode == VDSL_12))? (mode-1): mode;

   //SMS00972960  - Start
   pThreshTrimStg12 = &gt_adc_trim.uca_adc_threshtrim_stg12[i<<5];
   pFullScaleTrimStg12 = &gt_adc_trim.uca_adc_fullscaletrim_stg12[i<<1];
   //SMS00972960  - End

   //FullScale trim 1st Stage
   VR9_AfeWrite(ADC_COMP_QT, pFullScaleTrimStg12[0], 0x003F);
   if (mode != VDSL_5)
   {
      //FullScale trim 2nd Stage
      VR9_AfeWrite(ADC_COMP_QT, pFullScaleTrimStg12[1] << 8, 0x3F00);
      nstage = 1;  //2 stages
   }

   //Threshold trim 1st & 2nd stage loading
   for (j = 0; j <= nstage; j++)
   {
      //comparator selection
      for (i = 0; i < NUM_ADC_TRIM_COMP_PER_STG; i++)
      {
         ADC_Trim_Comp_Write(i, j, pThreshTrimStg12[Stg12BufIdx++]);
      }
   }
}

/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  ADC_Trim_Read_OverWrite()
**
** DESCRIPTION:    This function shall be called to read or overwrite the
**                 existing ADC trim comparator values.
**
** PARAMETERS:
**
** RETURN VALUE:
**
**
** NOTES:          Reference: (1) AFE_ADCtrimmings.doc;
**                 SMS00972960
** =============================================================================
*/
void ADC_Trim_Read_OverWrite (uint16 mode)
{
   uint16 i, j, nstage = 0;
   uint16 Idx = 0;

   //OverWrite option enabled, hence over write the values supplied through CMV
   if (gt_adc_trim.us_options & ADC_TRIM_OVER_WRITE_STAGE1)
   {
      if ((gt_adc_trim.us_options & ADC_TRIM_OVER_WRITE_STAGE12) == ADC_TRIM_OVER_WRITE_STAGE12)
         nstage = 1; //overwrite both stage 1 & 2

      //FullScale trim 1st Stage
      VR9_AfeWrite(ADC_COMP_QT, gt_adc_trim.uca_fscletrim_in_use_stg12[0], 0x003F);
      if (nstage)
      {
         //FullScale trim 2nd Stage
         VR9_AfeWrite(ADC_COMP_QT, gt_adc_trim.uca_fscletrim_in_use_stg12[1] << 8, 0x3F00);
      }

      //Threshold trim 1st & 2nd stage loading
      for (j = 0; j <= nstage; j++)
      {
         //comparator selection
         for (i = 0; i < NUM_ADC_TRIM_COMP_PER_STG; i++)
         {
            ADC_Trim_Comp_Write(i, j, gt_adc_trim.uca_thrshtrim_in_use_stg12[Idx++]);
         }
      } //for
   } //if

   ADC_Trim_Read(mode); //Do the Read operation at the end to reflect the AFE content

} //ADC_Trim_Read_OverWrite()


/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  ADC_Trim_Read()
**
** DESCRIPTION:    This function shall be called to read the trim values
**                 from the ADC comparators for each stage based on the
**                 selected mode of ADC operation.
**
** PARAMETERS:
**
** RETURN VALUE:
**
**
** NOTES:          Reference: (1) AFE_ADCtrimmings.doc;
**                 SMS00972960
** =============================================================================
*/
void ADC_Trim_Read(uint16 mode)
{
   uint16 i, j, us_temp,nstage = 0;
   uint16 Idx = 0;

   //Read the ADC comparator contents
   us_temp = VR9_AfeRead(ADC_COMP_QT);
   gt_adc_trim.uca_fscletrim_in_use_stg12[0] = us_temp & 0x003F; //1st Stage Full scale

   if (mode != VDSL_5)
   {
      nstage = 1;  //2 stages
      //2nd Stage Full scale
      gt_adc_trim.uca_fscletrim_in_use_stg12[1] = (us_temp & 0x3F00) >> 8;
   }

   //Threshold trim 1st & 2nd stage loading
   for (j = 0; j <= nstage; j++)
   {
      //comparator selection
      for (i = 0; i < NUM_ADC_TRIM_COMP_PER_STG; i++)
      {
         //Read back threshold comparator values of stage 1 & 2
         //Stored value will be read using CMV to know
         //threshold values written into the AFE
         gt_adc_trim.uca_thrshtrim_in_use_stg12[Idx++] =
            ADC_Trim_Comp_Read(i, j);
      } //for
   } //for
} //ADC_Trim_Read()

/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  VR9_ADC_Trim_Workaround()
**
** DESCRIPTION:    This function shall be called to verify the trim values.
**                 if any trim value is "0", this should be replaced with 0xF;
**                 This is the workaround. It will replaced with appropriate fix
**                 after the solution is found. This is done as per the I/P from Villagh.
**
** PARAMETERS:
**
** RETURN VALUE:
**
**
** =============================================================================
*/
void VR9_ADC_Trim_Workaround(void)
{
   uint8 *pThreshTrimStg12;
   uint8 *pFullScaleTrimStg12;
   uint32 i = 0;

   pThreshTrimStg12 = &gt_adc_trim.uca_adc_threshtrim_stg12[0];
   pFullScaleTrimStg12 = &gt_adc_trim.uca_adc_fullscaletrim_stg12[0];

   //First 32 loc for VDSL30, next 32 for VDSL17_12 and last 16 loc for VDSL05
   //uint8  uca_adc_threshtrim_stg12[ADC_THRESH_TRIM_LOC_STG12]; //32+32+16 = 80 bytes

   for (i=0; i < ADC_THRESH_TRIM_LOC_STG12; i++)
   {
      if (pThreshTrimStg12[i] == 0)
      {
         pThreshTrimStg12[i] = 0xF;
      }
   }

   //First 2 loc for VDSL30, next 2 for VDSL17_12 and last 1 loc for VDSL05
   //uint8  uca_adc_fullscaletrim_stg12[ADC_FULLSCALE_TRIM_LOC_STG12]; //2+2+1 = 5 bytes
   for (i=0; i < ADC_FULLSCALE_TRIM_LOC_STG12; i++)
   {
      if (pFullScaleTrimStg12[i] == 0)
         pFullScaleTrimStg12[i] = 0xF;
   }

} //VR9_ADC_Trim_Workaround()
