/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
// ******************************************************************
// AfeDigitalRegInit.c
//
// History
//
// 13-08-2010 Nihar: Only last 3 bits are used for controlling hybrid lif module
//              So, need to put the other 5 bits to 'Output enabled' state
//              otherwise they need to be 'pulled down'.
//              Grep for 'XDSLRTFW-161'
//
// 24/03/2011 Vinjam: Add "enable FIFOs" in FIFO init function and
//            renamed "VR9_A2x_FIFO_FIX( )" to "VR9_Init_AFE_FIFOs( )"
//            Grep for XDSLRTFW-271: Fix_AB_ALL_ALL_Init_FIFO
//
// 13/09/2011 Vinjam: Forced Oscillator control register "XO_CTL" to 0x400 as per Jira XDSLRTFW-331
//            Grep for XDSLRTFW-331 PERF_AB_ALL_ALL_Disable_OSC_AmplitudeControl
//
// 02/01/2013 Kannan: Uninitialized AFE registers were initilized as a part of AFE
//            power on initialization. Readonly registers were not touched and
//            clear on read (CLROR) registers were read to clear the sticky bits if any.
//            This is done to have the Reset values even while modifying some registers for
//            debug/measurement purpose.
//            Grep for 'XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values'
// ******************************************************************

/*
** =============================================================================
**                           INCLUDE FILES
** =============================================================================
*/
#include "common.h"
#include "typedef.h"
#include "AfeRdWrite.h"
#include "vr9_afe_analog_reg.h"
#include "vr9_afe_dig_global_reg.h"
#include "vr9_afe_dig_ch0_reg.h"
#include "vr9_afe_dig_filt_ch0_reg.h"
#include "vr9_afe_glp_slave_reg.h"
#include "VR9_AfeFunc.h"
#include "VR9_Afedata.h"
#include "VR9_Afe_const.h"
#include "delay.h"
#include "VRX_AfeCommonData.h"

/*
** =============================================================================
**                           LOCAL DEFINITIONS
** =============================================================================
*/


/*
** =============================================================================
**                           LOCAL TYPES
** =============================================================================
*/

/*
** =============================================================================
**                           LOCAL DATA
** =============================================================================
*/

/*
** =============================================================================
**                           LOCAL FUNCTION PROTOTYPES
** =============================================================================
*/
/*
** =============================================================================
**                           IMPLEMENTATION
** =============================================================================
*/

/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  VR9_Afe_Dig_Global_Regs_Reset
**
** DESCRIPTION:    This function writes the reset values into AFE digital global
**                 registers.
**
** PARAMETERS:     None.
**
** RETURN VALUE:   none
**
**
** NOTES:Reference: (1)XWAY_VRX208_V1.0_Design_Spec_Rev1.0_20090417.pdf
**                  (2)VR9_AFE_mode_settings_v1.xls;
**                  (3)UTP number SMS00920520
** =============================================================================
*/

void VR9_Afe_Dig_Global_Regs_Reset(void)
{
    uint16 us_temp;      //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START_END)
// Reset REG_BANK_A registers
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    PLL_CTL0_AFE_SET(0x301E);
    PLL_CTL1_AFE_SET(0x0037);
    PLL_CTL2_AFE_SET(0x0310);
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

    //PLL_ADC_CLK_OFF ADC clock is disabled,PLL_DAC1_CLK_OFF DAC1 clock is
    //disabled,PLL_DAC2_CLK_OFF DAC2 clock is disabled.
    PLL_CTL3_AFE_SET(0x0007);

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    us_temp = PLL_STAT_AFE_GET; //Readonly CLROR, Clear on read
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

    //XDSLRTFW-331 PERF_AB_ALL_ALL_Disable_OSC_AmplitudeControl (Start_End)
    //XDSLRTFW-2442 (Start_End)
    //XO_CTL_AFE_SET(0x0400); //moved to ConfigTaskLayerForLinkStart considering the CMV input.

    //ADC_FILT_CLK_OFF,DAC1_FILT_CLK_OFF Clock is disabled,DAC2_FILT_CLK_OFF
    //Clock is disabled,FCSI_CLK_ON Clock is enabled,FILT_GLP_TX1_CLK_OFF
    //FILT_GLP_TX2_CLK_OFF,FILT_GLP_RX_CLK_OFF,TPG_CLK_OFF Clock is disabled
    //TDA_CLK_OFF,TCLK_PHASE0 Phase 0 is selected,CLK_MON_DIS Clock monitor
    //disabled,DAC2_CLK_DEL3,DIS_INV_CORECLK
    CLOCKING_CTL_AFE_SET(0x3010);
    //ADC_FILT_RST_ON Sw-reset is enabled,DAC1_FILT_RST_ON Sw-reset is enabled
    //DAC2_FILT_RST_ON Sw-reset is enabled,
    //bit4:1b: FCSI_RST_OFF Sw-reset is disabled (****imp bit )
    //FILT_GLP_TX1_RST_ON Sw-reset is enabled,FILT_GLP_TX2_RST_ON Sw-reset
    //is enabled,FILT_GLP_RX_RST_ON Sw-reset is enabled.
    RESET_CTL_AFE_SET(0x0010); //TODO not matching with the comment in spec
    //XDSLRTFW-785: Bug_All_All_All_PsdAfeReg (START)
    //PAD_CTL<4> is set to "1" to have internal push pull operation.
    //if we set bit 4 to "0", Read operation to the AFE does not work. since
    //SCDO pin back to 'open drain' mode (PAD_CTL=0x0000).
    //The 'open drain' mode is only working if an external low ohmic pull-up exists.
    //Otherwise the SSC is limited in speed during read operations. Hence we can use
    //'push pull' setting all times, also in SSC point2multipoint applications,
    //i.e on chip bonding applications.
    // EN_PIN_CLK_OUT Pin CLK_OUT is enabled,EN_PIN_PLL_TEST_D,SCDO_OD SCDO
    // is open-drain
    //Make PAD_CTL<4> is set to "1" to have internal push pull operation.
    //if we set bit 4 to "0", Read operation to the AFE does not work.
    //PAD_CTL_AFE_SET(0x0000);
    //XDSLRTFW-785: Bug_All_All_All_PsdAfeReg (END)

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    GPIO_OMUX_CTL_0_AFE_SET(0x0000);
    GPIO_OMUX_CTL_1_AFE_SET(0x0000);
    GPIO_DOUT_AFE_SET(0x0000);
    //GPIO_DIN_AFE_GET; // Readonly Don't need to write
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

    // XDSLRTFW-161 (Start_End)
    // XDSLRTFW-2169 (START_END)
    // Configure Unused pins of GPIO from Bit 4 to 7 as input by setting 1 in all the bits
    // from Bit 4 to 7. This is done to reduce power consumption.
    GPIO_CTL_AFE_SET(0x00FF);

    //Register bank A & C initializations are done
    //Initialization of Reg Bank B, E & D.
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    //Reg Bank_B Global Registers
    RCMEAS_CTL_AFE_SET(0x0002);
    //RCMEAS_STAT_AFE_GET; //Readonly
    ADCTRIM_CTL_AFE_SET(0x0000);

    //Reg Bank_E Global Registers
    FUSE_DATA_0_AFE_SET(0x0000);
    FUSE_DATA_1_AFE_SET(0x0000);
    FUSE_DATA_2_AFE_SET(0x0000);
    FUSE_DATA_3_AFE_SET(0x0000);
    FUSE_DATA_4_AFE_SET(0x0000);
    FUSE_DATA_5_AFE_SET(0x0000);
    FUSE_DATA_6_AFE_SET(0x0000);
    FUSE_DATA_7_AFE_SET(0x0000);

    //FUSE_STAT_0_AFE_GET; //Readonly register
    //FUSE_STAT_1_AFE_GET; //Readonly register
    //FUSE_STAT_2_AFE_GET; //Readonly register
    //FUSE_STAT_3_AFE_GET; //Readonly register
    //FUSE_STAT_4_AFE_GET; //Readonly register
    //FUSE_STAT_5_AFE_GET; //Readonly register
    //FUSE_STAT_6_AFE_GET; //Readonly register
    //FUSE_STAT_7_AFE_GET; //Readonly register

    FUSE_CTL_AFE_SET(0x0013);
    //FUSE_RDY_STAT_AFE_GET; //Readonly register

    //Reg Bank_D Global Registers
    ADCTRIM_CNT_VAL_0_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_1_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_2_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_3_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_4_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_5_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_6_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_7_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_8_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_9_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_10_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_11_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_12_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_13_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_14_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_15_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_16_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_17_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_18_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_19_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_20_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_21_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_22_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_23_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_24_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_25_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_26_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_27_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_28_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_29_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_30_AFE_SET(0x0000);
    ADCTRIM_CNT_VAL_31_AFE_SET(0x0000);
    ADCTRIM_0_AFE_SET(0x1F00);
    ADCTRIM_1_AFE_SET(0x0000);
    ADCTRIM_2_AFE_SET(0x2000);
    //ADCTRIM_STAT_AFE_GET;  //read only
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

}

//XDSLRTFW-2442 OSC_AmplitudeControl (Start)

void ProgramOSCDriveLevel(uint16 us_data)
{
   XO_CTL_AFE_SET(us_data);
}
//XDSLRTFW-2442 OSC_AmplitudeControl (end)

/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  VR9_Glp_Slave_Regs_Reset
**
** DESCRIPTION:    This function writes the reset values into Glp Slave digital
**                 registers.
**
** PARAMETERS:     None.
**
** RETURN VALUE:   none
**
**
** NOTES:Reference: (1)XWAY_VRX208_V1.0_Design_Spec_Rev1.0_20090417.pdf
**                  (2)VR9_AFE_mode_settings_v1.xls;
**                  (3)UTP number SMS00920520
** =============================================================================
*/

void VR9_Glp_Slave_Regs_Reset(void)
{

// Reset GLP_REGFILE registers
    //Overrides CRC detection status of de-framers when OVR_CRC bits are OVR
    //Not Align for Word/Inband/Frame/Remote Frame alignment status.
    //Transmit not ready,enable Word Check,
    //Override inband/frame/remote/Transmit/word:NO_OVR,No override
    GLP_FR_CTRL_AFE_SET(0x0000);

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    //GLP_FR_STATUS_AFE_GET;  //Read only register
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

    //Overrides or reads Word Alignment Index of the deframer 1.
    //Overrides or reads Word Alignment Index of the deframer 2
    //Controls override of Word Alignment Index for corresponding de-framer
    GLP_FR_CTRL2_AFE_SET(0x0965);
    //Enables for framer module (EN),Enable RX Lane 1,Inband Control Override
    //disable, Override PMD/PHY (No override),Watchdog disable.
    GLP_FR_CTRL3_AFE_SET(0x0403);
    //Framer Bit Lock Counter Limit
    GLP_FR_BITLOCK_AFE_SET(0x000A);
    //Framer Watchdog Timer Limit
    GLP_FR_WDG_CFG_AFE_SET(0xFFFF);
    //Maximum CRC errors received: Sets the maximum number of CRC errors
    //received before framer restarts word alignment search.(0x64)
    //Minimum count of no error received:Sets the minimum count of "no errors"
    //condition received before resetting framer's error counter.(0xA
    GLP_FR_ERR_LIM_AFE_SET(0x0A64);
    //Framer Unique Word
    GLP_FR_UNIQ_WD_AFE_SET(0x6E78);

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    //GLP_FR_RX_UNIQ_WD_1_AFE_GET; // Read only register
    //GLP_FR_RX_UNIQ_WD_2_AFE_GET; // Read only register
    //GLP_FR_ERR_AFE_GET;          // Read only register
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

    //Bypass disabled,TX Enables(disabled),Enable Override of Bist(disabled)
    //Enable Bist(Enable)
    GLP_BIST_CTRL_AFE_SET(0x0000);
    //Phase Lock Count
    GLP_BIST_PHLOCK_AFE_SET(0x03E8);
    //Bist Endcount Register
    //Programs the number of words for the BIST to check for errors.
    GLP_BIST_END_AFE_SET(0x03E8);

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    GLP_BERT_TERM_AFE_SET(0x0000);
    //GLP_BERT_ERR_AFE_GET; //Read only Register
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

    //BERT TX Reset,BERT RX Reset,BERT Word Count(no Reset),BERT Error Count
    //(no reset),Scramble disable,
    //LSFR Data from BERTs will be used for scrambling data from PHY(set to 11'b)
    GLP_BERT_CTRL_AFE_SET(0x00C0);
    //3GLP Control Register:Clock switching(disabled),Data masking (disable)
    //Low Speed Test(disable)
    GLP_CTRL_AFE_SET(0x0000);

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    GLP_MISC_AFE_SET(0xC0C0);
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

// Reset PMA_REGFILE registers
    //Start Up Control Register:Bypass (disabled)
    //Skip Calibration enabled
    PMA_STARTUP_CTRL_AFE_SET(0x0400);

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    //PMA_STATE_AFE_GET; // Read only Register
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

// Reset PLL_REGFILE registers
    //PLL Control Register
    PLL_CTRL_AFE_SET(0x0000);

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    //PLL_ST_PLLINT_AFE_GET; //Read only Register
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

    //Startup control Register: tx_ibias_sel(001'b)rx_ibias_sel(011'b);
    PLL_STARTUP_CTRL_AFE_SET(0x1900);

// Reset TX_REGFILE registers
    //TX Control Register(Enable)
    TX_CTRL_AFE_SET(0x0001);
    //TX Analog Control Register;iout(0x7)
    TX_A_CTRL_AFE_SET(0x8027);
    //Programmable Data Register:Programmable TX Data(Disabled)
    TX_PROG_DATA_AFE_SET(0x0000);
    //Termination Resistor Register:termination resistors in PHY(0x19)
    TX_RTERM_AFE_SET(0x0019);
    //Modulation Configuration 1 Register
    TX_MOD1_AFE_SET(0x0000);
    //Modulation Configuration 2 Register (Skew Period:0)
    TX_MOD2_AFE_SET(0x0000);
    //Modulation Configuration 3 Register(The number of up or down modulation
    //pulses:0)
    TX_MOD3_AFE_SET(0x0000);
    //Test Control Register(Testbus Bist Enable)
    TX_TEST_CTRL_AFE_SET(0x0002);

// Reset RX1_REGFILE registers
    //Control Register:Enable (1);RES_SEL (1)
    RX1_CTRL_AFE_SET(0x0507);
    //Programmable Data Register,Programmable Data for RX1(0)
    //Programmable Data for RX1 (Disable).
    RX1_PROG_DATA_AFE_SET(0x0000);
    //CDR Configuration Register
    RX1_CDR_AFE_SET(0x0330);
    //Calibration Control Register:
    //The number of samples used for calibration(8)
    //The limit value for comparison used during calibration for hysteresis(8).
    RX1_CAL_CTRL_AFE_SET(0x0188);
    //Calibration Offsets Register
    //The OffsetP calibration value(0)The OffsetN calibration value.(0)
    RX1_CAL_READ_AFE_SET(0x0000);  //Read only ?
    //Calibration Bypass Register
    //BYPASS Disabled.
    RX1_CAL_BYP_AFE_SET(0x0000);
    //Termination Resistor Register
    RX1_RTERM_AFE_SET(0x0019);
    //Test Control Register,Testbus Bist Enable
    RX1_TEST_CTRL_AFE_SET(0x0002);

// Reset RX2_REGFILE registers
    //Control Register:Enable (1);RES_SEL (1)
    RX2_CTRL_AFE_SET(0x0507);
    //Programmable Data Register,Programmable Data for RX1(0)
    //Programmable Data for RX1 (Disable).
    RX2_PROG_DATA_AFE_SET(0x0000);
    //CDR Configuration Register
    RX2_CDR_AFE_SET(0x0330);
    //Calibration Control Register:
    //The number of samples used for calibration(8)
    //The limit value for comparison used during calibration for hysteresis(8).
    RX2_CAL_CTRL_AFE_SET(0x0188);
    //Calibration Offsets Register
    //The OffsetP calibration value(0)The OffsetN calibration value.(0)
    RX2_CAL_READ_AFE_SET(0x0000); //Read only ?
    //Calibration Bypass Register
    //BYPASS Disabled.
    RX2_CAL_BYP_AFE_SET(0x0000);
    //Termination Resistor Register
    RX2_RTERM_AFE_SET(0x0019);
    //Test Control Register,Testbus Bist Enable
    RX2_TEST_CTRL_AFE_SET(0x0002);

// Reset RX3_REGFILE registers
    //Control Register:Enable (1);RES_SEL (1)
    RX3_CTRL_AFE_SET(0x0507);
    //Programmable Data Register,Programmable Data for RX1(0)
    //Programmable Data for RX1 (Disable).
    RX3_PROG_DATA_AFE_SET(0x0000);
    //CDR Configuration Register
    RX3_CDR_AFE_SET(0x0330);
    //Calibration Control Register:
    //The number of samples used for calibration(8)
    //The limit value for comparison used during calibration for hysteresis(8).
    RX3_CAL_CTRL_AFE_SET(0x0188);
    //Calibration Offsets Register
    //The OffsetP calibration value(0)The OffsetN calibration value.(0)
    RX3_CAL_READ_AFE_SET(0x0000); //Read only ?
    //Calibration Bypass Register
    //BYPASS Disabled.
    RX3_CAL_BYP_AFE_SET(0x0000);
    //Termination Resistor Register
    RX3_RTERM_AFE_SET(0x0019);
    //Test Control Register,Testbus Bits Enable
    RX3_TEST_CTRL_AFE_SET(0x0002);

}

/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  VR9_Afe_Dig_Filt_CH0_Regs_Reset
**
** DESCRIPTION:    This function writes the reset values into Digital Filter CH0
**                 registers.
**
** PARAMETERS:     None.
**
** RETURN VALUE:   none
**
**
** NOTES:Reference: (1)XWAY_VRX208_V1.0_Design_Spec_Rev1.0_20090417.pdf
**                  (2)VR9_AFE_mode_settings_v1.xls;
**                  (3)UTP number SMS00920520
** =============================================================================
*/

void VR9_Afe_Dig_Filt_CH0_Regs_Reset(void)
{
   uint16 us_temp;      //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START_END)

   //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
   FGLB_DIFI_TLOOP_AFE_SET(0x0000);
   //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

// Reset FGLB registers
    //Sampling Frequency Selection for DAC1 216 MHz,
    //Sampling Frequency Selection for DAC2 216 MHz
    //Sampling Frequency Selection for ADC 288 MHz
    //Transmit Path Trigger initialization enabled
    //Echo Path Trigger initialization enabled
    //RX Path Trigger initialization enabled
    FGLB_CTRL1_AFE_SET(0x00F5);
    //Transmit Path Disable,Echo Path Disable,Receive Path Disable
    FGLB_CTRL2_AFE_SET(0x0000);

   //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values  (START)
   FGLB_CTRL3_AFE_SET(0x0E40);
   FGLB_CTRL4_AFE_SET(0x0000);
   FGLB_CTRL5_AFE_SET(0x0000);
   FGLB_ADC_INTCTRL_AFE_SET(0x0808);
   FGLB_TX_ECNT_THRES_AFE_SET(0xFFFF);
   FGLB_EC_ECNT_THRES_AFE_SET(0xFFFF);
   FGLB_RX_ECNT_THRES_AFE_SET(0xFFFF);
   FGLB_ADC_INTCTRL_THRES_AFE_SET(0x0100);
   us_temp = FGLB_FILT_STAT_AFE_GET; //Clear on Read, hence read it here.
   FGLB_FILT_IMASK_AFE_SET(0x0000);
   //FGLB_TX_ECNT_AFE_GET;  //Readonly
   //FGLB_EC_ECNT_AFE_GET;  //Echo filters overflow counter, Readonly
   //FGLB_RX_ECNT_AFE_GET;  //Rx filters overflow counter, Readonly
   //FGLB_ADC_OCNT_AFE_GET; //ADC Overflow counter, Readonly
   //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

// Reset RXCMB1 registers
    //RXCMB1 Filter disabled,initialization of RXCMB1 Filter enabled
    //Second Branch Cancellation logic enabled
    RXCMB1_CTRL_AFE_SET(0x0002);

    //RX CLCMB1 programmable subfilter coefficients minimum value (0)
    RXCMB1_COEFF_0_AFE_SET(0x0000);
    RXCMB1_COEFF_1_AFE_SET(0x0000);
    RXCMB1_COEFF_2_AFE_SET(0x0000);
    RXCMB1_COEFF_3_AFE_SET(0x0000);
    RXCMB1_COEFF_4_AFE_SET(0x0000);
    RXCMB1_COEFF_5_AFE_SET(0x0000);
    RXCMB1_COEFF_6_AFE_SET(0x0000);
    RXCMB1_COEFF_7_AFE_SET(0x0000);
    RXCMB1_COEFF_8_AFE_SET(0x0000);
    RXCMB1_COEFF_9_AFE_SET(0x0000);
    RXCMB1_COEFF_10_AFE_SET(0x0000);

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    us_temp = RXCMB1_STAT_AFE_GET;  //Saturation overflow flag, Clear on read, hence read should be OK.
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

// Reset RXBRWDF1 registers
    //RX BRWDF1 Filter Disabled,initialization of RX BRWDF1 Filter enabled
    //RX BRWDF1 decimators and filters are not bypassed
    //Resets the filter upon internal detection of an overflow enabled.
    RXBRWDF1_CTRL_AFE_SET(0x0102);
    //RX WDF1 programmable adaptor G1 coefficient startup value
    RXBRWDF1_COEFF_G1_AFE_SET(0x0047);
    //RX WDF1 programmable adaptor G3 coefficient startup value
    RXBRWDF1_COEFF_G3_AFE_SET(0x012C);

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    us_temp = RXBRWDF1_STAT_AFE_GET;  //Saturation overflow flag, Clear on read, hence read should be OK.
    RXADCT_CTRL_AFE_SET(0x0002);
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)


// Reset TX1ERRC registers
    //TX Error Correction Disabled
    //initialization of TX Error Correction enabled
    //TX 3GLP Error Correction Bypass Disabled
    TX1ERRC_CTRL_AFE_SET(0x0002);
    //Error Detection Control Register
    //MSB (3 bits) parity Error Detection Enabled
    //Source of the Parity Error Detection MSB parity error and normal parity
    //error,source for Parity Error Event Counter( Single)
    TX1ERRC_ERRD_CTRL_AFE_SET(0x0001);

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    us_temp = TX1ERRC_STAT_AFE_GET;  //Saturation overflow flag, Clear on read, hence read should be OK.
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

// Reset TX1BRWDF1 registers
    //TX BRWDF1 Filter Disabled,initialization of TX BRWDF1 Filter Enabled
    //TX BRWDF1 Filter Bypass Disabled,resets the filter upon internal detection
    // of an overflow of the adaptors Enabled,Interpolation Factor 1.5
    TX1BRWDF1_CTRL_AFE_SET(0x0102);
    //TX WDF1 programmable adaptor G01 coefficient Startup value
    TX1BRWDF1_COEFF_G01_AFE_SET(0x0020);
    //TX WDF1 programmable adaptor G02 coefficient Startup value
    TX1BRWDF1_COEFF_G02_AFE_SET(0x017C);
    //TX WDF1 programmable adaptor G11 coefficient Startup value
    TX1BRWDF1_COEFF_G11_AFE_SET(0x005C);
    //TX WDF1 programmable adaptor G21 coefficient Startup value
    TX1BRWDF1_COEFF_G21_AFE_SET(0x00F0);

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    us_temp = TX1BRWDF1_STAT_AFE_GET;  //Saturation overflow flag, Clear on read, hence read should be OK.
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

// Reset TX1NSH registers
    //TX Noise Shaper filter disabled,initialization of TX Noise Shaper enabled
    //TX Noise Shaper Filter Bypass disabled,TX Noiseshaper 1st Order
    //TX Noise Shaper Output Bitwidth 13 bits
    TX1NSH_CTRL_AFE_SET(0x0002);

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    us_temp = TX1NSH_STAT_AFE_GET;  //Saturation overflow flag, Clear on read, hence read should be OK.
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

// Reset TX2ERRC registers
    //EC Error Correction Disabled
    //initialization of EC Error Correction enabled
    //EC 3GLP Error Correction Bypass Disabled
    TX2ERRXC_CTRL_AFE_SET(0x0002);
    //Error Detection Control Register
    //MSB (3 bits) parity Error Detection Enabled
    //Source of the Parity Error Detection: MSB parity error and normal parity
    //error,source for Parity Error Event Counter( Single)
    TX2ERRXC_ERRD_CTRL_AFE_SET(0x0001);

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values  (START)
    us_temp = TX2ERRXC_STAT_AFE_GET;  //Saturation overflow flag, Clear on read, hence read should be OK.
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

// Reset TX2BRWDF1 registers
    //EC BRWDF1 Filter Disabled,initialization of EC BRWDF1 Filter Enabled
    //EC BRWDF1 Filter Bypass Disabled,Resets the filter upon internal detection
    // of an overflow of the adaptors Enabled,Interpolation Factor 1.5
    TX2BRWDF1_CTRL_AFE_SET(0x0102);
    //EC WDF1 programmable adaptor G01 coefficient Startup value
    TX2BRWDF1_COEFF_G01_AFE_SET(0x0020);
    //EC WDF1 programmable adaptor G02 coefficient Startup value
    TX2BRWDF1_COEFF_G02_AFE_SET(0x017C);
    //EC WDF1 programmable adaptor G11 coefficient Startup value
    TX2BRWDF1_COEFF_G11_AFE_SET(0x005C);
    //EC WDF1 programmable adaptor G21 coefficient Startup value
    TX2BRWDF1_COEFF_G21_AFE_SET(0x00F0);

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    us_temp = TX2BRWDF1_STAT_AFE_GET;  //Saturation overflow flag, Clear on read, hence read should be OK.
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

// Reset TX2NSH registers
    //EC Noise Shaper filter disabled,initialization of EC Noise Shaper enabled
    //EC Noise Shaper Filter Bypass disabled,EC Noiseshaper 1st Order
    //EC Noise Shaper Output Bitwidth 13 bits
    TX2NSH_CTRL_AFE_SET(0x0002);

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    us_temp = TX2NSH_STAT_AFE_GET;  //Saturation overflow flag, Clear on read, hence read should be OK.
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

    //AEC Programmable Gain Register
    //AEC Programmable Bypass Enabled
    //Gain Factor set to min value (0)
    TX2NSH_AEC_AFE_SET(0x0010);

}

/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  VR9_Afe_Dig_CH0_Regs_Reset
**
** DESCRIPTION:    This function writes the reset values into Digital CH0
**                 registers.
**
** PARAMETERS:     None.
**
** RETURN VALUE:   none
**
**
** NOTES:Reference: (1)XWAY_VRX208_V1.0_Design_Spec_Rev1.0_20090417.pdf
**                  (2)VR9_AFE_mode_settings_v1.xls;
**                  (3)UTP number SMS00920520
** =============================================================================
*/
void VR9_Afe_Dig_CH0_Regs_Reset(void)
{
    uint16 us_temp;      //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START_END)

    // Reset TX2NSH registers
    //ADC Functional Reset:Resets ADC digital data interface,
    //ADC digital data interface is in reset state,ADC digital data interface
    //clock is disabled,ADC digital data interface clock inversion is disabled
    ADC_CTL_AFE_SET(0x0000);
    //DAC1 digital core is in reset state,DAC1 digital core clock is disabled
    //DAC1 clock buffer is powerdown,DAC1 digital data interface clock is
    //disabled,DAC1 digital data interface clock inversion is disabled
    DAC1_CTL_AFE_SET(0x0004);
    //DAC2 digital core is in reset state,DAC2 digital core clock is disabled
    //DAC2 clock buffer is powerdown,DAC2 digital data interface clock is
    //disabled,DAC2 digital data interface clock inversion is disabled
    DAC2_CTL_AFE_SET(0x0004);

    //GPIO_LD Output Data:Sets the value of each GPIO line driver output pin
    //Constants. All line driver output pins are set to low.
    GPIO_LD_DOUT_AFE_SET(0x0000);

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    //GPIO_LD_DIN_AFE_GET;  //Read only
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

    //GPIO_LD Output Enable:Controls each GPIO line driver output pin
    //(All line driver output pins are disabled)
    //GPIO_LD Push-pull Enable:Enables GPIO line driver outputs push-pull
    //(All line driver output pins are opendrain).
    GPIO_LD_CTL_AFE_SET(0x000F);

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (START)
    //3GLP registers were initialized as a part of 3GLP AFE Init
    //GLP_CTL0(0x0001);
    //GLP_CTL1(0x0000);
    //GLP_CTL2(0x0000);
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)

    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values  (START)
    TPG_CTL_0_AFE_SET(0x0004);
    TPG_CTL_1_AFE_SET(0x0004);
    TPG_CTL_2_AFE_SET(0x0004);
    TPG_DAT_0_AFE_SET(0x0000);
    TPG_DAT_1_AFE_SET(0x0000);
    TPG_DAT_2_AFE_SET(0x0000);
    //TPG_STAT_0_AFE_GET; //Read only
    //TPG_STAT_1_AFE_GET; //Read only
    //TPG_STAT_2_AFE_GET; //Read only
    TDA_CTL_0_AFE_SET(0x0004);
    TDA_CTL_1_AFE_SET(0x0004);
    TDA_CTL_2_AFE_SET(0x0004);
    TDA_DAT_0_AFE_SET(0x0000);
    TDA_DAT_1_AFE_SET(0x0000);
    TDA_DAT_2_AFE_SET(0x0000);
    //TDA_STAT_0_AFE_GET; //Read only
    //TDA_STAT_1_AFE_GET; //Read only
    //TDA_STAT_2_AFE_GET; //Read only
    MISC_CTL_AFE_SET(0x00C0);
    us_temp = MISC_STAT_AFE_GET; //Read only, Clear on Read

    //Given below AFE FIFO Registers were initialized in "VR9_Init_AFE_FIFOs()"
    //GLP_RXFIFO_CTL, GLP_TX0FIFO_CTL, GLP_TX1FIFO_CTL, ADC_FIFO_CTL, DAC1_FIFO_CTL, DAC2_FIFO_CTL;
    //XDSLRTFW-226 VR9_VRX318_AFE_PowerOn_Reset_Values (END)
}


/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  VR9_ResetOutDigitalBlocks
**
** DESCRIPTION:    This function resetout the Digital blocks of the AFE.
**
** PARAMETERS:     None.
**
** RETURN VALUE:   none
**
**
** NOTES:Reference: Mail from Lusin Thomas, Dated 18/11/2009, 15/12/2009
** =============================================================================
*/

void VR9_ResetOutDigitalBlocks()
{

    // DAC2 Clock PowerUp & PowerDown
    if (!gft_AEC_On)
    {
        // These two commands simply switch on and off the clock in the DAC2.
        // If this is not done the output of the DAC2 is in an undefined state
        // and could disturb the receiver => a higher noise can be observed,
        // performance will be degraded
        DAC2_CTL_AFE_SET(0x0000);     // DAC2 clk PowerUp
        DAC2_CTL_AFE_SET(0x0004);     // DAC2 clk PowerDown
    }

    //% Reset ADC
    // VR9(r.ADC_LNDEC_STG1,'0800','0800');
    ADC_LNDEC_STG1_ADC_INVCLK_RSTQ_AFE_SET(ADC_INVCLK_RSTQ_ADC_RESET_CLK_INV);
    //VR9(r.ADC_LNDEC_STG1,'0000','0100');
    ADC_LNDEC_STG1_ADC_FORCE_RSTQ_AFE_SET(ADC_FORCE_RSTQ_ADC_NO_RESET);
    delay(10); //10 cycles delay to reset out the ADC, this delay was arrived by experiment

    //% Reset DAC1 calibration
    //VR9(r.DAC1_CAL_2,'1000','1000');
    DAC1_CAL_2_DAC1_RESET_CAL_AFE_SET(DAC1_RESET_CAL_DAC1_CAL_RESET_ON);
    delay(20); //20 cycles delay to reset out the DAC, this delay was arrived by experiment
    //VR9(r.DAC1_CAL_2,'0000','1000');
    DAC1_CAL_2_DAC1_RESET_CAL_AFE_SET(DAC1_RESET_CAL_DAC1_CAL_RESET_OFF);

    //% Reset DAC2 calibration
    //VR9(r.DAC2_CAL_2,'1000','1000');
    DAC2_CAL_2_DAC2_RESET_CAL_AFE_SET(DAC2_RESET_CAL_DAC2_CAL_RESET_ON);
    delay(20); //20 cycles delay to reset out the DAC, this delay was arrived by experiment
    //VR9(r.DAC2_CAL_2,'0000','1000');
    DAC2_CAL_2_DAC2_RESET_CAL_AFE_SET(DAC2_RESET_CAL_DAC2_CAL_RESET_OFF);

    //% Enable initialization
    //VR9_h(r.RXBRWDF1_CTRL, '0002','0002');  % RX
    //VR9_AfeWrite(RXBRWDF1_CTRL, 0x0002, RXBRWDF1_CTRL_INIT_MASK);
    RXBRWDF1_CTRL_INIT_AFE_SET(INIT_EN); // (1 << 1)

    //VR9_h(r.RXCMB1_CTRL,   '0002','0002');
    //VR9_AfeWrite(RXCMB1_CTRL, 0x0002, RXCMB1_CTRL_INIT_MASK);
    RXCMB1_CTRL_INIT_AFE_SET(INIT_EN);   // (1 << 1)

    //VR9_h(r.TX1NSH_CTRL,   '0002','0002');  % TX1
    //VR9_AfeWrite(TX1NSH_CTRL, 0x0002, TX1NSH_CTRL_INIT_MASK);
    TX1NSH_CTRL_INIT_AFE_SET(INIT_EN);   // (1 << 1)

    //VR9_h(r.TX1BRWDF1_CTRL,'0002','0002');
    //VR9_AfeWrite(TX1BRWDF1_CTRL, 0x0002, TX1BRWDF1_CTRL_INIT_MASK);
    TX1BRWDF1_CTRL_INIT_AFE_SET(INIT_ON); // (1 << 1)

    //VR9_h(r.TX1ERRC_CTRL,  '0002','0002');
    //VR9_AfeWrite(TX1ERRC_CTRL, 0x0002, TX1ERRC_CTRL_INIT_MASK);
    TX1ERRC_CTRL_INIT_AFE_SET(INIT_EN);

    //VR9_h(r.TX2NSH_CTRL,   '0002','0002');  % TX2
    //VR9_AfeWrite(TX2NSH_CTRL, 0x0002, TX2NSH_CTRL_INIT_MASK);
    TX2NSH_CTRL_INIT_AFE_SET(INIT_EN);

    //VR9_h(r.TX2BRWDF1_CTRL,'0002','0002');
    //VR9_AfeWrite(TX2BRWDF1_CTRL, 0x0002, TX2BRWDF1_CTRL_INIT_MASK);
    TX2BRWDF1_CTRL_INIT_AFE_SET(INIT_ON);

    //VR9_h(r.TX2ERRC_CTRL,  '0002','0002');
    //VR9_AfeWrite(TX2ERRXC_CTRL, 0x0002, TX2ERRXC_CTRL_INIT_MASK);
    TX2ERRXC_CTRL_INIT_AFE_SET(INIT_EN);

    delay(20); //20 cycles delay, this delay was arrived by experiment

    //%Disable initialization
    //VR9_h(r.RXBRWDF1_CTRL, '0000','0002');  % RX
    //VR9_AfeWrite(RXBRWDF1_CTRL, 0x0000, RXBRWDF1_CTRL_INIT_MASK);
    RXBRWDF1_CTRL_INIT_AFE_SET(INIT_DIS);

    //VR9_h(r.RXCMB1_CTRL,   '0000','0002');
    //VR9_AfeWrite(RXCMB1_CTRL, 0x0000, RXCMB1_CTRL_INIT_MASK);
    RXCMB1_CTRL_INIT_AFE_SET(INIT_DIS);

    //VR9_h(r.TX1NSH_CTRL,   '0000','0002');  % TX1
    //VR9_AfeWrite(TX1NSH_CTRL, 0x0000, TX1NSH_CTRL_INIT_MASK);
    TX1NSH_CTRL_INIT_AFE_SET(INIT_DIS);

    //VR9_h(r.TX1BRWDF1_CTRL,'0000','0002');
    //VR9_AfeWrite(TX1BRWDF1_CTRL, 0x0000, TX1BRWDF1_CTRL_INIT_MASK);
    TX1BRWDF1_CTRL_INIT_AFE_SET(INIT_OFF);

    //VR9_h(r.TX1ERRC_CTRL,  '0000','0002');
    //VR9_AfeWrite(TX1ERRC_CTRL, 0x0000, TX1ERRC_CTRL_INIT_MASK);
    TX1ERRC_CTRL_INIT_AFE_SET(INIT_DIS);

    //VR9_h(r.TX2NSH_CTRL,   '0000','0002');  % TX2
    //VR9_AfeWrite(TX2NSH_CTRL, 0x0000, TX2NSH_CTRL_INIT_MASK);
    TX2NSH_CTRL_INIT_AFE_SET(INIT_DIS);

    //VR9_h(r.TX2BRWDF1_CTRL,'0000','0002');
    //VR9_AfeWrite(TX2BRWDF1_CTRL, 0x0000, TX2BRWDF1_CTRL_INIT_MASK);
    TX2BRWDF1_CTRL_INIT_AFE_SET(INIT_OFF);

    //VR9_h(r.TX2ERRC_CTRL,  '0000','0002');
    //VR9_AfeWrite(TX2ERRXC_CTRL, 0x0000, TX2ERRXC_CTRL_INIT_MASK);
    TX2ERRXC_CTRL_INIT_AFE_SET(INIT_DIS);

    // % ADC release reset
    // VR9(r.ADC_LNDEC_STG1,'0100','0100');
    ADC_LNDEC_STG1_ADC_FORCE_RSTQ_AFE_SET(ADC_FORCE_RSTQ_ADC_FORCE_RESET);

    //XDSLRTFW-271: Fix_AB_ALL_ALL_Init_FIFO (Start)
    //Initialize AFE FIFOs for all chips. Older chips don't have FIFOs. Still It won't cause any problem
    //as writing to those memory locations is void - Comment from Stefan
    VR9_Init_AFE_FIFOs();
    //XDSLRTFW-271: Fix_AB_ALL_ALL_Init_FIFO (End)

}

/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  VR9_Afe_Digital_Regs_Reset
**
** DESCRIPTION:    This function writes resets values into all Digital
**                 registers.
**
** PARAMETERS:     None.
**
** RETURN VALUE:   none
**
**
** NOTES:Reference: (1)XWAY_VRX208_V1.0_Design_Spec_Rev1.0_20090417.pdf
**                  (2)VR9_AFE_mode_settings_v1.xls;
**                  (3)UTP number SMS00920520
** =============================================================================
*/

void VR9_AfeInitDigitalRegistersReset(void)
{

//Program reset values in to digital global register set
    VR9_Afe_Dig_Global_Regs_Reset();
    ////Program reset values in to Glp slave register set
    VR9_Glp_Slave_Regs_Reset();
    //Program reset values in to digital Filter CH0 register set
    VR9_Afe_Dig_Filt_CH0_Regs_Reset();
    //Program reset values in to digital CH0 register set
    VR9_Afe_Dig_CH0_Regs_Reset();

}
