/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
// ******************************************************************
// AfeRdWrite.c
//
// History
//
//15/01/2013 Kannan: AFE Read & Write Exception error indication
//Grep for: "XDSLRTFW-181: VR9_VRX318_AFE_Init_Fail_Indication"
//
//19/05/2013 Kannan: Added a function to do the AFE Fast write
//Grep for: "XDSLRTFW-938 Improvement_VR9_VRX318_AdcTrimException"
//
//18/07/2014 Balabth: AFE Read access failures are seen because of  AFE_DONE bit not getting cleared.
//           AFE_DONE bit is W1C (Write 1 to clear this bit). DSL FW write 1 to clear this bit for both AFE read and AFE write functions.
//           We are reading the register immediately after setting this bit some times not working.
//          added wait until AFE_DONE bit get cleared .
//          And There was a bug for reporting the sub-error codes if AFE failure occurs after LinkStart.Fixed the same
//          Grep for: "XDSLRTFW-1831"
//
// ******************************************************************
/*
** =============================================================================
**                           INCLUDE FILES
** =============================================================================
*/
#include "common.h"

#ifdef ADSL_62
    #include "const.h"
#else
    #include "vdsl_xception.h"
#endif

#include "typedef.h"
#include "str_memmap.h"
#include "VR9_Afedata.h"
#include "LL_IOf.h"
#include "delay.h"
#include "AfeRdWrite.h"
#include "VR9_AfeFunc.h"
#include "gdata.h"

/*
** =============================================================================
**                           LOCAL DEFINITIONS
** =============================================================================
*/
#define AFE_DONE_BIT                   (0x80000000)  //<Bit 31>
#define AFE_BUSY_BIT                   (0x40000000)  //<Bit 30>
#define AFE_FLUSH_BIT                  (0x00000800)  //<Bit 11>
#define SSC_IF_SCLK_SPEED              (0x00000008) //18Mhz;
#define AFE_RD_WR_TIMEOUT              (5000)   //TBD


/*
** =============================================================================
**                           LOCAL TYPES
** =============================================================================
*/

/*
** =============================================================================
**                           LOCAL DATA
** =============================================================================
*/


/*
** =============================================================================
**                           LOCAL FUNCTION PROTOTYPES
** =============================================================================
*/

/*
** =============================================================================
**                           IMPLEMENTATION
** =============================================================================
*/

/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  WaitForAfeBusy
**
** DESCRIPTION:   Wait in a loop as long as AFE done bit is "0" and
**                AFE Busy bit is "1" and timeout < threshold
**
** PARAMETERS:
**
** RETURN VALUE:
**
**
** NOTES:        Reference: (1) AFE_Control_IF_SSC_v1.doc;
**                          (2) Strymon-VFDF_xDSL_CPE_bonded.pdf
**                            SMS00917387
**
** =============================================================================
*/

void WaitForAfeBusy (void)
{
   uint16  us_timeout = 0;
   uint32  ul_cfg;

   //Wait in a loop as long as AFE done bit is "0" and
   //AFE Busy bit is "1" and timeout < threshold
   do
   {
      delay(1);  //1*10 = 10 cycles delay
      ReadCoreReg (V_AFE_SERIAL_CFG_ADDR, &ul_cfg);  //V_AFE_SERIAL_CFG_ADDR
      us_timeout++;
   }while( (!(ul_cfg & AFE_DONE_BIT)) && (ul_cfg & AFE_BUSY_BIT) && (us_timeout < AFE_RD_WR_TIMEOUT) );

   if (us_timeout >= AFE_RD_WR_TIMEOUT)
   {
      //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (START)
      if (gus_ExceptionCode == E_CODE_NO_ERROR)
         gus_ExceptionCode = E_CODE_AFE_NOT_READY;
      //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (END)
   }
}
//XDSLRTFW-1831 (start)
/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  WaitForAfeDone
**
** DESCRIPTION:   Wait in a loop as long as AFE done bit is "1". Have seen
**                Afe read failures otherwise.
**
** PARAMETERS:
**
** RETURN VALUE:
** =============================================================================
*/

uint16 gus_timeout =0;
void WaitForAfeDone (void)
{
   uint16  us_timeout = 0;
   uint32  ul_cfg;
   do
   {
      ReadCoreReg (V_AFE_SERIAL_CFG_ADDR, &ul_cfg);  //V_AFE_SERIAL_CFG_ADDR
      us_timeout++;
   }while(( ul_cfg & AFE_DONE_BIT)&& (us_timeout < AFE_RD_WR_TIMEOUT));

      // the below line also can be commented.
      gus_timeout = (us_timeout > gus_timeout) ? us_timeout : gus_timeout;
}
//XDSLRTFW-1831 (end)
/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  VR9_AfeRead
**
** DESCRIPTION:   It performs the read operation from the AFE register
**                if both AFE done bit and AFE busy bit are "0" in the
**                Register V_AFE_SERIAL_CFG.
**
** PARAMETERS:
**
** RETURN VALUE:
**
**
** NOTES:        Reference: (1) AFE_Control_IF_SSC_v1.doc;
**                          (2) Strymon-VFDF_xDSL_CPE_bonded.pdf
**                            SMS00917387
**
** =============================================================================
*/

uint16 gus_wrfail_done = 0;
uint16 gus_wrfail_busy = 0;
uint16 gus_rdfail_done = 0;
uint16 gus_rdfail_busy = 0;

uint16 VR9_AfeRead(uint16 us_addr)
{
  uint32 ul_cfg = 0;
  uint32 ul_data = 0;

  ReadCoreReg(V_AFE_SERIAL_CFG_ADDR, &ul_cfg);
  //Do the Read operation on AFE if both AFE done bit and AFE busy bit are "0"
  if((ul_cfg & (AFE_DONE_BIT | AFE_BUSY_BIT)) == 0)
  {
     WriteCoreReg(V_AFE_SERIAL_ADDR_ADDR, us_addr);  //Write Address
     WriteCoreReg(V_AFE_SERIAL_RDATA_ADDR, 0x0);     //Dummy write
     WaitForAfeBusy();

     ReadCoreReg(V_AFE_SERIAL_RDATA_ADDR, &ul_data);


     // set the afe done bit to clear the sticky bit !
     ReadCoreReg(V_AFE_SERIAL_CFG_ADDR, &ul_cfg);
     WriteCoreReg(V_AFE_SERIAL_CFG_ADDR, ul_cfg | AFE_DONE_BIT);
     WaitForAfeDone();
  }
  else
  {
     //AFE read Exception
      //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (START)
      if (gus_ExceptionCode == E_CODE_NO_ERROR)
         gus_ExceptionCode = E_CODE_AFE_READ_EXCEPTION_ERROR;
      //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (END)

    if (ul_cfg & AFE_DONE_BIT)
      gus_rdfail_done++;

    if (ul_cfg & AFE_BUSY_BIT)
      gus_rdfail_busy++;

  }
  return (uint16)ul_data;
}

/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  VR9_AfeWrite
**
** DESCRIPTION:  This function does the Read Modify Write operation if
**               the mask is != 0xFFFF. Otherwise it does write operation.
**               Does the write operation only if the AFE is not busy.
**
** PARAMETERS:
**
** RETURN VALUE:
**
**
** NOTES:        Reference: (1) AFE_Control_IF_SSC_v1.doc;
**                          (2) Strymon-VFDF_xDSL_CPE_bonded.pdf
**                            SMS00917387
**
** =============================================================================
*/


void VR9_AfeWrite(uint16 us_addr,
      uint16 us_data, uint16 us_mask)
{
   uint16 us_ReadData = 0;
   uint16 us_NewData;
   uint32 ul_cfg;

   if (us_mask != 0xFFFF)
    us_ReadData = VR9_AfeRead(us_addr);

   us_NewData = (us_ReadData & ~us_mask) | (us_data & us_mask);
   ReadCoreReg(V_AFE_SERIAL_CFG_ADDR, &ul_cfg);

   if((ul_cfg & (AFE_BUSY_BIT |AFE_DONE_BIT)) == 0)
   {
      WriteCoreReg(V_AFE_SERIAL_ADDR_ADDR, us_addr);
      WriteCoreReg(V_AFE_SERIAL_WDATA_ADDR, us_NewData);
      WaitForAfeBusy();

      // set the afe done bit to clear the sticky bit !
      ReadCoreReg(V_AFE_SERIAL_CFG_ADDR, &ul_cfg);
      WriteCoreReg(V_AFE_SERIAL_CFG_ADDR, ul_cfg | AFE_DONE_BIT);
      WaitForAfeDone();
   }
   else
   {
      //AFE write Exception
      //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (START)
      if (gus_ExceptionCode == E_CODE_NO_ERROR)
         gus_ExceptionCode = E_CODE_AFE_WRITE_EXCEPTION_ERROR;
      //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (END)

      if (ul_cfg & AFE_DONE_BIT)
        gus_wrfail_done++;

      if (ul_cfg & AFE_BUSY_BIT)
        gus_wrfail_busy++;


   }
}

/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  VR9_AfeFastWrite
**
** DESCRIPTION:  This function does the Write operation only into the AFE
**               registers. It can be used where read-modify is not
**               required into the AFE regiaters.
**
** PARAMETERS:
**
** RETURN VALUE:
**
**
** NOTES:        Reference: (1) AFE_Control_IF_SSC_v1.doc;
**                          (2) Strymon-VFDF_xDSL_CPE_bonded.pdf
**                            SMS00917387
**
** =============================================================================
*/

void VR9_AfeFastWrite(uint16 us_addr, uint16 us_data)
{
   uint32 ul_cfg;

   WriteCoreReg(V_AFE_SERIAL_ADDR_ADDR, us_addr);
   WriteCoreReg(V_AFE_SERIAL_WDATA_ADDR, us_data);
   WaitForAfeBusy();

   // set the afe done bit to clear the sticky bit !
   ReadCoreReg(V_AFE_SERIAL_CFG_ADDR, &ul_cfg);
   WriteCoreReg(V_AFE_SERIAL_CFG_ADDR, ul_cfg | AFE_DONE_BIT);
}

/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  VR9_SSC_init
**
** DESCRIPTION: This function programs the AFE serial config register
**              for the following:
**      Register: V_AFE_SERIAL_CFG0x20C698 :
**      Bits 0-7 AFE.SMULT  Serial S_CLK multiple. This multiple is in
**      terms of the 1/2 the VFDF_CLK rate to the S_CLK rate.
**      For example, S_CLK needs to be 9 MHz then SCLK_MULT would be set to 16.
**      (288/2/16 = 9) Currently setting to 18MHz( 288/2/8); Rest of the bits
**      are default config.
**
** PARAMETERS:
**
** RETURN VALUE:
**
**
** NOTES:        Reference: (1) AFE_Control_IF_SSC_v1.doc;
**                          (2) Strymon-VFDF_xDSL_CPE_bonded.pdf
**                            SMS00917387
**
** =============================================================================
*/

uint32 gul_SSC_Speed = 0xE;
void VR9_SSC_init(void)
{
   uint32 ul_cfg;
   ReadCoreReg(V_AFE_SERIAL_CFG_ADDR, &ul_cfg);

   // clear clk scale bits; clk bits are from 0-7 only ;
   ul_cfg &= 0xFFFFFF00;
   // set for 18MHz clk <bits 7:0>
//   WriteCoreReg(V_AFE_SERIAL_CFG_ADDR, ul_cfg | SSC_IF_SCLK_SPEED );
   WriteCoreReg(V_AFE_SERIAL_CFG_ADDR, ul_cfg | gul_SSC_Speed ); //VR9_BRINGUP: For Testing

}

/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  VR9_AFE_Flush
**
** DESCRIPTION:  Reg: V_AFE_SERIAL_CFG (0x20C698) <Bit 11; AFE_FLUSH_BIT>
**               1=Force S_SCDO output to '1' to enable flushing
**               of AFE serial if buffer. Used in conjunction
**               with a read operation.
**
** PARAMETERS:
**
** RETURN VALUE:
**
**
** NOTES:        Reference: (1) AFE_Control_IF_SSC_v1.doc;
**                          (2) Strymon-VFDF_xDSL_CPE_bonded.pdf
**                            SMS00917387
**
** =============================================================================
*/
void VR9_AFE_Flush(void)
{
   uint32 ul_data;
   ReadCoreReg(V_AFE_SERIAL_CFG_ADDR, &ul_data);
   WriteCoreReg(V_AFE_SERIAL_CFG_ADDR, ul_data | AFE_FLUSH_BIT);
   //wait atleast 42 cycles
   delay(5);//5*10 = 50 cycles; Note: delay() internally provides 10 cycles delay
}


/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:   wait_ms(uint16 msec)
**
** DESCRIPTION:  It was intended to wait mseconds but I assume that this function is not correct
**
**
** PARAMETERS:     NONE
**
** RETURN VALUE:   NONE
**
** NOTES:          OK!
**                 Reference: Vinax code
** =============================================================================
*/
void wait_ms(uint16 msec)
{
   uint32 NumberOfCycles;
   uint32 NopCount;

   // Calculate the clock cycles assuming a clock speed of 288 MHz and one instruction per loop circle
   // but is that true??
   NumberOfCycles = 288000 * msec;
#ifdef TARGET_HW
   for(NopCount = 0; NopCount <= NumberOfCycles-1; NopCount++)
   {
      _nop();
   }
#else
   while (NumberOfCycles > 0)
   NumberOfCycles--;
#endif
}
