/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C) 2005 - Infineon Technologies
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/***C***************************************************************************
** The 'CCASE' lines will be filled by ClearCase upon checkin,
** please do not change this information manually!
**
** =============================================================================
**
** All rights reserved.
** =============================================================================
**
** =============================================================================
**
** This document contains proprietary information belonging to Infineon
** Technologies. Passing on and copying of this document, and communication of
** its contents is not permitted without prior written authorisation.
**
** =============================================================================
**
**
** =============================================================================
**
** Revision Information :
**
** CCASE_FILE = "DebugMsgHandler.c"
** CCASE_DATE = "Wed Jul 26 09:02:59 2006"
**
** =============================================================================
**
** Contents:   Inside this file the DebugMsgHandler for incoming debug
**             messages is implemented. The implementation is according to the
**             Vinax Message catalouge.
**
** Functions:  DebugMemMapRead()
**             - Read from a address
**
**             DebugMemMapWrite()
**             - Write a value under a address
**
**             DebugAuxRegRead()
**             - Read from an auxiliary register
**
**             DebugAuxRegWrite()
**             - Write a value in an auxiliary register
**
**             ReadVR9AFE_Handler()
**             - Write data in VDSL-AFE (Analog Front End)
**
**             WriteVR9AFE_Handler()
**             - Write data in VDSL-AFE (Analog Front End)
**
**
** =============================================================================
**
** References: <List of related design documents>
**
** =============================================================================
** History:
**
** Date        Author        Comment
** 2009-12-18  Abraham.B     Creation
** =============================================================================
**
Additional version control information
********************************************************************************
 CCASE_USER    = "Bschor"
 CCASE_VERSION = "\main\dev_hh_2\xtc_codeswap\Rev_1.4\1"
***C*E*************************************************************************/

/*
** =============================================================================
**                           INCLUDE FILES
** =============================================================================
*/

#include "common.h"
#include "typedef.h"
#include "nmp.h"
#include "Cust_CmvDefinitions.h"

#include "VRX_AfeCommonConst.h"
#include "VRX_AfeCommonData.h"
#include "AFED_Constants.h"
#include "AFED_Data.h"
#include "AFED_Functions.h"
#include "AFED_ReadWriteModify.h"

#include "vrx5afe_mode_defines.h"
#include "vrx5afe_central_addrmap.h"
#include "vrx5afe_central_fcsi_regpkg.h"     // register for CENTRAL
#include "vrx5afe_dsl_addrmap.h"
#include "vrx5afe_dsl_fcsi_regpkg.h"         // register for DSL

/*
** =============================================================================
**                           LOCAL DEFINITIONS
** =============================================================================
*/

#define RESERVED_VALUE              0x0000

/*
** =============================================================================
**                           LOCAL DATA
** =============================================================================
*/

/*
** =============================================================================
**                           LOCAL FUNCTION PROTOTYPES
** =============================================================================
*/

uint8 DebugMemMapRead(uint16 *gus_PayLoadSize, uint16 gusa_PayLoad[]);
uint8 DebugMemMapWrite(uint16 *gus_PayLoadSize, uint16 gusa_PayLoad[]);
uint8 DebugAuxRegRead(uint16 *gus_PayLoadSize, uint16 gusa_PayLoad[]);
uint8 DebugAuxRegWrite(uint16 *gus_PayLoadSize, uint16 gusa_PayLoad[]);
uint8 ReadVR9AFE_Handler(uint16 *gus_PayLoadSize, uint16 gusa_PayLoad[]);
uint8 WriteVR9AFE_Handler(uint16 *gus_PayLoadSize, uint16 gusa_PayLoad[]);

/*
** =============================================================================
**                           IMPLEMENTATION
** =============================================================================
*/

/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  DebugMsgHandler()
**
** DESCRIPTION:    This function executes the incoming debug message.
**
** PARAMETERS:     MsgID (uint16 / unsigned short)
**                 - MsgID from actual message.
**
**                 *gus_PayLoadSize (uint16 / unsgned short)
**                 - Pointer to payload size.
**
**                 *HMP_BufferPointer (HerculesMPStruct_t)
**                 - Currently used HMP_Buffer.
**
** RETURN VALUE:   uc_MsgExecuted (uint8 / unsgned char)
**                 - Value for function code creation.
**
** NOTES:          Under construction!
** =============================================================================
*/
uint8 DebugMsgHandler(uint16 gus_MsgID, uint16 *gus_PayLoadSize, HerculesMPStruct_t *HMP_BufferPointer){

   uint8 uc_MsgExecuted = CMV_MSG_ID_UNKNOWN;

   /*
   ** Switch between different debug message events
   */
   switch(gus_MsgID){

      /*
      ** Read from memory / mapped devices
      */
      case CMD_DBG_MEM_MAP_READ:

         uc_MsgExecuted = DebugMemMapRead(&(*gus_PayLoadSize), HMP_BufferPointer->usa_HmpPayLoad);
         break;

      /*
      ** Write in memory / mapped devices
      */
      case CMD_DBG_MEM_MAP_WRITE:

         uc_MsgExecuted = DebugMemMapWrite(&(*gus_PayLoadSize), HMP_BufferPointer->usa_HmpPayLoad);
         break;

      /*
      ** Read from auxiliary register
      */
      case CMD_DBG_AUX_REG_READ:
         uc_MsgExecuted = DebugAuxRegRead(&(*gus_PayLoadSize), HMP_BufferPointer->usa_HmpPayLoad);
         break;

      /*
      ** Write in auxiliary register
      */
      case CMD_DBG_AUX_REG_WRITE:
         uc_MsgExecuted = DebugAuxRegWrite(&(*gus_PayLoadSize), HMP_BufferPointer->usa_HmpPayLoad);
         break;

      /*
      ** Read from VDSL-AFE
      */
      case CMD_DBG_READ_VDSL_AFE:

         uc_MsgExecuted = ReadVR9AFE_Handler(&(*gus_PayLoadSize), HMP_BufferPointer->usa_HmpPayLoad);
         break;

      /*
      ** Write in VDSL-AFE
      */
      case CMD_DBG_WRITE_VDSL_AFE:

         uc_MsgExecuted = WriteVR9AFE_Handler(&(*gus_PayLoadSize), HMP_BufferPointer->usa_HmpPayLoad);
         break;
   }

   return uc_MsgExecuted;
}

/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  DebugMemMapRead()
**
** DESCRIPTION:    Read from memory / mapped devices
**
** PARAMETERS:     *gus_PayLoadSize (uint16 / unsigned short)
**                 - Pointer to payload size.
**
**                 gusa_PayLoad[] (uint16 / unsigned short)
**                 - PayLoad in actual message.
**
** RETURN VALUE:   uc_MsgExecuted (uint8 / unsigned char)
**                 - Value for function code creation.
**
** NOTES:          OK!
** =============================================================================
*/
uint8 DebugMemMapRead(uint16 *gus_PayLoadSize, uint16 gusa_PayLoad[]){

   uint32 ul_DbgAddress, ul_DbgValue;
   uint8  uc_MsgExecuted = CMV_WITHOUT_ERRORS;

   /* Read the address from buffer  */
   ul_DbgAddress = gusa_PayLoad[1];
   ul_DbgAddress = ul_DbgAddress << 16;
   ul_DbgAddress += gusa_PayLoad[0];

   /* Execute read access */
   ul_DbgValue = *((uint32*) ul_DbgAddress);

   /* Write value in buffer */
   gusa_PayLoad[2] = ul_DbgValue;
   ul_DbgValue = ul_DbgValue >> 16;
   gusa_PayLoad[3] = ul_DbgValue;

   /* The gus_PayLoadSize has 1 element more */
   *gus_PayLoadSize += 2;

   return uc_MsgExecuted;
}


/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  DebugMemMapWrite()
**
** DESCRIPTION:    Write in memory / mapped devices
**
** PARAMETERS:     *gus_PayLoadSize (uint16 / unsigned short)
**                 - Pointer to payload size.
**
**                 gusa_PayLoad[] (uint16 / unsigned short)
**                 - PayLoad in actual message.
**
** RETURN VALUE:   uc_MsgExecuted (uint8 / unsigned char)
**                 - Value for function code creation.
**
** NOTES:          OK!
** =============================================================================
*/
uint8 DebugMemMapWrite(uint16 *gus_PayLoadSize, uint16 gusa_PayLoad[]){

   uint32 ul_DbgAddress, ul_DbgValue;
   uint8 uc_MsgExecuted = CMV_WITHOUT_ERRORS;

   /* Read the address from the xTC-Buffer  */
   ul_DbgAddress = gusa_PayLoad[1];
   ul_DbgAddress = ul_DbgAddress << 16;
   ul_DbgAddress += gusa_PayLoad[0];

   /* Create value for write access */
   ul_DbgValue = gusa_PayLoad[3];
   ul_DbgValue = ul_DbgValue << 16;
   ul_DbgValue += gusa_PayLoad[2];

   /* Execute write access */
   *((uint32*) ul_DbgAddress) = ul_DbgValue;

   /*
   ** Check written data
   */
   if (ul_DbgValue == *((uint32*) ul_DbgAddress)){
      /* The gus_PayLoadSize has 1 element less */
      *gus_PayLoadSize -= 2;
   }
   else{
      uc_MsgExecuted = CMV_ERR_DURING_EXEC;
   }

   return uc_MsgExecuted;
}


/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  DebugAuxRegRead()
**
** DESCRIPTION:    Read from auxiliary register
**
** PARAMETERS:     *gus_PayLoadSize (uint16 / unsigned short)
**                 - Pointer to payload size.
**
**                 gusa_PayLoad[] (uint16 / unsigned short)
**                 - PayLoad in actual message.
**
** RETURN VALUE:   uc_MsgExecuted (uint8 / unsigned char)
**                 - Value for function code creation.
**
** NOTES:          OK!
** =============================================================================
*/
uint8 DebugAuxRegRead(uint16 *gus_PayLoadSize, uint16 gusa_PayLoad[]){

   uint32 ul_DbgAddress, ul_DbgValue;
   uint8 uc_MsgExecuted = CMV_WITHOUT_ERRORS;

   /* Read the address from the Auxilliary space  */
   ul_DbgAddress = gusa_PayLoad[1];
   ul_DbgAddress = ul_DbgAddress << 16;
   ul_DbgAddress += gusa_PayLoad[0];

   /* Execute read access */
   ul_DbgValue = _lr (ul_DbgAddress);

   /* Write value in Auxilliary space */
   gusa_PayLoad[2] = ul_DbgValue;
   ul_DbgValue = ul_DbgValue >> 16;
   gusa_PayLoad[3] = ul_DbgValue;

   /* The gus_PayLoadSize has 1 element more */
   *gus_PayLoadSize += 2;

   return uc_MsgExecuted;
}


/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  DebugAuxRegWrite()
**
** DESCRIPTION:    Write in auxiliary register
**
** PARAMETERS:     *gus_PayLoadSize (uint16 / unsigned short)
**                 - Pointer to payload size.
**
**                 gusa_PayLoad[] (uint16 / unsigned short)
**                 - PayLoad in actual message.
**
** RETURN VALUE:   uc_MsgExecuted (uint8 / unsigned char)
**                 - Value for function code creation.
**
** NOTES:          OK!
** =============================================================================
*/
uint8 DebugAuxRegWrite(uint16 *gus_PayLoadSize, uint16 gusa_PayLoad[]){

   uint32 ul_DbgAddress, ul_DbgValue;
   uint8  uc_MsgExecuted = CMV_WITHOUT_ERRORS;

   /* Read the address from the xTC-Buffer  */
   ul_DbgAddress = gusa_PayLoad[1];
   ul_DbgAddress = ul_DbgAddress << 16;
   ul_DbgAddress += gusa_PayLoad[0];

   /* Create value for write access */
   ul_DbgValue = gusa_PayLoad[3];
   ul_DbgValue = ul_DbgValue << 16;
   ul_DbgValue += gusa_PayLoad[2];

   /* Execute write access */
   _sr (ul_DbgValue, ul_DbgAddress);

   /*
   ** Check written data
   */
   if (ul_DbgValue == _lr (ul_DbgAddress)){
      /* The gus_PayLoadSize has 1 element less */
      *gus_PayLoadSize -= 2;
   }
   else{
      uc_MsgExecuted = CMV_ERR_DURING_EXEC;
   }

   return uc_MsgExecuted;
}


/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  ReadVR9AFE_Handler()
**
** DESCRIPTION:    Read data from VDSL-AFE (Analog Front End)
**
** PARAMETERS:     *gus_PayLoadSize (uint16 / unsigned short)
**                 - Pointer to payload size.
**
**                 gusa_PayLoad[] (uint16 / unsigned short)
**                 - PayLoad in actual message.
**
** RETURN VALUE:   uc_MsgExecuted (uint8 / unsigned char)
**                 - Value for function code creation.
**
** NOTES:          OK!
** =============================================================================
*/
uint8 ReadVR9AFE_Handler(uint16 *gus_PayLoadSize, uint16 gusa_PayLoad[]){

   uint8 uc_MsgExecuted = CMV_WITHOUT_ERRORS;

   /* Read data from VDSL-AFE */
   gusa_PayLoad[2] = VR9_AfeRead((uint16)gusa_PayLoad[0]);

   /* Clean up the upper 16 bits in the address and payload word */
   gusa_PayLoad[1] = RESERVED_VALUE;
   gusa_PayLoad[3] = RESERVED_VALUE;

   /* The gus_PayLoadSize has 1 element more */
   *gus_PayLoadSize += 2;

   return uc_MsgExecuted;
}


/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  WriteVR9AFE_Handler()
**
** DESCRIPTION:    Write data in VDSL-AFE (Analog Front End)
**
** PARAMETERS:     *gus_PayLoadSize (uint16 / unsigned short)
**                 - Pointer to payload size.
**
**                 gusa_PayLoad[] (uint16 / unsigned short)
**                 - PayLoad in actual message.
**
** RETURN VALUE:   uc_MsgExecuted (uint8 / unsigned char)
**                 - Value for function code creation.
**
** NOTES:          OK!
** =============================================================================
*/
uint8 WriteVR9AFE_Handler(uint16 *gus_PayLoadSize, uint16 gusa_PayLoad[]){

   uint8 uc_MsgExecuted = CMV_WITHOUT_ERRORS;

   /* Write data in VDSL-AFE */
   VR9_AfeWrite((uint16)gusa_PayLoad[0], (uint16)gusa_PayLoad[2], 0xFFFF);

   /* Clean up the upper 16 bits in the address and payload word */
   gusa_PayLoad[1] = RESERVED_VALUE;
   gusa_PayLoad[3] = RESERVED_VALUE;

   /*
   ** Check written data
   */
   if (gusa_PayLoad[2] == VR9_AfeRead((uint16)gusa_PayLoad[0])){
      /* The gus_PayLoadSize has 1 element less */
      *gus_PayLoadSize -= 2;
   }
   else{
      uc_MsgExecuted = CMV_ERR_DURING_EXEC;
   }

   return uc_MsgExecuted;
}





